"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.splitHtml = void 0;
const htmlUtils_1 = require("./htmlUtils");
const linkReplacement_1 = require("./MdToHtml/linkReplacement");
const utils = require("./utils");
const InMemoryCache_1 = require("./InMemoryCache");
const noteStyle_1 = require("./noteStyle");
const md5 = require('md5');
// Renderered notes can potentially be quite large (for example
// when they come from the clipper) so keep the cache size
// relatively small.
const inMemoryCache = new InMemoryCache_1.default(10);
// https://github.com/es-shims/String.prototype.trimStart/blob/main/implementation.js
function trimStart(s) {
    // eslint-disable-next-line no-control-regex
    const startWhitespace = /^[\x09\x0A\x0B\x0C\x0D\x20\xA0\u1680\u180E\u2000\u2001\u2002\u2003\u2004\u2005\u2006\u2007\u2008\u2009\u200A\u202F\u205F\u3000\u2028\u2029\uFEFF]*/;
    return s.replace(startWhitespace, '');
}
class HtmlToHtml {
    constructor(options = null) {
        options = Object.assign({ ResourceModel: null }, options);
        this.resourceBaseUrl_ = 'resourceBaseUrl' in options ? options.resourceBaseUrl : null;
        this.ResourceModel_ = options.ResourceModel;
        this.cache_ = inMemoryCache;
        this.fsDriver_ = {
            writeFile: ( /* path, content, encoding = 'base64'*/) => { throw new Error('writeFile not set'); },
            exists: ( /* path*/) => { throw new Error('exists not set'); },
            cacheCssToFile: ( /* cssStrings*/) => { throw new Error('cacheCssToFile not set'); },
        };
        if (options.fsDriver) {
            if (options.fsDriver.writeFile)
                this.fsDriver_.writeFile = options.fsDriver.writeFile;
            if (options.fsDriver.exists)
                this.fsDriver_.exists = options.fsDriver.exists;
            if (options.fsDriver.cacheCssToFile)
                this.fsDriver_.cacheCssToFile = options.fsDriver.cacheCssToFile;
        }
    }
    fsDriver() {
        return this.fsDriver_;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async allAssets(theme, noteStyleOptions = null) {
        let cssStrings = [];
        if (noteStyleOptions.whiteBackgroundNoteRendering) {
            cssStrings = [(0, noteStyle_1.whiteBackgroundNoteStyle)()];
        }
        else {
            cssStrings = [(0, noteStyle_1.default)(theme, noteStyleOptions).join('\n')];
        }
        return [await this.fsDriver().cacheCssToFile(cssStrings)];
    }
    clearCache() {
        // TODO: Clear the in-memory cache
    }
    // Note: the "theme" variable is ignored and instead the light theme is
    // always used for HTML notes.
    // See: https://github.com/laurent22/joplin/issues/3698
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async render(markup, theme, options) {
        options = Object.assign({ splitted: false, postMessageSyntax: 'postMessage', enableLongPress: false }, options);
        const cacheKey = md5(escape(JSON.stringify({ markup, options, baseUrl: this.resourceBaseUrl_ })));
        let html = this.cache_.value(cacheKey);
        if (!html) {
            html = htmlUtils_1.default.sanitizeHtml(markup, {
                allowedFilePrefixes: options.allowedFilePrefixes,
            });
            html = htmlUtils_1.default.processImageTags(html, (data) => {
                if (!data.src)
                    return null;
                const r = utils.imageReplacement(this.ResourceModel_, data, options.resources, this.resourceBaseUrl_, options.itemIdToUrl);
                if (!r)
                    return null;
                if (typeof r === 'string') {
                    return {
                        type: 'replaceElement',
                        html: r,
                    };
                }
                else {
                    return {
                        type: 'setAttributes',
                        attrs: r,
                    };
                }
            });
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            html = htmlUtils_1.default.processAnchorTags(html, (data) => {
                var _a;
                if (!data.href)
                    return null;
                const r = (0, linkReplacement_1.default)(data.href, Object.assign({ resources: options.resources, ResourceModel: this.ResourceModel_, postMessageSyntax: options.postMessageSyntax, enableLongPress: options.enableLongPress }, (_a = options.plugins) === null || _a === void 0 ? void 0 : _a.link_open));
                if (!r.html)
                    return null;
                return {
                    type: 'replaceElement',
                    html: r.html,
                };
            });
        }
        this.cache_.setValue(cacheKey, html, 1000 * 60 * 10);
        if (options.bodyOnly) {
            return {
                html: html,
                pluginAssets: [],
                cssStrings: [],
            };
        }
        let cssStrings = options.whiteBackgroundNoteRendering ? [(0, noteStyle_1.whiteBackgroundNoteStyle)()] : (0, noteStyle_1.default)(theme);
        if (options.splitted) {
            const splitted = (0, exports.splitHtml)(html);
            cssStrings = [splitted.css].concat(cssStrings);
            const output = {
                html: splitted.html,
                pluginAssets: [],
                cssStrings: [],
            };
            if (options.externalAssetsOnly) {
                output.pluginAssets.push(await this.fsDriver().cacheCssToFile(cssStrings));
            }
            return output;
        }
        const styleHtml = `<style>${cssStrings.join('\n')}</style>`;
        return {
            html: styleHtml + html,
            pluginAssets: [],
            cssStrings: [],
        };
    }
}
exports.default = HtmlToHtml;
const splitHtmlRegex = /^<style>([\s\S]*?)<\/style>([\s\S]*)$/i;
// This function is designed to handle the narrow case of HTML generated by the
// HtmlToHtml class and used by the Rich Text editor, and that's with the STYLE
// tag at the top, followed by the HTML code. If it's anything else, we don't
// try to handle it and return the whole HTML code.
const splitHtml = (html) => {
    const trimmedHtml = trimStart(html);
    const result = trimmedHtml.match(splitHtmlRegex);
    if (!result)
        return { html, css: '' };
    return { html: result[2], css: result[1] };
};
exports.splitHtml = splitHtml;
//# sourceMappingURL=HtmlToHtml.js.map