"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const test_utils_1 = require("@joplin/lib/testing/test-utils");
const testUtils_1 = require("./utils/testUtils");
const Note_1 = require("@joplin/lib/models/Note");
const Folder_1 = require("@joplin/lib/models/Folder");
const app_1 = require("./app");
const trash_1 = require("@joplin/lib/services/trash");
const Command = require('./command-rmnote');
const setUpCommand = () => {
    const command = (0, testUtils_1.setupCommandForTesting)(Command);
    const promptMock = jest.fn(() => true);
    command.setPrompt(promptMock);
    return { command, promptMock };
};
const createTestNote = async () => {
    const folder = await Folder_1.default.save({ title: 'folder' });
    (0, app_1.default)().switchCurrentFolder(folder);
    return await Note_1.default.save({ title: 'note1', parent_id: folder.id });
};
describe('command-rmnote', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        await (0, testUtils_1.setupApplication)();
    });
    it('should move to the trash by default, without prompting', async () => {
        const { id: noteId } = await createTestNote();
        const { command, promptMock } = setUpCommand();
        await command.action({ 'note-pattern': 'note1', options: {} });
        expect(promptMock).not.toHaveBeenCalled();
        expect((await Note_1.default.allItemsInTrash()).noteIds.includes(noteId)).toBe(true);
    });
    it('should permanently delete trashed items by default, with prompting', async () => {
        const { id: noteId } = await createTestNote();
        const { command, promptMock } = setUpCommand();
        // Should not prompt when deleting from a folder
        await command.action({ 'note-pattern': 'note1', options: {} });
        expect(promptMock).toHaveBeenCalledTimes(0);
        // Should prompt when deleting from trash
        (0, app_1.default)().switchCurrentFolder(await Folder_1.default.load((0, trash_1.getTrashFolderId)()));
        await command.action({ 'note-pattern': 'note1', options: {} });
        expect(promptMock).toHaveBeenCalledTimes(1);
        expect(await Note_1.default.load(noteId, { includeDeleted: true })).toBe(undefined);
    });
});
//# sourceMappingURL=command-rmnote.test.js.map