"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.upgradeMasterKey = exports.determineKeyPassword = exports.useNeedMasterPassword = exports.usePasswordChecker = exports.useInputPasswords = exports.useInputMasterPassword = exports.onMasterPasswordSave = exports.onSavePasswordClick = exports.onToggleEnabledClick = exports.useToggleShowDisabledMasterKeys = exports.dontReencryptData = exports.reencryptData = exports.enableEncryptionConfirmationMessages = exports.decryptedStatText = exports.useStats = void 0;
const shim_1 = require("../../shim");
const locale_1 = require("../../locale");
const BaseItem_1 = require("../../models/BaseItem");
const useAsyncEffect_1 = require("../../hooks/useAsyncEffect");
const utils_1 = require("../../services/e2ee/utils");
const EncryptionService_1 = require("../../services/e2ee/EncryptionService");
const syncInfoUtils_1 = require("../../services/synchronizer/syncInfoUtils");
const MasterKey_1 = require("../../models/MasterKey");
const registry_1 = require("../../registry");
const Setting_1 = require("../../models/Setting");
const { useCallback, useEffect, useState } = shim_1.default.react();
const useStats = () => {
    const [stats, setStats] = useState({ encrypted: null, total: null });
    const [statsUpdateTime, setStatsUpdateTime] = useState(0);
    (0, useAsyncEffect_1.default)(async (event) => {
        const r = await BaseItem_1.default.encryptedItemsStats();
        if (event.cancelled)
            return;
        setStats(stats => {
            if (JSON.stringify(stats) === JSON.stringify(r))
                return stats;
            return r;
        });
    }, [statsUpdateTime]);
    useEffect(() => {
        const iid = shim_1.default.setInterval(() => {
            setStatsUpdateTime(Date.now());
        }, 3000);
        return () => {
            shim_1.default.clearInterval(iid);
        };
    }, []);
    return stats;
};
exports.useStats = useStats;
const decryptedStatText = (stats) => {
    const doneCount = stats.encrypted !== null ? stats.total - stats.encrypted : '-';
    const totalCount = stats.total !== null ? stats.total : '-';
    const result = (0, locale_1._)('Decrypted items: %s / %s', doneCount, totalCount);
    return result;
};
exports.decryptedStatText = decryptedStatText;
const enableEncryptionConfirmationMessages = (_masterKey, hasMasterPassword) => {
    const msg = [(0, locale_1._)('Enabling encryption means *all* your notes and attachments are going to be re-synchronised and sent encrypted to the sync target.')];
    if (hasMasterPassword) {
        msg.push((0, locale_1._)('To continue, please enter your master password below.'));
    }
    else {
        msg.push((0, locale_1._)('Do not lose the password as, for security purposes, this will be the *only* way to decrypt the data! To enable encryption, please enter your password below.'));
    }
    // if (masterKey) msg.push(_('Encryption will be enabled using the master key created on %s', time.unixMsToLocalDateTime(masterKey.created_time)));
    return msg;
};
exports.enableEncryptionConfirmationMessages = enableEncryptionConfirmationMessages;
const reencryptData = async () => {
    const ok = confirm((0, locale_1._)('Please confirm that you would like to re-encrypt your complete database.'));
    if (!ok)
        return;
    await BaseItem_1.default.forceSyncAll();
    void registry_1.reg.waitForSyncFinishedThenSync();
    Setting_1.default.setValue('encryption.shouldReencrypt', Setting_1.default.SHOULD_REENCRYPT_NO);
    alert((0, locale_1._)('Your data is going to be re-encrypted and synced again.'));
};
exports.reencryptData = reencryptData;
const dontReencryptData = () => {
    Setting_1.default.setValue('encryption.shouldReencrypt', Setting_1.default.SHOULD_REENCRYPT_NO);
};
exports.dontReencryptData = dontReencryptData;
const useToggleShowDisabledMasterKeys = () => {
    const [showDisabledMasterKeys, setShowDisabledMasterKeys] = useState(false);
    const toggleShowDisabledMasterKeys = () => {
        setShowDisabledMasterKeys((current) => !current);
    };
    return { showDisabledMasterKeys, toggleShowDisabledMasterKeys };
};
exports.useToggleShowDisabledMasterKeys = useToggleShowDisabledMasterKeys;
const onToggleEnabledClick = (mk) => {
    (0, syncInfoUtils_1.setMasterKeyEnabled)(mk.id, !(0, syncInfoUtils_1.masterKeyEnabled)(mk));
};
exports.onToggleEnabledClick = onToggleEnabledClick;
const onSavePasswordClick = (mk, passwords) => {
    const password = passwords[mk.id];
    if (!password) {
        Setting_1.default.deleteObjectValue('encryption.passwordCache', mk.id);
    }
    else {
        Setting_1.default.setObjectValue('encryption.passwordCache', mk.id, password);
    }
    // When setting a master key password, if the master password is not set, we
    // assume that this password is the master password. If it turns out it's
    // not, it's always possible to change it in the UI.
    if (password && !Setting_1.default.value('encryption.masterPassword')) {
        Setting_1.default.setValue('encryption.masterPassword', password);
    }
};
exports.onSavePasswordClick = onSavePasswordClick;
const onMasterPasswordSave = (masterPasswordInput) => {
    Setting_1.default.setValue('encryption.masterPassword', masterPasswordInput);
};
exports.onMasterPasswordSave = onMasterPasswordSave;
const useInputMasterPassword = (masterKeys, activeMasterKeyId) => {
    const [inputMasterPassword, setInputMasterPassword] = useState('');
    const onMasterPasswordSave = useCallback(async () => {
        Setting_1.default.setValue('encryption.masterPassword', inputMasterPassword);
        if (!(await (0, utils_1.masterPasswordIsValid)(inputMasterPassword, masterKeys.find(mk => mk.id === activeMasterKeyId)))) {
            alert('Password is invalid. Please try again.');
        }
        // eslint-disable-next-line @seiyab/react-hooks/exhaustive-deps -- Old code before rule was applied
    }, [inputMasterPassword]);
    const onMasterPasswordChange = useCallback((password) => {
        setInputMasterPassword(password);
    }, []);
    return { inputMasterPassword, onMasterPasswordSave, onMasterPasswordChange };
};
exports.useInputMasterPassword = useInputMasterPassword;
const useInputPasswords = (propsPasswords) => {
    const [inputPasswords, setInputPasswords] = useState(propsPasswords);
    useEffect(() => {
        setInputPasswords(propsPasswords);
    }, [propsPasswords]);
    const onInputPasswordChange = useCallback((mk, password) => {
        setInputPasswords(current => {
            return Object.assign(Object.assign({}, current), { [mk.id]: password });
        });
    }, []);
    return { inputPasswords, onInputPasswordChange };
};
exports.useInputPasswords = useInputPasswords;
const usePasswordChecker = (masterKeys, activeMasterKeyId, masterPassword, passwords) => {
    const [passwordChecks, setPasswordChecks] = useState({});
    // "masterPasswordKeys" are the master key that can be decrypted with the
    // master password. It should be all of them normally, but in previous
    // versions it was possible to have different passwords for different keys,
    // so we need this for backward compatibility.
    const [masterPasswordKeys, setMasterPasswordKeys] = useState({});
    const [masterPasswordStatus, setMasterPasswordStatus] = useState(utils_1.MasterPasswordStatus.Unknown);
    (0, useAsyncEffect_1.default)(async (event) => {
        const newPasswordChecks = {};
        const newMasterPasswordKeys = {};
        const masterPasswordOk = masterPassword ? await (0, utils_1.masterPasswordIsValid)(masterPassword, masterKeys.find(mk => mk.id === activeMasterKeyId)) : true;
        newPasswordChecks['master'] = masterPasswordOk;
        for (let i = 0; i < masterKeys.length; i++) {
            const mk = masterKeys[i];
            const password = await (0, utils_1.findMasterKeyPassword)(EncryptionService_1.default.instance(), mk, passwords);
            const ok = password ? await EncryptionService_1.default.instance().checkMasterKeyPassword(mk, password) : false;
            newPasswordChecks[mk.id] = ok;
            // Even if the password matches the master password, it isn't a master password key if the
            // master password can't decrypt this key.
            newMasterPasswordKeys[mk.id] = password === masterPassword && (ok || !masterPasswordOk);
        }
        if (event.cancelled)
            return;
        setPasswordChecks(passwordChecks => {
            if (JSON.stringify(newPasswordChecks) === JSON.stringify(passwordChecks))
                return passwordChecks;
            return newPasswordChecks;
        });
        setMasterPasswordKeys(masterPasswordKeys => {
            if (JSON.stringify(newMasterPasswordKeys) === JSON.stringify(masterPasswordKeys))
                return masterPasswordKeys;
            return newMasterPasswordKeys;
        });
        setMasterPasswordStatus(await (0, utils_1.getMasterPasswordStatus)(masterPassword));
    }, [masterKeys, masterPassword]);
    return { passwordChecks, masterPasswordKeys, masterPasswordStatus };
};
exports.usePasswordChecker = usePasswordChecker;
const useNeedMasterPassword = (passwordChecks, masterKeys) => {
    for (const [mkId, valid] of Object.entries(passwordChecks)) {
        const mk = masterKeys.find(mk => mk.id === mkId);
        if (!mk)
            continue;
        if (!(0, syncInfoUtils_1.masterKeyEnabled)(mk))
            continue;
        if (!valid)
            return true;
    }
    return false;
};
exports.useNeedMasterPassword = useNeedMasterPassword;
const determineKeyPassword = (masterKeyId, masterPasswordKeys, masterPassword, passwords) => {
    if (masterPasswordKeys[masterKeyId])
        return masterPassword;
    return passwords[masterKeyId];
};
exports.determineKeyPassword = determineKeyPassword;
const upgradeMasterKey = async (masterKey, password) => {
    if (!password) {
        return (0, locale_1._)('Please enter your password in the master key list below before upgrading the key.');
    }
    try {
        // Just re-encrypt the master key, but using the new encryption method
        // (which would be the default).
        const newMasterKey = await EncryptionService_1.default.instance().reencryptMasterKey(masterKey, password, password);
        await MasterKey_1.default.save(newMasterKey);
        void registry_1.reg.waitForSyncFinishedThenSync();
        return (0, locale_1._)('The master key has been upgraded successfully!');
    }
    catch (error) {
        return (0, locale_1._)('Could not upgrade master key: %s', error.message);
    }
};
exports.upgradeMasterKey = upgradeMasterKey;
//# sourceMappingURL=utils.js.map