"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const time_1 = require("@joplin/utils/time");
const test_utils_1 = require("../testing/test-utils");
const BaseItem_1 = require("./BaseItem");
const Folder_1 = require("./Folder");
const Note_1 = require("./Note");
describe('BaseItem', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    afterAll(async () => {
        await (0, test_utils_1.afterAllCleanUp)();
    });
    // This is to handle the case where a property is removed from a BaseItem table - in that case files in
    // the sync target will still have the old property but we don't need it locally.
    it('should ignore properties that are present in sync file but not in database when serialising', async () => {
        const folder = await Folder_1.default.save({ title: 'folder1' });
        let serialized = await Folder_1.default.serialize(folder);
        serialized += '\nignore_me: true';
        const unserialized = await Folder_1.default.unserialize(serialized);
        expect('ignore_me' in unserialized).toBe(false);
    });
    it('should not modify title when unserializing', async () => {
        const folder1 = await Folder_1.default.save({ title: '' });
        const folder2 = await Folder_1.default.save({ title: 'folder1' });
        const serialized1 = await Folder_1.default.serialize(folder1);
        const unserialized1 = await Folder_1.default.unserialize(serialized1);
        expect(unserialized1.title).toBe(folder1.title);
        const serialized2 = await Folder_1.default.serialize(folder2);
        const unserialized2 = await Folder_1.default.unserialize(serialized2);
        expect(unserialized2.title).toBe(folder2.title);
    });
    it.each([
        '',
        '\n\na\nb\nc\nç\nTest!\n Testing. \n',
        'Test! ☺',
        'Test! ☺\n\n\n',
    ])('should not modify body when unserializing (body: %j)', async (body) => {
        const note = await Note_1.default.save({ title: 'note1', body });
        expect(await Note_1.default.unserialize(await Note_1.default.serialize(note))).toMatchObject({
            body,
        });
    });
    it('should correctly unserialize note timestamps', async () => {
        const folder = await Folder_1.default.save({ title: 'folder' });
        const note = await Note_1.default.save({ title: 'note', parent_id: folder.id });
        const serialized = await Note_1.default.serialize(note);
        const unserialized = await Note_1.default.unserialize(serialized);
        expect(unserialized.created_time).toEqual(note.created_time);
        expect(unserialized.updated_time).toEqual(note.updated_time);
        expect(unserialized.user_created_time).toEqual(note.user_created_time);
        expect(unserialized.user_updated_time).toEqual(note.user_updated_time);
    });
    it('should unserialize a very large note quickly', async () => {
        const folder = await Folder_1.default.save({ title: 'folder' });
        const note = await Note_1.default.save({ title: 'note', parent_id: folder.id });
        const serialized = await Note_1.default.serialize(Object.assign(Object.assign({}, note), { 
            // 2 MiB
            body: '\n.'.repeat(1 * 1024 * 1024) }));
        const start = performance.now();
        await Note_1.default.unserialize(serialized);
        // Locally, this passes in in < 2s, so 30s should be a safe upper bound.
        expect(performance.now() - start).toBeLessThan(30 * time_1.Second);
    });
    it('should serialize geolocation fields', async () => {
        const folder = await Folder_1.default.save({ title: 'folder' });
        let note = await Note_1.default.save({ title: 'note', parent_id: folder.id });
        note = await Note_1.default.load(note.id);
        let serialized = await Note_1.default.serialize(note);
        let unserialized = await Note_1.default.unserialize(serialized);
        expect(unserialized.latitude).toEqual('0.00000000');
        expect(unserialized.longitude).toEqual('0.00000000');
        expect(unserialized.altitude).toEqual('0.0000');
        await Note_1.default.save({
            id: note.id,
            longitude: -3.459,
            altitude: 0,
            latitude: 48.732,
        });
        note = await Note_1.default.load(note.id);
        serialized = await Note_1.default.serialize(note);
        unserialized = await Note_1.default.unserialize(serialized);
        expect(unserialized.latitude).toEqual(note.latitude);
        expect(unserialized.longitude).toEqual(note.longitude);
        expect(unserialized.altitude).toEqual(note.altitude);
    });
    it('should serialize and unserialize notes', async () => {
        const folder = await Folder_1.default.save({ title: 'folder' });
        const note = await Note_1.default.save({ title: 'note', parent_id: folder.id });
        await Note_1.default.save({
            id: note.id,
            longitude: -3.459,
            altitude: 0,
            latitude: 48.732,
        });
        const noteBefore = await Note_1.default.load(note.id);
        const serialized = await Note_1.default.serialize(noteBefore);
        const noteAfter = await Note_1.default.unserialize(serialized);
        expect(noteAfter).toEqual(noteBefore);
    });
    it('should serialize and unserialize properties that contain new lines', async () => {
        const sourceUrl = `
https://joplinapp.org/ \\n
`;
        const note = await Note_1.default.save({ title: 'note', source_url: sourceUrl });
        const noteBefore = await Note_1.default.load(note.id);
        const serialized = await Note_1.default.serialize(noteBefore);
        const noteAfter = await Note_1.default.unserialize(serialized);
        expect(noteAfter).toEqual(noteBefore);
    });
    it('should not serialize the note title and body', async () => {
        const note = await Note_1.default.save({ title: 'my note', body: `one line
two line
three line \\n no escape` });
        const noteBefore = await Note_1.default.load(note.id);
        const serialized = await Note_1.default.serialize(noteBefore);
        expect(serialized.indexOf(`my note

one line
two line
three line \\n no escape`)).toBe(0);
    });
    it('should update item sync item', async () => {
        const note1 = await Note_1.default.save({});
        const syncTime = async (itemId) => {
            const syncItem = await BaseItem_1.default.syncItem((0, test_utils_1.syncTargetId)(), itemId, { fields: ['sync_time'] });
            return syncItem ? syncItem.sync_time : 0;
        };
        expect(await syncTime(note1.id)).toBe(0);
        await (0, test_utils_1.synchronizerStart)();
        const newTime = await syncTime(note1.id);
        expect(newTime).toBeLessThanOrEqual(Date.now());
        // Check that it doesn't change if we sync again
        await (0, test_utils_1.msleep)(1);
        await (0, test_utils_1.synchronizerStart)();
        expect(await syncTime(note1.id)).toBe(newTime);
    });
    it.each([
        'test-test!',
        'This ID has    spaces\ttabs\nand newlines',
        'Test`;',
        'Test"',
        'Test\'',
        'Test\'\'\'a\'\'',
        '% test',
    ])('should support querying items with IDs containing special characters (id: %j)', async (id) => {
        const note = await Note_1.default.save({ id }, { isNew: true });
        expect(await BaseItem_1.default.loadItemById(note.id)).toMatchObject({ id });
    });
});
//# sourceMappingURL=BaseItem.test.js.map