"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const trash_1 = require("../../services/trash");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../Folder");
const Note_1 = require("../Note");
const onFolderDrop_1 = require("./onFolderDrop");
describe('onFolderDrop', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should drop a note to the trash', async () => {
        const note = await Note_1.default.save({});
        const beforeTime = Date.now();
        await (0, onFolderDrop_1.default)([note.id], [], (0, trash_1.getTrashFolderId)());
        const n = await Note_1.default.load(note.id);
        expect(n.deleted_time).toBeGreaterThanOrEqual(beforeTime);
    });
    it('should drop a note in the trash to the root of the trash', async () => {
        const folder = await Folder_1.default.save({});
        const note = await Note_1.default.save({ parent_id: folder.id });
        const beforeTime = Date.now();
        await Folder_1.default.delete(folder.id, { toTrash: true });
        await (0, onFolderDrop_1.default)([note.id], [], (0, trash_1.getTrashFolderId)());
        const n = await Note_1.default.load(note.id);
        expect(n.deleted_time).toBeGreaterThan(beforeTime);
        expect(n.parent_id).toBe('');
    });
    it('should drop a folder in the trash to the root of the trash', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({ parent_id: folder1.id });
        await Folder_1.default.delete(folder1.id, { toTrash: true });
        await (0, onFolderDrop_1.default)([], [folder2.id], (0, trash_1.getTrashFolderId)());
        const f = await Folder_1.default.load(folder2.id);
        expect(f.deleted_time).toBeTruthy();
        expect(f.parent_id).toBe('');
    });
    it('should drop a deleted folder to a non-deleted one', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        await Folder_1.default.delete(folder2.id, { toTrash: true });
        await (0, onFolderDrop_1.default)([], [folder2.id], folder1.id);
        const f2 = await Folder_1.default.load(folder2.id);
        expect(f2.deleted_time).toBe(0);
        expect(f2.parent_id).toBe(folder1.id);
    });
    it('should drop a deleted note to a non-deleted folder', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder1.id });
        await Note_1.default.delete(note1.id, { toTrash: true });
        await (0, onFolderDrop_1.default)([note1.id], [], folder2.id);
        const n1 = await Note_1.default.load(note1.id);
        expect(n1.deleted_time).toBe(0);
        expect(n1.parent_id).toBe(folder2.id);
    });
    it('should drop a non-deleted folder to the virtual root notebook', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({ parent_id: folder1.id });
        await (0, onFolderDrop_1.default)([], [folder2.id], '');
        const folder2Reloaded = await Folder_1.default.load(folder2.id);
        expect(folder2Reloaded.parent_id).toBe(folder1.parent_id);
        expect(folder2Reloaded.parent_id).toBe('');
    });
    it('should drop a deleted folder to the virtual root notebook', async () => {
        const folder1 = await Folder_1.default.save({});
        await Folder_1.default.delete(folder1.id, { toTrash: true });
        await (0, onFolderDrop_1.default)([], [folder1.id], '');
        const folder1Reloaded = await Folder_1.default.load(folder1.id);
        expect(folder1Reloaded.parent_id).toBe('');
        expect(folder1Reloaded.deleted_time).toBe(0);
    });
});
//# sourceMappingURL=onFolderDrop.test.js.map