"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs_extra_1 = require("fs-extra");
const _1 = require(".");
const test_utils_1 = require("../../testing/test-utils");
const types_1 = require("./types");
describe('profileConfig/index', () => {
    it('should load a default profile config', async () => {
        const filePath = (0, test_utils_1.tempFilePath)('json');
        const config = await (0, _1.loadProfileConfig)(filePath);
        expect(config).toEqual((0, types_1.defaultProfileConfig)());
    });
    it('should load a profile config', async () => {
        const filePath = (0, test_utils_1.tempFilePath)('json');
        const config = {
            profiles: [
                {
                    name: 'Testing',
                    id: types_1.DefaultProfileId,
                },
            ],
        };
        await (0, fs_extra_1.writeFile)(filePath, JSON.stringify(config), 'utf8');
        const loadedConfig = await (0, _1.loadProfileConfig)(filePath);
        const expected = {
            version: types_1.CurrentProfileVersion,
            currentProfileId: types_1.DefaultProfileId,
            profiles: [
                {
                    name: 'Testing',
                    id: types_1.DefaultProfileId,
                },
            ],
        };
        expect(loadedConfig).toEqual(expected);
    });
    it('should load a save a config', async () => {
        const filePath = (0, test_utils_1.tempFilePath)('json');
        const config = (0, types_1.defaultProfileConfig)();
        await (0, _1.saveProfileConfig)(filePath, config);
        const loadedConfig = await (0, _1.loadProfileConfig)(filePath);
        expect(config).toEqual(loadedConfig);
    });
    it('should get a profile full path', async () => {
        const profile1 = Object.assign(Object.assign({}, (0, types_1.defaultProfile)()), { id: 'abcd' });
        const profile2 = Object.assign(Object.assign({}, (0, types_1.defaultProfile)()), { id: types_1.DefaultProfileId });
        expect((0, _1.getProfileFullPath)(profile1, '/test/root')).toBe('/test/root/profile-abcd');
        expect((0, _1.getProfileFullPath)(profile2, '/test/root')).toBe('/test/root');
    });
    it('should create a new profile', async () => {
        let config = (0, types_1.defaultProfileConfig)();
        const r1 = (0, _1.createNewProfile)(config, 'new profile 1');
        const r2 = (0, _1.createNewProfile)(r1.newConfig, 'new profile 2');
        config = r2.newConfig;
        expect(config.profiles.length).toBe(3);
        expect(config.profiles[1].name).toBe('new profile 1');
        expect(config.profiles[2].name).toBe('new profile 2');
        expect(config.profiles[1].id).not.toBe(config.profiles[2].id);
    });
    it('should migrate profile config - version 1 to 2', async () => {
        const migrated1 = (0, _1.migrateProfileConfig)({
            'version': 1,
            'currentProfile': 2,
            'profiles': [
                {
                    'name': 'Default',
                    'path': '.',
                },
                {
                    'name': 'sub1',
                    'path': 'profile-sjn25kuh',
                },
                {
                    'name': 'sub2',
                    'path': 'profile-yufzkns3',
                },
            ],
        }, 2);
        expect(migrated1).toEqual({
            'version': 2,
            'currentProfileId': 'yufzkns3',
            'profiles': [
                {
                    'name': 'Default',
                    'id': 'default',
                },
                {
                    'name': 'sub1',
                    'id': 'sjn25kuh',
                },
                {
                    'name': 'sub2',
                    'id': 'yufzkns3',
                },
            ],
        });
    });
});
//# sourceMappingURL=index.test.js.map