"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupAutoDeletion = void 0;
const Logger_1 = require("@joplin/utils/Logger");
const Folder_1 = require("../../models/Folder");
const Setting_1 = require("../../models/Setting");
const Note_1 = require("../../models/Note");
const time_1 = require("@joplin/utils/time");
const shim_1 = require("../../shim");
const readOnly_1 = require("../../models/utils/readOnly");
const BaseItem_1 = require("../../models/BaseItem");
const BaseModel_1 = require("../../BaseModel");
const ItemChange_1 = require("../../models/ItemChange");
const logger = Logger_1.default.create('permanentlyDeleteOldData');
const readOnlyItemsRemoved = async (itemIds, itemType) => {
    const result = [];
    for (const id of itemIds) {
        const item = await BaseItem_1.default.loadItem(itemType, id);
        // Only do the share-related read-only checks. If other checks are done,
        // readOnly will always be true because the item is in the trash.
        const shareChecksOnly = true;
        const readOnly = (0, readOnly_1.itemIsReadOnlySync)(itemType, ItemChange_1.default.SOURCE_UNSPECIFIED, item, Setting_1.default.value('sync.userId'), BaseItem_1.default.syncShareCache, shareChecksOnly);
        if (!readOnly) {
            result.push(id);
        }
    }
    return result;
};
const itemsToDelete = async (ttl = null) => {
    const result = await Folder_1.default.trashItemsOlderThan(ttl);
    const folderIds = await readOnlyItemsRemoved(result.folderIds, BaseModel_1.ModelType.Folder);
    const noteIds = await readOnlyItemsRemoved(result.noteIds, BaseModel_1.ModelType.Note);
    return { folderIds, noteIds };
};
const permanentlyDeleteOldItems = async (ttl = null) => {
    ttl = ttl === null ? Setting_1.default.value('trash.ttlDays') * time_1.Day : ttl;
    logger.info(`Processing items older than ${ttl}ms...`);
    if (!Setting_1.default.value('trash.autoDeletionEnabled')) {
        logger.info('Auto-deletion is not enabled - skipping');
        return;
    }
    const toDelete = await itemsToDelete(ttl);
    logger.info('Items to permanently delete:', toDelete);
    await Note_1.default.batchDelete(toDelete.noteIds, { sourceDescription: 'permanentlyDeleteOldItems' });
    // We only auto-delete folders if they are empty.
    for (const folderId of toDelete.folderIds) {
        const noteIds = await Folder_1.default.noteIds(folderId, { includeDeleted: true });
        if (!noteIds.length) {
            logger.info(`Deleting empty folder: ${folderId}`);
            await Folder_1.default.delete(folderId);
        }
        else {
            logger.info(`Skipping non-empty folder: ${folderId}`);
        }
    }
};
const setupAutoDeletion = async () => {
    await permanentlyDeleteOldItems();
    shim_1.default.setInterval(async () => {
        await permanentlyDeleteOldItems();
    }, 18 * time_1.Hour);
};
exports.setupAutoDeletion = setupAutoDeletion;
exports.default = permanentlyDeleteOldItems;
//# sourceMappingURL=permanentlyDeleteOldItems.js.map