"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Logger_1 = require("@joplin/utils/Logger");
const test_utils_1 = require("../testing/test-utils");
const Note_1 = require("../models/Note");
const ActionLogger_1 = require("./ActionLogger");
const Setting_1 = require("../models/Setting");
const fs_extra_1 = require("fs-extra");
const getLogPath = () => `${Setting_1.default.value('profileDir')}/log.txt`;
const logContainsEntryWith = async (...terms) => {
    const lines = (await (0, fs_extra_1.readFile)(getLogPath(), 'utf8')).split('\n');
    for (const line of lines) {
        if (terms.every(t => line.includes(t))) {
            return true;
        }
    }
    return false;
};
describe('ActionLogger', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        const logPath = getLogPath();
        if (await (0, fs_extra_1.pathExists)(logPath)) {
            await (0, fs_extra_1.remove)(logPath);
        }
        await (0, fs_extra_1.writeFile)(logPath, '', 'utf8');
        const logger = new Logger_1.default();
        logger.addTarget(Logger_1.TargetType.File, { path: logPath });
        logger.setLevel(Logger_1.LogLevel.Info);
        Logger_1.default.initializeGlobalLogger(logger);
    });
    it('should log deletions', async () => {
        const note = await Note_1.default.save({ title: 'MyTestNote' });
        await Note_1.default.delete(note.id, { toTrash: false });
        await Logger_1.default.globalLogger.waitForFileWritesToComplete_();
        expect(await logContainsEntryWith('DeleteAction', note.id, note.title)).toBe(true);
    });
    it('should be possible to disable ActionLogger globally', async () => {
        const note1 = await Note_1.default.save({ title: 'testNote1' });
        const note2 = await Note_1.default.save({ title: 'testNote2' });
        ActionLogger_1.default.enabled = true;
        await Note_1.default.delete(note1.id, { toTrash: false });
        ActionLogger_1.default.enabled = false;
        await Note_1.default.delete(note2.id, { toTrash: false });
        ActionLogger_1.default.enabled = true;
        await Logger_1.default.globalLogger.waitForFileWritesToComplete_();
        expect(await logContainsEntryWith('DeleteAction', note1.id)).toBe(true);
        expect(await logContainsEntryWith('DeleteAction', note2.id)).toBe(false);
    });
});
//# sourceMappingURL=ActionLogger.test.js.map