#ifndef JLCXX_MODULE_HPP
#define JLCXX_MODULE_HPP

#include <cassert>
#include <functional>
#include <map>
#include <memory>
#include <string>
#include <sstream>
#include <typeinfo>
#include <vector>

#include "array.hpp"
#include "attr.hpp"
#include "type_conversion.hpp"

namespace jlcxx
{

/// Wrappers for creating new datatype
JLCXX_API jl_datatype_t* new_datatype(jl_sym_t *name,
                            jl_module_t* module,
                            jl_datatype_t *super,
                            jl_svec_t *parameters,
                            jl_svec_t *fnames, jl_svec_t *ftypes,
                            int abstract, int mutabl,
                            int ninitialized);

JLCXX_API jl_datatype_t* new_bitstype(jl_sym_t *name,
                            jl_module_t* module,
                            jl_datatype_t *super,
                            jl_svec_t *parameters, const size_t nbits);

/// Some helper functions
namespace detail
{

// Need to treat void specially
template<typename R, typename... Args>
struct ReturnTypeAdapter
{
  using return_type = decltype(convert_to_julia(std::declval<R>()));

  inline return_type operator()(const void* functor, static_julia_type<Args>... args)
  {
    auto std_func = reinterpret_cast<const std::function<R(Args...)>*>(functor);
    assert(std_func != nullptr);
    return convert_to_julia((*std_func)(convert_to_cpp<Args>(args)...));
  }
};

template<typename... Args>
struct ReturnTypeAdapter<void, Args...>
{
  inline void operator()(const void* functor, static_julia_type<Args>... args)
  {
    auto std_func = reinterpret_cast<const std::function<void(Args...)>*>(functor);
    assert(std_func != nullptr);
    (*std_func)(convert_to_cpp<Args>(args)...);
  }
};

/// Call a C++ std::function, passed as a void pointer since it comes from Julia
template<typename R, typename... Args>
struct CallFunctor
{
  using return_type = std::remove_const_t<decltype(ReturnTypeAdapter<R, Args...>()(std::declval<const void*>(), std::declval<static_julia_type<Args>>()...))>;

  static return_type apply(const void* functor, static_julia_type<Args>... args)
  {
    try
    {
      return ReturnTypeAdapter<R, Args...>()(functor, args...);
    }
    catch(const std::exception& err)
    {
      jl_error(err.what());
    }

    return return_type();
  }
};

/// Make a vector with the types in the variadic template parameter pack
template<typename... Args>
std::vector<jl_datatype_t*> argtype_vector()
{
  return {julia_type<Args>()...};
}

template<typename... Args>
struct NeedConvertHelper
{
  bool operator()()
  {
    for(const bool b : {std::is_same_v<static_julia_type<Args>,Args>...})
    {
      if(!b)
        return true;
    }
    return false;
  }
};

template<>
struct NeedConvertHelper<>
{
  bool operator()()
  {
    return false;
  }
};

} // end namespace detail

/// Convenience function to create an object with a finalizer attached
template<typename T, bool finalize=true, typename... ArgsT>
BoxedValue<T> create(ArgsT&&... args)
{
  jl_datatype_t* dt = julia_type<T>();
  assert(jl_is_mutable_datatype(dt));


  T* cpp_obj = new T(std::forward<ArgsT>(args)...);

  return boxed_cpp_pointer(cpp_obj, dt, finalize);
}

/// Safe upcast to base type
template<typename T>
struct UpCast
{
  static inline supertype<T>& apply(T& derived)
  {
    return static_cast<supertype<T>&>(derived);
  }
};

// The CxxWrap Julia module
extern JLCXX_API jl_module_t* g_cxxwrap_module;
extern jl_datatype_t* g_cppfunctioninfo_type;

class JLCXX_API Module;

/// Abstract base class for storing any function
class JLCXX_API FunctionWrapperBase
{
public:
  FunctionWrapperBase(Module* mod, std::pair<jl_datatype_t*,jl_datatype_t*> return_type);

  /// Types of the arguments (used in the wrapper signature)
  virtual std::vector<jl_datatype_t*> argument_types() const = 0;

  /// Return type
  std::pair<jl_datatype_t*,jl_datatype_t*> return_type() const { return m_return_type; }

  void set_return_type(std::pair<jl_datatype_t*,jl_datatype_t*> dt) { m_return_type = dt; }

  virtual ~FunctionWrapperBase() {}

  inline void set_name(jl_value_t* name)
  {
    protect_from_gc(name);
    m_name = name;
  }

  inline jl_value_t* name() const
  {
    return m_name;
  }

  inline void set_doc(jl_value_t* doc)
  {
    protect_from_gc(doc);
    m_doc = doc;
  }

  inline jl_value_t* doc() const
  {
    return m_doc;
  }

  void set_extra_argument_data(std::vector<arg>&& posArgs, std::vector<kwarg>&& kwArgs)
  {
    m_number_of_keyword_args = kwArgs.size();

    // gather all argument names
    m_argument_names.clear();
    for(auto& a: posArgs)
      m_argument_names.push_back(jl_cstr_to_string(a.name));
    for(auto& a: kwArgs)
      m_argument_names.push_back(jl_cstr_to_string(a.name));
    // ensure the Julia GC doesn't throw away our strings:
    for(auto& s: m_argument_names)
      protect_from_gc(s);

    // gather all default values
    m_argument_default_values.clear();
    for(auto& a: posArgs)
      m_argument_default_values.push_back(a.defaultValue);
    for(auto& a: kwArgs)
      m_argument_default_values.push_back(a.defaultValue);
  }

  const std::vector<jl_value_t*>& argument_names() const {return m_argument_names;}
  int number_of_keyword_arguments() const {return m_number_of_keyword_args;}
  const std::vector<jl_value_t*>& argument_default_values() const {return m_argument_default_values;}

  inline void set_override_module(jl_module_t* mod) { m_override_module = (jl_value_t*)mod; }
  inline jl_value_t* override_module() const { return m_override_module; }

  /// Function pointer as void*, since that's what Julia expects
  virtual void *pointer() = 0;

  /// The thunk (i.e. std::function) to pass as first argument to the function pointed to by function_pointer
  virtual void *thunk() = 0;

private:
  jl_value_t* m_name = nullptr;
  jl_value_t* m_doc = nullptr;
  std::vector<jl_value_t*> m_argument_names;
  int m_number_of_keyword_args = 0;
  std::vector<jl_value_t*> m_argument_default_values;
  Module* m_module;
  std::pair<jl_datatype_t*,jl_datatype_t*> m_return_type = std::make_pair(nullptr,nullptr);

  // The module in which the function is overridden, e.g. jl_base_module when trying to override Base.getindex.
  jl_value_t* m_override_module = nullptr;

};

/// Implementation of function storage, case of std::function
template<typename R, typename... Args>
class FunctionWrapper : public FunctionWrapperBase
{
public:
  typedef std::function<R(Args...)> functor_t;

  FunctionWrapper(Module* mod, const functor_t &function) : FunctionWrapperBase(mod, julia_return_type<R>()), m_function(function)
  {
    (create_if_not_exists<Args>(), ...);
  }

  virtual std::vector<jl_datatype_t*> argument_types() const
  {
    return detail::argtype_vector<Args...>();
  }

protected:
  virtual void* pointer()
  {
    return reinterpret_cast<void*>(detail::CallFunctor<R, Args...>::apply);
  }

  virtual void* thunk()
  {
    return reinterpret_cast<void*>(&m_function);
  }

private:
  functor_t m_function;
};

/// Implementation of function storage, case of a function pointer
template<typename R, typename... Args>
class FunctionPtrWrapper : public FunctionWrapperBase
{
public:
  typedef std::function<R(Args...)> functor_t;

  FunctionPtrWrapper(Module* mod, R (*f)(Args...)) : FunctionWrapperBase(mod, julia_return_type<R>()), m_function(f)
  {
    (create_if_not_exists<Args>(), ...);
  }

  virtual std::vector<jl_datatype_t*> argument_types() const
  {
    return detail::argtype_vector<Args...>();
  }

protected:
  virtual void* pointer()
  {
    return reinterpret_cast<void*>(m_function);
  }

  virtual void* thunk()
  {
    return nullptr;
  }

private:
  R(*m_function)(Args...);
};

/// Indicate that a parametric type is to be added
template<typename... ParametersT>
struct Parametric
{
};

template<typename... T> struct IsMirroredType<Parametric<T...>> : std::false_type {};

template<typename T>
class TypeWrapper;

namespace detail
{

template<typename T>
struct GetJlType
{
  jl_value_t* operator()() const
  {
    using uncref_t = remove_const_ref<T>;
    if(has_julia_type<uncref_t>())
    {
      return (jl_value_t*)julia_base_type<remove_const_ref<T>>();
    }
    else
    {
      // The assumption here is that unmapped types are not needed, i.e. in default argument lists
      return nullptr;
    }
  }
};

template<int I>
struct GetJlType<TypeVar<I>>
{
  jl_value_t* operator()() const
  {
    return (jl_value_t*)TypeVar<I>::tvar();
  }
};

template<typename T, T Val>
struct GetJlType<std::integral_constant<T, Val>>
{
  jl_value_t* operator()() const
   {
    return box<T>(convert_to_julia(Val));
  }
};

template<typename T>
struct GetJlType<const T>
{
  jl_value_t* operator()() const
  {
    return (jl_value_t*)apply_type(jlcxx::julia_type("CxxConst"), (jl_datatype_t*)GetJlType<T>()());
  }
};

template<typename T>
struct IsParametric
{
  static constexpr bool value = false;
};

template<template<typename...> class T, int I, typename... ParametersT>
struct IsParametric<T<TypeVar<I>, ParametersT...>>
{
  static constexpr bool value = true;
};

template<typename... ArgsT>
inline jl_value_t* make_fname(const std::string& nametype, ArgsT... args)
{
  jl_value_t* name = nullptr;
  JL_GC_PUSH1(&name);
  name = jl_new_struct((jl_datatype_t*)julia_type(nametype), args...);
  protect_from_gc(name);
  JL_GC_POP();

  return name;
}

} // namespace detail

// Encapsulate a list of parameters, using types only
template<typename... ParametersT>
struct ParameterList
{
  static constexpr size_t nb_parameters = sizeof...(ParametersT);

  jl_svec_t* operator()(const size_t n = nb_parameters)
  {
    std::vector<jl_value_t*> paramlist({detail::GetJlType<ParametersT>()()...});
    for(size_t i = 0; i != n; ++i)
    {
      if(paramlist[i] == nullptr)
      {
        std::vector<std::string> typenames({(typeid(ParametersT).name())...});
        throw std::runtime_error("Attempt to use unmapped type " + typenames[i] + " in parameter list");
      }
    }
    jl_svec_t* result = jl_alloc_svec_uninit(n);
    JL_GC_PUSH1(&result);
    assert(paramlist.size() >= n);
    for(size_t i = 0; i != n; ++i)
    {
      jl_svecset(result, i, paramlist[i]);
    }
    JL_GC_POP();

    return result;
  }
};

namespace detail
{
  template<typename F, typename PL>
  struct ForEachParameterType
  {
  };

  template<typename F, typename... ParametersT>
  struct ForEachParameterType<F,ParameterList<ParametersT...>>
  {
    void operator()(F&& f)
    {
      (f.template apply<ParametersT>(),...);
    }
  };

  template<typename T, typename... ParametersT>
  struct has_type : std::bool_constant<(... || std::is_same_v<T, ParametersT>)> {};

  template<typename T, typename... ParametersT>
  constexpr bool has_type_v = has_type<T, ParametersT...>::value;

  template<typename T1, typename T2>
  struct CombineParameterLists;

  template<typename... Params1, typename... Params2>
  struct CombineParameterLists<ParameterList<Params1...>, ParameterList<Params2...>>
  {
    using type = ParameterList<Params1..., Params2...>;
  };

  template<typename... T>
  struct RemoveDuplicates;

  // empty/bottom case
  template<>
  struct RemoveDuplicates<ParameterList<>>
  {
    using type = ParameterList<>;
  };

  template<typename T, typename... Rest>
  struct RemoveDuplicates<ParameterList<T, Rest...>>
  {
  private:
    // recursive splitting of remaining parameters
    using rest_t = typename RemoveDuplicates<ParameterList<Rest...>>::type;

  public:
    using type = std::conditional_t<
      has_type_v<T, Rest...>,
      rest_t,  // T appears later, skip it
      typename CombineParameterLists<rest_t, ParameterList<T>>::type  // T is unique so far, append it
    >;
  };
}

template<typename ParametersT, typename F>
void for_each_parameter_type(F&& f)
{
  detail::ForEachParameterType<F,ParametersT>()(std::forward<F>(f));
}

template<typename T> using remove_duplicates = typename detail::RemoveDuplicates<T>::type;
template<typename T1, typename T2> using combine_parameterlists = typename detail::CombineParameterLists<T1,T2>::type;

using fundamental_int_types = remove_duplicates<ParameterList
<
  signed char,
  unsigned char,
  short int,
  unsigned short int,
  int,
  unsigned int,
  long,
  unsigned long,
  long long int,
  unsigned long long int
>>;

using fixed_int_types = ParameterList
<
  int8_t,uint8_t,
  int16_t,uint16_t,
  int32_t,uint32_t,
  int64_t,uint64_t
>;

template<typename T> inline std::string fundamental_int_type_name() { return "undefined"; }
template<> inline std::string fundamental_int_type_name<signed char>() { return "signed char"; }
template<> inline std::string fundamental_int_type_name<unsigned char>() { return "unsigned char"; }
template<> inline std::string fundamental_int_type_name<short>() { return "short"; }
template<> inline std::string fundamental_int_type_name<unsigned short>() { return "unsigned short"; }
template<> inline std::string fundamental_int_type_name<int>() { return "int"; }
template<> inline std::string fundamental_int_type_name<long>() { return "long"; }
template<> inline std::string fundamental_int_type_name<unsigned long>() { return "unsigned long"; }
template<> inline std::string fundamental_int_type_name<unsigned int>() { return "unsigned int"; }
template<> inline std::string fundamental_int_type_name<long long>() { return "long long"; }
template<> inline std::string fundamental_int_type_name<unsigned long long>() { return "unsigned long long"; }
template<typename T> inline std::string fixed_int_type_name() { return "undefined"; }
template<> inline std::string fixed_int_type_name<int8_t>() { return "int8_t"; }
template<> inline std::string fixed_int_type_name<uint8_t>() { return "uint8_t"; }
template<> inline std::string fixed_int_type_name<int16_t>() { return "int16_t"; }
template<> inline std::string fixed_int_type_name<uint16_t>() { return "uint16_t"; }
template<> inline std::string fixed_int_type_name<int32_t>() { return "int32_t"; }
template<> inline std::string fixed_int_type_name<uint32_t>() { return "uint32_t"; }
template<> inline std::string fixed_int_type_name<int64_t>() { return "int64_t"; }
template<> inline std::string fixed_int_type_name<uint64_t>() { return "uint64_t"; }

/// Trait to allow user-controlled disabling of the default constructor
template <typename T>
struct DefaultConstructible : std::bool_constant<std::is_default_constructible_v<T> && !std::is_abstract_v<T>>
{
};

/// Trait to allow user-controlled disabling of the copy constructor
template <typename T>
struct CopyConstructible : std::bool_constant<std::is_copy_constructible_v<T> && !std::is_abstract_v<T>>
{
};

/// Store all exposed C++ functions associated with a module
class JLCXX_API Module
{
public:

  Module(jl_module_t* jl_mod);

  void append_function(FunctionWrapperBase* f)
  {
    assert(f != nullptr);
    m_functions.push_back(std::shared_ptr<FunctionWrapperBase>(f));
    assert(m_functions.back() != nullptr);
    if(m_override_module != nullptr)
    {
      m_functions.back()->set_override_module(m_override_module);
    }
  }

  /// Define a new function
  template<typename R, typename... Args, typename... Extra>
  FunctionWrapperBase& method(const std::string& name,  std::function<R(Args...)> f, Extra... extra)
  {
    static_assert(detail::check_extra_argument_count<Extra...>(sizeof...(Args)), "Wrong number of annotated arguments (jlcxx::arg and jlcxx::kwarg arguments)!");

    detail::ExtraFunctionData extraData = detail::parse_attributes(extra...);
    return method_helper(name, f, extraData);
  }

  /// Define a new function. Overload for pointers
  template<typename R, typename... Args, typename... Extra>
  FunctionWrapperBase& method(const std::string& name,  R(*f)(Args...), Extra... extra)
  {
    static_assert(detail::check_extra_argument_count<Extra...>(sizeof...(Args)), "Wrong number of annotated arguments (jlcxx::arg and jlcxx::kwarg arguments)!");

    detail::ExtraFunctionData extraData = detail::parse_attributes<true>(extra...);
    const bool need_convert = bool(extraData.force_convert) || detail::NeedConvertHelper<R, Args...>()();

    // Conversion is automatic when using the std::function calling method, so if we need conversion we use that
    if(need_convert)
    {
      return method_helper(name, std::function<R(Args...)>(f), std::move(extraData));
    }

    // No conversion needed -> call can be through a naked function pointer
    auto* new_wrapper = new FunctionPtrWrapper<R, Args...>(this, f);
    new_wrapper->set_name((jl_value_t*)jl_symbol(name.c_str()));
    new_wrapper->set_doc(jl_cstr_to_string(extraData.doc.c_str()));
    new_wrapper->set_extra_argument_data(std::move(extraData.positionalArguments), std::move(extraData.keywordArguments));
    append_function(new_wrapper);
    return *new_wrapper;
  }

  /// Define a new function. Overload for lambda
  template<typename LambdaT, typename... Extra,
           std::enable_if_t<detail::has_call_operator<LambdaT>::value && !std::is_member_function_pointer_v<LambdaT>, bool> = true>
  FunctionWrapperBase& method(const std::string& name, LambdaT&& lambda, Extra... extra)
  {
    detail::ExtraFunctionData extraData = detail::parse_attributes(extra...);
    return lambda_helper(name, std::forward<LambdaT>(lambda), &LambdaT::operator(), std::move(extraData));
  }

  /// Add a constructor with the given argument types for the given datatype (used to get the name)
  template<typename T, typename... ArgsT, typename... Extra>
  void constructor(jl_datatype_t* dt, Extra... extra)
  {
    static_assert(detail::check_extra_argument_count<Extra...>(sizeof...(ArgsT)), "Wrong number of annotated arguments (jlcxx::arg and jlcxx::kwarg arguments)!");

    detail::ExtraFunctionData extraData = detail::parse_attributes<false,true>(extra...);
    FunctionWrapperBase &new_wrapper = bool(extraData.finalize) ? add_lambda("dummy", [](ArgsT... args) { return create<T, true>(args...); }, std::move(extraData)) : add_lambda("dummy", [](ArgsT... args) { return create<T, false>(args...); }, std::move(extraData));
    new_wrapper.set_name(detail::make_fname("ConstructorFname", dt));
    new_wrapper.set_doc(jl_cstr_to_string(extraData.doc.c_str()));
    new_wrapper.set_extra_argument_data(std::move(extraData.positionalArguments), std::move(extraData.keywordArguments));
  }

  template<typename T, typename R, typename LambdaT, typename... ArgsT, typename... Extra>
  void constructor(jl_datatype_t* dt, LambdaT&& lambda, R(LambdaT::*)(ArgsT...) const, Extra... extra)
  {
    static_assert(std::is_same_v<T*,R>, "Constructor lambda function must return a pointer to the constructed object, of the correct type");
    detail::ExtraFunctionData extraData = detail::parse_attributes<false,true>(extra...);
    FunctionWrapperBase &new_wrapper = add_lambda("dummy", [=](ArgsT... args)
    {
      jl_datatype_t* concrete_dt = julia_type<T>();
      assert(jl_is_mutable_datatype(concrete_dt));
      T* cpp_obj = lambda(std::forward<ArgsT>(args)...);
      return boxed_cpp_pointer(cpp_obj, concrete_dt, bool(extraData.finalize));
    }, std::move(extraData));
    new_wrapper.set_name(detail::make_fname("ConstructorFname", dt));
  }

  /// Loop over the functions
  template<typename F>
  void for_each_function(const F f) const
  {
    auto funcs_copy = m_functions;
    for(const auto &item : funcs_copy)
    {
      assert(item != nullptr);
      f(*item);
    }
    // Account for any new functions added during the loop
    while(funcs_copy.size() != m_functions.size())
    {
      const std::size_t oldsize = funcs_copy.size();
      const std::size_t newsize = m_functions.size();
      funcs_copy = m_functions;
      for(std::size_t i = oldsize; i != newsize; ++i)
      {
        assert(funcs_copy[i] != nullptr);
        f(*funcs_copy[i]);
      }
    }
  }

  inline jlcxx::FunctionWrapperBase& last_function()
  {
    return *m_functions.back();
  }

  /// Add a composite type
  template<typename T, typename SuperParametersT=ParameterList<>, typename JLSuperT=jl_datatype_t>
  TypeWrapper<T> add_type(const std::string& name, JLSuperT* super = jl_any_type);

  /// Add types that are directly mapped to a Julia struct
  template<typename T>
  void map_type(const std::string& name)
  {
    jl_datatype_t* dt = (jl_datatype_t*)julia_type(name, m_jl_mod);
    if(dt == nullptr)
    {
      throw std::runtime_error("Type for " + name + " was not found when mapping it.");
    }
    set_julia_type<T>(dt);
  }

  template<typename T, typename JLSuperT=jl_datatype_t>
  void add_bits(const std::string& name, JLSuperT* super = jl_any_type);

  template<typename EnumT, typename T>
  void add_enum(std::string name, std::vector<const char*> labels, std::vector<T> values)
  {
    const std::size_t nb_items = labels.size();
    if(nb_items != values.size())
    {
      throw std::runtime_error("Lengths of the labels and values vectors don't match for enum " + name);
    }

    create_if_not_exists<ArrayRef<const char*>>();
    create_if_not_exists<ArrayRef<T>>();
    auto labels_jl = ArrayRef<const char*>(&labels[0], nb_items);
    auto values_jl = ArrayRef<T>(&values[0], nb_items);

    jl_value_t* add_enum_fn = jl_get_function(g_cxxwrap_module, "add_enum");
    if(add_enum_fn == nullptr)
    {
      throw std::runtime_error("CxxWrapCore.add_enum function not found, ensure you are using at least CxxWrap 0.17.3");
    }

    jl_value_t** julia_args;
    JL_GC_PUSHARGS(julia_args, 4);
    julia_args[0] = (jl_value_t*)m_jl_mod;
    julia_args[1] = jl_cstr_to_string(name.c_str());
    julia_args[2] = (jl_value_t*)labels_jl.wrapped();
    julia_args[3] = (jl_value_t*)values_jl.wrapped();
    jl_value_t* dt = jl_call(add_enum_fn, julia_args, 4);
    JL_GC_POP();

    if(!jl_is_datatype(dt))
    {
      throw std::runtime_error("error adding enum type " + name);
    }
    set_julia_type<EnumT>((jl_datatype_t*)dt, false);
  }

  /// Set a global constant value at the module level
  template<typename T>
  void set_const(const std::string& name, T&& value)
  {
    using plain_type = remove_const_ref<T>;
    static_assert(IsMirroredType<plain_type>::value, "set_const can only be applied to IsMirrored types (e.g. numbers or standard layout structs)");
    if(get_constant(name) != nullptr)
    {
      throw std::runtime_error("Duplicate registration of constant " + name);
    }
    set_constant(name, box<plain_type>(value));
  }

  std::string name() const
  {
    return module_name(m_jl_mod);
  }

  void bind_constants(ArrayRef<jl_value_t*> symbols, ArrayRef<jl_value_t*> values);

  jl_datatype_t* get_julia_type(const char* name)
  {
    jl_value_t *cst = get_constant(name);
    if(cst != nullptr && jl_is_datatype(cst))
    {
      return (jl_datatype_t*)cst;
    }

    return nullptr;
  }

  void register_type(jl_datatype_t* box_type)
  {
    m_box_types.push_back(box_type);
  }

  const std::vector<jl_datatype_t*> box_types() const
  {
    return m_box_types;
  }

  jl_module_t* julia_module() const
  {
    return m_jl_mod;
  }

  inline void set_override_module(jl_module_t* mod) { m_override_module = mod; }
  inline void unset_override_module() { m_override_module = nullptr; }

private:

  template<typename T>
  void add_default_constructor(jl_datatype_t* dt);

  template<typename T>
  void add_copy_constructor(jl_datatype_t*)
  {
    if constexpr (CopyConstructible<T>::value)
    {
      set_override_module(jl_base_module);
      method("copy", [this](const T& other)
      {
        return create<T>(other);
      });
      unset_override_module();
    }
  }

  template<typename T, typename SuperParametersT, typename JLSuperT>
  TypeWrapper<T> add_type_internal(const std::string& name, JLSuperT* super);

  template<typename LambdaT>
  FunctionWrapperBase& add_lambda(const std::string& name, LambdaT&& lambda, detail::ExtraFunctionData&& extraData)
  {
    return lambda_helper(name, std::forward<LambdaT>(lambda), &LambdaT::operator(), std::move(extraData));
  }

  template<typename R, typename LambdaT, typename... ArgsT>
  FunctionWrapperBase& lambda_helper(const std::string& name, LambdaT&& lambda, R(LambdaT::*)(ArgsT...) const, detail::ExtraFunctionData&& extraData)
  {
    return method_helper(name, std::function<R(ArgsT...)>(std::forward<LambdaT>(lambda)), std::move(extraData));
  }

  template<typename R, typename... Args>
  FunctionWrapperBase& method_helper(const std::string& name,  std::function<R(Args...)> f, detail::ExtraFunctionData&& extraData)
  {
    auto* new_wrapper = new FunctionWrapper<R, Args...>(this, f);
    new_wrapper->set_name((jl_value_t*)jl_symbol(name.c_str()));
    new_wrapper->set_doc(jl_cstr_to_string(extraData.doc.c_str()));
    new_wrapper->set_extra_argument_data(std::move(extraData.positionalArguments), std::move(extraData.keywordArguments));
    append_function(new_wrapper);
    return *new_wrapper;
  }

  void set_constant(const std::string& name, jl_value_t* boxed_const);
  jl_value_t *get_constant(const std::string &name);

  jl_module_t* m_jl_mod;
  jl_module_t* m_override_module = nullptr;
  std::vector<std::shared_ptr<FunctionWrapperBase>> m_functions;
  std::map<std::string, size_t> m_jl_constants;
  std::vector<std::string> m_constant_names;
  Array<jl_value_t*> m_constant_values;
  std::vector<jl_datatype_t*> m_box_types;

  template<class T> friend class TypeWrapper;
  template<typename T, typename... AppliedTypesT> friend class ParametricTypeWrappers;
};

template<typename T>
void Module::add_default_constructor(jl_datatype_t* dt)
{
  if constexpr (DefaultConstructible<T>::value)
  {
    this->constructor<T>(dt);
  }
}

// Specialize this to build the correct parameter list, wrapping non-types in integral constants
// There is no way to provide a template here that matches all possible combinations of type and non-type arguments
template<typename T>
struct BuildParameterList
{
  typedef ParameterList<> type;
};

template<typename T> using parameter_list = typename BuildParameterList<T>::type;

// Match any combination of types only
template<template<typename...> class T, typename... ParametersT>
struct BuildParameterList<T<ParametersT...>>
{
  typedef ParameterList<ParametersT...> type;
};

// Match any number of int parameters
template<template<int...> class T, int... ParametersT>
struct BuildParameterList<T<ParametersT...>>
{
  typedef ParameterList<std::integral_constant<int, ParametersT>...> type;
};

namespace detail
{
  template<typename... Types>
  struct DoApply
  {
    template<typename WrapperT, typename FunctorT>
    void operator()(WrapperT& w, FunctorT&& ftor)
    {
      (..., DoApply<Types>()(w, std::forward<FunctorT>(ftor)));
    }
  };

  template<typename... Types>
  struct DoApply<ParameterList<Types...>>
  {
    template<typename WrapperT, typename FunctorT>
    void operator()(WrapperT& w, FunctorT&& ftor)
    {
      (..., DoApply<Types>()(w, std::forward<FunctorT>(ftor)));
    }
  };

  template<typename AppT>
  struct DoApply<AppT>
  {
    template<typename WrapperT, typename FunctorT>
    void operator()(WrapperT& w, FunctorT&& ftor)
    {
      w.template apply<AppT>(std::forward<FunctorT>(ftor));
    }
  };
}

/// Execute a functor on each type
template<typename... Types>
struct ForEachType
{
  template<typename FunctorT>
  void operator()(FunctorT&& ftor)
  {
    (..., ForEachType<Types>()(std::forward<FunctorT>(ftor)));
  }
};

template<typename... Types>
struct ForEachType<ParameterList<Types...>>
{
  template<typename FunctorT>
  void operator()(FunctorT&& ftor)
  {
    (..., ForEachType<Types>()(std::forward<FunctorT>(ftor)));
  }
};

template<typename AppT>
struct ForEachType<AppT>
{
  template<typename FunctorT>
  void operator()(FunctorT&& ftor)
  {
#ifdef _MSC_VER
    ftor.operator()<AppT>();
#else
    ftor.template operator()<AppT>();
#endif
  }
};

template<typename T, typename FunctorT>
void for_each_type(FunctorT&& f)
{
  ForEachType<T>()(f);
}

/// Type-level container for accumulating parameter packs during CombineTypes recursion.
/// Semantically equivalent to ParameterList but with a distinct typename to enable
/// template specialization in CombineTypes.
template<typename... Types>
struct UnpackedTypeList
{
};

template<typename... Types>
struct FlattenParameterLists;

// Base case: single non-ParameterList type
template<typename T>
struct FlattenParameterLists<T>
{
  using flatten = ParameterList<T>;
};

// Single ParameterList: flatten it recursively
template<typename... Types>
struct FlattenParameterLists<ParameterList<Types...>>
{
  using flatten = typename FlattenParameterLists<Types...>::flatten;
};

// Multiple types: recursively flatten each and concatenate
template<typename T1, typename T2, typename... Rest>
struct FlattenParameterLists<T1, T2, Rest...>
{
  using flatten = typename detail::CombineParameterLists<
      typename FlattenParameterLists<T1>::flatten,
      typename FlattenParameterLists<T2, Rest...>::flatten
    >::type;
};

/// Compute the cartesian product of multiple ParameterLists, applying ApplyT to each combination.
///
/// Given ApplyT and N ParameterLists, produces a ParameterList containing ApplyT applied to all
/// possible combinations of one type from each input list. ApplyT must provide a template type
/// alias `apply` that accepts the unpacked combination.
///
/// Example from parametric.cpp:
///   CombineTypes<ApplyType<Foo3>, ParameterList<int32_t, double>, ParameterList<P1,P2,bool>, ParameterList<float>>::type
///   => ParameterList<Foo3<int32_t,P1,float>, Foo3<int32_t,P2,float>, Foo3<int32_t,bool,float>,
///                    Foo3<double,P1,float>, Foo3<double,P2,float>, Foo3<double,bool,float>>
template<typename ApplyT, typename... TypeLists>
struct CombineTypes;

/// Primary specialization: Entry point that begins recursion on the first ParameterList.
/// Unpacks the first list and creates recursive CombineTypes instantiations for each type,
/// pairing it with all remaining lists.
///
/// Example instantiation:
///   CombineTypes<ApplyType<Foo3>, ParameterList<int32_t, double>, ParameterList<P1,P2,bool>, ParameterList<float>>
///   Types = {int32_t, double}
///   OtherTypeLists = {ParameterList<P1,P2,bool>, ParameterList<float>}
///   type = ParameterList<combined_t<int32_t>, combined_t<double>>
///   where combined_t<int32_t> = CombineTypes<ApplyType<Foo3>, UnpackedTypeList<int32_t>, ParameterList<P1,P2,bool>, ParameterList<float>>
template<typename ApplyT, typename... Types, typename... OtherTypeLists>
struct CombineTypes<ApplyT, ParameterList<Types...>, OtherTypeLists...>
{
  template<typename T1>
  using combined_t = typename CombineTypes<ApplyT, UnpackedTypeList<T1>, OtherTypeLists...>::type;

  using type = typename FlattenParameterLists<combined_t<Types>...>::flatten;
};

/// Secondary specialization: Accumulates types from remaining ParameterLists.
/// Takes types already accumulated in UnpackedTypeList, unpacks the next ParameterList,
/// and recursively combines each new type with the accumulated types.
///
/// Example instantiation:
///   CombineTypes<ApplyType<Foo3>, UnpackedTypeList<int32_t>, ParameterList<P1,P2,bool>, ParameterList<float>>
///   UnpackedTypes = {int32_t}
///   Types = {P1, P2, bool}
///   OtherTypeLists = {ParameterList<float>}
///   type = ParameterList<combined_t<P1>, combined_t<P2>, combined_t<bool>>
///   where combined_t<P1> = CombineTypes<ApplyType<Foo3>, UnpackedTypeList<int32_t, P1>, ParameterList<float>>
template<typename ApplyT, typename... UnpackedTypes, typename... Types, typename... OtherTypeLists>
struct CombineTypes<ApplyT, UnpackedTypeList<UnpackedTypes...>, ParameterList<Types...>, OtherTypeLists...>
{
  template<typename T1>
  using combined_t = typename CombineTypes<ApplyT, UnpackedTypeList<UnpackedTypes..., T1>, OtherTypeLists...>::type;

  using type = ParameterList<combined_t<Types>...>;
};

/// Base case: All ParameterLists exhausted, apply ApplyT to the accumulated type combination.
/// UnpackedTypes contains one complete combination of types selected from each input ParameterList.
///
/// Example instantiation:
///   CombineTypes<ApplyType<Foo3>, UnpackedTypeList<int32_t, P1, float>>
///   UnpackedTypes = {int32_t, P1, float}
///   type = Foo3<int32_t, P1, float>
template<typename ApplyT, typename... UnpackedTypes>
struct CombineTypes<ApplyT, UnpackedTypeList<UnpackedTypes...>>
{
  using type = typename ApplyT::template apply<UnpackedTypes...>;
};

// Default ApplyT implementation
template<template<typename...> class TemplateT>
struct ApplyType
{
  template<typename... Types> using apply = TemplateT<Types...>;
};

struct SpecializedFinalizer {};

template<typename T, typename Specializer=SpecializedFinalizer>
struct Finalizer
{
  static void finalize(T* to_delete)
  {
    delete to_delete;
  }
};

namespace detail
{
  template<typename T>
  struct CreateParameterType
  {
    inline void operator()()
    {
      create_if_not_exists<T>();
    }
  };

  template<typename T, T v>
  struct CreateParameterType<std::integral_constant<T, v>>
  {
    inline void operator()()
    {
    }
  };

  template<int N, typename T, std::size_t I>
  inline void create_parameter_type()
  {
    if(I < N)
    {
      CreateParameterType<T>()();
    }
  }

  template<int N, typename... ParametersT, std::size_t... Indices>
  void create_parameter_types(ParameterList<ParametersT...>, std::index_sequence<Indices...>)
  {
    (create_parameter_type<N, ParametersT,Indices>(), ...);
  }

}

/// Attempt downcast to derived type
template<typename SuperT, typename DerivedT>
struct DownCast
{
  static inline void apply(Module& mod)
  {
    mod.method("cxxdowncast", [](SingletonType<DerivedT>, SuperT* base)
    {
      if constexpr (std::is_polymorphic_v<DerivedT>)
      {
        return dynamic_cast<DerivedT*>(base);
      }
      else
      {
        return static_cast<DerivedT*>(base);
      }
    });
    using newsuper_t = supertype<SuperT>;
    if constexpr (!std::is_same_v<newsuper_t,SuperT>)
    {
      DownCast<supertype<SuperT>, DerivedT>::apply(mod);
    }
  }
};

template<typename T>
inline void add_default_methods(Module& mod)
{
  mod.set_override_module(get_cxxwrap_module());
  if constexpr(!std::is_same_v<supertype<T>, T>)
  {
    mod.method("cxxupcast", UpCast<T>::apply);
    DownCast<supertype<T>,T>::apply(mod);
  }
  if constexpr(std::is_destructible_v<T>)
  {
    mod.method("__delete", Finalizer<T>::finalize);
  }
  mod.unset_override_module();
}

template<typename T, typename... AppliedTypesT>
class ParametricTypeWrappers;

/// Helper class to wrap type methods
template<typename T>
class TypeWrapper
{
public:
  typedef T type;

  TypeWrapper(Module& mod, jl_datatype_t* dt, jl_datatype_t* box_dt) :
    m_module(mod),
    m_dt(dt),
    m_box_dt(box_dt)
  {
  }

  TypeWrapper(Module& mod, TypeWrapper<T>& other) :
    m_module(mod),
    m_dt(other.m_dt),
    m_box_dt(other.m_box_dt)
  {
  }

  /// Add a constructor with the given argument types
  template<typename... ArgsT, typename... Extra>
  TypeWrapper<T>& constructor(Extra... extra)
  {
    // Only add the default constructor if it wasn't added automatically
    if constexpr (!(DefaultConstructible<T>::value && sizeof...(ArgsT) == 0))
    {
      m_module.constructor<T, ArgsT...>(m_dt, extra...);
    }
    return *this;
  }

  /// Define a "constructor" using a lambda
  template<typename LambdaT, typename... Extra,
           std::enable_if_t<detail::has_call_operator<LambdaT>::value, bool> = true>
  TypeWrapper<T>& constructor(LambdaT&& lambda, Extra... extra)
  {
    m_module.constructor<T>(m_dt, std::forward<LambdaT>(lambda), &LambdaT::operator(), extra...);
    return *this;
  }

  /// Define a member function
  template<typename R, typename CT, typename... ArgsT, typename... Extra>
  TypeWrapper<T>& method(const std::string& name, R(CT::*f)(ArgsT...), Extra... extra)
  {
    m_module.method(name, [f](T& obj, ArgsT... args) -> R { return (obj.*f)(args...); }, extra... );
    m_module.method(name, [f](T* obj, ArgsT... args) -> R { return ((*obj).*f)(args...); }, extra... );
    return *this;
  }

  /// Define a member function, const version
  template<typename R, typename CT, typename... ArgsT, typename... Extra>
  TypeWrapper<T>& method(const std::string& name, R(CT::*f)(ArgsT...) const, Extra... extra)
  {
    m_module.method(name, [f](const T& obj, ArgsT... args) -> R { return (obj.*f)(args...); }, extra... );
    m_module.method(name, [f](const T* obj, ArgsT... args) -> R { return ((*obj).*f)(args...); }, extra... );
    return *this;
  }

  /// Define a "member" function using a lambda
  template<typename LambdaT, typename... Extra,
           std::enable_if_t<detail::has_call_operator<LambdaT>::value && !std::is_member_function_pointer_v<LambdaT>, bool> = true>
  TypeWrapper<T>& method(const std::string& name, LambdaT&& lambda, Extra... extra)
  {
    detail::ExtraFunctionData extraData = detail::parse_attributes(extra...);
    m_module.lambda_helper(name, std::forward<LambdaT>(lambda), &LambdaT::operator(), std::move(extraData));
    return *this;
  }

  /// Call operator overload. For concrete type box to work around https://github.com/JuliaLang/julia/issues/14919
  template<typename R, typename CT, typename... ArgsT, typename... Extra>
  TypeWrapper<T>& method(R(CT::*f)(ArgsT...), Extra... extra)
  {
    m_module.method("operator()", [f](T& obj, ArgsT... args) -> R { return (obj.*f)(args...); }, extra... )
      .set_name(detail::make_fname("CallOpOverload", m_box_dt));
    return *this;
  }
  template<typename R, typename CT, typename... ArgsT, typename... Extra>
  TypeWrapper<T>& method(R(CT::*f)(ArgsT...) const, Extra... extra)
  {
    m_module.method("operator()", [f](const T& obj, ArgsT... args) -> R { return (obj.*f)(args...); }, extra... )
      .set_name(detail::make_fname("CallOpOverload", m_box_dt));
    return *this;
  }

  /// Overload operator() using a lambda
  template<typename LambdaT, typename... Extra,
           std::enable_if_t<detail::has_call_operator<LambdaT>::value, bool> = true>
  TypeWrapper<T>& method(LambdaT&& lambda, Extra... extra)
  {
    detail::ExtraFunctionData extraData = detail::parse_attributes(extra...);
    m_module.lambda_helper("operator()", std::forward<LambdaT>(lambda), &LambdaT::operator(), std::move(extraData))
      .set_name(detail::make_fname("CallOpOverload", m_box_dt));
    return *this;
  }

  template<typename... AppliedTypesT>
  ParametricTypeWrappers<T, AppliedTypesT...> apply()
  {
    static_assert(detail::IsParametric<T>::value, "Apply can only be called on parametric types");
    return ParametricTypeWrappers<T, AppliedTypesT...>(*this);
  }

  template<typename... AppliedTypesT, typename FunctorT>
  TypeWrapper<T>& apply(FunctorT&& apply_ftor)
  {
    static_assert(detail::IsParametric<T>::value, "Apply can only be called on parametric types");
    apply<AppliedTypesT...>().apply(std::forward<FunctorT>(apply_ftor));
    return *this;
  }

  /// Apply all possible combinations of the given types (see example)
  template<template<typename...> class TemplateT, typename... TypeLists, typename FunctorT>
  void apply_combination(FunctorT&& ftor);

  template<typename ApplyT, typename... TypeLists, typename FunctorT>
  void apply_combination(FunctorT&& ftor);

  // Access to the module
  Module& module()
  {
    return m_module;
  }

  jl_datatype_t* dt()
  {
    return m_dt;
  }

private:

  Module& m_module;
  jl_datatype_t* m_dt;
  jl_datatype_t* m_box_dt;

  template<typename U, typename... AppliedTypesT> friend class ParametricTypeWrappers;
};

using TypeWrapper1 = TypeWrapper<Parametric<TypeVar<1>>>;

#ifdef JLCXX_USE_TYPE_MAP
JLCXX_API std::shared_ptr<TypeWrapper1>& jlcxx_smartpointer_type(std::type_index idx);
#endif

template<typename ApplyT, typename... TypeLists> using combine_types = typename CombineTypes<ApplyT, TypeLists...>::type;

template<typename T>
template<template<typename...> class TemplateT, typename... TypeLists, typename FunctorT>
void TypeWrapper<T>::apply_combination(FunctorT&& ftor)
{
  this->template apply_combination<ApplyType<TemplateT>, TypeLists...>(std::forward<FunctorT>(ftor));
}

template<typename T>
template<typename ApplyT, typename... TypeLists, typename FunctorT>
void TypeWrapper<T>::apply_combination(FunctorT&& ftor)
{
  detail::DoApply<combine_types<ApplyT, TypeLists...>>()(*this, std::forward<FunctorT>(ftor));
}

template<typename T, typename SuperParametersT, typename JLSuperT>
TypeWrapper<T> Module::add_type_internal(const std::string& name, JLSuperT* super_generic)
{
  static constexpr bool is_parametric = detail::IsParametric<T>::value;
  static_assert(!IsMirroredType<T>::value, "Mirrored types (marked with IsMirroredType) can't be added using add_type, map them directly to a struct instead and use map_type or explicitly disable mirroring for this type, e.g. define template<> struct IsMirroredType<Foo> : std::false_type { };");
  static_assert(!std::is_scalar_v<T>, "Scalar types must be added using add_bits");

  if(get_constant(name) != nullptr)
  {
    throw std::runtime_error("Duplicate registration of type or constant " + name);
  }

  jl_datatype_t* super = nullptr;

  jl_svec_t* parameters = nullptr;
  jl_svec_t* super_parameters = nullptr;
  jl_svec_t* fnames = nullptr;
  jl_svec_t* ftypes = nullptr;
  JL_GC_PUSH5(&super, &parameters, &super_parameters, &fnames, &ftypes);

  parameters = is_parametric ? parameter_list<T>()() : jl_emptysvec;
  fnames = jl_svec1(jl_symbol("cpp_object"));
  ftypes = jl_svec1(jl_voidpointer_type);

  if(jl_is_datatype(super_generic) && !jl_is_unionall(super_generic) && !(is_parametric && SuperParametersT::nb_parameters != 0))
  {
    super = (jl_datatype_t*)super_generic;
  }
  else
  {
    super_parameters = SuperParametersT::nb_parameters == 0 ? parameter_list<T>()() : SuperParametersT()();
    super = (jl_datatype_t*)apply_type((jl_value_t*)super_generic, super_parameters);
  }

  if (!jl_is_datatype(super) || !jl_is_abstracttype(super)||
    jl_subtype((jl_value_t*)super, (jl_value_t*)jl_vararg_type) ||
    jl_is_tuple_type(super) ||
    jl_is_namedtuple_type(super) ||
    jl_subtype((jl_value_t*)super, (jl_value_t*)jl_type_type) ||
    jl_subtype((jl_value_t*)super, (jl_value_t*)jl_builtin_type))
  {
      throw std::runtime_error("invalid subtyping in definition of " + name + " with supertype " + julia_type_name(super));
  }

  const std::string allocname = name+"Allocated";

  // Create the datatypes
  jl_datatype_t* base_dt = new_datatype(jl_symbol(name.c_str()), m_jl_mod, super, parameters, jl_emptysvec, jl_emptysvec, 1, 0, 0);
  protect_from_gc(base_dt);

  super = is_parametric ? (jl_datatype_t*)apply_type((jl_value_t*)base_dt, parameters) : base_dt;

  jl_datatype_t* box_dt = new_datatype(jl_symbol(allocname.c_str()), m_jl_mod, super, parameters, fnames, ftypes, 0, 1, 1);
  protect_from_gc(box_dt);

  // Register the type
  if(is_parametric)
  {
    set_const(name, std::forward<jl_value_t*>(base_dt->name->wrapper));
    set_const(allocname, std::forward<jl_value_t*>(box_dt->name->wrapper));
  }
  else
  {
    set_julia_type<T>(box_dt);
    add_default_constructor<T>(base_dt);
    add_copy_constructor<T>(base_dt);

    set_const(name, (jl_value_t*)base_dt);
    set_const(allocname, (jl_value_t*)box_dt);

    this->register_type(box_dt);
    add_default_methods<T>(*this);
  }

  JL_GC_POP();
  return TypeWrapper<T>(*this, base_dt, box_dt);
}

/// Add a composite type
template<typename T, typename SuperParametersT, typename JLSuperT>
TypeWrapper<T> Module::add_type(const std::string& name, JLSuperT* super)
{
  return add_type_internal<T, SuperParametersT>(name, super);
}

/// Collection of the wrappers of each class template instantiation, that is on
/// Julia side, each fully-parametrized type.
///
/// Use the TypeWrapper::apply<...>() method to produce this collection.
///
/// The method ParametricTypeWrappers::apply(FunctorT&&) can then be used to add
/// the wrappers for the class template methods.
///
template<typename T, typename... AppliedTypesT>
class ParametricTypeWrappers
{
public:
  ParametricTypeWrappers(TypeWrapper<T>& base_wrapper):
    m_generic_type(base_wrapper),
    m_specialized_types(std::make_tuple(create_type<AppliedTypesT>()...))
  {
  }

  ///Apply a functor, ftor, to each wrapper.
  ///
  ///Usage example.
  ///
  ///  Wrap methods, `method1` and `method2` of a class template.
  ///
  ///  ```
  ///  wrappers.apply([](auto wrapped)
  ///  {
  ///    typedef typename decltype(wrapped)::type WrappedT;
  ///    wrapped.method("method1", &WrappedT::method1);
  ///    wrapped.method("method2", &WrappedT::method2);
  ///  });
  ///  ```

  template<typename FunctorT>
  void apply(FunctorT&& ftor)
  {
    ssize_t i = 0;
    apply_internal(std::forward<FunctorT>(ftor), m_specialized_types,
                   std::index_sequence_for<AppliedTypesT...>{});
  }

private:

  template<typename FunctorT, typename Tuple, std::size_t... I>
  constexpr void apply_internal(FunctorT&& ftor, Tuple&& t, std::index_sequence<I...>)
  {
    //note: cast to decayed type required to allow in the functor statements like
    //`using WrappedT = typename TypeWrapperT::type`
    (void(ftor(static_cast<std::decay_t<decltype(std::get<I>(t))>>(std::get<I>(t)))), ...);
  }

  template<typename AppliedT>
  TypeWrapper<AppliedT>
  create_type()
  {
    static_assert(!IsMirroredType<AppliedT>::value, "Mirrored type templates can't be added using add_type and apply, you should explicitly disable mirroring for this type, e.g. define template<typename... T> struct IsMirroredType<Foo<T...>> : std::false_type { }; in the jlcxx namespace.");

    static constexpr int nb_julia_parameters = parameter_list<T>::nb_parameters;
    static constexpr int nb_cpp_parameters = parameter_list<AppliedT>::nb_parameters;
    static_assert(nb_cpp_parameters != 0, "No parameters found when applying type. Specialize jlcxx::BuildParameterList for your combination of type and non-type parameters.");
    static_assert(nb_cpp_parameters >= nb_julia_parameters, "Parametric type applied to wrong number of parameters.");
    const bool is_abstract = jl_is_abstracttype(m_generic_type.dt());

    detail::create_parameter_types<nb_julia_parameters>(parameter_list<AppliedT>(), std::make_index_sequence<nb_cpp_parameters>());

    jl_datatype_t* app_dt = (jl_datatype_t*)apply_type((jl_value_t*)m_generic_type.dt(), parameter_list<AppliedT>()(nb_julia_parameters));
    jl_datatype_t* app_box_dt = (jl_datatype_t*)apply_type((jl_value_t*)m_generic_type.m_box_dt, parameter_list<AppliedT>()(nb_julia_parameters));

    if(has_julia_type<AppliedT>())
      {
        std::cout << "existing type found : " << app_box_dt << " <-> " << julia_type<AppliedT>() << std::endl;
        assert(julia_type<AppliedT>() == app_box_dt);
      }
    else
      {
        set_julia_type<AppliedT>(app_box_dt);
        m_generic_type.module().register_type(app_box_dt);
      }

    m_generic_type.module().template add_default_constructor<AppliedT>(app_dt);
    m_generic_type.module().template add_copy_constructor<AppliedT>(app_dt);
    add_default_methods<AppliedT>(m_generic_type.module());

    return TypeWrapper<AppliedT>(m_generic_type.module(), app_dt, app_box_dt);
  }

private:
  TypeWrapper<T>& m_generic_type;
  std::tuple<TypeWrapper<AppliedTypesT>...> m_specialized_types;
};

namespace detail
{
  template<typename T, bool>
  struct dispatch_set_julia_type;

  // non-parametric
  template<typename T>
  struct dispatch_set_julia_type<T, false>
  {
    void operator()(jl_datatype_t* dt)
    {
      set_julia_type<T>(dt);
    }
  };

  // parametric
  template<typename T>
  struct dispatch_set_julia_type<T, true>
  {
    void operator()(jl_datatype_t*)
    {
    }
  };
}

/// Add a bits type
template<typename T, typename JLSuperT>
void Module::add_bits(const std::string& name, JLSuperT* super)
{
  assert(jl_is_datatype(super));
  static constexpr bool is_parametric = detail::IsParametric<T>::value;
  static_assert(std::is_scalar_v<T>, "Bits types must be a scalar type");
  jl_svec_t* params = is_parametric ? parameter_list<T>()() : jl_emptysvec;
  JL_GC_PUSH1(&params);
  jl_datatype_t* dt = new_bitstype(jl_symbol(name.c_str()), m_jl_mod, (jl_datatype_t*)super, params, 8*sizeof(T));
  protect_from_gc(dt);
  JL_GC_POP();
  detail::dispatch_set_julia_type<T, is_parametric>()(dt);
  set_const(name, (jl_value_t*)dt);
}

/// Registry containing different modules
class JLCXX_API ModuleRegistry
{
public:
  /// Create a module and register it
  Module& create_module(jl_module_t* jmod);

  Module& get_module(jl_module_t* mod) const
  {
    const auto iter = m_modules.find(mod);
    if(iter == m_modules.end())
    {
      throw std::runtime_error("Module with name " + module_name(mod) + " was not found in registry");
    }

    return *(iter->second);
  }

  bool has_module(jl_module_t* jmod) const
  {
    return m_modules.find(jmod) != m_modules.end();
  }

  bool has_current_module() { return m_current_module != nullptr; }
  Module& current_module();
  void reset_current_module() { m_current_module = nullptr; }

private:
  std::map<jl_module_t*, std::shared_ptr<Module>> m_modules;
  Module* m_current_module = nullptr;
};

JLCXX_API ModuleRegistry& registry();

JLCXX_API void register_core_types();
JLCXX_API void register_core_cxxwrap_types();
/// Initialize Julia and the CxxWrap module, optionally taking a path to an environment to load
JLCXX_API void cxxwrap_init(const std::string& envpath = "");

} // namespace jlcxx

/// Register a new module
extern "C" JLCXX_API void register_julia_module(jl_module_t* mod, void (*regfunc)(jlcxx::Module&));

#define JLCXX_MODULE extern "C" JLCXX_ONLY_EXPORTS void

#endif
