# 11 "driver/stage1/Lexer.mll"
 

open Lexing
open Parser
open Keyword
open Attribute
open Located

(* ------------------------------------------------------------------------ *)

(* Failing. *)

(* [blame range format ...] reports an error at [range]. *)

let blame range =
  Report.Just.error [range]

(* [error lexbuf format ...] reports an error at the current token. *)

let error lexbuf =
  blame (Range.current lexbuf)

(* ------------------------------------------------------------------------ *)

(* [int_of_string] raises [Failure] if its argument is too large. This is
   not a problem in practice, but causes false positives when fuzzing
   Menhir. We hide the problem by failing gracefully. *)

let int_of_string range i =
  try
    int_of_string i
  with Failure _ ->
    blame range "unreasonably large integer."

(* ------------------------------------------------------------------------ *)

(* This wrapper saves the current lexeme start, invokes its argument,
   and restores it. This allows transmitting better positions to the
   parser. *)

let savestart lexbuf f =
  let startp = lexbuf.lex_start_p in
  let token = f lexbuf in
  lexbuf.lex_start_p <- startp;
  token

(* ------------------------------------------------------------------------ *)

(* Overwrites an old character with a new one at a specified
   offset in a [bytes] buffer. *)

let overwrite content offset c1 c2 =
  assert (Bytes.get content offset = c1);
  Bytes.set content offset c2

(* ------------------------------------------------------------------------ *)

(* Keyword recognition and construction. *)

(* A monster is a spot where we have identified a keyword in concrete syntax.
   We describe a monster as an object with the following methods: *)

type monster = {

  (* The position of the monster. *)
  range: Range.range;

  (* This method is passed an array of (optional) names for the producers,
     that is, the elements of the production's right-hand side. It is also
     passed a flag which tells whether [$i] syntax is allowed or disallowed.
     It may perform some checks and is allowed to fail. *)
  check: check;

  (* This method transforms the keyword (in place) into a conventional
     OCaml identifier. This is done by replacing '$', '(', and ')' with
     '_'. Bloody. The arguments are [ofs1] and [content]. [ofs1] is the
     offset where [content] begins in the source file. *)
  transform: int -> bytes -> unit;

  (* This is the keyword, in abstract syntax. *)
  keyword: keyword option;

  (* If this is a [$i] monster, then the identifier [_i] is stored here. *)
  oid: string option;

}

and check =
  [`DollarsDisallowed | `DollarsAllowed] ->
  string option array ->
  unit

(* No check. *)

let none : check =
  fun _ _ -> ()

(* ------------------------------------------------------------------------ *)

(* We check that every [$i] is within range. Also, we forbid using [$i]
   when a producer has been given a name; this is bad style and may be
   a mistake. (Plus, this simplifies our life, as we rewrite [$i] to [_i],
   and we would have to rewrite it to a different identifier otherwise.) *)

let check_dollar range i : check = fun dollars producers ->
  (* If [i] is out of range, say so. *)
  if not (0 <= i - 1 && i - 1 < Array.length producers) then
    blame range "$%d refers to a nonexistent symbol." i;
  (* If [$i] could be referred to via a name, say so. *)
  producers.(i - 1) |> Option.iter (fun x ->
    blame range "please do not say: $%d. Instead, say: %s." i x
  );
  (* If [$i] syntax is disallowed, say so. *)
  match dollars with
  | `DollarsDisallowed ->
      blame range "please do not use $%d. Instead, name this value." i
  | `DollarsAllowed ->
      ()

(* We check that every reference to a producer [x] in a position keyword,
   such as [$startpos(x)], exists. *)

let check_producer range x : check = fun _ producers ->
  if not (List.mem (Some x) (Array.to_list producers)) then
    blame range "%s refers to a nonexistent symbol." x

(* ------------------------------------------------------------------------ *)

(* The [$i] monster. *)

let dollar range i : monster =
  let check : check = check_dollar range i
  and transform ofs1 content =
    (* [$i] is replaced with [_i]. Thus, it is no longer a keyword. *)
    let pos = Range.startp range in
    let ofs = pos.pos_cnum - ofs1 in
    overwrite content ofs '$' '_'
  and keyword =
    None
  and oid =
    Some (Printf.sprintf "_%d" i)
  in
  { range; check; transform; keyword; oid }

(* ------------------------------------------------------------------------ *)

(* The position-keyword monster. The most horrible of all. *)

let position range
  (where : string)
  (flavor : string)
  (i : string option) (x : string option)
=
  let check_no_parameter () =
    if i <> None || x <> None then
      blame range "$%s%s does not take a parameter." where flavor
  in
  let ofslpar = (* offset of the opening parenthesis, if there is one *)
    1 + (* for the initial "$" *)
    String.length where +
    3   (* for "pos" or "ofs" or "loc" *)
  in
  let where =
    match where with
    | "symbolstart"
    | "s"           -> check_no_parameter(); WhereSymbolStart
    | "start"       -> WhereStart
    | "end"         -> WhereEnd
    | ""            -> WhereStart
    | _             -> assert false
  in
  let flavor =
    match flavor with
    | "pos"   -> FlavorPosition
    | "ofs"   -> FlavorOffset
    | "loc"   -> FlavorLocation
    | _       -> assert false
  in
  let subject, check =
    match i, x with
    | Some i, None ->
        let ii = int_of_string range i in
        if ii = 0 && where = WhereEnd then
          (* [$endpos($0)] *)
          Before, none
        else
          (* [$startpos($i)] is rewritten to [$startpos(_i)]. *)
          RightNamed ("_" ^ i), check_dollar range ii
    | None, Some x ->
        (* [$startpos(x)] *)
        RightNamed x, check_producer range x
    | None, None ->
        (* [$startpos] *)
        Left, none
    | Some _, Some _ ->
        assert false
  in
  let transform ofs1 content =
    let pos = Range.startp range in
    let ofs = pos.pos_cnum - ofs1 in
    overwrite content ofs '$' '_';
    let ofslpar = ofs + ofslpar in
    match i, x with
    | None, Some x ->
        overwrite content ofslpar '(' '_';
        overwrite content (ofslpar + 1 + String.length x) ')' '_'
    | Some i, None ->
        overwrite content ofslpar '(' '_';
        overwrite content (ofslpar + 1) '$' '_';
        overwrite content (ofslpar + 2 + String.length i) ')' '_'
    | _, _ ->
        ()
  in
  let keyword =
    Some (Position (subject, where, flavor))
  and oid =
    None
  in
  { range; check; transform; keyword; oid }

(* ------------------------------------------------------------------------ *)

(* In an OCaml header, there should be no monsters. This is just a sanity
   check. *)

let no_monsters monsters =
  match monsters with
  | [] ->
      ()
  | monster :: _ ->
      blame monster.range
        "a Menhir keyword cannot be used in an OCaml header."

(* ------------------------------------------------------------------------ *)

(* Gathering all of the identifiers in an array of optional identifiers. *)

let gather_oid xs oid =
  match oid with
  | Some x ->
      StringSet.add x xs
  | None ->
      xs

let gather_oids oids =
  Array.fold_left gather_oid StringSet.empty oids

(* Gathering all of the [oid] identifiers in a list of monsters. *)

let gather_monsters monsters =
  List.fold_left (fun xs monster ->
    gather_oid xs monster.oid
  ) StringSet.empty monsters

(* ------------------------------------------------------------------------ *)

(* Create a fragment. *)

let fragment pos1 pos2 =
  (* Read the specified chunk of the file. *)
  let content = InputFile.chunk (pos1, pos2) in
  (* Construct a fragment. *)
  locate (Range.make (pos1, pos2)) content

(* ------------------------------------------------------------------------ *)

(* Create a semantic action. *)

let transform ofs1 content monsters : string =
  match monsters with
  | [] ->
      content
  | _ :: _ ->
      let content = Bytes.of_string content in
      List.iter (fun monster -> monster.transform ofs1 content) monsters;
      Bytes.unsafe_to_string content

(* This reference can be modified from the outside. Ugly but convenient.
   It determines the priority of the semantic actions that we construct. *)
let priority =
  ref 0

let make_action pos1 pos2 monsters dollars producers =
  (* Check that the monsters are well-formed. *)
  List.iter (fun monster -> monster.check dollars producers) monsters;
  (* Gather all of the identifiers that the semantic action may use to refer
     to a semantic value. This includes the identifiers that are explicitly
     bound by the user (these appear in the array [producers]) and the
     identifiers [_i] when the semantic action uses [$i]. *)
  let ids = StringSet.union (gather_oids producers) (gather_monsters monsters) in
  (* Collect their keywords. *)
  let keywords = MList.filter_map (fun monster -> monster.keyword) monsters in
  (* Read the specified chunk of the file. *)
  let content = InputFile.chunk (pos1, pos2) in
  (* Transform the monsters, if there are any. *)
  let ofs1 = pos1.pos_cnum in
  let content = transform ofs1 content monsters in
  (* Construct a fragment. *)
  let fragment = locate (Range.make (pos1, pos2)) content in
  (* Add parentheses to delimit the semantic action. *)
  let fragment = Located.parenthesize fragment in
  (* Build a semantic action. *)
  Action.make !priority ids keywords (IL.ETextual fragment)

(* ------------------------------------------------------------------------ *)

(* OCaml's reserved words. *)

let table words =
  let table = Hashtbl.create 149 in
  List.iter (fun word -> Hashtbl.add table word ()) words;
  table

let reserved =
  table [
    "and";
    "as";
    "assert";
    "begin";
    "class";
    "constraint";
    "do";
    "done";
    "downto";
    "effect";
    "else";
    "end";
    "exception";
    "external";
    "false";
    "for";
    "fun";
    "function";
    "functor";
    "if";
    "in";
    "include";
    "inherit";
    "initializer";
    "lazy";
    "let";
    "match";
    "method";
    "module";
    "mutable";
    "new";
    "nonrec";
    "object";
    "of";
    "open";
    "or";
    "private";
    "rec";
    "sig";
    "struct";
    "then";
    "to";
    "true";
    "try";
    "type";
    "val";
    "virtual";
    "when";
    "while";
    "with";
    "lor";
    "lxor";
    "mod";
    "land";
    "lsl";
    "lsr";
    "asr";
  ]

(* ------------------------------------------------------------------------ *)

(* Menhir's percent-directives. *)

let table directives =
  let table = Hashtbl.create 149 in
  List.iter (fun (word, token) -> Hashtbl.add table word token) directives;
  table

let directives =
  table [
    "token", TOKEN;
    "type", TYPE;
    "left", LEFT;
    "right", RIGHT;
    "nonassoc", NONASSOC;
    "start", START;
    "prec", PREC;
    "public", PUBLIC;
    "parameter", PARAMETER;
    "inline", INLINE;
    "attribute", PERCENTATTRIBUTE;
    "on_error_reduce", ON_ERROR_REDUCE;
    "merge", MERGE;
  ]


# 404 "driver/stage2/Lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\226\255\227\255\000\000\229\255\231\255\003\000\001\000\
    \235\255\003\000\236\255\209\000\161\001\113\002\065\003\243\255\
    \244\255\245\255\246\255\247\255\248\255\004\000\001\000\251\255\
    \031\000\253\255\019\004\029\000\230\255\254\255\227\004\181\005\
    \163\006\006\000\241\255\240\255\232\255\115\007\067\008\233\255\
    \011\000\004\000\012\000\252\255\253\255\001\000\006\000\255\255\
    \254\255\124\000\250\255\251\255\007\000\252\255\007\000\254\255\
    \032\000\034\000\255\255\253\255\096\001\243\255\244\255\008\000\
    \245\255\247\255\248\255\236\001\009\000\000\000\254\255\255\255\
    \246\255\013\000\017\000\019\000\022\000\096\000\034\000\023\000\
    \019\000\038\000\028\000\032\000\026\000\026\000\057\000\041\000\
    \040\000\046\000\057\000\049\000\046\000\012\000\122\008\081\009\
    \122\000\035\001\251\255\251\255\047\000\066\000\045\000\065\000\
    \053\000\054\000\069\000\067\000\249\255\068\000\067\000\084\000\
    \086\000\086\000\070\000\084\000\079\000\074\000\132\000\168\000\
    \156\000\157\000\161\000\159\000\250\255\000\003\243\255\244\255\
    \009\000\245\255\247\255\248\255\140\003\253\255\254\255\011\000\
    \246\255\186\000\190\000\192\000\242\000\062\001\206\000\224\000\
    \189\000\237\000\225\000\229\000\223\000\233\000\254\000\238\000\
    \237\000\243\000\254\000\246\000\244\000\015\000\136\009\095\010\
    \246\001\007\002\251\255\251\255\245\000\008\001\244\000\019\001\
    \007\001\008\001\012\001\011\001\249\255\012\001\011\001\028\001\
    \030\001\030\001\016\001\033\001\028\001\023\001\026\001\041\001\
    \029\001\030\001\057\001\086\001\250\255\085\011\246\255\247\255\
    \010\000\248\255\250\255\251\255\014\000\253\255\254\255\255\255\
    \249\255\228\002\249\255\250\255\013\000\251\255\252\255\253\255\
    \015\000\179\001\255\255\254\255\188\003\251\255\252\255\016\000\
    \254\255\242\001\255\255\253\255\017\000\142\004\249\255\250\255\
    \018\000\251\255\065\011\255\255\252\255\253\255\254\255\236\002\
    \000\000\214\001\183\001\146\004\184\001\055\005\204\002\215\001\
    \218\001\255\255\010\002\017\002\254\255\011\002\000\006\012\002\
    \161\002\253\255\254\255\019\000\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\029\000\255\255\255\255\021\000\020\000\
    \255\255\029\000\255\255\018\000\017\000\017\000\013\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\005\000\255\255\
    \003\000\255\255\029\000\255\255\255\255\255\255\000\000\255\255\
    \027\000\027\000\255\255\255\255\255\255\017\000\016\000\255\255\
    \255\255\020\000\255\255\255\255\255\255\003\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\005\000\255\255\
    \005\000\005\000\255\255\255\255\255\255\255\255\255\255\010\000\
    \255\255\255\255\255\255\012\000\002\000\012\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\003\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\004\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \010\000\255\255\255\255\255\255\012\000\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\003\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\004\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \007\000\255\255\255\255\255\255\003\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\004\000\255\255\255\255\255\255\
    \006\000\006\000\255\255\255\255\255\255\255\255\255\255\001\000\
    \255\255\004\000\255\255\255\255\001\000\255\255\255\255\255\255\
    \004\000\255\255\006\000\255\255\255\255\255\255\255\255\001\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\000\000\000\000\255\255\255\255\
    \000\000\255\255\000\000\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\000\000\
    \255\255\000\000\255\255\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\255\255\255\255\000\000\
    \040\000\255\255\043\000\000\000\000\000\255\255\255\255\000\000\
    \000\000\050\000\000\000\000\000\255\255\000\000\255\255\000\000\
    \255\255\255\255\000\000\000\000\061\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\126\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\255\255\000\000\000\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\190\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\202\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \255\255\255\255\000\000\000\000\213\000\000\000\000\000\255\255\
    \000\000\219\000\000\000\000\000\255\255\222\000\000\000\000\000\
    \255\255\000\000\228\000\000\000\000\000\000\000\000\000\232\000\
    \255\255\255\255\255\255\236\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \249\000\000\000\000\000\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\008\000\008\000\006\000\007\000\008\000\033\000\
    \047\000\053\000\064\000\129\000\193\000\008\000\047\000\205\000\
    \041\000\046\000\216\000\216\000\225\000\252\000\000\000\000\000\
    \006\000\000\000\010\000\006\000\000\000\026\000\033\000\244\000\
    \021\000\020\000\017\000\016\000\023\000\039\000\036\000\009\000\
    \048\000\059\000\040\000\072\000\094\000\136\000\045\000\158\000\
    \200\000\211\000\024\000\025\000\005\000\022\000\035\000\018\000\
    \031\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\003\000\034\000\031\000\058\000\014\000\
    \058\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\013\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\004\000\019\000\070\000\015\000\113\000\
    \112\000\111\000\074\000\109\000\081\000\080\000\053\000\101\000\
    \082\000\052\000\078\000\083\000\084\000\085\000\086\000\079\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\087\000\088\000\089\000\090\000\091\000\100\000\
    \092\000\093\000\093\000\102\000\054\000\103\000\104\000\105\000\
    \106\000\057\000\097\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\107\000\108\000\110\000\089\000\
    \089\000\093\000\055\000\114\000\115\000\116\000\117\000\118\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\056\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\119\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \002\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\255\255\044\000\120\000\121\000\122\000\
    \123\000\124\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\177\000\176\000\175\000\173\000\
    \011\000\165\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\098\000\145\000\144\000\146\000\
    \147\000\148\000\149\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\097\000\150\000\138\000\151\000\
    \152\000\153\000\154\000\155\000\164\000\156\000\142\000\157\000\
    \157\000\166\000\064\000\143\000\167\000\063\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \168\000\169\000\170\000\171\000\051\000\172\000\174\000\153\000\
    \153\000\157\000\066\000\178\000\067\000\069\000\179\000\065\000\
    \068\000\062\000\180\000\181\000\182\000\183\000\184\000\185\000\
    \186\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \187\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \188\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\071\000\210\000\070\000\241\000\244\000\
    \242\000\242\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\216\000\241\000\241\000\220\000\
    \012\000\241\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\161\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \162\000\241\000\244\000\244\000\000\000\000\000\000\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\245\000\245\000\245\000\245\000\245\000\245\000\245\000\
    \245\000\245\000\245\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\075\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \074\000\000\000\000\000\000\000\073\000\000\000\000\000\076\000\
    \062\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\252\000\000\000\000\000\251\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \012\000\000\000\012\000\012\000\012\000\012\000\037\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\205\000\000\000\
    \000\000\204\000\255\255\244\000\000\000\000\000\234\000\000\000\
    \000\000\233\000\000\000\000\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\207\000\000\000\
    \000\000\000\000\129\000\206\000\208\000\128\000\209\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\131\000\000\000\132\000\000\000\000\000\130\000\
    \135\000\134\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \235\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\133\000\000\000\127\000\000\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \012\000\250\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\216\000\000\000\
    \000\000\215\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\218\000\000\000\
    \000\000\000\000\000\000\000\000\203\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\139\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \138\000\000\000\000\000\000\000\137\000\000\000\000\000\140\000\
    \127\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \217\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \029\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\000\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\027\000\000\000\
    \000\000\000\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\028\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \225\000\000\000\000\000\224\000\240\000\000\000\000\000\239\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \227\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\214\000\000\000\000\000\000\000\
    \000\000\000\000\238\000\238\000\238\000\238\000\238\000\238\000\
    \238\000\238\000\238\000\238\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\226\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\237\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\000\000\
    \000\000\000\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\244\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\246\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\246\000\246\000\
    \246\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\223\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\000\000\000\000\
    \000\000\000\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\032\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \000\000\000\000\000\000\000\000\032\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \247\000\247\000\247\000\247\000\247\000\247\000\247\000\247\000\
    \247\000\247\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\247\000\247\000\247\000\247\000\247\000\247\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\247\000\247\000\247\000\247\000\247\000\247\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\033\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\033\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\000\000\
    \000\000\000\000\032\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\000\000\012\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\038\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\096\000\000\000\
    \000\000\000\000\012\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\000\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\099\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\160\000\000\000\000\000\000\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \000\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\000\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \163\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\000\000\000\000\000\000\000\000\159\000\000\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\193\000\
    \000\000\230\000\192\000\230\000\000\000\000\000\000\000\000\000\
    \230\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\195\000\
    \000\000\000\000\000\000\000\000\194\000\196\000\191\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\230\000\000\000\000\000\
    \000\000\000\000\000\000\230\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\229\000\
    \199\000\000\000\198\000\230\000\000\000\230\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \197\000\000\000\191\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\191\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\007\000\006\000\000\000\041\000\033\000\
    \046\000\052\000\063\000\128\000\192\000\040\000\042\000\204\000\
    \040\000\042\000\215\000\220\000\224\000\251\000\255\255\255\255\
    \000\000\255\255\000\000\006\000\255\255\000\000\033\000\232\000\
    \000\000\000\000\000\000\000\000\000\000\009\000\021\000\000\000\
    \045\000\054\000\009\000\068\000\093\000\135\000\042\000\157\000\
    \196\000\208\000\000\000\000\000\000\000\000\000\022\000\000\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\024\000\027\000\056\000\000\000\
    \057\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\069\000\000\000\073\000\
    \074\000\075\000\076\000\078\000\079\000\079\000\049\000\080\000\
    \081\000\049\000\076\000\082\000\083\000\084\000\085\000\076\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\086\000\087\000\088\000\089\000\089\000\090\000\
    \091\000\092\000\100\000\101\000\049\000\102\000\103\000\104\000\
    \105\000\049\000\096\000\096\000\096\000\096\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\106\000\107\000\109\000\110\000\
    \111\000\112\000\049\000\113\000\114\000\115\000\116\000\117\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\049\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\118\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\040\000\042\000\119\000\120\000\121\000\
    \122\000\123\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\137\000\138\000\139\000\142\000\
    \011\000\144\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\097\000\143\000\143\000\145\000\
    \146\000\147\000\148\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\097\000\149\000\140\000\150\000\
    \151\000\152\000\153\000\153\000\154\000\155\000\140\000\156\000\
    \164\000\165\000\060\000\140\000\166\000\060\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \167\000\168\000\169\000\170\000\049\000\171\000\173\000\174\000\
    \175\000\176\000\060\000\177\000\060\000\060\000\178\000\060\000\
    \060\000\060\000\179\000\180\000\181\000\182\000\183\000\184\000\
    \185\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \186\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \187\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\060\000\209\000\060\000\234\000\236\000\
    \233\000\239\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\217\000\233\000\239\000\217\000\
    \012\000\240\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \161\000\242\000\245\000\247\000\255\255\255\255\255\255\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\067\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \067\000\255\255\255\255\255\255\067\000\255\255\255\255\067\000\
    \060\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \255\255\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \255\255\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\248\000\255\255\255\255\248\000\255\255\
    \255\255\255\255\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\255\255\255\255\255\255\255\255\
    \013\000\255\255\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\255\255\255\255\201\000\255\255\
    \255\255\201\000\217\000\238\000\255\255\255\255\231\000\255\255\
    \255\255\231\000\255\255\255\255\238\000\238\000\238\000\238\000\
    \238\000\238\000\238\000\238\000\238\000\238\000\201\000\255\255\
    \255\255\255\255\125\000\201\000\201\000\125\000\201\000\255\255\
    \255\255\255\255\255\255\231\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\125\000\255\255\125\000\255\255\255\255\125\000\
    \125\000\125\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \231\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \255\255\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\125\000\255\255\125\000\255\255\255\255\
    \255\255\255\255\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\255\255\255\255\255\255\255\255\
    \014\000\248\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\212\000\255\255\
    \255\255\212\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\212\000\255\255\
    \255\255\255\255\255\255\255\255\201\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\231\000\255\255\255\255\255\255\
    \255\255\132\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \132\000\255\255\255\255\255\255\132\000\255\255\255\255\132\000\
    \125\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \212\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \026\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\255\255\255\255\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\255\255\
    \255\255\255\255\026\000\255\255\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \221\000\255\255\255\255\221\000\235\000\255\255\255\255\235\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \221\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\212\000\255\255\255\255\255\255\
    \255\255\255\255\235\000\235\000\235\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\221\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\235\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\255\255\255\255\
    \255\255\255\255\030\000\255\255\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\237\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \237\000\237\000\237\000\237\000\237\000\237\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\221\000\255\255\
    \255\255\255\255\235\000\255\255\255\255\255\255\255\255\255\255\
    \237\000\237\000\237\000\237\000\237\000\237\000\255\255\255\255\
    \255\255\255\255\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\031\000\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\255\255\255\255\255\255\031\000\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\246\000\246\000\
    \246\000\246\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\246\000\246\000\246\000\246\000\246\000\246\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\246\000\246\000\246\000\246\000\246\000\246\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\032\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \255\255\255\255\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\255\255\255\255\
    \255\255\255\255\037\000\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\094\000\255\255\
    \255\255\255\255\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\094\000\255\255\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\255\255\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\158\000\255\255\255\255\255\255\
    \095\000\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \255\255\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\255\255\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \159\000\255\255\255\255\255\255\255\255\255\255\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\255\255\255\255\255\255\255\255\159\000\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\189\000\
    \255\255\226\000\189\000\226\000\255\255\255\255\255\255\255\255\
    \226\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\000\
    \255\255\255\255\255\255\255\255\189\000\189\000\189\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\226\000\255\255\255\255\
    \255\255\255\255\255\255\226\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\226\000\
    \189\000\255\255\189\000\226\000\255\255\226\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \189\000\255\255\189\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\226\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\189\000";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \210\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\000\000\000\000\000\000\010\000\000\000\009\001\217\001\
    \078\000\088\000\052\000\067\000\031\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\031\000\048\000\
    \050\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\035\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\000\000\000\000\000\000\037\000\001\000\016\002\224\002\
    \105\000\115\000\052\000\067\000\058\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\058\000\075\000\
    \077\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \007\000\007\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\027\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\027\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\000\001\000\012\000\000\000\
    \040\000\040\000\000\000\000\000\000\000\000\000\004\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\000\000\000\000\000\000\000\000\004\000\
    \000\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\019\000\019\000\024\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\024\000\019\000\019\000\024\000\019\000\019\000\
    \024\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\024\000\019\000\019\000\
    \024\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\043\000\000\000\000\000\
    \000\000\004\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\000\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\043\000\000\000\000\000\000\000\
    \046\000\000\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\060\000\000\000\125\000\255\255\
    \093\000\157\000\255\255\255\255\255\255\255\255\031\000\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\255\255\255\255\255\255\255\255\031\000\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\067\000\088\000\092\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\096\000\096\000\096\000\096\000\
    \097\000\097\000\097\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\100\000\110\000\111\000\112\000\132\000\152\000\
    \156\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\164\000\174\000\175\000\
    \176\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\094\000\255\255\255\255\
    \255\255\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \094\000\255\255\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \255\255\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\158\000\255\255\255\255\255\255\
    \095\000\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \255\255\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\255\255\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\255\255\255\255\255\255\255\255\159\000\
    \255\255\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \255\255";
  Lexing.lex_code =
   "\255\002\255\255\003\255\255\000\002\001\003\255\010\255\009\255\
    \008\255\255\010\255\009\255\255\011\255\255\007\255\003\255\005\
    \008\004\010\001\009\000\011\255\012\255\255\014\255\255\013\255\
    \255\015\255\255\007\255\005\008\004\010\003\014\002\015\001\009\
    \000\011\255\003\255\007\012\006\013\005\008\004\010\001\009\000\
    \011\255";
}

let rec main lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 4 (-1);(* L=1 [2] <- p ;  *)
  lexbuf.Lexing.lex_mem.(2) <- lexbuf.Lexing.lex_curr_pos ;
 __ocaml_lex_main_rec lexbuf 0
and __ocaml_lex_main_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 452 "driver/stage1/Lexer.mll"
                     directive
# 1667 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 453 "driver/stage1/Lexer.mll"
    ( try Hashtbl.find directives directive
      with Not_found -> error lexbuf "unknown directive: %s." directive )
# 1672 "driver/stage2/Lexer.ml"

  | 1 ->
# 456 "driver/stage1/Lexer.mll"
    ( (* The token [PERCENTPERCENT] carries a fragment that contains
         everything that follows %% in the input file. This string
         must be created lazily. The parser decides (based on the
         context) whether this fragment is needed. If it is indeed
         needed, then constructing this fragment drives the lexer
         to the end of the file. *)
      PERCENTPERCENT (lazy (
        let openingpos = lexeme_end_p lexbuf in
        let closingpos = finish lexbuf in
        fragment openingpos closingpos
      )) )
# 1687 "driver/stage2/Lexer.ml"

  | 2 ->
# 468 "driver/stage1/Lexer.mll"
    ( SEMI )
# 1692 "driver/stage2/Lexer.ml"

  | 3 ->
# 470 "driver/stage1/Lexer.mll"
    ( COLON )
# 1697 "driver/stage2/Lexer.ml"

  | 4 ->
# 472 "driver/stage1/Lexer.mll"
    ( COMMA )
# 1702 "driver/stage2/Lexer.ml"

  | 5 ->
# 474 "driver/stage1/Lexer.mll"
    ( EQUAL )
# 1707 "driver/stage2/Lexer.ml"

  | 6 ->
# 476 "driver/stage1/Lexer.mll"
    ( LPAREN )
# 1712 "driver/stage2/Lexer.ml"

  | 7 ->
# 478 "driver/stage1/Lexer.mll"
    ( RPAREN )
# 1717 "driver/stage2/Lexer.ml"

  | 8 ->
# 480 "driver/stage1/Lexer.mll"
    ( BAR )
# 1722 "driver/stage2/Lexer.ml"

  | 9 ->
# 482 "driver/stage1/Lexer.mll"
    ( QUESTION )
# 1727 "driver/stage2/Lexer.ml"

  | 10 ->
# 484 "driver/stage1/Lexer.mll"
    ( STAR )
# 1732 "driver/stage2/Lexer.ml"

  | 11 ->
# 486 "driver/stage1/Lexer.mll"
    ( PLUS )
# 1737 "driver/stage2/Lexer.ml"

  | 12 ->
# 488 "driver/stage1/Lexer.mll"
    ( TILDE )
# 1742 "driver/stage2/Lexer.ml"

  | 13 ->
# 490 "driver/stage1/Lexer.mll"
    ( UNDERSCORE )
# 1747 "driver/stage2/Lexer.ml"

  | 14 ->
# 492 "driver/stage1/Lexer.mll"
    ( COLONEQUAL )
# 1752 "driver/stage2/Lexer.ml"

  | 15 ->
# 494 "driver/stage1/Lexer.mll"
    ( EQUALEQUAL )
# 1757 "driver/stage2/Lexer.ml"

  | 16 ->
# 496 "driver/stage1/Lexer.mll"
    ( LET )
# 1762 "driver/stage2/Lexer.ml"

  | 17 ->
let
# 497 "driver/stage1/Lexer.mll"
                             id
# 1768 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 498 "driver/stage1/Lexer.mll"
    ( if Hashtbl.mem reserved id then
        error lexbuf "this is an OCaml reserved word."
      else
        LID (locate (Range.current lexbuf) id)
    )
# 1776 "driver/stage2/Lexer.ml"

  | 18 ->
let
# 503 "driver/stage1/Lexer.mll"
                             id
# 1782 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 504 "driver/stage1/Lexer.mll"
    ( UID (locate (Range.current lexbuf) id) )
# 1786 "driver/stage2/Lexer.ml"

  | 19 ->
# 509 "driver/stage1/Lexer.mll"
    ( let buffer = Buffer.create 16 in
      let openingrange = Range.current lexbuf in
      let content = record_string openingrange buffer lexbuf in
      let id = Printf.sprintf "\"%s\"" content in
      let range = Range.make (Range.startp openingrange, lexbuf.lex_curr_p) in
      QID (locate range id) )
# 1796 "driver/stage2/Lexer.ml"

  | 20 ->
# 517 "driver/stage1/Lexer.mll"
    ( new_line lexbuf; main lexbuf )
# 1801 "driver/stage2/Lexer.ml"

  | 21 ->
# 519 "driver/stage1/Lexer.mll"
    ( main lexbuf )
# 1806 "driver/stage2/Lexer.ml"

  | 22 ->
# 521 "driver/stage1/Lexer.mll"
    ( comment (Range.current lexbuf) lexbuf; main lexbuf )
# 1811 "driver/stage2/Lexer.ml"

  | 23 ->
# 523 "driver/stage1/Lexer.mll"
    ( ocamlcomment (Range.current lexbuf) lexbuf; main lexbuf )
# 1816 "driver/stage2/Lexer.ml"

  | 24 ->
# 525 "driver/stage1/Lexer.mll"
    ( savestart lexbuf (angled (Range.current lexbuf)) )
# 1821 "driver/stage2/Lexer.ml"

  | 25 ->
# 527 "driver/stage1/Lexer.mll"
    ( savestart lexbuf (fun lexbuf ->
        let openingrange = Range.current lexbuf in
        let startpos = lexeme_end_p lexbuf in
        let closingpos, monsters = action true openingrange [] lexbuf in
        no_monsters monsters;
        HEADER (fragment startpos closingpos)
      ) )
# 1832 "driver/stage2/Lexer.ml"

  | 26 ->
# 535 "driver/stage1/Lexer.mll"
    ( savestart lexbuf @@ fun lexbuf ->
      let openingrange = Range.current lexbuf in
      let startpos = lexeme_end_p lexbuf in
      let closingpos, monsters = action false openingrange [] lexbuf in
      ACTION (make_action startpos closingpos monsters)
        (* Partial application: [dollars] and [producers] are supplied by the
           parser once they are available. *)
    )
# 1844 "driver/stage2/Lexer.ml"

  | 27 ->
let
# 543 "driver/stage1/Lexer.mll"
           percent
# 1850 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 543 "driver/stage1/Lexer.mll"
                                            key
# 1855 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_mem.(0) + 2) lexbuf.Lexing.lex_mem.(1) in
# 544 "driver/stage1/Lexer.mll"
    ( let openingrange = Range.current lexbuf in
      let startpos = lexeme_end_p lexbuf in
      let closingpos = attribute openingrange lexbuf in
      let origin = Range.make (Range.startp openingrange, lexeme_end_p lexbuf) in
      let payload = InputFile.chunk (startpos, closingpos) in
      let attr : attribute = { key; payload; origin } in
      if percent = "" then
        (* No [%] sign: this is a normal attribute. *)
        ATTRIBUTE attr
      else
        (* A [%] sign is present: this is a grammar-wide attribute. *)
        GRAMMARATTRIBUTE attr
    )
# 1871 "driver/stage2/Lexer.ml"

  | 28 ->
# 558 "driver/stage1/Lexer.mll"
    ( EOF )
# 1876 "driver/stage2/Lexer.ml"

  | 29 ->
# 560 "driver/stage1/Lexer.mll"
    ( error lexbuf "unexpected character(s)." )
# 1881 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_main_rec lexbuf __ocaml_lex_state

and comment openingrange lexbuf =
   __ocaml_lex_comment_rec openingrange lexbuf 42
and __ocaml_lex_comment_rec openingrange lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 568 "driver/stage1/Lexer.mll"
    ( new_line lexbuf; comment openingrange lexbuf )
# 1893 "driver/stage2/Lexer.ml"

  | 1 ->
# 570 "driver/stage1/Lexer.mll"
    ( () )
# 1898 "driver/stage2/Lexer.ml"

  | 2 ->
# 572 "driver/stage1/Lexer.mll"
    ( blame openingrange "unterminated comment." )
# 1903 "driver/stage2/Lexer.ml"

  | 3 ->
# 574 "driver/stage1/Lexer.mll"
    ( comment openingrange lexbuf )
# 1908 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec openingrange lexbuf __ocaml_lex_state

and angled openingrange lexbuf =
   __ocaml_lex_angled_rec openingrange lexbuf 49
and __ocaml_lex_angled_rec openingrange lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 586 "driver/stage1/Lexer.mll"
    ( angled openingrange lexbuf )
# 1920 "driver/stage2/Lexer.ml"

  | 1 ->
# 588 "driver/stage1/Lexer.mll"
    ( let openingpos = Range.endp openingrange
      and closingpos = lexeme_start_p lexbuf in
      let fragment = fragment openingpos closingpos in
      (* Note: no parentheses are inserted. This could be a point-free
         semantic action, which must consist of a single identifier. *)
      ANGLED fragment )
# 1930 "driver/stage2/Lexer.ml"

  | 2 ->
# 595 "driver/stage1/Lexer.mll"
    ( ocamlcomment (Range.current lexbuf) lexbuf;
      angled openingrange lexbuf )
# 1936 "driver/stage2/Lexer.ml"

  | 3 ->
# 598 "driver/stage1/Lexer.mll"
    ( new_line lexbuf; angled openingrange lexbuf )
# 1941 "driver/stage2/Lexer.ml"

  | 4 ->
# 600 "driver/stage1/Lexer.mll"
    ( blame openingrange "unterminated opening bracket." )
# 1946 "driver/stage2/Lexer.ml"

  | 5 ->
# 602 "driver/stage1/Lexer.mll"
    ( angled openingrange lexbuf )
# 1951 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_angled_rec openingrange lexbuf __ocaml_lex_state

and action percent openingrange monsters lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 16 (-1); __ocaml_lex_action_rec percent openingrange monsters lexbuf 60
and __ocaml_lex_action_rec percent openingrange monsters lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 613 "driver/stage1/Lexer.mll"
    ( let _, monsters = action false (Range.current lexbuf) monsters lexbuf in
      action percent openingrange monsters lexbuf )
# 1964 "driver/stage2/Lexer.ml"

  | 1 ->
let
# 615 "driver/stage1/Lexer.mll"
                  delimiter
# 1970 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 616 "driver/stage1/Lexer.mll"
    ( match percent, delimiter with
      | true, "%}"
      | false, "}" ->
          (* This is the delimiter we were instructed to look for. *)
          lexeme_start_p lexbuf, monsters
      | _, _ ->
          (* This is not it. *)
          blame openingrange "unbalanced opening brace."
    )
# 1982 "driver/stage2/Lexer.ml"

  | 2 ->
# 626 "driver/stage1/Lexer.mll"
    ( let _, monsters = parentheses (Range.current lexbuf) monsters lexbuf in
      action percent openingrange monsters lexbuf )
# 1988 "driver/stage2/Lexer.ml"

  | 3 ->
let
# 628 "driver/stage1/Lexer.mll"
                     i
# 1994 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 629 "driver/stage1/Lexer.mll"
    ( let i = int_of_string (Range.current lexbuf) i in
      let monster = dollar (Range.current lexbuf) i in
      action percent openingrange (monster :: monsters) lexbuf )
# 2000 "driver/stage2/Lexer.ml"

  | 4 ->
let
# 430 "driver/stage1/Lexer.mll"
                     i
# 2006 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2)
and
# 431 "driver/stage1/Lexer.mll"
                             x
# 2011 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(7) lexbuf.Lexing.lex_mem.(6)
and
# 436 "driver/stage1/Lexer.mll"
                                          where
# 2016 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(5) lexbuf.Lexing.lex_mem.(4)
and
# 436 "driver/stage1/Lexer.mll"
                                                                     flavor
# 2021 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0) in
# 633 "driver/stage1/Lexer.mll"
    ( let monster = position (Range.current lexbuf) where flavor i x in
      action percent openingrange (monster :: monsters) lexbuf )
# 2026 "driver/stage2/Lexer.ml"

  | 5 ->
# 636 "driver/stage1/Lexer.mll"
    ( error lexbuf "$previouserror is no longer supported." )
# 2031 "driver/stage2/Lexer.ml"

  | 6 ->
# 638 "driver/stage1/Lexer.mll"
    ( error lexbuf "$syntaxerror is no longer supported." )
# 2036 "driver/stage2/Lexer.ml"

  | 7 ->
# 640 "driver/stage1/Lexer.mll"
    ( string (Range.current lexbuf) lexbuf;
      action percent openingrange monsters lexbuf )
# 2042 "driver/stage2/Lexer.ml"

  | 8 ->
# 643 "driver/stage1/Lexer.mll"
    ( char lexbuf;
      action percent openingrange monsters lexbuf )
# 2048 "driver/stage2/Lexer.ml"

  | 9 ->
# 646 "driver/stage1/Lexer.mll"
    ( ocamlcomment (Range.current lexbuf) lexbuf;
      action percent openingrange monsters lexbuf )
# 2054 "driver/stage2/Lexer.ml"

  | 10 ->
# 649 "driver/stage1/Lexer.mll"
    ( new_line lexbuf;
      action percent openingrange monsters lexbuf )
# 2060 "driver/stage2/Lexer.ml"

  | 11 ->
# 653 "driver/stage1/Lexer.mll"
    ( blame openingrange "unbalanced opening brace." )
# 2065 "driver/stage2/Lexer.ml"

  | 12 ->
# 655 "driver/stage1/Lexer.mll"
    ( action percent openingrange monsters lexbuf )
# 2070 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_action_rec percent openingrange monsters lexbuf __ocaml_lex_state

and parentheses openingrange monsters lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 16 (-1); __ocaml_lex_parentheses_rec openingrange monsters lexbuf 125
and __ocaml_lex_parentheses_rec openingrange monsters lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 664 "driver/stage1/Lexer.mll"
    ( let _, monsters = parentheses (Range.current lexbuf) monsters lexbuf in
      parentheses openingrange monsters lexbuf )
# 2083 "driver/stage2/Lexer.ml"

  | 1 ->
# 667 "driver/stage1/Lexer.mll"
    ( lexeme_start_p lexbuf, monsters )
# 2088 "driver/stage2/Lexer.ml"

  | 2 ->
# 669 "driver/stage1/Lexer.mll"
    ( let _, monsters = action false (Range.current lexbuf) monsters lexbuf in
      parentheses openingrange monsters lexbuf )
# 2094 "driver/stage2/Lexer.ml"

  | 3 ->
let
# 671 "driver/stage1/Lexer.mll"
                     i
# 2100 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 672 "driver/stage1/Lexer.mll"
    ( let i = int_of_string (Range.current lexbuf) i in
      let monster = dollar (Range.current lexbuf) i in
      parentheses openingrange (monster :: monsters) lexbuf )
# 2106 "driver/stage2/Lexer.ml"

  | 4 ->
let
# 430 "driver/stage1/Lexer.mll"
                     i
# 2112 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2)
and
# 431 "driver/stage1/Lexer.mll"
                             x
# 2117 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(7) lexbuf.Lexing.lex_mem.(6)
and
# 436 "driver/stage1/Lexer.mll"
                                          where
# 2122 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(5) lexbuf.Lexing.lex_mem.(4)
and
# 436 "driver/stage1/Lexer.mll"
                                                                     flavor
# 2127 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0) in
# 676 "driver/stage1/Lexer.mll"
    ( let monster = position (Range.current lexbuf) where flavor i x in
      parentheses openingrange (monster :: monsters) lexbuf )
# 2132 "driver/stage2/Lexer.ml"

  | 5 ->
# 679 "driver/stage1/Lexer.mll"
    ( error lexbuf "$previouserror is no longer supported." )
# 2137 "driver/stage2/Lexer.ml"

  | 6 ->
# 681 "driver/stage1/Lexer.mll"
    ( error lexbuf "$syntaxerror is no longer supported." )
# 2142 "driver/stage2/Lexer.ml"

  | 7 ->
# 683 "driver/stage1/Lexer.mll"
    ( string (Range.current lexbuf) lexbuf;
      parentheses openingrange monsters lexbuf )
# 2148 "driver/stage2/Lexer.ml"

  | 8 ->
# 686 "driver/stage1/Lexer.mll"
    ( char lexbuf; parentheses openingrange monsters lexbuf )
# 2153 "driver/stage2/Lexer.ml"

  | 9 ->
# 688 "driver/stage1/Lexer.mll"
    ( ocamlcomment (Range.current lexbuf) lexbuf;
      parentheses openingrange monsters lexbuf )
# 2159 "driver/stage2/Lexer.ml"

  | 10 ->
# 691 "driver/stage1/Lexer.mll"
    ( new_line lexbuf;
      parentheses openingrange monsters lexbuf )
# 2165 "driver/stage2/Lexer.ml"

  | 11 ->
# 695 "driver/stage1/Lexer.mll"
    ( blame openingrange "unbalanced opening parenthesis." )
# 2170 "driver/stage2/Lexer.ml"

  | 12 ->
# 697 "driver/stage1/Lexer.mll"
    ( parentheses openingrange monsters lexbuf )
# 2175 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_parentheses_rec openingrange monsters lexbuf __ocaml_lex_state

and attribute openingrange lexbuf =
   __ocaml_lex_attribute_rec openingrange lexbuf 189
and __ocaml_lex_attribute_rec openingrange lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 708 "driver/stage1/Lexer.mll"
    ( let _ = attribute (Range.current lexbuf) lexbuf in
      attribute openingrange lexbuf )
# 2188 "driver/stage2/Lexer.ml"

  | 1 ->
# 711 "driver/stage1/Lexer.mll"
    ( lexeme_start_p lexbuf )
# 2193 "driver/stage2/Lexer.ml"

  | 2 ->
# 713 "driver/stage1/Lexer.mll"
    ( let _, _ = action false (Range.current lexbuf) [] lexbuf in
      attribute openingrange lexbuf )
# 2199 "driver/stage2/Lexer.ml"

  | 3 ->
# 716 "driver/stage1/Lexer.mll"
    ( let _, _ = parentheses (Range.current lexbuf) [] lexbuf in
      attribute openingrange lexbuf )
# 2205 "driver/stage2/Lexer.ml"

  | 4 ->
# 719 "driver/stage1/Lexer.mll"
    ( string (Range.current lexbuf) lexbuf;
      attribute openingrange lexbuf )
# 2211 "driver/stage2/Lexer.ml"

  | 5 ->
# 722 "driver/stage1/Lexer.mll"
    ( char lexbuf; attribute openingrange lexbuf )
# 2216 "driver/stage2/Lexer.ml"

  | 6 ->
# 724 "driver/stage1/Lexer.mll"
    ( ocamlcomment (Range.current lexbuf) lexbuf;
      attribute openingrange lexbuf )
# 2222 "driver/stage2/Lexer.ml"

  | 7 ->
# 727 "driver/stage1/Lexer.mll"
    ( new_line lexbuf;
      attribute openingrange lexbuf )
# 2228 "driver/stage2/Lexer.ml"

  | 8 ->
# 732 "driver/stage1/Lexer.mll"
    ( blame openingrange "unbalanced opening bracket." )
# 2233 "driver/stage2/Lexer.ml"

  | 9 ->
# 734 "driver/stage1/Lexer.mll"
    ( attribute openingrange lexbuf )
# 2238 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_attribute_rec openingrange lexbuf __ocaml_lex_state

and ocamlcomment openingrange lexbuf =
   __ocaml_lex_ocamlcomment_rec openingrange lexbuf 201
and __ocaml_lex_ocamlcomment_rec openingrange lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 746 "driver/stage1/Lexer.mll"
    ( () )
# 2250 "driver/stage2/Lexer.ml"

  | 1 ->
# 748 "driver/stage1/Lexer.mll"
    ( ocamlcomment (Range.current lexbuf) lexbuf;
      ocamlcomment openingrange lexbuf )
# 2256 "driver/stage2/Lexer.ml"

  | 2 ->
# 751 "driver/stage1/Lexer.mll"
    ( string (Range.current lexbuf) lexbuf;
      ocamlcomment openingrange lexbuf )
# 2262 "driver/stage2/Lexer.ml"

  | 3 ->
# 754 "driver/stage1/Lexer.mll"
    ( char lexbuf; ocamlcomment openingrange lexbuf )
# 2267 "driver/stage2/Lexer.ml"

  | 4 ->
# 756 "driver/stage1/Lexer.mll"
    ( new_line lexbuf;
      ocamlcomment openingrange lexbuf )
# 2273 "driver/stage2/Lexer.ml"

  | 5 ->
# 759 "driver/stage1/Lexer.mll"
    ( blame openingrange "unterminated OCaml comment." )
# 2278 "driver/stage2/Lexer.ml"

  | 6 ->
# 761 "driver/stage1/Lexer.mll"
    ( ocamlcomment openingrange lexbuf )
# 2283 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_ocamlcomment_rec openingrange lexbuf __ocaml_lex_state

and string openingrange lexbuf =
   __ocaml_lex_string_rec openingrange lexbuf 212
and __ocaml_lex_string_rec openingrange lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 769 "driver/stage1/Lexer.mll"
    ( () )
# 2295 "driver/stage2/Lexer.ml"

  | 1 ->
# 772 "driver/stage1/Lexer.mll"
    ( new_line lexbuf;
      string openingrange lexbuf )
# 2301 "driver/stage2/Lexer.ml"

  | 2 ->
# 777 "driver/stage1/Lexer.mll"
    ( string openingrange lexbuf )
# 2306 "driver/stage2/Lexer.ml"

  | 3 ->
# 779 "driver/stage1/Lexer.mll"
    ( blame openingrange "unterminated OCaml string." )
# 2311 "driver/stage2/Lexer.ml"

  | 4 ->
# 781 "driver/stage1/Lexer.mll"
    ( string openingrange lexbuf )
# 2316 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_rec openingrange lexbuf __ocaml_lex_state

and record_string openingrange buffer lexbuf =
   __ocaml_lex_record_string_rec openingrange buffer lexbuf 221
and __ocaml_lex_record_string_rec openingrange buffer lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 789 "driver/stage1/Lexer.mll"
    ( Buffer.contents buffer )
# 2328 "driver/stage2/Lexer.ml"

  | 1 ->
let
# 790 "driver/stage1/Lexer.mll"
                                            sequence
# 2334 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 791 "driver/stage1/Lexer.mll"
    ( (* This escape sequence is recognized as such, but not decoded. *)
      Buffer.add_string buffer sequence;
      record_string openingrange buffer lexbuf )
# 2340 "driver/stage2/Lexer.ml"

  | 2 ->
# 799 "driver/stage1/Lexer.mll"
    ( error lexbuf "'\\n' is not permitted in a token alias." )
# 2345 "driver/stage2/Lexer.ml"

  | 3 ->
# 801 "driver/stage1/Lexer.mll"
    ( error lexbuf "illegal backslash escape in string." )
# 2350 "driver/stage2/Lexer.ml"

  | 4 ->
# 803 "driver/stage1/Lexer.mll"
    ( error lexbuf "illegal newline in string." )
# 2355 "driver/stage2/Lexer.ml"

  | 5 ->
# 805 "driver/stage1/Lexer.mll"
    ( blame openingrange "unterminated string." )
# 2360 "driver/stage2/Lexer.ml"

  | 6 ->
let
# 806 "driver/stage1/Lexer.mll"
       c
# 2366 "driver/stage2/Lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 807 "driver/stage1/Lexer.mll"
    ( Buffer.add_char buffer c;
      record_string openingrange buffer lexbuf )
# 2371 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_record_string_rec openingrange buffer lexbuf __ocaml_lex_state

and char lexbuf =
   __ocaml_lex_char_rec lexbuf 231
and __ocaml_lex_char_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 818 "driver/stage1/Lexer.mll"
   ( new_line lexbuf )
# 2383 "driver/stage2/Lexer.ml"

  | 1 ->
# 824 "driver/stage1/Lexer.mll"
   ( () )
# 2388 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_char_rec lexbuf __ocaml_lex_state

and finish lexbuf =
   __ocaml_lex_finish_rec lexbuf 248
and __ocaml_lex_finish_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 835 "driver/stage1/Lexer.mll"
    ( new_line lexbuf; finish lexbuf )
# 2400 "driver/stage2/Lexer.ml"

  | 1 ->
# 837 "driver/stage1/Lexer.mll"
    ( lexeme_start_p lexbuf )
# 2405 "driver/stage2/Lexer.ml"

  | 2 ->
# 839 "driver/stage1/Lexer.mll"
    ( finish lexbuf )
# 2410 "driver/stage2/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_finish_rec lexbuf __ocaml_lex_state

;;

