/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.panoply.task;

import gov.nasa.giss.data.nc.NcArray;
import gov.nasa.giss.data.nc.NcDimension;
import gov.nasa.giss.img.GImageIO;
import gov.nasa.giss.panoply.data.PanData;
import gov.nasa.giss.panoply.plot.PanPlot;
import gov.nasa.giss.panoply.plot.PanPlotMeta;
import gov.nasa.giss.panoply.plotui.PanPlotFrame;
import gov.nasa.giss.panoply.util.PanAnimationFormat;
import gov.nasa.giss.panoply.util.PanConstants;
import gov.nasa.giss.panoply.util.PanExportAnimationOption;
import gov.nasa.giss.text.PrintfFormat;
import gov.nasa.giss.ui.SlowMessage;
import gov.nasa.giss.ui.SlowMessageListener;
import gov.nasa.giss.util.task.Task;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.util.Hashtable;
import org.jcodec.api.awt.AWTSequenceEncoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PanExportAnimationTask
extends Task
implements SlowMessageListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final PrintfFormat PFORMAT_5D = new PrintfFormat("%05d");
    private static final String ANIM_ERR = "Animation Error";
    private static final String TERMINATED = "The animation has been terminated.";
    private final PanPlotFrame pframe_;
    private final File file_;
    private final Hashtable<PanExportAnimationOption, Object> options_;
    private final NcDimension[] dimensions_;
    private final PanAnimationFormat format_;
    private PanPlotMeta pmeta_;
    private PanData pdata_;
    private int[] sliceDims_;
    private int[] sliceIndices_;
    private int[] dimlengths_;
    private final boolean showSubtitle_;
    private final boolean subBounds_;
    private File oldFile_;
    private BufferedImage bimage_;

    public PanExportAnimationTask(File file, PanPlotFrame pframe, NcDimension[] dimensions, Hashtable<PanExportAnimationOption, Object> options) {
        super("Exporting plot animation");
        this.file_ = file;
        this.pframe_ = pframe;
        this.dimensions_ = dimensions;
        this.options_ = options;
        this.format_ = (PanAnimationFormat)options.get((Object)PanExportAnimationOption.FORMAT);
        this.showSubtitle_ = (Boolean)options.get((Object)PanExportAnimationOption.SUBTITLE_STEP);
        this.subBounds_ = (Boolean)options.get((Object)PanExportAnimationOption.SUBTITLE_BOUNDS);
    }

    @Override
    protected Object beginTask() {
        if (this.isCancelled()) {
            return null;
        }
        if (!this.pframe_.isVisible()) {
            LOGGER.debug("Frame no longer visible; canceling.");
            this.cancel(ANIM_ERR, "Plot window no longer visible. The animation has been terminated.");
            return null;
        }
        String oldSubtitle = this.pframe_.getMeta().getString("subtitle.text");
        boolean successful = false;
        try {
            if (this.file_.exists()) {
                this.oldFile_ = new File(this.file_.getParent(), this.file_.getName() + "_old");
                this.file_.renameTo(this.oldFile_);
            }
        }
        catch (Exception exc) {
            LOGGER.warn("Failed to move existing file to temp address: {}", (Object)exc.toString());
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            this.cancel(ANIM_ERR, "Failed to save animation to a file. " + exc.toString());
        }
        try {
            successful = this.writeAnimation();
        }
        catch (NullPointerException exc) {
            LOGGER.warn("Null pointer exception");
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            this.cancel(ANIM_ERR, "Null pointer exception occurred while writing animation.");
        }
        catch (Exception exc) {
            LOGGER.warn("Uncaught exception writing animation: {}", (Object)exc.toString());
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            this.cancel(ANIM_ERR, "Could not save animation to a file: " + exc.toString());
        }
        if (successful) {
            if (this.oldFile_ != null && this.oldFile_.exists()) {
                this.oldFile_.delete();
            }
        } else {
            if (this.file_.exists()) {
                this.file_.delete();
            }
            if (this.oldFile_ != null && this.oldFile_.exists()) {
                this.oldFile_.renameTo(this.file_);
            }
        }
        this.pframe_.getMeta().setString("subtitle.text", oldSubtitle);
        return null;
    }

    @Override
    protected void finishTask() {
        this.pframe_.setAnimating(false);
        PanPlotFrame.setControlWindoidsEnabled(true);
        this.pframe_.setControlWindoidsVisible(true);
    }

    private boolean writeAnimation() throws InterruptedException, IOException {
        if (this.isCancelled()) {
            return false;
        }
        if (this.format_ == null) {
            throw new RuntimeException("Null format");
        }
        this.pdata_ = this.pframe_.getData();
        this.pmeta_ = this.pframe_.getPlot().getMeta();
        Dimension psize = this.pmeta_.getLayout().getDimension("size.pixels");
        int frameCount = 1;
        this.sliceDims_ = new int[]{-9999, -9999};
        this.sliceIndices_ = new int[]{-9999, -9999};
        this.dimlengths_ = new int[]{-1, -1};
        for (int i = 0; i < 2; ++i) {
            if (this.dimensions_[i] == null) continue;
            NcArray array = this.pframe_.getData().getArrays()[i];
            this.sliceDims_[i] = array.getDimensionId(this.dimensions_[i]);
            this.sliceIndices_[i] = this.pdata_.getSlice(i, this.sliceDims_[i]);
            this.dimlengths_[i] = this.dimensions_[i].getLength();
            frameCount = Math.max(frameCount, this.dimlengths_[i]);
        }
        this.bimage_ = new BufferedImage(psize.width, psize.height, 1);
        boolean success = true;
        switch (this.format_) {
            case MP4: {
                success = this.writeVideo();
                break;
            }
            case PNG: 
            case JPEG: 
            case TIFF: {
                success = this.writeStills();
                break;
            }
            default: {
                throw new RuntimeException("Unknown animation format.");
            }
        }
        this.cleanup();
        return success;
    }

    private boolean writeVideo() throws IOException {
        int frameRate = (Integer)this.options_.get((Object)PanExportAnimationOption.FRAME_RATE);
        int startIndex = (Integer)this.options_.get((Object)PanExportAnimationOption.START_INDEX);
        int stopIndex = (Integer)this.options_.get((Object)PanExportAnimationOption.STOP_INDEX);
        int stride = (Integer)this.options_.get((Object)PanExportAnimationOption.STRIDE);
        AWTSequenceEncoder encoder = AWTSequenceEncoder.createSequenceEncoder((File)this.file_, (int)frameRate);
        for (int i = startIndex - 1; i < stopIndex; i += stride) {
            if (this.isCancelled()) {
                this.cleanup();
                return false;
            }
            boolean good = this.writeFrame(this.bimage_, i);
            if (good) {
                try {
                    encoder.encodeImage(this.bimage_);
                    continue;
                }
                catch (Exception exc) {
                    if (!this.isCancelled()) {
                        LOGGER.error("Failed to encode image frame due to exception. Canceling.");
                        if (LOGGER.isTraceEnabled()) {
                            exc.printStackTrace();
                        }
                        this.cancel(ANIM_ERR, "Failed to encode image frame due to exception. The animation has been terminated.");
                    }
                    this.cleanup();
                    return false;
                }
            }
            if (!this.isCancelled()) {
                LOGGER.error("Failed to generate video frame. Canceling.");
                this.cancel(ANIM_ERR, "Failed to encode image frame due to exception. The animation has been terminated.");
                this.cleanup();
            }
            return false;
        }
        encoder.finish();
        return true;
    }

    private boolean writeStills() throws IOException {
        Object fname = this.file_.getName();
        int lastDot = ((String)fname).lastIndexOf(46);
        String froot = ((String)fname).substring(0, lastDot);
        String fext = ((String)fname).substring(lastDot);
        int quality = (Integer)this.options_.get((Object)PanExportAnimationOption.QUALITY);
        float qualityF = quality < 95 ? 0.01f * (float)quality : 1.0f;
        int startIndex = (Integer)this.options_.get((Object)PanExportAnimationOption.START_INDEX);
        int stopIndex = (Integer)this.options_.get((Object)PanExportAnimationOption.STOP_INDEX);
        int stride = (Integer)this.options_.get((Object)PanExportAnimationOption.STRIDE);
        for (int i = startIndex - 1; i < stopIndex; i += stride) {
            if (this.isCancelled()) {
                this.cleanup();
                return false;
            }
            boolean good = this.writeFrame(this.bimage_, i);
            if (!good) {
                if (!this.isCancelled()) {
                    LOGGER.warn("Failed to write still frame. Canceling.");
                    this.cancel(ANIM_ERR, "Failed to write still frame. The animation has been terminated.");
                    this.cleanup();
                }
                return false;
            }
            if (this.format_ != PanAnimationFormat.PNG && this.format_ != PanAnimationFormat.JPEG && this.format_ != PanAnimationFormat.TIFF) continue;
            fname = froot + "_" + PFORMAT_5D.sprintfx(i + 1) + fext;
            File f = new File(this.file_.getParent(), (String)fname);
            if (this.format_ == PanAnimationFormat.PNG) {
                GImageIO.saveAsPng(f, this.bimage_);
                continue;
            }
            if (this.format_ == PanAnimationFormat.JPEG) {
                GImageIO.saveAsJpeg(f, this.bimage_, qualityF);
                continue;
            }
            if (this.format_ != PanAnimationFormat.TIFF) continue;
            GImageIO.saveAsTiff(f, this.bimage_, PanConstants.SOFTWARE);
        }
        return true;
    }

    private boolean writeFrame(BufferedImage bimage, int findex) {
        PanPlot plot;
        String fbds;
        String fval;
        boolean fromBounds;
        if (!this.pframe_.isVisible()) {
            LOGGER.debug("Plot window closed; canceling animation.");
            this.cancel(ANIM_ERR, "Plot window no longer visible. The animation has been terminated.");
            return false;
        }
        String s0 = null;
        String s1 = null;
        if (this.dimlengths_[0] > findex) {
            this.pdata_.setSlice(0, this.sliceDims_[0], findex);
            fromBounds = this.subBounds_ && this.dimensions_[0].hasBounds();
            fval = this.dimensions_[0].formattedValueAt(findex);
            fbds = this.dimensions_[0].formattedBoundsAt(findex);
            s0 = this.dimensions_[0].getLongName() + ": " + (fromBounds ? fbds : fval);
        }
        if (this.dimlengths_[1] > findex) {
            this.pdata_.setSlice(1, this.sliceDims_[1], findex);
            fromBounds = this.subBounds_ && this.dimensions_[1].hasBounds();
            fval = this.dimensions_[1].formattedValueAt(findex);
            fbds = this.dimensions_[1].formattedBoundsAt(findex);
            s1 = this.dimensions_[1].getLongName() + ": " + (fromBounds ? fbds : fval);
        }
        if (this.showSubtitle_) {
            if (s0 != null && s1 != null) {
                this.pmeta_.setString("subtitle.text", s0 + " : " + s1);
            } else if (s0 != null) {
                this.pmeta_.setString("subtitle.text", s0);
            } else if (s1 != null) {
                this.pmeta_.setString("subtitle.text", s1);
            } else {
                this.pmeta_.setString("subtitle.text", "");
            }
        }
        if ((plot = this.pframe_.getPlot()) == null) {
            if (!this.isCancelled()) {
                LOGGER.warn("Plot object is null. Skipping frame write.");
            }
            return false;
        }
        try {
            plot.paint(bimage);
        }
        catch (NullPointerException npe) {
            this.cancel(ANIM_ERR, "Null pointer exception occurred while writing animation.");
        }
        try {
            if (this.pframe_.isVisible()) {
                this.pframe_.getPlotHolder().repaint();
            }
        }
        catch (NullPointerException npe) {
            this.cancel();
        }
        return true;
    }

    private void cleanup() {
        PanPlot plot;
        if (this.sliceDims_[0] > -9999) {
            this.pdata_.setSlice(0, this.sliceDims_[0], this.sliceIndices_[0]);
        }
        if (this.sliceDims_[1] > -9999) {
            this.pdata_.setSlice(1, this.sliceDims_[1], this.sliceIndices_[1]);
        }
        if ((plot = this.pframe_.getPlot()) == null) {
            if (!this.isCancelled()) {
                LOGGER.warn("Frame says plot is null. Terminating clean-up.");
            }
            return;
        }
        this.pframe_.getPlot().dataChanged(null);
        EventQueue.invokeLater(() -> {
            if (this.pframe_ == null) {
                LOGGER.trace("pframe_ is null?");
                return;
            }
            this.pframe_.getPlotHolder().repaint();
        });
    }

    public void cancel() {
        this.cancel("Animation Canceled", "The animation was canceled before completion.");
    }

    public void cancel(String title, String message) {
        super.cancel(true);
        if (this.pframe_ != null && this.pframe_.isVisible()) {
            EventQueue.invokeLater(() -> this.pframe_.setAnimating(false));
        }
        SlowMessage.show(this.pframe_, title, message, this);
    }

    @Override
    public void slowMessageAcknowledged() {
        if (this.pframe_ == null) {
            LOGGER.trace("pframe_ is null?");
            return;
        }
        PanPlotFrame.setControlWindoidsEnabled(true);
        if (this.pframe_.isVisible()) {
            EventQueue.invokeLater(() -> this.pframe_.setWindoidsVisible(true));
        }
    }
}

