/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.retry;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;
import org.springframework.core.log.LogAccessor;
import org.springframework.core.retry.RetryException;
import org.springframework.core.retry.RetryListener;
import org.springframework.core.retry.RetryOperations;
import org.springframework.core.retry.RetryPolicy;
import org.springframework.core.retry.RetryState;
import org.springframework.core.retry.Retryable;
import org.springframework.util.Assert;
import org.springframework.util.backoff.BackOffExecution;

public class RetryTemplate
implements RetryOperations {
    private static final LogAccessor logger = new LogAccessor(RetryTemplate.class);
    private RetryPolicy retryPolicy = RetryPolicy.withDefaults();
    private RetryListener retryListener = new RetryListener(){};

    public RetryTemplate() {
    }

    public RetryTemplate(RetryPolicy retryPolicy) {
        Assert.notNull((Object)retryPolicy, "RetryPolicy must not be null");
        this.retryPolicy = retryPolicy;
    }

    public void setRetryPolicy(RetryPolicy retryPolicy) {
        Assert.notNull((Object)retryPolicy, "RetryPolicy must not be null");
        this.retryPolicy = retryPolicy;
    }

    public RetryPolicy getRetryPolicy() {
        return this.retryPolicy;
    }

    public void setRetryListener(RetryListener retryListener) {
        Assert.notNull((Object)retryListener, "Retry listener must not be null");
        this.retryListener = retryListener;
    }

    public RetryListener getRetryListener() {
        return this.retryListener;
    }

    @Override
    public <R> R execute(Retryable<R> retryable) throws RetryException {
        R result;
        long startTime = System.currentTimeMillis();
        String retryableName = retryable.getName();
        MutableRetryState retryState = new MutableRetryState();
        logger.debug(() -> "Preparing to execute retryable operation '%s'".formatted(retryableName));
        try {
            result = retryable.execute();
        }
        catch (Throwable initialException) {
            logger.debug(initialException, () -> "Execution of retryable operation '%s' failed; initiating the retry process".formatted(retryableName));
            retryState.addException(initialException);
            this.retryListener.onRetryableExecution(this.retryPolicy, retryable, retryState);
            BackOffExecution backOffExecution = this.retryPolicy.getBackOff().start();
            Throwable lastException = initialException;
            long timeout = this.retryPolicy.getTimeout().toMillis();
            while (this.retryPolicy.shouldRetry(lastException) && retryState.getRetryCount() < Integer.MAX_VALUE) {
                R result2;
                this.checkIfTimeoutExceeded(timeout, startTime, 0L, retryable, retryState);
                try {
                    long sleepTime = backOffExecution.nextBackOff();
                    if (sleepTime == -1L) break;
                    this.checkIfTimeoutExceeded(timeout, startTime, sleepTime, retryable, retryState);
                    logger.debug(() -> "Backing off for %dms after retryable operation '%s'".formatted(sleepTime, retryableName));
                    Thread.sleep(sleepTime);
                }
                catch (InterruptedException interruptedException) {
                    Thread.currentThread().interrupt();
                    RetryException retryException = new RetryException("Interrupted during back-off for " + "retryable operation '%s'; aborting execution".formatted(retryableName), retryState);
                    this.retryListener.onRetryPolicyInterruption(this.retryPolicy, retryable, retryException);
                    throw retryException;
                }
                logger.debug(() -> "Preparing to retry operation '%s'".formatted(retryableName));
                retryState.increaseRetryCount();
                this.retryListener.beforeRetry(this.retryPolicy, retryable, retryState);
                try {
                    result2 = retryable.execute();
                }
                catch (Throwable currentException) {
                    logger.debug(currentException, () -> "Retry attempt for operation '%s' failed due to '%s'".formatted(retryableName, currentException));
                    retryState.addException(currentException);
                    this.retryListener.onRetryFailure(this.retryPolicy, retryable, currentException);
                    this.retryListener.onRetryableExecution(this.retryPolicy, retryable, retryState);
                    lastException = currentException;
                    continue;
                }
                logger.debug(() -> "Retryable operation '%s' completed successfully after retry".formatted(retryableName));
                this.retryListener.onRetrySuccess(this.retryPolicy, retryable, result2);
                this.retryListener.onRetryableExecution(this.retryPolicy, retryable, retryState);
                return result2;
            }
            RetryException retryException = new RetryException("Retry policy for operation '%s' exhausted; aborting execution".formatted(retryableName), retryState);
            this.retryListener.onRetryPolicyExhaustion(this.retryPolicy, retryable, retryException);
            throw retryException;
        }
        logger.debug(() -> "Retryable operation '%s' completed successfully".formatted(retryableName));
        this.retryListener.onRetryableExecution(this.retryPolicy, retryable, retryState);
        return result;
    }

    private void checkIfTimeoutExceeded(long timeout, long startTime, long sleepTime, Retryable<?> retryable, RetryState retryState) throws RetryException {
        long elapsedTime;
        if (timeout > 0L && (elapsedTime = System.currentTimeMillis() + sleepTime - startTime) >= timeout) {
            String message = sleepTime > 0L ? "Retry policy for operation '%s' would exceed timeout (%dms) due to pending sleep time (%dms); preemptively aborting execution".formatted(retryable.getName(), timeout, sleepTime) : "Retry policy for operation '%s' exceeded timeout (%dms); aborting execution".formatted(retryable.getName(), timeout);
            RetryException retryException = new RetryException(message, retryState);
            this.retryListener.onRetryPolicyTimeout(this.retryPolicy, retryable, retryException);
            throw retryException;
        }
    }

    @Override
    public <R> R invoke(final Supplier<R> retryable) {
        try {
            return this.execute(new Retryable<R>(this){
                final /* synthetic */ RetryTemplate this$0;
                {
                    this.this$0 = this$0;
                }

                @Override
                public R execute() {
                    return retryable.get();
                }

                @Override
                public String getName() {
                    return retryable.getClass().getName();
                }
            });
        }
        catch (RetryException retryException) {
            Throwable ex = retryException.getCause();
            if (ex instanceof RuntimeException) {
                RuntimeException runtimeException = (RuntimeException)ex;
                throw runtimeException;
            }
            if (ex instanceof Error) {
                Error error = (Error)ex;
                throw error;
            }
            throw new UndeclaredThrowableException(ex);
        }
    }

    @Override
    public void invoke(final Runnable retryable) {
        try {
            this.execute(new Retryable<Object>(this){
                final /* synthetic */ RetryTemplate this$0;
                {
                    this.this$0 = this$0;
                }

                @Override
                public Void execute() {
                    retryable.run();
                    return null;
                }

                @Override
                public String getName() {
                    return retryable.getClass().getName();
                }
            });
        }
        catch (RetryException retryException) {
            Throwable ex = retryException.getCause();
            if (ex instanceof RuntimeException) {
                RuntimeException runtimeException = (RuntimeException)ex;
                throw runtimeException;
            }
            if (ex instanceof Error) {
                Error error = (Error)ex;
                throw error;
            }
            throw new UndeclaredThrowableException(ex);
        }
    }

    private static class MutableRetryState
    implements RetryState {
        private int retryCount;
        private final List<Throwable> exceptions = new ArrayList<Throwable>(4);

        private MutableRetryState() {
        }

        public void increaseRetryCount() {
            ++this.retryCount;
        }

        @Override
        public int getRetryCount() {
            return this.retryCount;
        }

        public void addException(Throwable exception) {
            this.exceptions.add(exception);
        }

        @Override
        public List<Throwable> getExceptions() {
            return Collections.unmodifiableList(this.exceptions);
        }

        public String toString() {
            return "RetryState: retryCount=" + this.retryCount + ", exceptions=" + String.valueOf(this.exceptions);
        }
    }
}

