/**
 *  @brief  Unicode TR29 Word Boundary detection for UTF-8 text.
 *  @file   utf8_word.h
 *  @author Ash Vardanian
 *
 *  Includes core APIs:
 *
 *  - `sz_rune_word_break_property` - get TR29 Word_Break property for a codepoint (16 values)
 *  - `sz_utf8_word_find_boundary` - find next word boundary position (TR29 compliant)
 *  - `sz_utf8_word_rfind_boundary` - find previous word boundary position
 *
 *  StringZilla implements full Unicode TR29 word boundary detection:
 *  - 16 Word_Break property values encoded in 4 bits
 *  - ASCII: 128-byte direct table for O(1) lookup
 *  - BMP: Two-stage table with nibble-packed blocks (~6.4 KB)
 *  - SMP: Transition-based encoding (~3.4 KB)
 *  Total table size is approximately 10 KB, providing 100% TR29 compliance.
 */
#ifndef STRINGZILLA_UTF8_WORD_H_
#define STRINGZILLA_UTF8_WORD_H_

#include "types.h"

#ifdef __cplusplus
extern "C" {
#endif

#pragma region Word_Break Property Encoding

/**
 *  @brief Unicode TR29 Word_Break property values (4-bit encoding, 0-15).
 *
 *  These values correspond to the Word_Break property from Unicode TR29.
 *  Used by `sz_rune_word_break_property()` for full TR29-compliant boundary detection.
 */
typedef enum sz_tr29_word_break_t {
    sz_tr29_word_break_other_k = 0,         /**< Default - creates word boundary */
    sz_tr29_word_break_cr_k = 1,            /**< Carriage Return (U+000D) */
    sz_tr29_word_break_lf_k = 2,            /**< Line Feed (U+000A) */
    sz_tr29_word_break_newline_k = 3,       /**< Other newlines (VT, FF, NEL, LS, PS) */
    sz_tr29_word_break_extend_k = 4,        /**< Combining marks (Mn, Me, Mc) */
    sz_tr29_word_break_zwj_k = 5,           /**< Zero Width Joiner (U+200D) */
    sz_tr29_word_break_format_k = 6,        /**< Format characters (Cf) */
    sz_tr29_word_break_regional_ind_k = 7,  /**< Regional Indicator (U+1F1E6-U+1F1FF) */
    sz_tr29_word_break_aletter_k = 8,       /**< Alphabetic letters */
    sz_tr29_word_break_hebrew_letter_k = 9, /**< Hebrew script letters */
    sz_tr29_word_break_numeric_k = 10,      /**< Digits (0-9 and other scripts) */
    sz_tr29_word_break_katakana_k = 11,     /**< Japanese Katakana */
    sz_tr29_word_break_extendnumlet_k = 12, /**< Underscore, connector punctuation */
    sz_tr29_word_break_midletter_k = 13,    /**< Mid-letter punctuation (colon, etc.) */
    sz_tr29_word_break_midnum_k = 14,       /**< Mid-number punctuation (comma, etc.) */
    sz_tr29_word_break_mid_quotes_k = 15,   /**< MidNumLet + Single_Quote + Double_Quote */
} sz_tr29_word_break_t;

#pragma endregion

#pragma region Core API

/**
 *  @brief  Get the Unicode TR29 Word_Break property for a codepoint.
 *
 *  Returns one of the 16 Word_Break property values (sz_tr29_word_break_other_k through
 *  sz_tr29_word_break_mid_quotes_k). This is the foundation for TR29-compliant word boundary detection.
 *
 *  @param[in] rune The Unicode codepoint to classify.
 *  @return The Word_Break property value (0-15).
 *
 *  @see https://www.unicode.org/reports/tr29/ - Unicode Text Segmentation
 */
SZ_PUBLIC sz_u8_t sz_rune_word_break_property(sz_rune_t rune);

/**
 *  @brief  Check if a codepoint is a "word character" (has word-forming property).
 *
 *  Returns true if the codepoint has a Word_Break property that typically forms words:
 *  ALetter, Hebrew_Letter, Numeric, Katakana, ExtendNumLet, or mid-word punctuation.
 *
 *  @param[in] rune The Unicode codepoint to check.
 *  @return sz_true_k if the codepoint is a word character, sz_false_k otherwise.
 */
SZ_PUBLIC sz_bool_t sz_rune_is_word_char(sz_rune_t rune);

/**
 *  @brief  Find the next word boundary in UTF-8 text (dispatch function).
 *
 *  Scans forward from the start of text to find the first word boundary position.
 *  Returns a pointer to the boundary position and optionally outputs the boundary width.
 *
 *  @param[in] text UTF-8 encoded text.
 *  @param[in] length Byte length of text.
 *  @param[out] boundary_width Optional output: bytes spanning the boundary.
 *  @return Pointer to boundary position, or text+length at end.
 */
SZ_DYNAMIC sz_cptr_t sz_utf8_word_find_boundary(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width);

/**
 *  @brief  Find the previous word boundary in UTF-8 text (dispatch function).
 *
 *  @param[in] text UTF-8 encoded text.
 *  @param[in] length Byte length of text.
 *  @param[out] boundary_width Optional output: bytes spanning the boundary.
 *  @return Pointer to boundary position, or text at start.
 */
SZ_DYNAMIC sz_cptr_t sz_utf8_word_rfind_boundary(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width);

/**
 *  @brief  Check if a position in UTF-8 text is a word boundary per Unicode TR29.
 *
 *  Implements the full TR29 word boundary algorithm including:
 *  - WB3: Do not break between CR and LF
 *  - WB4: Ignore Extend/Format/ZWJ characters for boundary purposes
 *  - WB5-WB13: Letter, number, and punctuation rules
 *  - WB15-WB16: Regional Indicator pair rules
 *
 *  @param[in] text UTF-8 encoded text.
 *  @param[in] length Byte length of text.
 *  @param[in] pos Byte offset to check (must be start of a UTF-8 codepoint).
 *  @return sz_true_k if pos is a word boundary, sz_false_k otherwise.
 *
 *  @note Position 0 and position == length are always boundaries (SOT/EOT).
 *  @note This is an internal helper used by the iterators; not part of stable ABI.
 */
SZ_PUBLIC sz_bool_t sz_utf8_is_word_boundary_serial(sz_cptr_t text, sz_size_t length, sz_size_t pos);

/*  Serial implementations */
SZ_PUBLIC sz_cptr_t sz_utf8_word_find_boundary_serial(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width);
SZ_PUBLIC sz_cptr_t sz_utf8_word_rfind_boundary_serial(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width);

/*  Ice Lake (AVX-512) implementations - placeholders for future */
SZ_PUBLIC sz_cptr_t sz_utf8_word_find_boundary_ice(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width);
SZ_PUBLIC sz_cptr_t sz_utf8_word_rfind_boundary_ice(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width);

#pragma endregion

/*  Implementation Section
 *  Following the same pattern as find.h with implementations in the header file.
 */

#pragma region Serial Implementation

/**
 *  Unicode 17.0 TR29 Word_Break property tables (~10 KB, full 4-bit property encoding).
 *
 *  Structure:
 *  - ASCII direct: 128 bytes (1 byte per codepoint, contains 4-bit property)
 *  - BMP Stage1: 256 bytes (block index: 0=OTHER, 1=ALetter, 255=fast-path, 0x80+=stage2)
 *  - BMP Stage2: 6400 bytes (50 nibble-packed blocks, 128 bytes each = 256 codepoints)
 *  - SMP transitions: 3368 bytes (842 transition points, packed as (codepoint << 4) | property)
 *
 *  4-bit property values (see sz_tr29_word_break_t enum):
 *    0=Other, 1=CR, 2=LF, 3=Newline, 4=Extend, 5=ZWJ, 6=Format, 7=Regional_Indicator,
 *    8=ALetter, 9=Hebrew_Letter, 10=Numeric, 11=Katakana, 12=ExtendNumLet, 13=MidLetter,
 *    14=MidNum, 15=MidNumLet+Quotes
 *
 *  Fast paths (algorithmic, bypass table lookup):
 *    U+0100-017F Latin Extended-A: ALetter (8)
 *    U+0180-024F Latin Extended-B: ALetter (8)
 *    U+AC00-D7A3 Hangul Syllables: ALetter (8)
 *    U+3400-4DBF CJK Extension A: Other (0)
 *    U+4E00-9FFF CJK Unified: Other (0)
 *
 *  Generation script (Python 3.6+, stdlib only - regenerates tables from Unicode 17.0):
 *  @code{.py}
 *  import urllib.request, re
 *  url = "https://www.unicode.org/Public/17.0.0/ucd/auxiliary/WordBreakProperty.txt"
 *  PROPS = {'CR':1,'LF':2,'Newline':3,'Extend':4,'ZWJ':5,'Format':6,'Regional_Indicator':7,
 *           'ALetter':8,'Hebrew_Letter':9,'Numeric':10,'Katakana':11,'ExtendNumLet':12,
 *           'MidLetter':13,'MidNum':14,'MidNumLet':15,'Single_Quote':15,'Double_Quote':15}
 *  cp_props = {}
 *  with urllib.request.urlopen(url) as f:
 *      for line in f.read().decode().splitlines():
 *          m = re.match(r'([0-9A-F]+)(?:\.\.([0-9A-F]+))?\s*;\s*(\w+)', line)
 *          if m and m.group(3) in PROPS:
 *              s, e, p = int(m.group(1),16), int(m.group(2) or m.group(1),16), PROPS[m.group(3)]
 *              for cp in range(s, e+1): cp_props[cp] = p
 *  # ASCII direct table
 *  ascii_tbl = [cp_props.get(cp, 0) for cp in range(128)]
 *  # BMP: stage1 index + stage2 nibble-packed blocks
 *  fast_ranges = [(0x100,0x24F,8),(0x3400,0x4DBF,0),(0x4E00,0x9FFF,0),(0xAC00,0xD7A3,8)]
 *  def in_fast(cp): return any(lo<=cp<=hi for lo,hi,_ in fast_ranges)
 *  def fast_prop(cp): return next((p for lo,hi,p in fast_ranges if lo<=cp<=hi), None)
 *  stage1, stage2, next_block = [0]*256, [], 0x80
 *  for hi in range(256):
 *      block = [cp_props.get((hi<<8)|lo, 0) for lo in range(256)]
 *      if any(in_fast((hi<<8)|lo) for lo in range(256)): stage1[hi] = 255; continue
 *      if all(p==0 for p in block): stage1[hi] = 0; continue
 *      if all(p==8 for p in block): stage1[hi] = 1; continue
 *      stage1[hi] = next_block; next_block += 1
 *      for i in range(0,256,2): stage2.append((block[i+1]<<4)|block[i])
 *  # SMP: transition-based encoding
 *  smp_trans = []
 *  prev_prop = 0
 *  for cp in range(0x10000, 0x110000):
 *      prop = cp_props.get(cp, 0)
 *      if prop != prev_prop: smp_trans.append((cp<<4)|prop); prev_prop = prop
 *  # Output
 *  def p8(nm,d,w=16): print(f"static const sz_u8_t {nm}[{len(d)}] = {{"+"".join(
 *      f"\n    "+", ".join(f"0x{x:02X}" for x in d[i:i+w])+"," for i in range(0,len(d),w))+"\n};")
 *  def p32(nm,d,w=6): print(f"static const sz_u32_t {nm}[{len(d)}] = {{"+"".join(
 *      f"\n    "+", ".join(f"0x{x:06X}" for x in d[i:i+w])+"," for i in range(0,len(d),w))+"\n};")
 *  p8("sz_wb_prop_ascii_", ascii_tbl, 16)
 *  p8("sz_wb_prop_stage1_", stage1, 16)
 *  p8("sz_wb_prop_stage2_", stage2, 16)
 *  print(f"#define SZ_TR29_WB_SMP_TRANSITION_COUNT_ {len(smp_trans)}")
 *  p32("sz_wb_prop_smp_transitions_", smp_trans, 6)
 *  @endcode
 */

/* ASCII direct property table (128 bytes) - one property byte per codepoint */
static const sz_u8_t sz_wb_prop_ascii_[128] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x03, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x00, 0x00, 0x00,
    0x00, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x0F, 0x00, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
    0x0A, 0x0D, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
    0x0C, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
};

/* BMP block index (256 bytes): 0=all-OTHER, 1=all-ALetter, 255=fast-path, 0x80+=stage2 offset */
static const sz_u8_t sz_wb_prop_stage1_[256] = {
    0x80, 0xFF, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x01, 0x90,
    0x91, 0x92, 0x01, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0x01, 0x9B, 0x9C, 0x9D, 0x00, 0x00, 0x9E, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9F, 0xA0, 0xA1, 0x00, 0xA2, 0xA3, 0xA4, 0xA5, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x01, 0x01, 0x01, 0xA6, 0x01, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB,
    0xAC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xAE, 0x01, 0xAF, 0xB0, 0xB1,
};

/* BMP Stage 2 nibble-packed property data (6400 bytes = 50 blocks × 128 bytes/block) */
static const sz_u8_t sz_wb_prop_stage2_[6400] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x00,
    0xF0, 0x00, 0x00, 0x0E, 0x0F, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xED, 0x00, 0x00, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0xC0, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x60, 0x00, 0x00, 0x00, 0x80, 0xD0, 0x08, 0x08, 0x00,
    0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x88, 0x88, 0x08, 0x88, 0x00, 0x88, 0x88, 0x8E, 0x00, 0x00, 0x00,
    0xD8, 0x88, 0x08, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x40, 0x44, 0x44, 0x44, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x08, 0x80, 0x88, 0x08, 0xD8, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0xE8, 0x08, 0x00, 0x00, 0x40, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x40,
    0x40, 0x04, 0x44, 0x40, 0x00, 0x00, 0x00, 0x00, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
    0x99, 0x99, 0x09, 0x00, 0x90, 0x99, 0x89, 0x0D, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00,
    0xEE, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x04, 0x06, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xA0, 0x0E, 0x88, 0x84, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x44, 0x44, 0x44, 0xA4, 0x40, 0x44, 0x44, 0x84, 0x48, 0x04, 0x44, 0x44, 0x88,
    0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x88, 0x08, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x48, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x04, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44, 0x44, 0x44, 0x88, 0x00, 0x0E,
    0x08, 0x40, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x48, 0x44, 0x44,
    0x44, 0x44, 0x48, 0x44, 0x48, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08,
    0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x88, 0x88, 0xAA,
    0x00, 0x00, 0x40, 0x44, 0x44, 0x44, 0x44, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x4A, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x84, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x48, 0x44, 0x44, 0x44, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x80, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x80, 0x88, 0x88, 0x88, 0x08, 0x80, 0x08, 0x80, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x08, 0x00, 0x88, 0x88, 0x00, 0x84, 0x44, 0x44,
    0x44, 0x04, 0x40, 0x04, 0x40, 0x44, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x88, 0x80, 0x88, 0x44, 0x00, 0xAA,
    0xAA, 0xAA, 0xAA, 0xAA, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x04, 0x40, 0x44, 0x80, 0x88, 0x88, 0x08, 0x00,
    0x80, 0x08, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x88,
    0x80, 0x08, 0x88, 0x00, 0x04, 0x44, 0x44, 0x04, 0x00, 0x40, 0x04, 0x40, 0x44, 0x00, 0x40, 0x00, 0x00, 0x00, 0x80,
    0x88, 0x08, 0x08, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x44, 0x88, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x40, 0x44, 0x80, 0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x88, 0x80, 0x88, 0x88, 0x00, 0x84, 0x44, 0x44, 0x44, 0x44, 0x40, 0x44, 0x40,
    0x44, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x44, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x44, 0x44, 0x44, 0x40, 0x44, 0x80, 0x88, 0x88, 0x88, 0x08, 0x80, 0x08, 0x80, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x88, 0x80, 0x88, 0x88, 0x00, 0x84,
    0x44, 0x44, 0x44, 0x04, 0x40, 0x04, 0x40, 0x44, 0x00, 0x00, 0x00, 0x40, 0x44, 0x00, 0x00, 0x88, 0x80, 0x88, 0x44,
    0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x84, 0x80, 0x88, 0x88,
    0x08, 0x00, 0x88, 0x08, 0x88, 0x88, 0x00, 0x80, 0x08, 0x08, 0x88, 0x00, 0x80, 0x08, 0x00, 0x88, 0x08, 0x00, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x44, 0x44, 0x04, 0x00, 0x44, 0x04, 0x44, 0x44, 0x00, 0x08, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x44, 0x44, 0x84, 0x88, 0x88, 0x88, 0x08, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x84, 0x44, 0x44, 0x44, 0x04, 0x44,
    0x04, 0x44, 0x44, 0x00, 0x00, 0x00, 0x40, 0x04, 0x88, 0x08, 0x88, 0x00, 0x88, 0x44, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
    0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x44, 0x80, 0x88, 0x88, 0x88, 0x08, 0x88, 0x08, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x88,
    0x00, 0x84, 0x44, 0x44, 0x44, 0x04, 0x44, 0x04, 0x44, 0x44, 0x00, 0x00, 0x00, 0x40, 0x04, 0x00, 0x00, 0x88, 0x08,
    0x88, 0x44, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x80, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x88,
    0x88, 0x88, 0x88, 0x08, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x84, 0x44, 0x44, 0x44, 0x04, 0x44, 0x04, 0x44, 0x44, 0x08, 0x00,
    0x00, 0x88, 0x48, 0x00, 0x00, 0x00, 0x80, 0x88, 0x44, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x88, 0x88, 0x88, 0x40, 0x44, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x88, 0x88, 0x88, 0x80, 0x00, 0x88, 0x88,
    0x88, 0x08, 0x00, 0x04, 0x00, 0x40, 0x44, 0x44, 0x04, 0x04, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0xAA, 0xAA,
    0xAA, 0xAA, 0xAA, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x44,
    0x44, 0x44, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x44, 0x44, 0x44, 0x04, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x44, 0x44, 0x44, 0x44, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44,
    0x04, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44,
    0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x00, 0x00, 0x44,
    0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x08, 0x00, 0x40, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x04, 0x44, 0x88, 0x88,
    0x48, 0x44, 0x44, 0x44, 0x44, 0x44, 0x40, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x04, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x40, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x04, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x00, 0x00, 0x44, 0x04, 0x44, 0x04, 0x40, 0x44, 0x44, 0x44, 0x00,
    0x40, 0x44, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x40, 0xAA, 0xAA, 0xAA,
    0xAA, 0xAA, 0x44, 0x44, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x00, 0x00, 0x80, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x00, 0x88, 0x88,
    0x88, 0x08, 0x08, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x08, 0x08, 0x88, 0x88,
    0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08,
    0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x40,
    0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x00, 0x88, 0x88, 0x88, 0x00, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x80, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x08, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x08, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x44, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x08, 0x44, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x44, 0x46, 0xAA, 0xAA,
    0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x88, 0x88, 0x48,
    0x84, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x08,
    0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x08, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x0A,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x40, 0x44, 0x44, 0x44, 0x44, 0x04, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x04, 0x40, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x84, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x84, 0x88, 0x88, 0x88, 0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
    0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x44, 0x84, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x88, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00,
    0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x80, 0x88, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x08, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x80, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x04,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x84, 0x88, 0x48, 0x88, 0x88, 0x88, 0x84, 0x48, 0x44,
    0x08, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x00,
    0x88, 0x88, 0x88, 0x88, 0x80, 0x80, 0x80, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x08, 0x00,
    0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x00, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x08, 0x00, 0x00, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54,
    0x66, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0xD0, 0x33, 0x66, 0x66, 0xC6, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x0C, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x66, 0x66, 0x06, 0x66, 0x66, 0x66, 0x66, 0x66, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x04,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x80, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x00,
    0x80, 0x88, 0x88, 0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x88, 0x88, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x88,
    0x88, 0x00, 0x00, 0x80, 0x88, 0x88, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00,
    0x80, 0x88, 0x48, 0x44, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x80, 0x00, 0x00, 0x80, 0x00, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x08,
    0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88,
    0x08, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0xB0, 0xBB, 0xBB, 0x00, 0x00, 0x80, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x40, 0xB4, 0x0B, 0x00, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
    0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
    0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0x0B, 0xBB, 0xBB, 0x00, 0x00, 0x80, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
    0xBB, 0xBB, 0xBB, 0xBB, 0x0B, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
    0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
    0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x08, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x88, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x04, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x84, 0x88, 0x84, 0x88, 0x48, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x48, 0x44, 0x44, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00,
    0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x88, 0x88, 0x88, 0x00, 0x80, 0x80, 0x48, 0xAA,
    0xAA, 0xAA, 0xAA, 0xAA, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x44,
    0x44, 0x44, 0x44, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x44, 0x44, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x04, 0x00, 0x00, 0x00, 0x00, 0x88, 0x48, 0x88, 0x88, 0x88, 0x88, 0x44, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
    0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x44,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x44, 0x04, 0x40, 0x04, 0x00, 0x00, 0x44, 0x40, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44,
    0x00, 0x88, 0x48, 0x04, 0x00, 0x00, 0x00, 0x00, 0x80, 0x88, 0x88, 0x08, 0x80, 0x88, 0x88, 0x08, 0x80, 0x88, 0x88,
    0x08, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x48, 0x44, 0x44, 0x44, 0x04, 0x44, 0x00, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
    0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x88, 0x88,
    0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x88, 0x88, 0x00, 0x00, 0x90, 0x94, 0x99, 0x99, 0x99, 0x99, 0x09,
    0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x09, 0x99, 0x99, 0x09, 0x09, 0x99, 0x90, 0x09, 0x99, 0x99, 0x99, 0x99, 0x99,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0xD0, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00, 0xC0, 0x0C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xCC, 0x0E, 0x0F, 0xDE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x88, 0x08, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x08, 0x60, 0x00, 0x00, 0x00, 0xF0, 0x00, 0x00, 0x0E, 0x0F, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xED, 0x00, 0x00, 0x80,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0xC0, 0x80, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBB, 0xBB, 0xBB, 0xBB,
    0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
    0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0x44, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
    0x88, 0x88, 0x08, 0x00, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x00, 0x88, 0x88, 0x88, 0x00, 0x88, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x66, 0x00, 0x00};

/* SMP transition-based property encoding: packed as (codepoint << 4) | property */
#define SZ_TR29_WB_SMP_TRANSITION_COUNT_ 842
static const sz_u32_t sz_wb_prop_smp_transitions_[842] = {
    0x100008, 0x1000C0, 0x1000D8, 0x100270, 0x100288, 0x1003B0, 0x1003C8, 0x1003E0, 0x1003F8, 0x1004E0, 0x100508,
    0x1005E0, 0x100808, 0x100FB0, 0x101408, 0x101750, 0x101FD4, 0x101FE0, 0x102808, 0x1029D0, 0x102A08, 0x102D10,
    0x102E04, 0x102E10, 0x103008, 0x103200, 0x1032D8, 0x1034B0, 0x103508, 0x103764, 0x1037B0, 0x103808, 0x1039E0,
    0x103A08, 0x103C40, 0x103C88, 0x103D00, 0x103D18, 0x103D60, 0x104008, 0x1049E0, 0x104A0A, 0x104AA0, 0x104B08,
    0x104D40, 0x104D88, 0x104FC0, 0x105008, 0x105280, 0x105308, 0x105640, 0x105708, 0x1057B0, 0x1057C8, 0x1058B0,
    0x1058C8, 0x105930, 0x105948, 0x105960, 0x105978, 0x105A20, 0x105A38, 0x105B20, 0x105B38, 0x105BA0, 0x105BB8,
    0x105BD0, 0x105C08, 0x105F40, 0x106008, 0x107370, 0x107408, 0x107560, 0x107608, 0x107680, 0x107808, 0x107860,
    0x107878, 0x107B10, 0x107B28, 0x107BB0, 0x108008, 0x108060, 0x108088, 0x108090, 0x1080A8, 0x108360, 0x108378,
    0x108390, 0x1083C8, 0x1083D0, 0x1083F8, 0x108560, 0x108608, 0x108770, 0x108808, 0x1089F0, 0x108E08, 0x108F30,
    0x108F48, 0x108F60, 0x109008, 0x109160, 0x109208, 0x1093A0, 0x109408, 0x1095A0, 0x109808, 0x109B80, 0x109BE8,
    0x109C00, 0x10A008, 0x10A014, 0x10A040, 0x10A054, 0x10A070, 0x10A0C4, 0x10A108, 0x10A140, 0x10A158, 0x10A180,
    0x10A198, 0x10A360, 0x10A384, 0x10A3B0, 0x10A3F4, 0x10A400, 0x10A608, 0x10A7D0, 0x10A808, 0x10A9D0, 0x10AC08,
    0x10AC80, 0x10AC98, 0x10AE54, 0x10AE70, 0x10B008, 0x10B360, 0x10B408, 0x10B560, 0x10B608, 0x10B730, 0x10B808,
    0x10B920, 0x10C008, 0x10C490, 0x10C808, 0x10CB30, 0x10CC08, 0x10CF30, 0x10D008, 0x10D244, 0x10D280, 0x10D30A,
    0x10D3A0, 0x10D40A, 0x10D4A8, 0x10D660, 0x10D694, 0x10D6E0, 0x10D6F8, 0x10D860, 0x10E808, 0x10EAA0, 0x10EAB4,
    0x10EAD0, 0x10EB08, 0x10EB20, 0x10EC28, 0x10EC80, 0x10EFA4, 0x10F008, 0x10F1D0, 0x10F278, 0x10F280, 0x10F308,
    0x10F464, 0x10F510, 0x10F708, 0x10F824, 0x10F860, 0x10FB08, 0x10FC50, 0x10FE08, 0x10FF70, 0x110004, 0x110038,
    0x110384, 0x110470, 0x11066A, 0x110704, 0x110718, 0x110734, 0x110758, 0x110760, 0x1107F4, 0x110838, 0x110B04,
    0x110BB0, 0x110BDA, 0x110BE0, 0x110C24, 0x110C30, 0x110CDA, 0x110CE0, 0x110D08, 0x110E90, 0x110F0A, 0x110FA0,
    0x111004, 0x111038, 0x111274, 0x111350, 0x11136A, 0x111400, 0x111448, 0x111454, 0x111478, 0x111480, 0x111508,
    0x111734, 0x111740, 0x111768, 0x111770, 0x111804, 0x111838, 0x111B34, 0x111C18, 0x111C50, 0x111C94, 0x111CD0,
    0x111CE4, 0x111D0A, 0x111DA8, 0x111DB0, 0x111DC8, 0x111DD0, 0x112008, 0x112120, 0x112138, 0x1122C4, 0x112380,
    0x1123E4, 0x1123F8, 0x112414, 0x112420, 0x112808, 0x112870, 0x112888, 0x112890, 0x1128A8, 0x1128E0, 0x1128F8,
    0x1129E0, 0x1129F8, 0x112A90, 0x112B08, 0x112DF4, 0x112EB0, 0x112F0A, 0x112FA0, 0x113004, 0x113040, 0x113058,
    0x1130D0, 0x1130F8, 0x113110, 0x113138, 0x113290, 0x1132A8, 0x113310, 0x113328, 0x113340, 0x113358, 0x1133A0,
    0x1133B4, 0x1133D8, 0x1133E4, 0x113450, 0x113474, 0x113490, 0x1134B4, 0x1134E0, 0x113508, 0x113510, 0x113574,
    0x113580, 0x1135D8, 0x113624, 0x113640, 0x113664, 0x1136D0, 0x113704, 0x113750, 0x113808, 0x1138A0, 0x1138B8,
    0x1138C0, 0x1138E8, 0x1138F0, 0x113908, 0x113B60, 0x113B78, 0x113B84, 0x113C10, 0x113C24, 0x113C30, 0x113C54,
    0x113C60, 0x113C74, 0x113CB0, 0x113CC4, 0x113D18, 0x113D24, 0x113D38, 0x113D40, 0x113E14, 0x113E30, 0x114008,
    0x114354, 0x114478, 0x1144B0, 0x11450A, 0x1145A0, 0x1145E4, 0x1145F8, 0x114620, 0x114808, 0x114B04, 0x114C48,
    0x114C60, 0x114C78, 0x114C80, 0x114D0A, 0x114DA0, 0x115808, 0x115AF4, 0x115B60, 0x115B84, 0x115C10, 0x115D88,
    0x115DC4, 0x115DE0, 0x116008, 0x116304, 0x116410, 0x116448, 0x116450, 0x11650A, 0x1165A0, 0x116808, 0x116AB4,
    0x116B88, 0x116B90, 0x116C0A, 0x116CA0, 0x116D0A, 0x116E40, 0x1171D4, 0x1172C0, 0x11730A, 0x1173A0, 0x118008,
    0x1182C4, 0x1183B0, 0x118A08, 0x118E0A, 0x118EA0, 0x118FF8, 0x119070, 0x119098, 0x1190A0, 0x1190C8, 0x119140,
    0x119158, 0x119170, 0x119188, 0x119304, 0x119360, 0x119374, 0x119390, 0x1193B4, 0x1193F8, 0x119404, 0x119418,
    0x119424, 0x119440, 0x11950A, 0x1195A0, 0x119A08, 0x119A80, 0x119AA8, 0x119D14, 0x119D80, 0x119DA4, 0x119E18,
    0x119E20, 0x119E38, 0x119E44, 0x119E50, 0x11A008, 0x11A014, 0x11A0B8, 0x11A334, 0x11A3A8, 0x11A3B4, 0x11A3F0,
    0x11A474, 0x11A480, 0x11A508, 0x11A514, 0x11A5C8, 0x11A8A4, 0x11A9A0, 0x11A9D8, 0x11A9E0, 0x11AB08, 0x11AF90,
    0x11B604, 0x11B680, 0x11BC08, 0x11BE10, 0x11BF0A, 0x11BFA0, 0x11C008, 0x11C090, 0x11C0A8, 0x11C2F4, 0x11C370,
    0x11C384, 0x11C408, 0x11C410, 0x11C50A, 0x11C5A0, 0x11C728, 0x11C900, 0x11C924, 0x11CA80, 0x11CA94, 0x11CB70,
    0x11D008, 0x11D070, 0x11D088, 0x11D0A0, 0x11D0B8, 0x11D314, 0x11D370, 0x11D3A4, 0x11D3B0, 0x11D3C4, 0x11D3E0,
    0x11D3F4, 0x11D468, 0x11D474, 0x11D480, 0x11D50A, 0x11D5A0, 0x11D608, 0x11D660, 0x11D678, 0x11D690, 0x11D6A8,
    0x11D8A4, 0x11D8F0, 0x11D904, 0x11D920, 0x11D934, 0x11D988, 0x11D990, 0x11DA0A, 0x11DAA0, 0x11DB08, 0x11DDC0,
    0x11DE0A, 0x11DEA0, 0x11EE08, 0x11EF34, 0x11EF70, 0x11F004, 0x11F028, 0x11F034, 0x11F048, 0x11F110, 0x11F128,
    0x11F344, 0x11F3B0, 0x11F3E4, 0x11F430, 0x11F50A, 0x11F5A4, 0x11F5B0, 0x11FB08, 0x11FB10, 0x120008, 0x1239A0,
    0x124008, 0x1246F0, 0x124808, 0x125440, 0x12F908, 0x12FF10, 0x130008, 0x134306, 0x134404, 0x134418, 0x134474,
    0x134560, 0x134608, 0x143FB0, 0x144008, 0x146470, 0x161008, 0x1611E4, 0x16130A, 0x1613A0, 0x168008, 0x16A390,
    0x16A408, 0x16A5F0, 0x16A60A, 0x16A6A0, 0x16A708, 0x16ABF0, 0x16AC0A, 0x16ACA0, 0x16AD08, 0x16AEE0, 0x16AF04,
    0x16AF50, 0x16B008, 0x16B304, 0x16B370, 0x16B408, 0x16B440, 0x16B50A, 0x16B5A0, 0x16B638, 0x16B780, 0x16B7D8,
    0x16B900, 0x16D408, 0x16D6D0, 0x16D70A, 0x16D7A0, 0x16E408, 0x16E800, 0x16EA08, 0x16EB90, 0x16EBB8, 0x16ED40,
    0x16F008, 0x16F4B0, 0x16F4F4, 0x16F508, 0x16F514, 0x16F880, 0x16F8F4, 0x16F938, 0x16FA00, 0x16FE08, 0x16FE20,
    0x16FE38, 0x16FE44, 0x16FE50, 0x16FF04, 0x16FF20, 0x1AFF0B, 0x1AFF40, 0x1AFF5B, 0x1AFFC0, 0x1AFFDB, 0x1AFFF0,
    0x1B000B, 0x1B0010, 0x1B120B, 0x1B1230, 0x1B155B, 0x1B1560, 0x1B164B, 0x1B1680, 0x1BC008, 0x1BC6B0, 0x1BC708,
    0x1BC7D0, 0x1BC808, 0x1BC890, 0x1BC908, 0x1BC9A0, 0x1BC9D4, 0x1BC9F0, 0x1BCA06, 0x1BCA40, 0x1CCF0A, 0x1CCFA0,
    0x1CF004, 0x1CF2E0, 0x1CF304, 0x1CF470, 0x1D1654, 0x1D16A0, 0x1D16D4, 0x1D1736, 0x1D17B4, 0x1D1830, 0x1D1854,
    0x1D18C0, 0x1D1AA4, 0x1D1AE0, 0x1D2424, 0x1D2450, 0x1D4008, 0x1D4550, 0x1D4568, 0x1D49D0, 0x1D49E8, 0x1D4A00,
    0x1D4A28, 0x1D4A30, 0x1D4A58, 0x1D4A70, 0x1D4A98, 0x1D4AD0, 0x1D4AE8, 0x1D4BA0, 0x1D4BB8, 0x1D4BC0, 0x1D4BD8,
    0x1D4C40, 0x1D4C58, 0x1D5060, 0x1D5078, 0x1D50B0, 0x1D50D8, 0x1D5150, 0x1D5168, 0x1D51D0, 0x1D51E8, 0x1D53A0,
    0x1D53B8, 0x1D53F0, 0x1D5408, 0x1D5450, 0x1D5468, 0x1D5470, 0x1D54A8, 0x1D5510, 0x1D5528, 0x1D6A60, 0x1D6A88,
    0x1D6C10, 0x1D6C28, 0x1D6DB0, 0x1D6DC8, 0x1D6FB0, 0x1D6FC8, 0x1D7150, 0x1D7168, 0x1D7350, 0x1D7368, 0x1D74F0,
    0x1D7508, 0x1D76F0, 0x1D7708, 0x1D7890, 0x1D78A8, 0x1D7A90, 0x1D7AA8, 0x1D7C30, 0x1D7C48, 0x1D7CC0, 0x1D7CEA,
    0x1D8000, 0x1DA004, 0x1DA370, 0x1DA3B4, 0x1DA6D0, 0x1DA754, 0x1DA760, 0x1DA844, 0x1DA850, 0x1DA9B4, 0x1DAA00,
    0x1DAA14, 0x1DAB00, 0x1DF008, 0x1DF1F0, 0x1DF258, 0x1DF2B0, 0x1E0004, 0x1E0070, 0x1E0084, 0x1E0190, 0x1E01B4,
    0x1E0220, 0x1E0234, 0x1E0250, 0x1E0264, 0x1E02B0, 0x1E0308, 0x1E06E0, 0x1E08F4, 0x1E0900, 0x1E1008, 0x1E12D0,
    0x1E1304, 0x1E1378, 0x1E13E0, 0x1E140A, 0x1E14A0, 0x1E14E8, 0x1E14F0, 0x1E2908, 0x1E2AE4, 0x1E2AF0, 0x1E2C08,
    0x1E2EC4, 0x1E2F0A, 0x1E2FA0, 0x1E4D08, 0x1E4EC4, 0x1E4F0A, 0x1E4FA0, 0x1E5D08, 0x1E5EE4, 0x1E5F08, 0x1E5F1A,
    0x1E5FB0, 0x1E6C08, 0x1E6DF0, 0x1E6E08, 0x1E6E34, 0x1E6E48, 0x1E6E64, 0x1E6E78, 0x1E6EE4, 0x1E6F08, 0x1E6F54,
    0x1E6F60, 0x1E6FE8, 0x1E7000, 0x1E7E08, 0x1E7E70, 0x1E7E88, 0x1E7EC0, 0x1E7ED8, 0x1E7EF0, 0x1E7F08, 0x1E7FF0,
    0x1E8008, 0x1E8C50, 0x1E8D04, 0x1E8D70, 0x1E9008, 0x1E9444, 0x1E94B8, 0x1E94C0, 0x1E950A, 0x1E95A0, 0x1EE008,
    0x1EE040, 0x1EE058, 0x1EE200, 0x1EE218, 0x1EE230, 0x1EE248, 0x1EE250, 0x1EE278, 0x1EE280, 0x1EE298, 0x1EE330,
    0x1EE348, 0x1EE380, 0x1EE398, 0x1EE3A0, 0x1EE3B8, 0x1EE3C0, 0x1EE428, 0x1EE430, 0x1EE478, 0x1EE480, 0x1EE498,
    0x1EE4A0, 0x1EE4B8, 0x1EE4C0, 0x1EE4D8, 0x1EE500, 0x1EE518, 0x1EE530, 0x1EE548, 0x1EE550, 0x1EE578, 0x1EE580,
    0x1EE598, 0x1EE5A0, 0x1EE5B8, 0x1EE5C0, 0x1EE5D8, 0x1EE5E0, 0x1EE5F8, 0x1EE600, 0x1EE618, 0x1EE630, 0x1EE648,
    0x1EE650, 0x1EE678, 0x1EE6B0, 0x1EE6C8, 0x1EE730, 0x1EE748, 0x1EE780, 0x1EE798, 0x1EE7D0, 0x1EE7E8, 0x1EE7F0,
    0x1EE808, 0x1EE8A0, 0x1EE8B8, 0x1EE9C0, 0x1EEA18, 0x1EEA40, 0x1EEA58, 0x1EEAA0, 0x1EEAB8, 0x1EEBC0, 0x1F1308,
    0x1F14A0, 0x1F1508, 0x1F16A0, 0x1F1708, 0x1F18A0, 0x1F1E67, 0x1F2000, 0x1F3FB4, 0x1F4000, 0x1FBF0A, 0x1FBFA0,
    0xE00016, 0xE00020, 0xE00204, 0xE00800, 0xE01004, 0xE01F00};

/**
 *  @brief Get the Unicode TR29 Word_Break property for a codepoint.
 *
 *  Returns one of the 16 Word_Break property values (0-15):
 *  - sz_tr29_word_break_other_k (0): Default - creates word boundary
 *  - sz_tr29_word_break_cr_k (1): Carriage Return
 *  - sz_tr29_word_break_lf_k (2): Line Feed
 *  - sz_tr29_word_break_newline_k (3): Other newlines
 *  - sz_tr29_word_break_extend_k (4): Combining marks
 *  - sz_tr29_word_break_zwj_k (5): Zero Width Joiner
 *  - sz_tr29_word_break_format_k (6): Format characters
 *  - sz_tr29_word_break_regional_ind_k (7): Regional Indicators
 *  - sz_tr29_word_break_aletter_k (8): Alphabetic letters
 *  - sz_tr29_word_break_hebrew_letter_k (9): Hebrew letters
 *  - sz_tr29_word_break_numeric_k (10): Digits
 *  - sz_tr29_word_break_katakana_k (11): Japanese Katakana
 *  - sz_tr29_word_break_extendnumlet_k (12): Underscore/connector
 *  - sz_tr29_word_break_midletter_k (13): Mid-letter punctuation
 *  - sz_tr29_word_break_midnum_k (14): Mid-number punctuation
 *  - sz_tr29_word_break_mid_quotes_k (15): MidNumLet + quotes
 */
SZ_PUBLIC sz_u8_t sz_rune_word_break_property(sz_rune_t rune) {
    // Fast path 1: ASCII - direct table lookup
    if (rune < 0x80) { return sz_wb_prop_ascii_[rune]; }

    // Fast path 2: Latin Extended-A (U+0100-U+017F) - all ALetter
    if (rune >= 0x0100 && rune <= 0x017F) return sz_tr29_word_break_aletter_k;

    // Fast path 3: Latin Extended-B (U+0180-U+024F) - all ALetter
    if (rune >= 0x0180 && rune <= 0x024F) return sz_tr29_word_break_aletter_k;

    // Fast path 4: Hangul Syllables (U+AC00-U+D7A3) - all ALetter
    if (rune >= 0xAC00 && rune <= 0xD7A3) return sz_tr29_word_break_aletter_k;

    // Fast path 5: CJK Extension A (U+3400-U+4DBF) - all Other/boundary
    if (rune >= 0x3400 && rune <= 0x4DBF) return sz_tr29_word_break_other_k;

    // Fast path 6: CJK Unified (U+4E00-U+9FFF) - all Other/boundary
    if (rune >= 0x4E00 && rune <= 0x9FFF) return sz_tr29_word_break_other_k;

    // BMP two-stage lookup with nibble-packed stage2
    if (rune < 0x10000) {
        sz_u8_t block_idx = sz_wb_prop_stage1_[rune >> 8];

        // Special indices
        if (block_idx == 0) return sz_tr29_word_break_other_k;   // All boundary
        if (block_idx == 1) return sz_tr29_word_break_aletter_k; // All ALetter
        if (block_idx == 255) return sz_tr29_word_break_other_k; // Fast path handled above

        // Nibble-packed lookup: each byte holds 2 properties
        sz_size_t stage2_offset = ((sz_size_t)(block_idx - 0x80)) * 128;
        sz_u8_t low_byte = (sz_u8_t)(rune & 0xFF);
        sz_u8_t packed = sz_wb_prop_stage2_[stage2_offset + (low_byte >> 1)];
        return (low_byte & 1) ? (packed >> 4) : (packed & 0x0F);
    }

    // SMP: Binary search over transition points
    // Each entry is packed as (codepoint << 4) | property
    // Property changes at each transition point
    sz_size_t lo = 0, hi = SZ_TR29_WB_SMP_TRANSITION_COUNT_;
    while (lo < hi) {
        sz_size_t mid = (lo + hi) / 2;
        sz_rune_t trans_cp = sz_wb_prop_smp_transitions_[mid] >> 4;
        if (rune < trans_cp) { hi = mid; }
        else { lo = mid + 1; }
    }
    // lo is now the index of first transition > rune, so lo-1 is our transition
    if (lo == 0) return sz_tr29_word_break_other_k;
    return (sz_u8_t)(sz_wb_prop_smp_transitions_[lo - 1] & 0x0F);
}

/**
 *  @brief  Check if a codepoint is a "word character" (has word-forming property).
 *
 *  This is a convenience function that returns true if the codepoint has a
 *  Word_Break property that typically forms words (ALetter, Hebrew_Letter,
 *  Numeric, Katakana, ExtendNumLet, or mid-word punctuation).
 *
 *  @param[in] rune The Unicode codepoint to check.
 *  @return sz_true_k if the codepoint is a word character, sz_false_k otherwise.
 */
SZ_PUBLIC sz_bool_t sz_rune_is_word_char(sz_rune_t rune) {
    sz_u8_t prop = sz_rune_word_break_property(rune);
    // Word characters: ALetter(8), Hebrew_Letter(9), Numeric(10), Katakana(11),
    // ExtendNumLet(12), MidLetter(13), MidNum(14), MidNumLet/Quotes(15)
    return (sz_bool_t)(prop >= sz_tr29_word_break_aletter_k);
}

/**
 *  @brief  Decode a UTF-8 codepoint starting at the given position.
 *  @return The decoded codepoint, or 0xFFFD on error. Updates pos to next codepoint.
 */
SZ_INTERNAL sz_rune_t sz_utf8_decode_(sz_cptr_t text, sz_size_t length, sz_size_t *pos) {
    if (*pos >= length) return 0;
    sz_u8_t b0 = (sz_u8_t)text[*pos];
    if (b0 < 0x80) {
        (*pos)++;
        return b0;
    }
    if ((b0 & 0xE0) == 0xC0 && *pos + 1 < length) {
        sz_rune_t r = ((b0 & 0x1F) << 6) | (text[*pos + 1] & 0x3F);
        *pos += 2;
        return r;
    }
    if ((b0 & 0xF0) == 0xE0 && *pos + 2 < length) {
        sz_rune_t r = ((b0 & 0x0F) << 12) | ((text[*pos + 1] & 0x3F) << 6) | (text[*pos + 2] & 0x3F);
        *pos += 3;
        return r;
    }
    if ((b0 & 0xF8) == 0xF0 && *pos + 3 < length) {
        sz_rune_t r = ((b0 & 0x07) << 18) | ((text[*pos + 1] & 0x3F) << 12) | ((text[*pos + 2] & 0x3F) << 6) |
                      (text[*pos + 3] & 0x3F);
        *pos += 4;
        return r;
    }
    (*pos)++;
    return 0xFFFD;
}

/**
 *  @brief  Get UTF-8 sequence length from lead byte.
 */
SZ_INTERNAL sz_size_t sz_utf8_char_length_(sz_u8_t lead) {
    if (lead < 0x80) return 1;
    if ((lead & 0xE0) == 0xC0) return 2;
    if ((lead & 0xF0) == 0xE0) return 3;
    if ((lead & 0xF8) == 0xF0) return 4;
    return 1; // Invalid, treat as single byte
}

/**
 *  @brief  Check if property is WB4-ignorable (Extend, Format, ZWJ).
 */
SZ_INTERNAL sz_bool_t sz_wb_is_ignorable_(sz_u8_t prop) {
    return (sz_bool_t)(prop == sz_tr29_word_break_extend_k || prop == sz_tr29_word_break_format_k ||
                       prop == sz_tr29_word_break_zwj_k);
}

/**
 *  @brief  Check if property is AHLetter (ALetter or Hebrew_Letter).
 */
SZ_INTERNAL sz_bool_t sz_wb_is_ahletter_(sz_u8_t prop) {
    return (sz_bool_t)(prop == sz_tr29_word_break_aletter_k || prop == sz_tr29_word_break_hebrew_letter_k);
}

/**
 *  @brief  Check if property is MidNumLetQ (MidNumLet or Single_Quote).
 *          In our encoding, MID_QUOTES (15) covers MidNumLet + quotes.
 */
SZ_INTERNAL sz_bool_t sz_wb_is_midnumletq_(sz_u8_t prop) {
    return (sz_bool_t)(prop == sz_tr29_word_break_mid_quotes_k);
}

/**
 *  @brief  Skip forward past WB4-ignorable characters (Extend, Format, ZWJ).
 *  @return Position after ignorables, or original position if none.
 */
SZ_INTERNAL sz_size_t sz_utf8_skip_ignorables_forward_(sz_cptr_t text, sz_size_t length, sz_size_t pos) {
    while (pos < length) {
        sz_size_t next_pos = pos;
        sz_rune_t rune = sz_utf8_decode_(text, length, &next_pos);
        sz_u8_t prop = sz_rune_word_break_property(rune);
        if (!sz_wb_is_ignorable_(prop)) break;
        pos = next_pos;
    }
    return pos;
}

/**
 *  @brief  Get the effective property at position, skipping WB4 ignorables.
 *          Returns the property and updates next_pos to position after ignorables.
 */
SZ_INTERNAL sz_u8_t sz_wb_get_effective_prop_(sz_cptr_t text, sz_size_t length, sz_size_t pos, sz_size_t *next_pos) {
    sz_size_t cur = pos;
    sz_rune_t rune = sz_utf8_decode_(text, length, &cur);
    sz_u8_t prop = sz_rune_word_break_property(rune);

    // Skip ignorables to find effective next
    while (cur < length && sz_wb_is_ignorable_(prop)) {
        rune = sz_utf8_decode_(text, length, &cur);
        prop = sz_rune_word_break_property(rune);
    }

    if (next_pos) *next_pos = cur;
    return prop;
}

/**
 *  @brief  Look back to find the previous non-ignorable property.
 *  @return The property of the previous significant character, or sz_tr29_word_break_other_k if none.
 */
SZ_INTERNAL sz_u8_t sz_wb_prev_prop_(sz_cptr_t text, sz_size_t pos) {
    if (pos == 0) return sz_tr29_word_break_other_k;

    // Scan backward to find start of previous codepoint
    sz_size_t prev = pos - 1;
    while (prev > 0 && ((sz_u8_t)text[prev] & 0xC0) == 0x80) prev--;

    // Decode and get property
    sz_size_t decode_pos = prev;
    sz_rune_t rune = sz_utf8_decode_(text, pos, &decode_pos);
    sz_u8_t prop = sz_rune_word_break_property(rune);

    // Skip back over ignorables
    while (sz_wb_is_ignorable_(prop) && prev > 0) {
        prev--;
        while (prev > 0 && ((sz_u8_t)text[prev] & 0xC0) == 0x80) prev--;
        decode_pos = prev;
        rune = sz_utf8_decode_(text, pos, &decode_pos);
        prop = sz_rune_word_break_property(rune);
    }

    return prop;
}

/**
 *  @brief  Count Regional Indicators before position (for WB15/16).
 */
SZ_INTERNAL sz_size_t sz_wb_count_ri_before_(sz_cptr_t text, sz_size_t pos) {
    sz_size_t count = 0;
    sz_size_t cur = pos;

    while (cur > 0) {
        // Find previous codepoint
        sz_size_t prev = cur - 1;
        while (prev > 0 && ((sz_u8_t)text[prev] & 0xC0) == 0x80) prev--;

        sz_size_t decode_pos = prev;
        sz_rune_t rune = sz_utf8_decode_(text, cur, &decode_pos);
        sz_u8_t prop = sz_rune_word_break_property(rune);

        if (prop == sz_tr29_word_break_regional_ind_k) {
            count++;
            cur = prev;
        }
        else if (sz_wb_is_ignorable_(prop)) {
            cur = prev; // Skip ignorables
        }
        else { break; }
    }
    return count;
}

/**
 *  @brief  Check if position is a word boundary per Unicode TR29.
 *
 *  Implements the TR29 word boundary algorithm. Position 0 and position == length
 *  are always boundaries (WB1/WB2).
 */
SZ_PUBLIC sz_bool_t sz_utf8_is_word_boundary_serial(sz_cptr_t text, sz_size_t length, sz_size_t pos) {
    // WB1: Break at start of text
    if (pos == 0) return sz_true_k;
    // WB2: Break at end of text
    if (pos >= length) return sz_true_k;

    // Never break at UTF-8 continuation bytes (0x80-0xBF)
    if (((sz_u8_t)text[pos] & 0xC0) == 0x80) return sz_false_k;

    // Get properties of characters before and after the boundary
    sz_u8_t prev_prop = sz_wb_prev_prop_(text, pos);

    sz_size_t after_pos = pos;
    sz_rune_t after_rune = sz_utf8_decode_(text, length, &after_pos);
    sz_u8_t after_prop = sz_rune_word_break_property(after_rune);

    // WB3: Do not break between CR and LF
    if (prev_prop == sz_tr29_word_break_cr_k && after_prop == sz_tr29_word_break_lf_k) return sz_false_k;

    // WB3a: Break after Newline, CR, LF
    if (prev_prop == sz_tr29_word_break_newline_k || prev_prop == sz_tr29_word_break_cr_k ||
        prev_prop == sz_tr29_word_break_lf_k)
        return sz_true_k;

    // WB3b: Break before Newline, CR, LF
    if (after_prop == sz_tr29_word_break_newline_k || after_prop == sz_tr29_word_break_cr_k ||
        after_prop == sz_tr29_word_break_lf_k)
        return sz_true_k;

    // WB3c: Do not break within emoji ZWJ sequences
    // (Simplified: don't break ZWJ × anything)
    if (prev_prop == sz_tr29_word_break_zwj_k) return sz_false_k;

    // WB4: Ignore Format and Extend characters - get effective properties
    // Skip ignorables after position to get effective "after" property
    if (sz_wb_is_ignorable_(after_prop)) {
        sz_size_t skip_pos = pos;
        after_prop = sz_wb_get_effective_prop_(text, length, skip_pos, (sz_size_t *)0);
    }

    // WB5: Do not break between AHLetter
    if (sz_wb_is_ahletter_(prev_prop) && sz_wb_is_ahletter_(after_prop)) return sz_false_k;

    // WB6: Do not break AHLetter × (MidLetter|MidNumLetQ) × AHLetter
    if (sz_wb_is_ahletter_(prev_prop) &&
        (after_prop == sz_tr29_word_break_midletter_k || sz_wb_is_midnumletq_(after_prop))) {
        // Look ahead to see if followed by AHLetter
        sz_size_t lookahead = after_pos;
        lookahead = sz_utf8_skip_ignorables_forward_(text, length, lookahead);
        if (lookahead < length) {
            sz_size_t la_pos = lookahead;
            sz_rune_t la_rune = sz_utf8_decode_(text, length, &la_pos);
            sz_u8_t la_prop = sz_rune_word_break_property(la_rune);
            if (sz_wb_is_ahletter_(la_prop)) return sz_false_k;
        }
    }

    // WB7: Do not break AHLetter (MidLetter|MidNumLetQ) × AHLetter
    if ((prev_prop == sz_tr29_word_break_midletter_k || sz_wb_is_midnumletq_(prev_prop)) &&
        sz_wb_is_ahletter_(after_prop)) {
        // Look back to see if preceded by AHLetter
        // This requires looking at the character before prev
        sz_size_t prev_prev_end = pos - 1;
        while (prev_prev_end > 0 && ((sz_u8_t)text[prev_prev_end] & 0xC0) == 0x80) prev_prev_end--;
        if (prev_prev_end > 0) {
            sz_size_t pp_start = prev_prev_end - 1;
            while (pp_start > 0 && ((sz_u8_t)text[pp_start] & 0xC0) == 0x80) pp_start--;
            sz_size_t pp_pos = pp_start;
            sz_rune_t pp_rune = sz_utf8_decode_(text, prev_prev_end, &pp_pos);
            sz_u8_t pp_prop = sz_rune_word_break_property(pp_rune);
            if (sz_wb_is_ahletter_(pp_prop)) return sz_false_k;
        }
    }

    // WB7a: Do not break Hebrew_Letter × Single_Quote
    if (prev_prop == sz_tr29_word_break_hebrew_letter_k && after_prop == sz_tr29_word_break_mid_quotes_k)
        return sz_false_k;

    // WB8: Do not break Numeric × Numeric
    if (prev_prop == sz_tr29_word_break_numeric_k && after_prop == sz_tr29_word_break_numeric_k) return sz_false_k;

    // WB9: Do not break AHLetter × Numeric
    if (sz_wb_is_ahletter_(prev_prop) && after_prop == sz_tr29_word_break_numeric_k) return sz_false_k;

    // WB10: Do not break Numeric × AHLetter
    if (prev_prop == sz_tr29_word_break_numeric_k && sz_wb_is_ahletter_(after_prop)) return sz_false_k;

    // WB11: Do not break Numeric × (MidNum|MidNumLetQ) × Numeric
    if (prev_prop == sz_tr29_word_break_numeric_k &&
        (after_prop == sz_tr29_word_break_midnum_k || sz_wb_is_midnumletq_(after_prop))) {
        sz_size_t lookahead = after_pos;
        lookahead = sz_utf8_skip_ignorables_forward_(text, length, lookahead);
        if (lookahead < length) {
            sz_size_t la_pos = lookahead;
            sz_rune_t la_rune = sz_utf8_decode_(text, length, &la_pos);
            sz_u8_t la_prop = sz_rune_word_break_property(la_rune);
            if (la_prop == sz_tr29_word_break_numeric_k) return sz_false_k;
        }
    }

    // WB12: Do not break Numeric (MidNum|MidNumLetQ) × Numeric (reverse of WB11)
    if ((prev_prop == sz_tr29_word_break_midnum_k || sz_wb_is_midnumletq_(prev_prop)) &&
        after_prop == sz_tr29_word_break_numeric_k) {
        // Check if preceded by Numeric
        sz_size_t prev_prev_end = pos - 1;
        while (prev_prev_end > 0 && ((sz_u8_t)text[prev_prev_end] & 0xC0) == 0x80) prev_prev_end--;
        if (prev_prev_end > 0) {
            sz_size_t pp_start = prev_prev_end - 1;
            while (pp_start > 0 && ((sz_u8_t)text[pp_start] & 0xC0) == 0x80) pp_start--;
            sz_size_t pp_pos = pp_start;
            sz_rune_t pp_rune = sz_utf8_decode_(text, prev_prev_end, &pp_pos);
            sz_u8_t pp_prop = sz_rune_word_break_property(pp_rune);
            if (pp_prop == sz_tr29_word_break_numeric_k) return sz_false_k;
        }
    }

    // WB13: Do not break Katakana × Katakana
    if (prev_prop == sz_tr29_word_break_katakana_k && after_prop == sz_tr29_word_break_katakana_k) return sz_false_k;

    // WB13a: Do not break (AHLetter|Numeric|Katakana|ExtendNumLet) × ExtendNumLet
    if ((sz_wb_is_ahletter_(prev_prop) || prev_prop == sz_tr29_word_break_numeric_k ||
         prev_prop == sz_tr29_word_break_katakana_k || prev_prop == sz_tr29_word_break_extendnumlet_k) &&
        after_prop == sz_tr29_word_break_extendnumlet_k)
        return sz_false_k;

    // WB13b: Do not break ExtendNumLet × (AHLetter|Numeric|Katakana)
    if (prev_prop == sz_tr29_word_break_extendnumlet_k &&
        (sz_wb_is_ahletter_(after_prop) || after_prop == sz_tr29_word_break_numeric_k ||
         after_prop == sz_tr29_word_break_katakana_k))
        return sz_false_k;

    // WB15/16: Do not break between Regional Indicators (keep pairs together)
    if (prev_prop == sz_tr29_word_break_regional_ind_k && after_prop == sz_tr29_word_break_regional_ind_k) {
        // Count RI before - if odd, don't break (we're in the middle of a pair)
        sz_size_t ri_count = sz_wb_count_ri_before_(text, pos);
        if (ri_count % 2 == 1) return sz_false_k;
    }

    // WB999: Otherwise, break everywhere
    return sz_true_k;
}

/**
 *  @brief  Find the next word boundary in UTF-8 text.
 *
 *  Scans forward from the start of text to find the first word boundary position
 *  after position 0 (since position 0 is always a boundary).
 *
 *  @param[in] text UTF-8 encoded text.
 *  @param[in] length Byte length of text.
 *  @param[out] boundary_width Outputs bytes consumed to reach boundary.
 *  @return Pointer to boundary position, or text+length if at end.
 */
SZ_PUBLIC sz_cptr_t sz_utf8_word_find_boundary_serial(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width) {
    if (length == 0) {
        if (boundary_width) *boundary_width = 0;
        return text;
    }

    sz_size_t pos = 0;
    // Skip first codepoint (position 0 is always a boundary)
    if (pos < length) { pos += sz_utf8_char_length_((sz_u8_t)text[pos]); }

    // Scan for next boundary
    while (pos < length) {
        if (sz_utf8_is_word_boundary_serial(text, length, pos)) {
            if (boundary_width) *boundary_width = pos;
            return text + pos;
        }
        pos += sz_utf8_char_length_((sz_u8_t)text[pos]);
    }

    // End of text is always a boundary
    if (boundary_width) *boundary_width = length;
    return text + length;
}

/**
 *  @brief  Find the previous word boundary in UTF-8 text (reverse search).
 *
 *  Scans backward from the end of text to find the last word boundary position
 *  before position length (since position length is always a boundary).
 *
 *  @param[in] text UTF-8 encoded text.
 *  @param[in] length Byte length of text.
 *  @param[out] boundary_width Outputs bytes from boundary to end.
 *  @return Pointer to boundary position, or text if at start.
 */
SZ_PUBLIC sz_cptr_t sz_utf8_word_rfind_boundary_serial(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width) {
    if (length == 0) {
        if (boundary_width) *boundary_width = 0;
        return text;
    }

    sz_size_t pos = length;
    // Move back one codepoint (position length is always a boundary)
    if (pos > 0) {
        pos--;
        while (pos > 0 && ((sz_u8_t)text[pos] & 0xC0) == 0x80) pos--;
    }

    // Scan backward for previous boundary
    while (pos > 0) {
        if (sz_utf8_is_word_boundary_serial(text, length, pos)) {
            if (boundary_width) *boundary_width = length - pos;
            return text + pos;
        }
        pos--;
        while (pos > 0 && ((sz_u8_t)text[pos] & 0xC0) == 0x80) pos--;
    }

    // Start of text is always a boundary
    if (boundary_width) *boundary_width = length;
    return text;
}

#pragma endregion // Serial Implementation

#pragma region Dynamic Dispatch

/**
 *  When dynamic dispatch is disabled, the SZ_DYNAMIC functions simply forward to serial implementations.
 *  When dynamic dispatch is enabled, these are provided by the shared library.
 */
#if !SZ_DYNAMIC_DISPATCH

SZ_DYNAMIC sz_cptr_t sz_utf8_word_find_boundary(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width) {
    return sz_utf8_word_find_boundary_serial(text, length, boundary_width);
}

SZ_DYNAMIC sz_cptr_t sz_utf8_word_rfind_boundary(sz_cptr_t text, sz_size_t length, sz_size_t *boundary_width) {
    return sz_utf8_word_rfind_boundary_serial(text, length, boundary_width);
}

#endif // !SZ_DYNAMIC_DISPATCH

#pragma endregion // Dynamic Dispatch

#ifdef __cplusplus
}
#endif

#endif // STRINGZILLA_UTF8_WORD_H_
