
/* Copyright (C) 2011-2024 Michael Lugmair (Lucio Carreras)
 *
 * This file is part of sayonara player
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ColumnHeader.h"
#include "Utils/Language/Language.h"
#include "Utils/Settings/Settings.h"

#include <QAction>

#include <algorithm>

using namespace Library;

struct ColumnHeader::Private
{
	int defaultSize;
	QMap<Qt::SortOrder, VariableSortorder> sortorderMap;
	ColumnIndex::IntegerType columnIndex;
	bool isSwitchable;
	bool isStretchable;

	Private(const ColumnIndex::IntegerType columnIndex, const bool isSwitchable,
	        const VariableSortorder sortAsc, const VariableSortorder sortDesc,
	        const int preferredWidth, const bool isStretchable) :
		defaultSize{preferredWidth},
		sortorderMap{
			{Qt::SortOrder::AscendingOrder, sortAsc},
			{Qt::SortOrder::DescendingOrder, sortDesc},
		},
		columnIndex{columnIndex},
		isSwitchable{isSwitchable},
		isStretchable{isStretchable} {}
};

ColumnHeader::~ColumnHeader() = default;

ColumnHeader::ColumnHeader(const ColumnIndex::IntegerType columnIndex, const bool isSwitchable,
                           const VariableSortorder sortAsc, const VariableSortorder sortDesc, const int preferredWidth,
                           const bool isStretchable)
{
	m = Pimpl::make<Private>(columnIndex, isSwitchable, sortAsc, sortDesc, preferredWidth, isStretchable);
}

QString ColumnHeader::hash() const
{
	return hashPrefix() + QString::number(static_cast<int>(m->columnIndex));
}

bool ColumnHeader::isSwitchable() const { return m->isSwitchable; }

bool ColumnHeader::isStretchable() const { return m->isStretchable; }

int ColumnHeader::defaultSize() const { return m->defaultSize; }

VariableSortorder ColumnHeader::sortorder(const Qt::SortOrder qtSortorder) const
{
	return m->sortorderMap[qtSortorder];
}

ColumnIndex::IntegerType ColumnHeader::columnIndex() const { return m->columnIndex; }

ColumnHeaderTrack::ColumnHeaderTrack(const ColumnIndex::Track columnIndex, const bool isSwitchable,
                                     const TrackSortorder sortAsc, const TrackSortorder sortDesc,
                                     const int preferredWidth, const bool isStretchable) :
	ColumnHeader(static_cast<ColumnIndex::IntegerType>(columnIndex), isSwitchable, sortAsc, sortDesc, preferredWidth,
	             isStretchable) {}

QString ColumnHeaderTrack::title() const
{
	switch(static_cast<ColumnIndex::Track>(columnIndex()))
	{
		case ColumnIndex::Track::TrackNumber:
			return QStringLiteral("#");
		case ColumnIndex::Track::Title:
			return Lang::get(Lang::Title).toFirstUpper();
		case ColumnIndex::Track::Artist:
			return Lang::get(Lang::Artist).toFirstUpper();
		case ColumnIndex::Track::AlbumArtist:
			return Lang::get(Lang::AlbumArtist).toFirstUpper();
		case ColumnIndex::Track::Album:
			return Lang::get(Lang::Album).toFirstUpper();
		case ColumnIndex::Track::Discnumber:
			return Lang::get(Lang::Disc);
		case ColumnIndex::Track::Year:
			return Lang::get(Lang::Year).toFirstUpper();
		case ColumnIndex::Track::Length:
			return Lang::get(Lang::DurationShort).toFirstUpper();
		case ColumnIndex::Track::Bitrate:
			return Lang::get(Lang::Bitrate).toFirstUpper();
		case ColumnIndex::Track::Filesize:
			return Lang::get(Lang::Filesize).toFirstUpper();
		case ColumnIndex::Track::Filetype:
			return Lang::get(Lang::Filetype).toFirstUpper();
		case ColumnIndex::Track::AddedDate:
			return Lang::get(Lang::Created).toFirstUpper();
		case ColumnIndex::Track::ModifiedDate:
			return Lang::get(Lang::Modified).toFirstUpper();
		case ColumnIndex::Track::Rating:
			return Lang::get(Lang::Rating).toFirstUpper();
		case ColumnIndex::Track::Genres:
			return Lang::get(Lang::Genres).toFirstUpper();
		default:
			return {};
	}
}

QString ColumnHeaderTrack::hashPrefix() const { return QStringLiteral("track"); }

ColumnHeaderAlbum::ColumnHeaderAlbum(const ColumnIndex::Album columnIndex, const bool isSwitchable,
                                     const AlbumSortorder sortAsc, const AlbumSortorder sortDesc,
                                     const int preferredWidth, const bool isStretchable) :
	ColumnHeader(static_cast<ColumnIndex::IntegerType>(columnIndex), isSwitchable, sortAsc, sortDesc, preferredWidth,
	             isStretchable) {}

QString ColumnHeaderAlbum::title() const
{
	switch(static_cast<ColumnIndex::Album>(columnIndex()))
	{
		case ColumnIndex::Album::MultiDisc:
			return QStringLiteral("#");
		case ColumnIndex::Album::Name:
			return Lang::get(Lang::Album).toFirstUpper();
		case ColumnIndex::Album::AlbumArtist:
			return Lang::get(Lang::AlbumArtist).toFirstUpper();
		case ColumnIndex::Album::Duration:
			return Lang::get(Lang::DurationShort).toFirstUpper();
		case ColumnIndex::Album::NumSongs:
			return Lang::get(Lang::NumTracks).toFirstUpper();
		case ColumnIndex::Album::Year:
			return Lang::get(Lang::Year).toFirstUpper();
		case ColumnIndex::Album::Rating:
			return Lang::get(Lang::Rating).toFirstUpper();
		default:
			return QString();
	}
}

QString ColumnHeaderAlbum::hashPrefix() const { return QStringLiteral("album"); }

ColumnHeaderArtist::ColumnHeaderArtist(const ColumnIndex::Artist columnIndex, const bool isSwitchable,
                                       const ArtistSortorder sortAsc, const ArtistSortorder sortDesc,
                                       const int preferredWidth, const bool isStretchable) :
	ColumnHeader(static_cast<ColumnIndex::IntegerType>(columnIndex), isSwitchable, sortAsc, sortDesc, preferredWidth,
	             isStretchable) {}

QString ColumnHeaderArtist::title() const
{
	switch(static_cast<ColumnIndex::Artist>(columnIndex()))
	{
		case ColumnIndex::Artist::Name:
			return GetSetting(Set::Lib_ShowAlbumArtists)
			       ? Lang::get(Lang::AlbumArtist)
			       : Lang::get(Lang::Artist);

		case ColumnIndex::Artist::Tracks:
			return Lang::get(Lang::NumTracks).toFirstUpper();
		default:
			return QString();
	}
}

QString ColumnHeaderArtist::hashPrefix() const { return QStringLiteral("artist"); }

