/*
 Copyright © 2021-2023  TokiNoBug
This file is part of SlopeCraft.

    SlopeCraft is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    SlopeCraft is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with SlopeCraft. If not, see <https://www.gnu.org/licenses/>.

    Contact with me:
    github:https://github.com/SlopeCraft/SlopeCraft
    bilibili:https://space.bilibili.com/351429231
*/

#ifndef SCHEM_SCHEM_H
#define SCHEM_SCHEM_H

#include <MCDataVersion.h>
#include <string>
#include <string_view>
#include <type_traits>
#include <Eigen/Dense>
#include <unsupported/Eigen/CXX11/Tensor>
#include <vector>
#include <span>
#include <cereal/access.hpp>
#include <cereal/cereal.hpp>
#include <cereal/types/string.hpp>
#include <cereal/types/vector.hpp>
#include <exception>
#include <concepts>

#include <boost/multi_array.hpp>

#include "SC_GlobalEnums.h"
#include "entity.h"

namespace libSchem {
// template <int64_t max_block_count = 256>

struct litematic_info {
  litematic_info();
  std::string litename_utf8{"Litematic generated by SlopeCraft."};
  std::string regionname_utf8{"Software developer : TokiNoBug."};
  std::string author_utf8{"SlopeCraft"};
  std::string destricption_utf8{"This litematic is generated by SlopeCraft."};
  uint64_t time_created;   //< Miliseconds since 1970
  uint64_t time_modified;  //< Miliseconds since 1970
};

struct WorldEditSchem_info {
  WorldEditSchem_info();
  std::array<int, 3> offset{0, 0, 0};
  std::array<int, 3> WE_offset{0, 0, 0};
  std::string schem_name_utf8{
      "WorldEdit schem generated by SlopeCraft, deveploer TokiNoBug."};
  std::string author_utf8{"SlopeCraft"};
  std::vector<std::string> required_mods_utf8{};
  uint64_t date;  //< Miliseconds since 1970
};

class Schem {
 public:
  // using ele_t = std::conditional_t<(max_block_count > 256), uint16_t,
  // uint8_t>;
  using ele_t = uint16_t;

  static constexpr ele_t invalid_ele_t = ~ele_t(0);

 private:
  /// The 3 indices are stored in [x][z][y] col-major, and in minecraft the
  /// best storage is [y][z][x] row-major
  Eigen::Tensor<ele_t, 3> xzy;

  std::vector<std::string> block_id_list;

  ::SCL_gameVersion MC_major_ver;
  MCDataVersion::MCDataVersion_t MC_data_ver;

  std::vector<std::unique_ptr<entity>> entities;

 public:
  Schem() { xzy.resize(0, 0, 0); }
  Schem(const Schem &src) noexcept
      : block_id_list{src.block_id_list},
        MC_major_ver{src.MC_major_ver},
        MC_data_ver{src.MC_data_ver},
        xzy{src.xzy} {
    this->entities.reserve(src.entities.size());
    for (auto &entity : src.entities) {
      this->entities.emplace_back(entity->clone());
    }
  }
  Schem(Schem &&) = default;
  Schem(int64_t x, int64_t y, int64_t z) {
    xzy.resize(x, z, y);
    xzy.setZero();
  }

  Schem &operator=(Schem &&src) = default;
  Schem &operator=(const Schem &src) noexcept {
    Schem temp{src};
    std::swap(*this, temp);
    return *this;
  }

  std::string check_size() const noexcept;
  static std::string check_size(int64_t x, int64_t y, int64_t z) noexcept;

  inline void set_zero() noexcept { xzy.setZero(); }

  inline ele_t *data() noexcept { return xzy.data(); }

  inline const ele_t *data() const noexcept { return xzy.data(); }

  void resize(int64_t x, int64_t y, int64_t z);

  inline bool check_version_id() const noexcept {
    return MCDataVersion::is_data_version_suitable(this->MC_major_ver,
                                                   this->MC_data_ver);
  }

  inline const auto &tensor() const noexcept { return this->xzy; }

  inline const auto &palette() const noexcept { return this->block_id_list; }

  inline ele_t &operator()(int64_t x, int64_t y, int64_t z) noexcept {
    assert(x >= 0 && x < this->x_range());
    assert(y >= 0 && y < this->y_range());
    assert(z >= 0 && z < this->z_range());
    return xzy(x, z, y);
  }

  inline const ele_t &operator()(int64_t x, int64_t y,
                                 int64_t z) const noexcept {
    assert(x >= 0 && x < this->x_range());
    assert(y >= 0 && y < this->y_range());
    assert(z >= 0 && z < this->z_range());
    return xzy(x, z, y);
  }

  inline ele_t &operator()(int64_t idx) noexcept {
    assert(idx >= 0 && idx < this->size());
    return xzy(idx);
  }

  inline const ele_t &operator()(int64_t idx) const noexcept {
    assert(idx >= 0 && idx < this->size());
    return xzy(idx);
  }

  inline const char *id_at(int64_t x, int64_t y, int64_t z) const noexcept {
    assert(x >= 0 && x < this->x_range());
    assert(y >= 0 && y < this->y_range());
    assert(z >= 0 && z < this->z_range());
    return block_id_list[xzy(x, z, y)].c_str();
  }

  inline const char *id_at(int64_t idx) const noexcept {
    assert(idx >= 0 && idx < this->size());
    return block_id_list[xzy(idx)].c_str();
  }

  inline ele_t *begin() noexcept { return xzy.data(); }

  inline const ele_t *begin() const noexcept { return xzy.data(); }

  inline ele_t *end() noexcept { return xzy.data() + xzy.size(); }
  inline const ele_t *end() const noexcept { return xzy.data() + xzy.size(); }

  inline void fill(const ele_t _) noexcept {
    for (uint16_t &val : *this) {
      val = _;
    }
  }

  void stat_blocks(std::vector<size_t> &dest) const noexcept;
  [[nodiscard]] std::vector<size_t> stat_blocks() const noexcept {
    std::vector<size_t> buf;
    this->stat_blocks(buf);
    return buf;
  }

  inline int64_t x_range() const noexcept { return xzy.dimension(0); }
  inline int64_t y_range() const noexcept { return xzy.dimension(2); }
  inline int64_t z_range() const noexcept { return xzy.dimension(1); }

  inline size_t palette_size() const noexcept { return block_id_list.size(); }

  inline int64_t size() const noexcept { return xzy.size(); }

  int64_t non_zero_count() const noexcept;

  /// Schem will copy these strings
  void set_block_id(const char *const *const block_ids, const int num) noexcept;
  void set_block_id(std::span<std::string_view> id) noexcept;

  inline MCDataVersion::MCDataVersion_t MC_version_number() const noexcept {
    return this->MC_data_ver;
  }

  inline void set_MC_version_number(
      const MCDataVersion::MCDataVersion_t _) noexcept {
    this->MC_data_ver = _;
  }

  inline ::SCL_gameVersion MC_major_version_number() const noexcept {
    return this->MC_major_ver;
  }

  inline void set_MC_major_version_number(const ::SCL_gameVersion _) noexcept {
    this->MC_major_ver = _;
  }

  /**
   * \brief Search for invalid block.
   *
   * \note Invalid block are blocks which has an id(uint16_t) greater than or
   * equal to the block type number. For example, if there are only 5 kinds of
   * blocks, then id 5 and greater are invalid.
   *
   * \param first_invalid_block_idx The index of the first invalid block
   * \return true The schem contains invalid block(s)
   * \return false The schem don't contain any invalid block.
   */
  bool have_invalid_block(
      int64_t *first_invalid_block_idx = nullptr) const noexcept;

  /**
   * \brief Search for invalid block.
   *
   * \param first_invalid_block_x_pos The x position of the first invalid block
   * \param first_invalid_block_y_pos The y position of the first invalid block
   * \param first_invalid_block_z_pos The z position of the first invalid block
   * \return true The schem contains invalid block(s)
   * \return false The schem don't contain any invalid block.
   */
  bool have_invalid_block(int64_t *first_invalid_block_x_pos,
                          int64_t *first_invalid_block_y_pos,
                          int64_t *first_invalid_block_z_pos) const noexcept;

  void process_mushroom_states() noexcept;

  void process_mushroom_states_fast() noexcept;

  auto &entity_list() noexcept { return this->entities; }
  auto &entity_list() const noexcept { return this->entities; }

  template <class T>
  struct schem_slice {
    Eigen::Array<int64_t, 3, 1> offset;
    T content;
  };

  [[nodiscard]] tl::expected<boost::multi_array<schem_slice<Schem>, 3>,
                             std::string>
  split_by_block_size(std::span<const uint64_t> x_block_length,
                      std::span<const uint64_t> y_block_length,
                      std::span<const uint64_t> z_block_length) const noexcept;

  struct remove_unused_id_result {
    size_t id_count_before;
    size_t id_count_after;
  };

  [[nodiscard]] tl::expected<remove_unused_id_result, std::string>
  remove_unused_ids() noexcept;

 protected:
  [[nodiscard]] Schem slice_no_check(
      std::span<const std::pair<int64_t, int64_t>, 3> xyz_index_range)
      const noexcept;

 public:
  static void update_error_dest(
      const tl::expected<void, std::pair<SCL_errorFlag, std::string>> &result,
      SCL_errorFlag *const error_flag, std::string *const error_str) noexcept {
    if (result) {
      if (error_flag not_eq nullptr) {
        *error_flag = SCL_errorFlag::NO_ERROR_OCCUR;
      }
      if (error_str not_eq nullptr) {
        error_str->clear();
      }
    } else {
      auto &err = result.error();
      if (error_flag not_eq nullptr) {
        *error_flag = err.first;
      }
      if (error_str not_eq nullptr) {
        *error_str = err.second;
      }
    }
  }
  tl::expected<void, std::pair<SCL_errorFlag, std::string>> export_litematic(
      std::string_view filename, const litematic_info &info) const noexcept;

  tl::expected<void, std::pair<SCL_errorFlag, std::string>> export_structure(
      std::string_view filename,
      const bool is_air_structure_void) const noexcept;

  tl::expected<void, std::pair<SCL_errorFlag, std::string>> export_WESchem(
      std::string_view filename,
      const WorldEditSchem_info &info) const noexcept;

  [[deprecated]] bool export_litematic(
      std::string_view filename, const litematic_info &info,
      SCL_errorFlag *const error_flag,
      std::string *const error_str) const noexcept {
    auto res = this->export_litematic(filename, info);
    update_error_dest(res, error_flag, error_str);
    return res.has_value();
  }

  [[deprecated]] bool export_structure(
      std::string_view filename, const bool is_air_structure_void,
      SCL_errorFlag *const error_flag,
      std::string *const error_str) const noexcept {
    auto res = this->export_structure(filename, is_air_structure_void);
    update_error_dest(res, error_flag, error_str);
    return res.has_value();
  }

  [[deprecated]] bool export_WESchem(
      std::string_view filename, const WorldEditSchem_info &info,
      SCL_errorFlag *const error_flag,
      std::string *const error_str) const noexcept {
    auto res = this->export_WESchem(filename, info);
    update_error_dest(res, error_flag, error_str);
    return res.has_value();
  }

 private:
  friend class cereal::access;

  tl::expected<void, std::pair<SCL_errorFlag, std::string>> pre_check(
      std::string_view filename, std::string_view extension) const noexcept;

  template <class archive>
  void save(archive &ar) const {
    ar(this->MC_major_ver);
    ar(this->MC_data_ver);
    ar(this->block_id_list);
    const int64_t x{this->x_range()}, y{this->y_range()}, z{this->z_range()};
    ar(x, y, z);
    ar(cereal::binary_data(this->xzy.data(),
                           this->xzy.size() * sizeof(uint16_t)));
  }

  template <class archive>
  void load(archive &ar) {
    ar(this->MC_major_ver);
    ar(this->MC_data_ver);
    ar(this->block_id_list);
    int64_t x, y, z;
    ar(x, y, z);
    {
      std::string err = check_size(x, y, z);
      if (!err.empty()) {
        throw std::runtime_error{err};
      }
    }
    this->resize(x, y, z);
    ar(cereal::binary_data(this->xzy.data(),
                           this->xzy.size() * sizeof(uint16_t)));
  }
};

/**
 * Find minimum value >= a that can is multiple of b
 * @tparam int_t
 * @param a
 * @param b
 * @return
 */
template <typename int_t, typename uint_t>
  requires std::integral<int_t> && std::integral<uint_t>
constexpr int_t ceil_up_to(int_t a, uint_t b) {
  if (a % b == 0) {
    return a;
  }
  return ((a / b) + 1) * b;
}

}  // namespace libSchem

#endif  // SCHEM_SCHEM_H