#include "nowplaying.hpp"
#include "clickableLabel.hpp"
#include <QMenu>
#include <QGraphicsDropShadowEffect>

#include "mainwindow.hpp"


Context::NowPlaying::NowPlaying(QWidget *parent)
	: QWidget(parent)
{
	layout = new QVBoxLayout(this);
	layout->setSpacing(0);

	name = newLabel(nameScale);
	artist = newLabel(artistScale);

	if (lib::developer_mode::enabled)
	{
		setContextMenuPolicy(Qt::ContextMenuPolicy::CustomContextMenu);

		QLabel::connect(this, &QWidget::customContextMenuRequested,
			this, &Context::NowPlaying::onMenu);
	}
}

void Context::NowPlaying::setTrack(const lib::spt::track &track)
{
	if (name != nullptr)
	{
        track_id = track.id;
        album_id = track.album.id;
		name->setVisible(true);
		setText(name, track.name);
	}

	if (artist != nullptr)
	{
        artist_id = track.artists[0].id;
		const auto names = lib::spt::entity::combine_names(track.artists);
		setText(artist, names);
	}
}


void Context::NowPlaying::setNoPlaying()
{
	if (name != nullptr)
	{
		name->setVisible(false);
	}

	if (artist != nullptr)
	{
		setText(artist, "No music playing");
	}
}

auto Context::NowPlaying::newLabel(float scale) -> QLabel *
{
	auto *label = new ClickableLabel(this);
	label->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Fixed);

	auto font = label->font();
	auto pointSize = font.pointSizeF() * scale;
	font.setPointSizeF(pointSize);
	label->setFont(font);

	layout->addWidget(label);

    connect(label, &ClickableLabel::clicked, this, [this, label](){
        handleLabelClick(label);
    });
	return label;
}



void Context::NowPlaying::setText(QLabel *label, const std::string &text)
{
	auto qText = QString::fromStdString(text);
	label->setText(qText);
	label->setToolTip(qText);
}

auto Context::NowPlaying::getTextShadow() const -> bool
{
	return name->graphicsEffect() != nullptr
		&& artist->graphicsEffect() != nullptr;
}

void Context::NowPlaying::setTextShadow(bool value)
{
	if (value)
	{
		addTextShadow(name);
		addTextShadow(artist);
	}
	else
	{
		removeTextShadow(name);
		removeTextShadow(artist);
	}
}

void Context::NowPlaying::addTextShadow(QLabel *label)
{
	if (label->graphicsEffect() != nullptr)
	{
		return;
	}

	// Force white text to match text shadow
	auto palette = label->palette();
	palette.setColor(label->foregroundRole(), Qt::white);
	label->setPalette(palette);

	// Always dark text shadow
	auto *dropShadow = new QGraphicsDropShadowEffect(label);
	dropShadow->setColor(Qt::black);
	dropShadow->setOffset(1);
	dropShadow->setBlurRadius(2);
	label->setGraphicsEffect(dropShadow);
}

void Context::NowPlaying::removeTextShadow(QLabel *label)
{
	label->setPalette(QLabel().palette());
	label->setGraphicsEffect(nullptr);
}

void Context::NowPlaying::onMenu(const QPoint &pos)
{
	auto *menu = new QMenu(this);
	auto *textShadow = menu->addAction(QStringLiteral("Text shadow"));
	textShadow->setCheckable(true);
	textShadow->setChecked(getTextShadow());

	QAction::connect(textShadow, &QAction::triggered, [this](bool checked)
	{
		setTextShadow(checked);
	});

	menu->popup(mapToGlobal(pos));
}

void Context::NowPlaying::handleLabelClick(QLabel *clickedLabel){
    if(clickedLabel == artist){
    MainWindow::find(parentWidget())->openArtist(artist_id);
    }
    else if(clickedLabel == name){
        MainWindow::find(parentWidget())->loadAlbum(album_id, track_id);
    }
}
