# scale_spin_button.py: Spin button for setting factors.
#
# Copyright (C) 2024 Upscaler Contributors
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-only

from typing import Any, cast

from gi.repository import GObject, Gdk, Gtk

from upscaler.logger import logger


@Gtk.Template.from_resource(
    "/io/gitlab/theevilskeleton/Upscaler/scale-spin-button.ui",
)
class ScaleSpinButton(Gtk.Widget, Gtk.Accessible, Gtk.AccessibleRange):  # type: ignore[misc]
    """A custom spin button widget."""

    __gtype_name__ = "ScaleSpinButton"

    factor: Gtk.Text = Gtk.Template.Child()
    decrement: Gtk.Button = Gtk.Template.Child()
    increment: Gtk.Button = Gtk.Template.Child()

    def __init__(self, **kwargs: Any) -> None:
        """Initialize custom spin button widget."""
        super().__init__(**kwargs)

        self.factor.set_accessible_parent(self, None)

    def _value_changed_cb(self, *_args: Any) -> None:
        value = self.adjustment.props.value
        text = str(int(value))
        accessible_text = cast("Gtk.AccessibleProperty", text)

        self.update_property(
            *zip(
                (Gtk.AccessibleProperty.VALUE_NOW, value),
                (Gtk.AccessibleProperty.VALUE_TEXT, accessible_text),
            )
        )

        self.decrement.props.sensitive = value != self.adjustment.props.lower
        self.increment.props.sensitive = value != self.adjustment.props.upper

        buffer = self.factor.props.buffer
        buffer.props.text = text

        logger.info(f"Scale value: {text}")

    @Gtk.Template.Callback()
    def _key_pressed_cb(self, _: Any, key: int, *_args: Any) -> bool:
        value = self.get_value()
        is_greater_than_lower = value > self.adjustment.props.lower
        is_smaller_than_upper = value < self.adjustment.props.upper
        position = self.factor.get_position()

        match key:
            case Gdk.KEY_Down if is_greater_than_lower:
                self.adjustment.props.value -= self.adjustment.props.step_increment
            case Gdk.KEY_Up if is_smaller_than_upper:
                self.adjustment.props.value += self.adjustment.props.step_increment
            case Gdk.KEY_Page_Down if is_greater_than_lower:
                self.adjustment.props.value -= self.adjustment.props.page_increment
            case Gdk.KEY_Page_Up if is_smaller_than_upper:
                self.adjustment.props.value += self.adjustment.props.page_increment
            case Gdk.KEY_Up | Gdk.KEY_Page_Up:
                self.keynav_failed(Gtk.DirectionType.UP)
            case Gdk.KEY_Down | Gdk.KEY_Page_Down:
                self.keynav_failed(Gtk.DirectionType.DOWN)
            case _:
                return False

        self.factor.set_position(position)
        return True

    @Gtk.Template.Callback()
    def _has_focus_cb(self, *_args: Any) -> None:
        if not self.factor.has_focus():
            self._enter_clicked_cb()

    @Gtk.Template.Callback()
    def _decrement_button_clicked_cb(self, *_args: Any) -> None:
        self.adjustment.props.value -= self.adjustment.props.step_increment

        self.grab_focus()
        self.factor.set_position(-1)

    @Gtk.Template.Callback()
    def _increment_button_clicked_cb(self, *_args: Any) -> None:
        self.adjustment.props.value += self.adjustment.props.step_increment

        self.grab_focus()
        self.factor.set_position(-1)

    @Gtk.Template.Callback()
    def _enter_clicked_cb(self, *_args: Any) -> None:
        text = self.factor.props.text
        position = self.factor.get_position()

        try:
            value = int(text)
        except ValueError:
            self.factor.props.text = str(self.adjustment.props.value)
        else:
            self.adjustment.props.value = value
            buffer = self.factor.props.buffer
            buffer.props.text = str(self.adjustment.props.value)

        self.factor.set_position(position)

    @GObject.Property(type=Gtk.Adjustment)
    def adjustment(self) -> Gtk.Adjustment | None:
        """Get or set the adjustment that holds the value of the spin button."""
        return self._adjustment

    @adjustment.setter  # type: ignore [no-redef]
    def adjustment(self, adjustment: Gtk.Adjustment) -> None:
        self._adjustment = adjustment

        self.update_property(
            *zip(
                (Gtk.AccessibleProperty.VALUE_MAX, self.adjustment.props.upper),
                (Gtk.AccessibleProperty.VALUE_MIN, self.adjustment.props.lower),
            )
        )

        adjustment.connect("value-changed", self._value_changed_cb)
        self._value_changed_cb()

    def do_get_platform_state(self, state: Gtk.AccessiblePlatformState) -> bool:
        """Query a platform state."""
        if state == Gtk.AccessiblePlatformState.FOCUSED:
            return self.factor.has_focus()

        return self.factor.get_platform_state(state)

    def do_grab_focus(self) -> bool:
        """Grab focus."""
        return self.factor.grab_focus()

    def get_value(self) -> int:
        """Return the current value as an integer."""
        return int(self.adjustment.props.value)
