# window.py: main window
#
# Copyright (C) 2022 Upscaler Contributors
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-only

import asyncio
import contextlib
import io
import os
import re
import sys
import tempfile
from gettext import gettext as _
from locale import atof
from pathlib import Path
from typing import TYPE_CHECKING, Any, Literal, cast

import vulkan
from PIL import Image, ImageChops, ImageOps
from gi.repository import Adw, GLib, Gdk, Gio, Gtk, Pango

from upscaler import ALG_WARNINGS, APP_ID
from upscaler.exceptions import (
    AlgorithmFailedError,
    AlgorithmWarning,
    InvalidApplicationError,
)
from upscaler.logger import logger
from upscaler.media import (
    ImageFile,
    MediaFile,
    create_texture_from_img,
    image_formats,
)
from upscaler.queue_row import QueueRow
from upscaler.scale_comparison_frame import ScaleComparisonFrame
from upscaler.scale_spin_button import ScaleSpinButton

if TYPE_CHECKING:
    from upscaler.main import Application


@Gtk.Template.from_resource("/io/gitlab/theevilskeleton/Upscaler/window.ui")
class Window(Adw.ApplicationWindow):
    """Application window."""

    __gtype_name__ = "Window"

    nth_temporary_file = 1

    # Declare child widgets
    toast: Adw.ToastOverlay = Gtk.Template.Child()
    stack_upscaler: Gtk.Stack = Gtk.Template.Child()
    stack_picture: Gtk.Stack = Gtk.Template.Child()
    status_welcome: Adw.StatusPage = Gtk.Template.Child()
    button_open: Gtk.Button = Gtk.Template.Child()
    loading_page_spinner: Adw.Spinner = Gtk.Template.Child()
    loading_image_loading: Adw.Spinner = Gtk.Template.Child()
    # video = Gtk.Template.Child()
    combo_models: Adw.ComboRow = Gtk.Template.Child()
    string_models: Gtk.StringList = Gtk.Template.Child()
    spin_scale: ScaleSpinButton = Gtk.Template.Child()
    drag_revealer: Gtk.Revealer = Gtk.Template.Child()
    main_toolbar_view: Adw.ToolbarView = Gtk.Template.Child()
    main_nav_view: Adw.NavigationView = Gtk.Template.Child()
    options_page: Adw.PreferencesPage = Gtk.Template.Child()
    queue_list_box: Gtk.ListBox = Gtk.Template.Child()
    progress_list_box: Gtk.ListBox = Gtk.Template.Child()
    completed_list_box: Gtk.ListBox = Gtk.Template.Child()
    main_view_header_bar: Adw.HeaderBar = Gtk.Template.Child()
    image_carousel: Adw.Carousel = Gtk.Template.Child()
    previous_button: Gtk.Button = Gtk.Template.Child()
    next_button: Gtk.Button = Gtk.Template.Child()

    def __init__(self, **kwargs: Any) -> None:
        """Initialize application window."""
        super().__init__(**kwargs)

        self.application = cast("Application", self.get_application())
        if not self.application:
            raise InvalidApplicationError(self.application)

        self.application.create_action("quit", lambda *_: self.close(), ("<primary>q",))
        self.application.create_action("open", self._open_file, ("<primary>o",))
        self.application.create_action("paste-image", self._on_paste, ("<primary>v",))

        settings = Gio.Settings.new("io.gitlab.theevilskeleton.Upscaler")
        settings.bind("width", self, "default-width", Gio.SettingsBindFlags.DEFAULT)
        settings.bind("height", self, "default-height", Gio.SettingsBindFlags.DEFAULT)
        settings.bind("is-maximized", self, "maximized", Gio.SettingsBindFlags.DEFAULT)

        # Check if hardware is supported
        if os.environ.get("DEBUG_DISABLE_VULKAN_CHECK") != "1":
            GLib.idle_add(self._vulkaninfo)
        else:
            logger.warning("Skipping Vulkan check")

        # Set icon on welcome page
        self.status_welcome.set_icon_name(APP_ID)

        # Declare default models and variables
        self.model_images = {
            "realesrgan-x4plus": _("Photo"),
            "realesrgan-x4plus-anime": _("Digital Art"),
        }

        # Suppress "DecompressionBombWarning" warning
        # https://pillow.readthedocs.io/en/stable/reference/Image.html#PIL.Image.open
        Image.MAX_IMAGE_PIXELS = None

        self.media_file = MediaFile(Path())

        self.string_models.splice(0, 0, tuple(self.model_images.values()))
        self.busy: bool = False

        self.stack_upscaler.set_visible_child_name("stack_welcome_page")
        self.current_stack_page_name: str = "stack_welcome_page"

        # self.model_videos = [
        #     'realesr-animevideov3',
        # ]

    @Gtk.Template.Callback()
    def _go_previous(self, *_args: Any) -> None:
        new_position = int(self.image_carousel.get_position()) - 1
        self.image_carousel.scroll_to(
            self.image_carousel.get_nth_page(new_position), True
        )

    @Gtk.Template.Callback()
    def _go_next(self, *_args: Any) -> None:
        new_position = int(self.image_carousel.get_position()) + 1
        self.image_carousel.scroll_to(
            self.image_carousel.get_nth_page(new_position), True
        )

    @Gtk.Template.Callback()
    def _on_position_changed(self, *_args: Any) -> None:
        position = self.image_carousel.get_position()
        if 0 <= position <= 1:
            self.previous_button.set_opacity(position)
            self.previous_button.set_visible(bool(position))
        n_pages = self.image_carousel.get_n_pages()
        if n_pages - 2 <= position <= n_pages - 1:
            difference = self.image_carousel.get_n_pages() - 1 - position
            self.next_button.set_opacity(difference)
            self.next_button.set_visible(bool(difference))

    def _create_thumbnail_texture(self, image: Image.Image) -> Gdk.Texture:
        image.thumbnail((512, 512))
        return create_texture_from_img(image)

    async def on_load_texture(self, file: Gio.File, texture: Gdk.Texture) -> None:
        """Load and save a given texture."""
        try:
            data: tuple[Any, ...]

            self.start_loading_animation()
            file_path = Path(str(file.get_path()))
            data = await asyncio.gather(asyncio.to_thread(texture.save_to_png_bytes))
            gbytes = data[0]
            buffer = cast("bytes", gbytes.get_data())

            with Image.open(io.BytesIO(buffer)) as image:
                thumbnail_thread = asyncio.to_thread(
                    self._create_thumbnail_texture, image
                )
                save_thread = asyncio.to_thread(
                    # Instance method
                    file.replace_contents_bytes_async,
                    # Parameters
                    gbytes,
                    None,
                    False,
                    Gio.FileCreateFlags.REPLACE_DESTINATION,
                    None,
                    None,
                )
                data = await asyncio.gather(thumbnail_thread, save_thread)
        except Exception as error:
            logger.exception(error)
        else:
            texture = data[0]

            self.media_file = ImageFile(file_path)

            self.image_carousel.append(
                ScaleComparisonFrame(
                    self.media_file, texture, self.spin_scale.get_value()
                ),
            )
            self._on_position_changed()

            transposed_image = ImageOps.exif_transpose(image)
            diff = ImageChops.difference(transposed_image, image)
            if diff.getbbox():
                temporary_path = tempfile.mkstemp(suffix=f".{image.format}")[1]
                transposed_image.save(temporary_path, quality=100, subsampling=0)
                self.media_file.temporary_path = Path(temporary_path)

            self._set_post_upscale_image_size()

            if self.main_nav_view.get_visible_page_tag() != "upscaling-options":
                self.main_nav_view.push_by_tag("upscaling-options")
        finally:
            self.stop_loading_animation()

    async def on_load_file(self, file_path: Path) -> None:
        """Load a given file."""
        data: tuple[Any, ...]

        try:
            with Image.open(file_path) as image:
                self.start_loading_animation()
                thumbnail = image

                data = await asyncio.gather(
                    asyncio.to_thread(ImageOps.exif_transpose, image)
                )
                transposed_image = data[0]
                diff = ImageChops.difference(transposed_image, image)
                if diff.getbbox():
                    temporary_path = tempfile.mkstemp(suffix=f".{image.format}")[1]
                    await asyncio.to_thread(
                        transposed_image.save,
                        temporary_path,
                        quality=100,
                        subsampling=0,
                    )
                    self.media_file.temporary_path = Path(temporary_path)
                    thumbnail = transposed_image

                data = await asyncio.gather(
                    asyncio.to_thread(self._create_thumbnail_texture, thumbnail)
                )
        except Exception as error:
            message = _("“{path}” is not a valid image").format(
                path=file_path.name or "?",
            )
            self.toast.add_toast(Adw.Toast.new(message))
            logger.exception(error)
        else:
            texture = data[0]

            self.media_file = ImageFile(file_path)

            self.image_carousel.append(
                ScaleComparisonFrame(
                    self.media_file, texture, self.spin_scale.get_value()
                ),
            )
            self._on_position_changed()

            self._set_post_upscale_image_size()

            if self.main_nav_view.get_visible_page_tag() != "upscaling-options":
                self.main_nav_view.push_by_tag("upscaling-options")
        finally:
            self.stop_loading_animation()

    @Gtk.Template.Callback()
    def _open_file(self, *_args: Any) -> None:
        """Open the file chooser to load the file."""

        def load_files(_dialog: Gtk.FileDialog, result: Gio.AsyncResult) -> None:
            try:
                files = dialog.open_multiple_finish(result)
            except GLib.GError:  # type: ignore [misc]
                return

            for file in files:
                if isinstance(file, Gio.File):
                    file_path = Path(str(file.get_path()))
                    task = asyncio.create_task(self.on_load_file(file_path))
                    self.application.tasks.add(task)

        filters = Gio.ListStore.new(Gtk.FileFilter)

        filters.append(
            Gtk.FileFilter(name=_("Supported Image Files"), mime_types=image_formats)
        )

        dialog = Gtk.FileDialog.new()
        dialog.set_title(_("Select Image"))
        dialog.set_filters(filters)

        dialog.open_multiple(self, callback=load_files)

    def _on_paste(self, *_args: Any) -> None:
        """Attempt to load the file from the clipboard."""

        def on_file_pasted(clipboard: Gdk.Clipboard, result: Gio.Task) -> None:
            try:
                files = clipboard.read_value_finish(result)
            except GLib.GError:  # type: ignore [misc]
                clipboard.read_texture_async(None, on_texture_pasted)
            else:
                for file in files:
                    file_path = Path(str(file.get_path()))
                    task = asyncio.create_task(self.on_load_file(file_path))
                    self.application.tasks.add(task)

        def on_texture_pasted(clipboard: Gdk.Clipboard, result: Gio.Task) -> None:
            try:
                paste_as_texture = cast(
                    "Gdk.Texture", clipboard.read_texture_finish(result)
                )
            except GLib.GError:  # type: ignore [misc]
                message = _("No image found in clipboard")
                self.toast.add_toast(Adw.Toast.new(message))
            else:
                file = Gio.File.new_tmp()[0]
                task = asyncio.create_task(self.on_load_texture(file, paste_as_texture))
                self.application.tasks.add(task)

        display = cast("Gdk.Display", Gdk.Display.get_default())
        clipboard = display.get_clipboard()
        clipboard.read_value_async(Gdk.FileList, 0, None, on_file_pasted)

    @Gtk.Template.Callback()
    def _on_drop(
        self,
        _: Any,
        contents: Gdk.FileList | Gdk.Texture,
        *_args: Any,
    ) -> None:
        """Load image when it has been dropped into the app."""
        if isinstance(contents, Gdk.FileList):
            for file in contents.get_files():
                file_path = Path(str(file.get_path()))
                task = asyncio.create_task(self.on_load_file(file_path))
                self.application.tasks.add(task)
        elif isinstance(contents, Gdk.Texture):
            file = Gio.File.new_tmp()[0]
            task = asyncio.create_task(self.on_load_texture(file, contents))
            self.application.tasks.add(task)

    @Gtk.Template.Callback()
    def _on_enter(self, *_args: Any) -> Literal[Gdk.DragAction.COPY]:
        self.main_nav_view.add_css_class("blurred")
        self.drag_revealer.set_reveal_child(True)
        return Gdk.DragAction.COPY

    @Gtk.Template.Callback()
    def _on_leave(self, *_args: Any) -> None:
        self.main_nav_view.remove_css_class("blurred")
        self.drag_revealer.set_reveal_child(False)

    @Gtk.Template.Callback()
    def _on_child_changed(self, *_args: Any) -> None:
        child_name = self.stack_upscaler.get_visible_child_name()
        match child_name:
            case "stack_loading" | None:
                pass
            case _:
                self.current_stack_page_name = child_name
                is_welcome_page = child_name != "stack_welcome_page"
                self.main_view_header_bar.set_show_title(is_welcome_page)
                self.button_open.set_visible(is_welcome_page)

    def _upscale_progress(self, queue_row: QueueRow, progress: str) -> None:
        """Update upscale progress."""
        queue_row.progressbar.set_fraction(atof(progress) / 100)

    def _run_next(self) -> None:
        def get_first_child(list_box: Gtk.ListBox) -> QueueRow | None:
            row = list_box.get_first_child()
            return cast("QueueRow", row)

        queue_row = get_first_child(self.progress_list_box)
        queued_row = get_first_child(self.queue_list_box)
        self.progress_list_box.set_visible(True)

        if hasattr(queue_row, "process"):
            self.queue_list_box.set_visible(bool(self.queue_list_box.get_first_child()))
            return
        if not queue_row:
            if not queued_row:
                if not bool(self.completed_list_box.get_row_at_index(0)):
                    self.stack_upscaler.set_visible_child_name("stack_welcome_page")
                self.busy = False
                self.progress_list_box.set_visible(False)
                return

            queue_row = queued_row
            self.queue_list_box.remove(queued_row)
            self.queue_list_box.set_visible(bool(self.queue_list_box.get_first_child()))

        self.progress_list_box.append(queue_row)

        queue_row.destination_path = Path(tempfile.mkstemp(suffix=".webp")[1])

        selected_model = tuple(self.model_images)[self.combo_models.get_selected()]

        queue_row.progressbar.set_visible(True)

        async def run() -> None:
            def start_process(running_row: QueueRow) -> None:
                running_row.command = (
                    "upscayl-bin",
                    "-i",
                    str(running_row.media_file.get_preferred_input_path()),
                    "-o",
                    str(running_row.destination_path),
                    "-n",
                    str(selected_model),
                    "-s",
                    str(self.spin_scale.get_value()),
                )

                running_row.run()

                cmd = " ".join(running_row.command)
                logger.info(f"Running: {cmd}")

                # Read each line, query the percentage and update the progress bar
                output = ""
                bad = False
                if running_row.process.stderr is not None:
                    for line in iter(running_row.process.stderr.readline, ""):
                        logger.info(line.strip())
                        output += line
                        if (res := re.match(r"^(\d*.\d+)%$", line)) is not None:
                            GLib.idle_add(
                                self._upscale_progress, running_row, res.group(1)
                            )
                            continue
                        # Check if this line is a warning
                        if bad:
                            continue
                        for warn in ALG_WARNINGS:
                            bad = bad or re.match(warn, line) is not None

                # Process algorithm output
                result = running_row.process.poll()
                if running_row.canceled:
                    logger.info("Manually canceled upscaling by the user")

                elif result != 0:
                    error_value = 0 if result is None else result
                    raise AlgorithmFailedError(error_value, output)

                if bad:
                    raise AlgorithmWarning

            try:
                await asyncio.gather(asyncio.to_thread(start_process, queue_row))
                error_message = None
            except Exception as error:
                logger.exception(error)
                error_message = error

            self.progress_list_box.remove(queue_row)

            if not queue_row.canceled:
                self.upscaling_completed_dialog(queue_row, error_message)

            with contextlib.suppress(TypeError):
                queue_row.media_file.remove_temporary_path()

            self._run_next()

        task = asyncio.create_task(run())
        self.application.tasks.add(task)
        self.busy = True

    @Gtk.Template.Callback()
    def _upscale(self, *_args: Any) -> None:
        """Initialize algorithm and updates widgets."""

        async def set_thumbnail(thumbnail: Gtk.Picture, texture: Gdk.Texture) -> None:
            data: Any
            data = await asyncio.gather(asyncio.to_thread(texture.save_to_png_bytes))
            gbytes = data[0]
            buffer = cast("bytes", gbytes.get_data())

            with Image.open(io.BytesIO(buffer)) as image:
                image.thumbnail(thumbnail.get_size_request())
                data = await asyncio.gather(
                    asyncio.to_thread(create_texture_from_img, image)
                )

            thumbnail.set_paintable(data[0])

        self.progress_list_box.set_visible(True)
        self.stack_upscaler.set_visible_child_name("stack_queue_page")
        self.main_nav_view.pop()

        while item := self.image_carousel.get_first_child():
            if not isinstance(item, ScaleComparisonFrame):
                break

            queue_row = QueueRow(item.media_file)
            self.image_carousel.remove(item)

            thumbnail = cast("Gtk.Picture", queue_row.thumbnail.get_child())
            texture = cast("Gdk.Texture", item.image.get_paintable())

            task = asyncio.create_task(set_thumbnail(thumbnail, texture))
            self.application.tasks.add(task)

            if str(item.media_file.original_path.parent) != GLib.get_tmp_dir():
                title = item.media_file.original_path.name
            else:
                title = _("Image – {number}").format(number=self.nth_temporary_file)
                self.nth_temporary_file += 1

            queue_row.set_title(title)
            queue_row.connect("aborted", self._cancel)
            queue_row.progressbar.set_visible(False)
            queue_row.button_cancel.set_valign(Gtk.Align.CENTER)
            self.queue_list_box.append(queue_row)

        self._run_next()

    def _on_copy_clicked_cb(
        self,
        _button: Gtk.Button,
        queue_row: QueueRow,
    ) -> None:
        async def load_file() -> None:
            def create_and_save() -> Gdk.ContentProvider:
                texture = create_texture_from_img(image)
                gbytes = texture.save_to_png_bytes()
                return Gdk.ContentProvider.new_for_bytes("image/png", gbytes)

            try:
                data = await asyncio.gather(asyncio.to_thread(create_and_save))
                content = data[0]

                clipboard.set_content(content)
            except Exception as error:
                logger.exception(error)
                return

            toast = Adw.Toast(title=_("Image copied to clipboard"))
            self.toast.add_toast(toast)

        try:
            image = Image.open(queue_row.destination_path)
        except FileNotFoundError as error:
            logger.exception(error)
            title = _("“{path}” does not exist").format(
                path=queue_row.destination_path.name
            )
            toast = Adw.Toast(title=title, timeout=0)
            self.toast.add_toast(toast)
            return

        clipboard = cast("Gdk.Display", Gdk.Display.get_default()).get_clipboard()
        task = asyncio.create_task(load_file())
        self.application.tasks.add(task)

    def _on_save_clicked_cb(
        self,
        _button: Gtk.Button,
        queue_row: QueueRow,
    ) -> None:
        def save_file(_dialog: Gtk.FileDialog, result: Gio.AsyncResult) -> None:
            try:
                destination_file = dialog.save_finish(result)
            except GLib.GError:  # type: ignore [misc]
                return

            destination_path = Path(str(destination_file.get_path()))

            with Image.open(queue_row.destination_path) as source:
                destination_image = Image.new(source.mode, source.size)
                destination_image.paste(source)
                try:
                    destination_image.save(destination_path)
                except ValueError:
                    if not destination_path.suffix:
                        title = _("File extension required")
                    else:
                        title = _("“{extension}” is not a valid file extension").format(
                            extension=destination_path.suffix
                        )
                    toast = Adw.Toast(title=title, timeout=0)
                    self.toast.add_toast(toast)
                    return

            queue_row.button_save.set_visible(False)
            queue_row.button_open.set_visible(True)
            queue_row.set_activatable(True)
            queue_row.destination_path = destination_path
            queue_row.button_open.connect(
                "clicked",
                lambda *_args: self._on_open_file_external(queue_row),
            )

        dialog = Gtk.FileDialog.new()
        dialog.set_title(_("Select Image"))
        dialog.set_initial_name(queue_row.get_title())

        dialog.save(self, callback=save_file)

    def upscaling_completed_dialog(
        self,
        queue_row: QueueRow,
        error: Exception | None,
    ) -> None:
        """Ask the user if they want to open the file."""
        toast = None

        temporary_output_path = str(queue_row.destination_path)
        notification = Gio.Notification()
        notification.set_body(
            _("Upscaled {path}").format(path=temporary_output_path),
        )

        operation = _("Save")
        action_name = "app.open-output"
        output_file_variant = GLib.Variant("s", temporary_output_path)

        self.completed_list_box.append(queue_row)
        self.completed_list_box.set_visible(True)
        queue_row.button_cancel.set_visible(False)

        # Display success
        if error is None:
            queue_row.add_css_class("success")
            queue_row.button_copy.set_visible(True)
            queue_row.button_save.set_visible(True)
            queue_row.button_save.connect(
                "clicked",
                self._on_save_clicked_cb,
                queue_row,
            )
            queue_row.button_copy.connect(
                "clicked",
                self._on_copy_clicked_cb,
                queue_row,
            )

            notification.set_title(_("Upscaling Completed"))

            notification.set_default_action_and_target(action_name, output_file_variant)
            notification.add_button_with_target(
                operation,
                action_name,
                output_file_variant,
            )

        # Display success with warnings
        elif isinstance(error, AlgorithmWarning):
            queue_row.add_css_class("warning")

            toast = Adw.Toast(
                title=_("Image upscaled with warnings"),
                timeout=0,
            )
            self.toast.add_toast(toast)

            notification.set_title(_("Upscaling Completed With Warnings"))

            notification.set_default_action_and_target(action_name, output_file_variant)
            notification.add_button_with_target(
                operation,
                action_name,
                output_file_variant,
            )

        # Display error dialog with error message
        else:
            queue_row.add_css_class("error")
            dialog = Adw.AlertDialog.new(_("Error While Upscaling"))
            sw = Gtk.ScrolledWindow()
            sw.set_min_content_height(200)
            sw.set_min_content_width(400)
            sw.add_css_class("card")

            text = Gtk.Label()
            text.set_label(str(error))
            text.set_margin_top(12)
            text.set_margin_bottom(12)
            text.set_margin_start(12)
            text.set_margin_end(12)
            text.set_xalign(0)
            text.set_yalign(0)
            text.add_css_class("monospace")
            text.set_wrap(True)
            text.set_wrap_mode(Pango.WrapMode.WORD_CHAR)

            sw.set_child(text)
            dialog.set_extra_child(sw)

            def error_response(dialog: Adw.AlertDialog, response_id: str) -> None:
                dialog.close()

                if response_id != "copy":
                    return

                if (display := Gdk.Display.get_default()) is not None:
                    clipboard = display.get_clipboard()
                    clipboard.set(str(error))
                    toast = Adw.Toast.new(_("Error copied to clipboard"))
                    self.toast.add_toast(toast)

            dialog.add_response("ok", _("_Dismiss"))
            dialog.connect("response", error_response)
            dialog.add_response("copy", _("_Copy to Clipboard"))
            dialog.set_response_appearance("copy", Adw.ResponseAppearance.SUGGESTED)
            dialog.present(self)

            notification.set_title(_("Upscaling Failed"))
            notification.set_body(_("Error while processing"))

        if not self.props.is_active:
            self.application.send_notification("upscaling-done", notification)

    @Gtk.Template.Callback()
    def _set_model(self, *_args: Any) -> None:
        """Set model and print."""
        selected_model_name = tuple(self.model_images)[self.combo_models.get_selected()]
        message = f"Model name: {selected_model_name}"
        logger.info(message)

    # Update post-upscale image size as the user adjusts the spinner
    @Gtk.Template.Callback()
    def _set_post_upscale_image_size(self, *_args: Any) -> None:
        factor = self.spin_scale.adjustment.get_value()

        for index in range(self.image_carousel.get_n_pages()):
            item = self.image_carousel.get_nth_page(index)
            if isinstance(item, ScaleComparisonFrame):
                item.set_output_dimension_by_scale_factor(factor)

    def start_loading_animation(self) -> None:
        """Show loading screen."""
        self.stack_upscaler.set_visible_child_name("stack_loading")
        self.stack_picture.set_visible_child_name("stack_loading")
        self.options_page.set_sensitive(False)

    def stop_loading_animation(self) -> None:
        """Close loading screen."""
        self.stack_upscaler.set_visible_child_name(self.current_stack_page_name)
        self.stack_picture.set_visible_child_name("stack_picture")
        self.options_page.set_sensitive(True)

    def _cancel(self, queue_row: QueueRow, *_args: Any) -> None:
        """Stop algorithm."""
        if hasattr(queue_row, "process"):
            queue_row.canceled = True
            queue_row.process.kill()
        else:
            self.queue_list_box.remove(queue_row)

        with contextlib.suppress(TypeError):
            queue_row.media_file.remove_temporary_path()

        self._run_next()

    def _vulkaninfo(self) -> None:
        """Check if Vulkan works."""
        try:
            vulkan.vkCreateInstance(vulkan.VkInstanceCreateInfo(), None)
        except (vulkan.VkErrorIncompatibleDriver, OSError):
            logger.critical("Error: Vulkan drivers not found")
            title = _("Incompatible or Missing Vulkan Drivers")
            subtitle = _(
                "The Vulkan drivers are either not installed or incompatible with the hardware. Please make sure that the correct Vulkan drivers are installed for the appropriate hardware.",
            )

            dialog = Adw.AlertDialog.new(title, subtitle)
            dialog.add_response("exit", _("_Exit Upscaler"))
            dialog.connect("response", lambda *_args: sys.exit(1))
            dialog.present(self)

    @Gtk.Template.Callback()
    def _remove_notifications(self, *_args: Any) -> None:
        self.application.withdraw_notification("upscaling-done")

    @Gtk.Template.Callback()
    def _on_item_removed(self, *_args: Any) -> None:
        if self.image_carousel.get_n_pages() == 0:
            self.main_nav_view.pop()

    @Gtk.Template.Callback()
    def _on_open_file_external(self, *args: QueueRow) -> None:
        queue_row = args[-1]
        try:
            self.application.open_file_in_external_program(queue_row.destination_path)
        except FileNotFoundError:
            title = _("“{path}” does not exist").format(queue_row.destination_path.name)
            toast = Adw.Toast(title=title, timeout=0)
            self.toast.add_toast(toast)

    def do_close_request(self) -> bool:
        """Prompt user to stop the algorithm if it's running."""
        if not self.busy:
            return False

        dialog = Adw.AlertDialog.new(
            _("Stop Upscaling?"),
            _("All progress will be lost"),
        )

        def response(_dialog: Adw.AlertDialog, response_id: str) -> None:
            if response_id != "stop":
                return

            self.busy = False
            self.close()

        dialog.add_response("cancel", _("_Cancel"))
        dialog.add_response("stop", _("_Stop"))
        dialog.set_response_appearance("stop", Adw.ResponseAppearance.DESTRUCTIVE)
        dialog.connect("response", response)
        dialog.present(self)

        return True
