'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');
var middlewareSerde = require('@smithy/middleware-serde');
var core$1 = require('@aws-sdk/core');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "cognito-identity",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class CognitoIdentityClient extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_6);
        const _config_8 = resolveRuntimeExtensions(_config_7, configuration?.extensions || []);
        this.config = _config_8;
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultCognitoIdentityHttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class CognitoIdentityServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, CognitoIdentityServiceException.prototype);
    }
}

const AmbiguousRoleResolutionType = {
    AUTHENTICATED_ROLE: "AuthenticatedRole",
    DENY: "Deny",
};
class InternalErrorException extends CognitoIdentityServiceException {
    name = "InternalErrorException";
    $fault = "server";
    constructor(opts) {
        super({
            name: "InternalErrorException",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, InternalErrorException.prototype);
    }
}
class InvalidParameterException extends CognitoIdentityServiceException {
    name = "InvalidParameterException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidParameterException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidParameterException.prototype);
    }
}
class LimitExceededException extends CognitoIdentityServiceException {
    name = "LimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "LimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, LimitExceededException.prototype);
    }
}
class NotAuthorizedException extends CognitoIdentityServiceException {
    name = "NotAuthorizedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "NotAuthorizedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NotAuthorizedException.prototype);
    }
}
class ResourceConflictException extends CognitoIdentityServiceException {
    name = "ResourceConflictException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ResourceConflictException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceConflictException.prototype);
    }
}
class TooManyRequestsException extends CognitoIdentityServiceException {
    name = "TooManyRequestsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TooManyRequestsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyRequestsException.prototype);
    }
}
const ErrorCode = {
    ACCESS_DENIED: "AccessDenied",
    INTERNAL_SERVER_ERROR: "InternalServerError",
};
class ResourceNotFoundException extends CognitoIdentityServiceException {
    name = "ResourceNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ResourceNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotFoundException.prototype);
    }
}
class ExternalServiceException extends CognitoIdentityServiceException {
    name = "ExternalServiceException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ExternalServiceException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ExternalServiceException.prototype);
    }
}
class InvalidIdentityPoolConfigurationException extends CognitoIdentityServiceException {
    name = "InvalidIdentityPoolConfigurationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidIdentityPoolConfigurationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidIdentityPoolConfigurationException.prototype);
    }
}
const MappingRuleMatchType = {
    CONTAINS: "Contains",
    EQUALS: "Equals",
    NOT_EQUAL: "NotEqual",
    STARTS_WITH: "StartsWith",
};
const RoleMappingType = {
    RULES: "Rules",
    TOKEN: "Token",
};
class DeveloperUserAlreadyRegisteredException extends CognitoIdentityServiceException {
    name = "DeveloperUserAlreadyRegisteredException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "DeveloperUserAlreadyRegisteredException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, DeveloperUserAlreadyRegisteredException.prototype);
    }
}
class ConcurrentModificationException extends CognitoIdentityServiceException {
    name = "ConcurrentModificationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ConcurrentModificationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ConcurrentModificationException.prototype);
    }
}
const GetCredentialsForIdentityInputFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Logins && { Logins: smithyClient.SENSITIVE_STRING }),
});
const CredentialsFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.SecretKey && { SecretKey: smithyClient.SENSITIVE_STRING }),
});
const GetCredentialsForIdentityResponseFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Credentials && { Credentials: CredentialsFilterSensitiveLog(obj.Credentials) }),
});
const GetIdInputFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Logins && { Logins: smithyClient.SENSITIVE_STRING }),
});
const GetOpenIdTokenInputFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Logins && { Logins: smithyClient.SENSITIVE_STRING }),
});
const GetOpenIdTokenResponseFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Token && { Token: smithyClient.SENSITIVE_STRING }),
});
const GetOpenIdTokenForDeveloperIdentityInputFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Logins && { Logins: smithyClient.SENSITIVE_STRING }),
});
const GetOpenIdTokenForDeveloperIdentityResponseFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Token && { Token: smithyClient.SENSITIVE_STRING }),
});
const UnlinkIdentityInputFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Logins && { Logins: smithyClient.SENSITIVE_STRING }),
});

const se_CreateIdentityPoolCommand = async (input, context) => {
    const headers = sharedHeaders("CreateIdentityPool");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_DeleteIdentitiesCommand = async (input, context) => {
    const headers = sharedHeaders("DeleteIdentities");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_DeleteIdentityPoolCommand = async (input, context) => {
    const headers = sharedHeaders("DeleteIdentityPool");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_DescribeIdentityCommand = async (input, context) => {
    const headers = sharedHeaders("DescribeIdentity");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_DescribeIdentityPoolCommand = async (input, context) => {
    const headers = sharedHeaders("DescribeIdentityPool");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_GetCredentialsForIdentityCommand = async (input, context) => {
    const headers = sharedHeaders("GetCredentialsForIdentity");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_GetIdCommand = async (input, context) => {
    const headers = sharedHeaders("GetId");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_GetIdentityPoolRolesCommand = async (input, context) => {
    const headers = sharedHeaders("GetIdentityPoolRoles");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_GetOpenIdTokenCommand = async (input, context) => {
    const headers = sharedHeaders("GetOpenIdToken");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_GetOpenIdTokenForDeveloperIdentityCommand = async (input, context) => {
    const headers = sharedHeaders("GetOpenIdTokenForDeveloperIdentity");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_GetPrincipalTagAttributeMapCommand = async (input, context) => {
    const headers = sharedHeaders("GetPrincipalTagAttributeMap");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_ListIdentitiesCommand = async (input, context) => {
    const headers = sharedHeaders("ListIdentities");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_ListIdentityPoolsCommand = async (input, context) => {
    const headers = sharedHeaders("ListIdentityPools");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_ListTagsForResourceCommand = async (input, context) => {
    const headers = sharedHeaders("ListTagsForResource");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_LookupDeveloperIdentityCommand = async (input, context) => {
    const headers = sharedHeaders("LookupDeveloperIdentity");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_MergeDeveloperIdentitiesCommand = async (input, context) => {
    const headers = sharedHeaders("MergeDeveloperIdentities");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_SetIdentityPoolRolesCommand = async (input, context) => {
    const headers = sharedHeaders("SetIdentityPoolRoles");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_SetPrincipalTagAttributeMapCommand = async (input, context) => {
    const headers = sharedHeaders("SetPrincipalTagAttributeMap");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_TagResourceCommand = async (input, context) => {
    const headers = sharedHeaders("TagResource");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_UnlinkDeveloperIdentityCommand = async (input, context) => {
    const headers = sharedHeaders("UnlinkDeveloperIdentity");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_UnlinkIdentityCommand = async (input, context) => {
    const headers = sharedHeaders("UnlinkIdentity");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_UntagResourceCommand = async (input, context) => {
    const headers = sharedHeaders("UntagResource");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const se_UpdateIdentityPoolCommand = async (input, context) => {
    const headers = sharedHeaders("UpdateIdentityPool");
    let body;
    body = JSON.stringify(smithyClient._json(input));
    return buildHttpRpcRequest(context, headers, "/", undefined, body);
};
const de_CreateIdentityPoolCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_DeleteIdentitiesCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_DeleteIdentityPoolCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    await smithyClient.collectBody(output.body, context);
    const response = {
        $metadata: deserializeMetadata(output),
    };
    return response;
};
const de_DescribeIdentityCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = de_IdentityDescription(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_DescribeIdentityPoolCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_GetCredentialsForIdentityCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = de_GetCredentialsForIdentityResponse(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_GetIdCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_GetIdentityPoolRolesCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_GetOpenIdTokenCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_GetOpenIdTokenForDeveloperIdentityCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_GetPrincipalTagAttributeMapCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_ListIdentitiesCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = de_ListIdentitiesResponse(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_ListIdentityPoolsCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_ListTagsForResourceCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_LookupDeveloperIdentityCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_MergeDeveloperIdentitiesCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_SetIdentityPoolRolesCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    await smithyClient.collectBody(output.body, context);
    const response = {
        $metadata: deserializeMetadata(output),
    };
    return response;
};
const de_SetPrincipalTagAttributeMapCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_TagResourceCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_UnlinkDeveloperIdentityCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    await smithyClient.collectBody(output.body, context);
    const response = {
        $metadata: deserializeMetadata(output),
    };
    return response;
};
const de_UnlinkIdentityCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    await smithyClient.collectBody(output.body, context);
    const response = {
        $metadata: deserializeMetadata(output),
    };
    return response;
};
const de_UntagResourceCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_UpdateIdentityPoolCommand = async (output, context) => {
    if (output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const data = await core$1.parseJsonBody(output.body, context);
    let contents = {};
    contents = smithyClient._json(data);
    const response = {
        $metadata: deserializeMetadata(output),
        ...contents,
    };
    return response;
};
const de_CommandError = async (output, context) => {
    const parsedOutput = {
        ...output,
        body: await core$1.parseJsonErrorBody(output.body, context),
    };
    const errorCode = core$1.loadRestJsonErrorCode(output, parsedOutput.body);
    switch (errorCode) {
        case "InternalErrorException":
        case "com.amazonaws.cognitoidentity#InternalErrorException":
            throw await de_InternalErrorExceptionRes(parsedOutput);
        case "InvalidParameterException":
        case "com.amazonaws.cognitoidentity#InvalidParameterException":
            throw await de_InvalidParameterExceptionRes(parsedOutput);
        case "LimitExceededException":
        case "com.amazonaws.cognitoidentity#LimitExceededException":
            throw await de_LimitExceededExceptionRes(parsedOutput);
        case "NotAuthorizedException":
        case "com.amazonaws.cognitoidentity#NotAuthorizedException":
            throw await de_NotAuthorizedExceptionRes(parsedOutput);
        case "ResourceConflictException":
        case "com.amazonaws.cognitoidentity#ResourceConflictException":
            throw await de_ResourceConflictExceptionRes(parsedOutput);
        case "TooManyRequestsException":
        case "com.amazonaws.cognitoidentity#TooManyRequestsException":
            throw await de_TooManyRequestsExceptionRes(parsedOutput);
        case "ResourceNotFoundException":
        case "com.amazonaws.cognitoidentity#ResourceNotFoundException":
            throw await de_ResourceNotFoundExceptionRes(parsedOutput);
        case "ExternalServiceException":
        case "com.amazonaws.cognitoidentity#ExternalServiceException":
            throw await de_ExternalServiceExceptionRes(parsedOutput);
        case "InvalidIdentityPoolConfigurationException":
        case "com.amazonaws.cognitoidentity#InvalidIdentityPoolConfigurationException":
            throw await de_InvalidIdentityPoolConfigurationExceptionRes(parsedOutput);
        case "DeveloperUserAlreadyRegisteredException":
        case "com.amazonaws.cognitoidentity#DeveloperUserAlreadyRegisteredException":
            throw await de_DeveloperUserAlreadyRegisteredExceptionRes(parsedOutput);
        case "ConcurrentModificationException":
        case "com.amazonaws.cognitoidentity#ConcurrentModificationException":
            throw await de_ConcurrentModificationExceptionRes(parsedOutput);
        default:
            const parsedBody = parsedOutput.body;
            return throwDefaultError({
                output,
                parsedBody,
                errorCode,
            });
    }
};
const de_ConcurrentModificationExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new ConcurrentModificationException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_DeveloperUserAlreadyRegisteredExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new DeveloperUserAlreadyRegisteredException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_ExternalServiceExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new ExternalServiceException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_InternalErrorExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new InternalErrorException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_InvalidIdentityPoolConfigurationExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new InvalidIdentityPoolConfigurationException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_InvalidParameterExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new InvalidParameterException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_LimitExceededExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new LimitExceededException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_NotAuthorizedExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new NotAuthorizedException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_ResourceConflictExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new ResourceConflictException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_ResourceNotFoundExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new ResourceNotFoundException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_TooManyRequestsExceptionRes = async (parsedOutput, context) => {
    const body = parsedOutput.body;
    const deserialized = smithyClient._json(body);
    const exception = new TooManyRequestsException({
        $metadata: deserializeMetadata(parsedOutput),
        ...deserialized,
    });
    return smithyClient.decorateServiceException(exception, body);
};
const de_Credentials = (output, context) => {
    return smithyClient.take(output, {
        AccessKeyId: smithyClient.expectString,
        Expiration: (_) => smithyClient.expectNonNull(smithyClient.parseEpochTimestamp(smithyClient.expectNumber(_))),
        SecretKey: smithyClient.expectString,
        SessionToken: smithyClient.expectString,
    });
};
const de_GetCredentialsForIdentityResponse = (output, context) => {
    return smithyClient.take(output, {
        Credentials: (_) => de_Credentials(_),
        IdentityId: smithyClient.expectString,
    });
};
const de_IdentitiesList = (output, context) => {
    const retVal = (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_IdentityDescription(entry);
    });
    return retVal;
};
const de_IdentityDescription = (output, context) => {
    return smithyClient.take(output, {
        CreationDate: (_) => smithyClient.expectNonNull(smithyClient.parseEpochTimestamp(smithyClient.expectNumber(_))),
        IdentityId: smithyClient.expectString,
        LastModifiedDate: (_) => smithyClient.expectNonNull(smithyClient.parseEpochTimestamp(smithyClient.expectNumber(_))),
        Logins: smithyClient._json,
    });
};
const de_ListIdentitiesResponse = (output, context) => {
    return smithyClient.take(output, {
        Identities: (_) => de_IdentitiesList(_),
        IdentityPoolId: smithyClient.expectString,
        NextToken: smithyClient.expectString,
    });
};
const deserializeMetadata = (output) => ({
    httpStatusCode: output.statusCode,
    requestId: output.headers["x-amzn-requestid"] ?? output.headers["x-amzn-request-id"] ?? output.headers["x-amz-request-id"],
    extendedRequestId: output.headers["x-amz-id-2"],
    cfId: output.headers["x-amz-cf-id"],
});
const throwDefaultError = smithyClient.withBaseException(CognitoIdentityServiceException);
const buildHttpRpcRequest = async (context, headers, path, resolvedHostname, body) => {
    const { hostname, protocol = "https", port, path: basePath } = await context.endpoint();
    const contents = {
        protocol,
        hostname,
        port,
        method: "POST",
        path: basePath.endsWith("/") ? basePath.slice(0, -1) + path : basePath + path,
        headers,
    };
    if (body !== undefined) {
        contents.body = body;
    }
    return new protocolHttp.HttpRequest(contents);
};
function sharedHeaders(operation) {
    return {
        "content-type": "application/x-amz-json-1.1",
        "x-amz-target": `AWSCognitoIdentityService.${operation}`,
    };
}

class CreateIdentityPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "CreateIdentityPool", {})
    .n("CognitoIdentityClient", "CreateIdentityPoolCommand")
    .f(void 0, void 0)
    .ser(se_CreateIdentityPoolCommand)
    .de(de_CreateIdentityPoolCommand)
    .build() {
}

class DeleteIdentitiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "DeleteIdentities", {})
    .n("CognitoIdentityClient", "DeleteIdentitiesCommand")
    .f(void 0, void 0)
    .ser(se_DeleteIdentitiesCommand)
    .de(de_DeleteIdentitiesCommand)
    .build() {
}

class DeleteIdentityPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "DeleteIdentityPool", {})
    .n("CognitoIdentityClient", "DeleteIdentityPoolCommand")
    .f(void 0, void 0)
    .ser(se_DeleteIdentityPoolCommand)
    .de(de_DeleteIdentityPoolCommand)
    .build() {
}

class DescribeIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "DescribeIdentity", {})
    .n("CognitoIdentityClient", "DescribeIdentityCommand")
    .f(void 0, void 0)
    .ser(se_DescribeIdentityCommand)
    .de(de_DescribeIdentityCommand)
    .build() {
}

class DescribeIdentityPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "DescribeIdentityPool", {})
    .n("CognitoIdentityClient", "DescribeIdentityPoolCommand")
    .f(void 0, void 0)
    .ser(se_DescribeIdentityPoolCommand)
    .de(de_DescribeIdentityPoolCommand)
    .build() {
}

class GetCredentialsForIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "GetCredentialsForIdentity", {})
    .n("CognitoIdentityClient", "GetCredentialsForIdentityCommand")
    .f(GetCredentialsForIdentityInputFilterSensitiveLog, GetCredentialsForIdentityResponseFilterSensitiveLog)
    .ser(se_GetCredentialsForIdentityCommand)
    .de(de_GetCredentialsForIdentityCommand)
    .build() {
}

class GetIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "GetId", {})
    .n("CognitoIdentityClient", "GetIdCommand")
    .f(GetIdInputFilterSensitiveLog, void 0)
    .ser(se_GetIdCommand)
    .de(de_GetIdCommand)
    .build() {
}

class GetIdentityPoolRolesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "GetIdentityPoolRoles", {})
    .n("CognitoIdentityClient", "GetIdentityPoolRolesCommand")
    .f(void 0, void 0)
    .ser(se_GetIdentityPoolRolesCommand)
    .de(de_GetIdentityPoolRolesCommand)
    .build() {
}

class GetOpenIdTokenCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "GetOpenIdToken", {})
    .n("CognitoIdentityClient", "GetOpenIdTokenCommand")
    .f(GetOpenIdTokenInputFilterSensitiveLog, GetOpenIdTokenResponseFilterSensitiveLog)
    .ser(se_GetOpenIdTokenCommand)
    .de(de_GetOpenIdTokenCommand)
    .build() {
}

class GetOpenIdTokenForDeveloperIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "GetOpenIdTokenForDeveloperIdentity", {})
    .n("CognitoIdentityClient", "GetOpenIdTokenForDeveloperIdentityCommand")
    .f(GetOpenIdTokenForDeveloperIdentityInputFilterSensitiveLog, GetOpenIdTokenForDeveloperIdentityResponseFilterSensitiveLog)
    .ser(se_GetOpenIdTokenForDeveloperIdentityCommand)
    .de(de_GetOpenIdTokenForDeveloperIdentityCommand)
    .build() {
}

class GetPrincipalTagAttributeMapCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "GetPrincipalTagAttributeMap", {})
    .n("CognitoIdentityClient", "GetPrincipalTagAttributeMapCommand")
    .f(void 0, void 0)
    .ser(se_GetPrincipalTagAttributeMapCommand)
    .de(de_GetPrincipalTagAttributeMapCommand)
    .build() {
}

class ListIdentitiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "ListIdentities", {})
    .n("CognitoIdentityClient", "ListIdentitiesCommand")
    .f(void 0, void 0)
    .ser(se_ListIdentitiesCommand)
    .de(de_ListIdentitiesCommand)
    .build() {
}

class ListIdentityPoolsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "ListIdentityPools", {})
    .n("CognitoIdentityClient", "ListIdentityPoolsCommand")
    .f(void 0, void 0)
    .ser(se_ListIdentityPoolsCommand)
    .de(de_ListIdentityPoolsCommand)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "ListTagsForResource", {})
    .n("CognitoIdentityClient", "ListTagsForResourceCommand")
    .f(void 0, void 0)
    .ser(se_ListTagsForResourceCommand)
    .de(de_ListTagsForResourceCommand)
    .build() {
}

class LookupDeveloperIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "LookupDeveloperIdentity", {})
    .n("CognitoIdentityClient", "LookupDeveloperIdentityCommand")
    .f(void 0, void 0)
    .ser(se_LookupDeveloperIdentityCommand)
    .de(de_LookupDeveloperIdentityCommand)
    .build() {
}

class MergeDeveloperIdentitiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "MergeDeveloperIdentities", {})
    .n("CognitoIdentityClient", "MergeDeveloperIdentitiesCommand")
    .f(void 0, void 0)
    .ser(se_MergeDeveloperIdentitiesCommand)
    .de(de_MergeDeveloperIdentitiesCommand)
    .build() {
}

class SetIdentityPoolRolesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "SetIdentityPoolRoles", {})
    .n("CognitoIdentityClient", "SetIdentityPoolRolesCommand")
    .f(void 0, void 0)
    .ser(se_SetIdentityPoolRolesCommand)
    .de(de_SetIdentityPoolRolesCommand)
    .build() {
}

class SetPrincipalTagAttributeMapCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "SetPrincipalTagAttributeMap", {})
    .n("CognitoIdentityClient", "SetPrincipalTagAttributeMapCommand")
    .f(void 0, void 0)
    .ser(se_SetPrincipalTagAttributeMapCommand)
    .de(de_SetPrincipalTagAttributeMapCommand)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "TagResource", {})
    .n("CognitoIdentityClient", "TagResourceCommand")
    .f(void 0, void 0)
    .ser(se_TagResourceCommand)
    .de(de_TagResourceCommand)
    .build() {
}

class UnlinkDeveloperIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "UnlinkDeveloperIdentity", {})
    .n("CognitoIdentityClient", "UnlinkDeveloperIdentityCommand")
    .f(void 0, void 0)
    .ser(se_UnlinkDeveloperIdentityCommand)
    .de(de_UnlinkDeveloperIdentityCommand)
    .build() {
}

class UnlinkIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "UnlinkIdentity", {})
    .n("CognitoIdentityClient", "UnlinkIdentityCommand")
    .f(UnlinkIdentityInputFilterSensitiveLog, void 0)
    .ser(se_UnlinkIdentityCommand)
    .de(de_UnlinkIdentityCommand)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "UntagResource", {})
    .n("CognitoIdentityClient", "UntagResourceCommand")
    .f(void 0, void 0)
    .ser(se_UntagResourceCommand)
    .de(de_UntagResourceCommand)
    .build() {
}

class UpdateIdentityPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("AWSCognitoIdentityService", "UpdateIdentityPool", {})
    .n("CognitoIdentityClient", "UpdateIdentityPoolCommand")
    .f(void 0, void 0)
    .ser(se_UpdateIdentityPoolCommand)
    .de(de_UpdateIdentityPoolCommand)
    .build() {
}

const commands = {
    CreateIdentityPoolCommand,
    DeleteIdentitiesCommand,
    DeleteIdentityPoolCommand,
    DescribeIdentityCommand,
    DescribeIdentityPoolCommand,
    GetCredentialsForIdentityCommand,
    GetIdCommand,
    GetIdentityPoolRolesCommand,
    GetOpenIdTokenCommand,
    GetOpenIdTokenForDeveloperIdentityCommand,
    GetPrincipalTagAttributeMapCommand,
    ListIdentitiesCommand,
    ListIdentityPoolsCommand,
    ListTagsForResourceCommand,
    LookupDeveloperIdentityCommand,
    MergeDeveloperIdentitiesCommand,
    SetIdentityPoolRolesCommand,
    SetPrincipalTagAttributeMapCommand,
    TagResourceCommand,
    UnlinkDeveloperIdentityCommand,
    UnlinkIdentityCommand,
    UntagResourceCommand,
    UpdateIdentityPoolCommand,
};
class CognitoIdentity extends CognitoIdentityClient {
}
smithyClient.createAggregatedClient(commands, CognitoIdentity);

const paginateListIdentityPools = core.createPaginator(CognitoIdentityClient, ListIdentityPoolsCommand, "NextToken", "NextToken", "MaxResults");

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AmbiguousRoleResolutionType = AmbiguousRoleResolutionType;
exports.CognitoIdentity = CognitoIdentity;
exports.CognitoIdentityClient = CognitoIdentityClient;
exports.CognitoIdentityServiceException = CognitoIdentityServiceException;
exports.ConcurrentModificationException = ConcurrentModificationException;
exports.CreateIdentityPoolCommand = CreateIdentityPoolCommand;
exports.CredentialsFilterSensitiveLog = CredentialsFilterSensitiveLog;
exports.DeleteIdentitiesCommand = DeleteIdentitiesCommand;
exports.DeleteIdentityPoolCommand = DeleteIdentityPoolCommand;
exports.DescribeIdentityCommand = DescribeIdentityCommand;
exports.DescribeIdentityPoolCommand = DescribeIdentityPoolCommand;
exports.DeveloperUserAlreadyRegisteredException = DeveloperUserAlreadyRegisteredException;
exports.ErrorCode = ErrorCode;
exports.ExternalServiceException = ExternalServiceException;
exports.GetCredentialsForIdentityCommand = GetCredentialsForIdentityCommand;
exports.GetCredentialsForIdentityInputFilterSensitiveLog = GetCredentialsForIdentityInputFilterSensitiveLog;
exports.GetCredentialsForIdentityResponseFilterSensitiveLog = GetCredentialsForIdentityResponseFilterSensitiveLog;
exports.GetIdCommand = GetIdCommand;
exports.GetIdInputFilterSensitiveLog = GetIdInputFilterSensitiveLog;
exports.GetIdentityPoolRolesCommand = GetIdentityPoolRolesCommand;
exports.GetOpenIdTokenCommand = GetOpenIdTokenCommand;
exports.GetOpenIdTokenForDeveloperIdentityCommand = GetOpenIdTokenForDeveloperIdentityCommand;
exports.GetOpenIdTokenForDeveloperIdentityInputFilterSensitiveLog = GetOpenIdTokenForDeveloperIdentityInputFilterSensitiveLog;
exports.GetOpenIdTokenForDeveloperIdentityResponseFilterSensitiveLog = GetOpenIdTokenForDeveloperIdentityResponseFilterSensitiveLog;
exports.GetOpenIdTokenInputFilterSensitiveLog = GetOpenIdTokenInputFilterSensitiveLog;
exports.GetOpenIdTokenResponseFilterSensitiveLog = GetOpenIdTokenResponseFilterSensitiveLog;
exports.GetPrincipalTagAttributeMapCommand = GetPrincipalTagAttributeMapCommand;
exports.InternalErrorException = InternalErrorException;
exports.InvalidIdentityPoolConfigurationException = InvalidIdentityPoolConfigurationException;
exports.InvalidParameterException = InvalidParameterException;
exports.LimitExceededException = LimitExceededException;
exports.ListIdentitiesCommand = ListIdentitiesCommand;
exports.ListIdentityPoolsCommand = ListIdentityPoolsCommand;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.LookupDeveloperIdentityCommand = LookupDeveloperIdentityCommand;
exports.MappingRuleMatchType = MappingRuleMatchType;
exports.MergeDeveloperIdentitiesCommand = MergeDeveloperIdentitiesCommand;
exports.NotAuthorizedException = NotAuthorizedException;
exports.ResourceConflictException = ResourceConflictException;
exports.ResourceNotFoundException = ResourceNotFoundException;
exports.RoleMappingType = RoleMappingType;
exports.SetIdentityPoolRolesCommand = SetIdentityPoolRolesCommand;
exports.SetPrincipalTagAttributeMapCommand = SetPrincipalTagAttributeMapCommand;
exports.TagResourceCommand = TagResourceCommand;
exports.TooManyRequestsException = TooManyRequestsException;
exports.UnlinkDeveloperIdentityCommand = UnlinkDeveloperIdentityCommand;
exports.UnlinkIdentityCommand = UnlinkIdentityCommand;
exports.UnlinkIdentityInputFilterSensitiveLog = UnlinkIdentityInputFilterSensitiveLog;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UpdateIdentityPoolCommand = UpdateIdentityPoolCommand;
exports.paginateListIdentityPools = paginateListIdentityPools;
