"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _errors = require("../errors");
var _writableTrackingBuffer = _interopRequireDefault(require("../tracking-buffer/writable-tracking-buffer"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
const TVP_ROW_TOKEN = Buffer.from([0x01]);
const TVP_END_TOKEN = Buffer.from([0x00]);
const NULL_LENGTH = Buffer.from([0xFF, 0xFF]);
const TVP = {
  id: 0xF3,
  type: 'TVPTYPE',
  name: 'TVP',
  declaration: function (parameter) {
    const value = parameter.value; // Temporary solution. Remove 'any' later.
    return value.name + ' readonly';
  },
  generateTypeInfo(parameter) {
    const databaseName = '';
    const schema = parameter.value?.schema ?? '';
    const typeName = parameter.value?.name ?? '';
    const bufferLength = 1 + 1 + Buffer.byteLength(databaseName, 'ucs2') + 1 + Buffer.byteLength(schema, 'ucs2') + 1 + Buffer.byteLength(typeName, 'ucs2');
    const buffer = new _writableTrackingBuffer.default(bufferLength, 'ucs2');
    buffer.writeUInt8(this.id);
    buffer.writeBVarchar(databaseName);
    buffer.writeBVarchar(schema);
    buffer.writeBVarchar(typeName);
    return buffer.data;
  },
  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      return NULL_LENGTH;
    }
    const {
      columns
    } = parameter.value;
    const buffer = Buffer.alloc(2);
    buffer.writeUInt16LE(columns.length, 0);
    return buffer;
  },
  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      yield TVP_END_TOKEN;
      yield TVP_END_TOKEN;
      return;
    }
    const {
      columns,
      rows
    } = parameter.value;
    for (let i = 0, len = columns.length; i < len; i++) {
      const column = columns[i];
      const buff = Buffer.alloc(6);
      // UserType
      buff.writeUInt32LE(0x00000000, 0);

      // Flags
      buff.writeUInt16LE(0x0000, 4);
      yield buff;

      // TYPE_INFO
      yield column.type.generateTypeInfo(column);

      // ColName
      yield Buffer.from([0x00]);
    }
    yield TVP_END_TOKEN;
    for (let i = 0, length = rows.length; i < length; i++) {
      yield TVP_ROW_TOKEN;
      const row = rows[i];
      for (let k = 0, len2 = row.length; k < len2; k++) {
        const column = columns[k];
        const value = row[k];
        let paramValue;
        try {
          paramValue = column.type.validate(value, parameter.collation);
        } catch (error) {
          throw new _errors.InputError(`TVP column '${column.name}' has invalid data at row index ${i}`, {
            cause: error
          });
        }
        const param = {
          value: paramValue,
          length: column.length,
          scale: column.scale,
          precision: column.precision
        };

        // TvpColumnData
        yield column.type.generateParameterLength(param, options);
        yield* column.type.generateParameterData(param, options);
      }
    }
    yield TVP_END_TOKEN;
  },
  validate: function (value) {
    if (value == null) {
      return null;
    }
    if (typeof value !== 'object') {
      throw new TypeError('Invalid table.');
    }
    if (!Array.isArray(value.columns)) {
      throw new TypeError('Invalid table.');
    }
    if (!Array.isArray(value.rows)) {
      throw new TypeError('Invalid table.');
    }
    return value;
  }
};
var _default = exports.default = TVP;
module.exports = TVP;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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