"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _token = require("./token");
var _colmetadataTokenParser = _interopRequireDefault(require("./colmetadata-token-parser"));
var _doneTokenParser = require("./done-token-parser");
var _envChangeTokenParser = _interopRequireDefault(require("./env-change-token-parser"));
var _infoerrorTokenParser = require("./infoerror-token-parser");
var _fedauthInfoParser = _interopRequireDefault(require("./fedauth-info-parser"));
var _featureExtAckParser = _interopRequireDefault(require("./feature-ext-ack-parser"));
var _loginackTokenParser = _interopRequireDefault(require("./loginack-token-parser"));
var _orderTokenParser = _interopRequireDefault(require("./order-token-parser"));
var _returnstatusTokenParser = _interopRequireDefault(require("./returnstatus-token-parser"));
var _returnvalueTokenParser = _interopRequireDefault(require("./returnvalue-token-parser"));
var _rowTokenParser = _interopRequireDefault(require("./row-token-parser"));
var _nbcrowTokenParser = _interopRequireDefault(require("./nbcrow-token-parser"));
var _sspiTokenParser = _interopRequireDefault(require("./sspi-token-parser"));
var _helpers = require("./helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
class Parser {
  debug;
  colMetadata;
  options;
  iterator;
  buffer;
  position;
  static async *parseTokens(iterable, debug, options, colMetadata = []) {
    const parser = new Parser(iterable, debug, options);
    parser.colMetadata = colMetadata;
    while (true) {
      try {
        await parser.waitForChunk();
      } catch (err) {
        if (parser.position === parser.buffer.length) {
          return;
        }
        throw err;
      }
      while (parser.buffer.length >= parser.position + 1) {
        const type = parser.buffer.readUInt8(parser.position);
        parser.position += 1;
        const token = parser.readToken(type);
        if (token !== undefined) {
          yield token;
        }
      }
    }
  }
  readToken(type) {
    switch (type) {
      case _token.TYPE.DONE:
        {
          return this.readDoneToken();
        }
      case _token.TYPE.DONEPROC:
        {
          return this.readDoneProcToken();
        }
      case _token.TYPE.DONEINPROC:
        {
          return this.readDoneInProcToken();
        }
      case _token.TYPE.ERROR:
        {
          return this.readErrorToken();
        }
      case _token.TYPE.INFO:
        {
          return this.readInfoToken();
        }
      case _token.TYPE.ENVCHANGE:
        {
          return this.readEnvChangeToken();
        }
      case _token.TYPE.LOGINACK:
        {
          return this.readLoginAckToken();
        }
      case _token.TYPE.RETURNSTATUS:
        {
          return this.readReturnStatusToken();
        }
      case _token.TYPE.ORDER:
        {
          return this.readOrderToken();
        }
      case _token.TYPE.FEDAUTHINFO:
        {
          return this.readFedAuthInfoToken();
        }
      case _token.TYPE.SSPI:
        {
          return this.readSSPIToken();
        }
      case _token.TYPE.COLMETADATA:
        {
          return this.readColMetadataToken();
        }
      case _token.TYPE.RETURNVALUE:
        {
          return this.readReturnValueToken();
        }
      case _token.TYPE.ROW:
        {
          return this.readRowToken();
        }
      case _token.TYPE.NBCROW:
        {
          return this.readNbcRowToken();
        }
      case _token.TYPE.FEATUREEXTACK:
        {
          return this.readFeatureExtAckToken();
        }
      default:
        {
          throw new Error('Unknown type: ' + type);
        }
    }
  }
  readFeatureExtAckToken() {
    let result;
    try {
      result = (0, _featureExtAckParser.default)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readFeatureExtAckToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  async readNbcRowToken() {
    return await (0, _nbcrowTokenParser.default)(this);
  }
  async readReturnValueToken() {
    return await (0, _returnvalueTokenParser.default)(this);
  }
  async readColMetadataToken() {
    const token = await (0, _colmetadataTokenParser.default)(this);
    this.colMetadata = token.columns;
    return token;
  }
  readSSPIToken() {
    let result;
    try {
      result = (0, _sspiTokenParser.default)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readSSPIToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readFedAuthInfoToken() {
    let result;
    try {
      result = (0, _fedauthInfoParser.default)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readFedAuthInfoToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readOrderToken() {
    let result;
    try {
      result = (0, _orderTokenParser.default)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readOrderToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readReturnStatusToken() {
    let result;
    try {
      result = (0, _returnstatusTokenParser.default)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readReturnStatusToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readLoginAckToken() {
    let result;
    try {
      result = (0, _loginackTokenParser.default)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readLoginAckToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readEnvChangeToken() {
    let result;
    try {
      result = (0, _envChangeTokenParser.default)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readEnvChangeToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readRowToken() {
    return (0, _rowTokenParser.default)(this);
  }
  readInfoToken() {
    let result;
    try {
      result = (0, _infoerrorTokenParser.infoParser)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readInfoToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readErrorToken() {
    let result;
    try {
      result = (0, _infoerrorTokenParser.errorParser)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readErrorToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readDoneInProcToken() {
    let result;
    try {
      result = (0, _doneTokenParser.doneInProcParser)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readDoneInProcToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readDoneProcToken() {
    let result;
    try {
      result = (0, _doneTokenParser.doneProcParser)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readDoneProcToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  readDoneToken() {
    let result;
    try {
      result = (0, _doneTokenParser.doneParser)(this.buffer, this.position, this.options);
    } catch (err) {
      if (err instanceof _helpers.NotEnoughDataError) {
        return this.waitForChunk().then(() => {
          return this.readDoneToken();
        });
      }
      throw err;
    }
    this.position = result.offset;
    return result.value;
  }
  constructor(iterable, debug, options) {
    this.debug = debug;
    this.colMetadata = [];
    this.options = options;
    this.iterator = (iterable[Symbol.asyncIterator] || iterable[Symbol.iterator]).call(iterable);
    this.buffer = Buffer.alloc(0);
    this.position = 0;
  }
  async waitForChunk() {
    const result = await this.iterator.next();
    if (result.done) {
      throw new Error('unexpected end of data');
    }
    if (this.position === this.buffer.length) {
      this.buffer = result.value;
    } else {
      this.buffer = Buffer.concat([this.buffer.slice(this.position), result.value]);
    }
    this.position = 0;
  }
}
var _default = exports.default = Parser;
module.exports = Parser;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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