import { createRequire } from "node:module";
import { StructureModel, detectInteractiveSession, fromRef, isNode, isRef, isSymbol, loadConfigFile, log, nodeBrand, ref, refs } from "@hey-api/codegen-core";
import { ConfigError, OperationPath, OperationStrategy, applyNaming, createOperationKey, createSchemaProcessor, deduplicateSchema, definePluginConfig, dependencyFactory, ensureDirSync, escapeComment, findTsConfigPath, getInput, getLogs, getParser, hasOperationDataRequired, hasParameterGroupObjectRequired, loadTsConfig, mappers, operationPagination, operationResponsesMap, parseUrl, pathToJsonPointer, pathToName, refToName, resolveSource, satisfies, statusCodeToGroup, toCase, valueToObject } from "@hey-api/shared";
import colors from "ansi-colors";
import path from "node:path";
import { fileURLToPath } from "node:url";
import fs from "node:fs";
import ts from "typescript";

//#region rolldown:runtime
var __require = /* @__PURE__ */ createRequire(import.meta.url);

//#endregion
//#region src/config/expand.ts
function expandToJobs(configs) {
	const jobs = [];
	let jobIndex = 0;
	for (const config of configs) {
		const inputs = getInput(config);
		const outputs = config.output instanceof Array ? config.output : [config.output];
		if (outputs.length === 1) jobs.push({
			config: {
				...config,
				input: inputs,
				output: outputs[0]
			},
			index: jobIndex++
		});
		else if (outputs.length > 1 && inputs.length !== outputs.length) {
			console.warn(`⚙️ ${colors.yellow("Warning:")} You provided ${colors.cyan(String(inputs.length))} ${colors.cyan(inputs.length === 1 ? "input" : "inputs")} and ${colors.yellow(String(outputs.length))} ${colors.yellow("outputs")}. This will produce identical output in multiple locations. You likely want to provide a single output or the same number of outputs as inputs.`);
			for (const output of outputs) jobs.push({
				config: {
					...config,
					input: inputs,
					output
				},
				index: jobIndex++
			});
		} else if (outputs.length > 1) outputs.forEach((output, index) => {
			jobs.push({
				config: {
					...config,
					input: inputs[index],
					output
				},
				index: jobIndex++
			});
		});
	}
	return jobs;
}

//#endregion
//#region src/config/packages.ts
/**
* Finds and reads the project's package.json file by searching upwards from the config file location,
* or from process.cwd() if no config file is provided.
* This ensures we get the correct dependencies even in monorepo setups.
*
* @param configFilePath - The path to the configuration file (e.g., openapi-ts.config.ts)
* @returns An object containing all project dependencies (dependencies, devDependencies, peerDependencies, optionalDependencies)
*/
const getProjectDependencies = (configFilePath) => {
	let currentDir = configFilePath ? path.dirname(configFilePath) : process.cwd();
	while (currentDir !== path.dirname(currentDir)) {
		const packageJsonPath = path.join(currentDir, "package.json");
		if (fs.existsSync(packageJsonPath)) try {
			const packageJson = JSON.parse(fs.readFileSync(packageJsonPath, "utf8"));
			return {
				...packageJson.dependencies,
				...packageJson.devDependencies,
				...packageJson.peerDependencies,
				...packageJson.optionalDependencies
			};
		} catch {}
		const parentDir = path.dirname(currentDir);
		if (parentDir === currentDir) break;
		currentDir = parentDir;
	}
	return {};
};

//#endregion
//#region src/config/output/postprocess.ts
const postProcessors = {
	"biome:format": {
		args: [
			"format",
			"--write",
			"{{path}}"
		],
		command: "biome",
		name: "Biome (Format)"
	},
	"biome:lint": {
		args: [
			"lint",
			"--apply",
			"{{path}}"
		],
		command: "biome",
		name: "Biome (Lint)"
	},
	eslint: {
		args: ["{{path}}", "--fix"],
		command: "eslint",
		name: "ESLint"
	},
	oxfmt: {
		args: ["{{path}}"],
		command: "oxfmt",
		name: "Oxfmt"
	},
	oxlint: {
		args: ["--fix", "{{path}}"],
		command: "oxlint",
		name: "Oxlint"
	},
	prettier: {
		args: [
			"--ignore-unknown",
			"{{path}}",
			"--write",
			"--ignore-path",
			"./.prettierignore"
		],
		command: "prettier",
		name: "Prettier"
	}
};

//#endregion
//#region src/config/output/config.ts
const __filename$1 = fileURLToPath(import.meta.url);
const __dirname$1 = path.dirname(__filename$1);
function getOutput(userConfig) {
	if (userConfig.output instanceof Array) throw new Error("Unexpected array of outputs in user configuration. This should have been expanded already.");
	const userOutput = typeof userConfig.output === "string" ? { path: userConfig.output } : userConfig.output ?? {};
	const legacyPostProcess = resolveLegacyPostProcess(userOutput);
	const output = valueToObject({
		defaultValue: {
			clean: true,
			entryFile: true,
			fileName: {
				case: "preserve",
				name: "{{name}}",
				suffix: ".gen"
			},
			format: null,
			header: "// This file is auto-generated by @hey-api/openapi-ts",
			lint: null,
			path: "",
			postProcess: [],
			preferExportAll: false
		},
		mappers: { object: (fields, defaultValue) => ({
			...fields,
			fileName: valueToObject({
				defaultValue: { ...defaultValue.fileName },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: fields.fileName
			})
		}) },
		value: userOutput
	});
	output.tsConfig = loadTsConfig(findTsConfigPath(__dirname$1, output.tsConfigPath));
	if (output.importFileExtension === void 0 && (output.tsConfig?.options.moduleResolution === ts.ModuleResolutionKind.NodeNext || output.tsConfig?.options.moduleResolution === ts.ModuleResolutionKind.Node16 || output.tsConfig?.options.module === ts.ModuleKind.NodeNext || output.tsConfig?.options.module === ts.ModuleKind.Node16)) output.importFileExtension = ".js";
	if (output.importFileExtension && !output.importFileExtension.startsWith(".")) output.importFileExtension = `.${output.importFileExtension}`;
	output.postProcess = normalizePostProcess(userOutput.postProcess ?? legacyPostProcess);
	output.source = resolveSource(output);
	return output;
}
function resolveLegacyPostProcess(config) {
	const result = [];
	if (config.lint !== void 0) {
		let processor;
		let preset;
		if (config.lint) {
			preset = config.lint === "biome" ? "biome:lint" : config.lint;
			processor = postProcessors[preset];
			if (processor) result.push(processor);
		}
		log.warnDeprecated({
			context: "output",
			field: "lint",
			replacement: `postProcess: [${processor && preset ? `'${preset}'` : ""}]`
		});
	}
	if (config.format !== void 0) {
		let processor;
		let preset;
		if (config.format) {
			preset = config.format === "biome" ? "biome:format" : config.format;
			processor = postProcessors[preset];
			if (processor) result.push(processor);
		}
		log.warnDeprecated({
			context: "output",
			field: "format",
			replacement: `postProcess: [${processor && preset ? `'${preset}'` : ""}]`
		});
	}
	return result;
}
function normalizePostProcess(input) {
	if (!input) return [];
	return input.map((item) => {
		if (typeof item === "string") {
			const preset = postProcessors[item];
			if (!preset) throw new Error(`Unknown post-processor preset: "${item}"`);
			return preset;
		}
		return {
			name: item.name ?? item.command,
			...item
		};
	});
}

//#endregion
//#region src/plugins/@angular/common/httpRequests/config.ts
function resolveHttpRequests(config, context) {
	let input = config.httpRequests;
	if (typeof input === "string" || typeof input === "function") input = { strategy: input };
	else if (typeof input === "boolean" || !input) input = { enabled: Boolean(input) };
	const strategy = input.strategy ?? "flat";
	return context.valueToObject({
		defaultValue: {
			container: "class",
			enabled: true,
			methods: "instance",
			nesting: "operationId",
			nestingDelimiters: /[./]/,
			strategy,
			strategyDefaultTag: "default"
		},
		mappers: { object(value) {
			value.containerName = context.valueToObject({
				defaultValue: strategy === "single" ? {
					casing: "PascalCase",
					name: "HttpRequests"
				} : { casing: "PascalCase" },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.containerName
			});
			value.methodName = context.valueToObject({
				defaultValue: strategy === "flat" ? {
					casing: "camelCase",
					name: "{{name}}Request"
				} : { casing: "camelCase" },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.methodName
			});
			value.segmentName = context.valueToObject({
				defaultValue: {
					casing: "PascalCase",
					name: "{{name}}Requests"
				},
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.segmentName
			});
			return value;
		} },
		value: input
	});
}

//#endregion
//#region src/plugins/@angular/common/httpRequests/resolve.ts
function resolvePath$2(plugin) {
	if (plugin.config.httpRequests.nesting === "id") return OperationPath.id();
	if (plugin.config.httpRequests.nesting === "operationId") return OperationPath.fromOperationId({
		delimiters: plugin.config.httpRequests.nestingDelimiters,
		fallback: OperationPath.id()
	});
	return plugin.config.httpRequests.nesting;
}
function resolveHttpRequestsStrategy(plugin) {
	if (plugin.config.httpRequests.strategy === "flat") return OperationStrategy.flat({ path: (operation) => [resolvePath$2(plugin)(operation).join(".")] });
	if (plugin.config.httpRequests.strategy === "single") {
		const root = plugin.config.httpRequests.containerName;
		return OperationStrategy.single({
			path: resolvePath$2(plugin),
			root: typeof root.name === "string" ? root.name : root.name?.("") ?? ""
		});
	}
	if (plugin.config.httpRequests.strategy === "byTags") return OperationStrategy.byTags({
		fallback: plugin.config.httpRequests.strategyDefaultTag,
		path: resolvePath$2(plugin)
	});
	return plugin.config.httpRequests.strategy;
}

//#endregion
//#region src/plugins/@angular/common/httpResources/config.ts
function resolveHttpResources(config, context) {
	let input = config.httpResources;
	if (typeof input === "string" || typeof input === "function") input = { strategy: input };
	else if (typeof input === "boolean" || !input) input = { enabled: Boolean(input) };
	const strategy = input.strategy ?? "flat";
	return context.valueToObject({
		defaultValue: {
			container: "class",
			enabled: true,
			methods: "instance",
			nesting: "operationId",
			nestingDelimiters: /[./]/,
			strategy,
			strategyDefaultTag: "default"
		},
		mappers: { object(value) {
			value.containerName = context.valueToObject({
				defaultValue: strategy === "single" ? {
					casing: "PascalCase",
					name: "HttpResources"
				} : { casing: "PascalCase" },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.containerName
			});
			value.methodName = context.valueToObject({
				defaultValue: strategy === "flat" ? {
					casing: "camelCase",
					name: "{{name}}Resource"
				} : { casing: "camelCase" },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.methodName
			});
			value.segmentName = context.valueToObject({
				defaultValue: {
					casing: "PascalCase",
					name: "{{name}}Resources"
				},
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.segmentName
			});
			return value;
		} },
		value: input
	});
}

//#endregion
//#region src/plugins/@angular/common/httpResources/resolve.ts
function resolvePath$1(plugin) {
	if (plugin.config.httpResources.nesting === "id") return OperationPath.id();
	if (plugin.config.httpResources.nesting === "operationId") return OperationPath.fromOperationId({
		delimiters: plugin.config.httpResources.nestingDelimiters,
		fallback: OperationPath.id()
	});
	return plugin.config.httpResources.nesting;
}
function resolveHttpResourcesStrategy(plugin) {
	if (plugin.config.httpResources.strategy === "flat") return OperationStrategy.flat({ path: (operation) => [resolvePath$1(plugin)(operation).join(".")] });
	if (plugin.config.httpResources.strategy === "single") {
		const root = plugin.config.httpResources.containerName;
		return OperationStrategy.single({
			path: resolvePath$1(plugin),
			root: typeof root.name === "string" ? root.name : root.name?.("") ?? ""
		});
	}
	if (plugin.config.httpResources.strategy === "byTags") return OperationStrategy.byTags({
		fallback: plugin.config.httpResources.strategyDefaultTag,
		path: resolvePath$1(plugin)
	});
	return plugin.config.httpResources.strategy;
}

//#endregion
//#region src/config/utils.ts
function getTypedConfig(plugin) {
	if ("context" in plugin) return plugin.context.config;
	return plugin.config;
}

//#endregion
//#region src/plugins/@hey-api/client-core/utils.ts
function getClientBaseUrlKey(config) {
	const client = getClientPlugin(config);
	if (client.name === "@hey-api/client-axios" || client.name === "@hey-api/client-nuxt") return "baseURL";
	return "baseUrl";
}
function getClientPlugin(config) {
	for (const name of config.pluginOrder) {
		const plugin = config.plugins[name];
		if (plugin?.tags?.includes("client")) return plugin;
	}
	return {
		config: { name: "" },
		name: ""
	};
}

//#endregion
//#region src/ts-dsl/base.ts
var TsDsl = class {
	analyze(_) {}
	clone() {
		const cloned = Object.create(Object.getPrototypeOf(this));
		Object.assign(cloned, this);
		return cloned;
	}
	exported;
	file;
	get name() {
		return {
			...this._name,
			set: (value) => {
				this._name = ref(value);
				if (isSymbol(value)) value.setNode(this);
			},
			toString: () => this._name ? this.$name(this._name) : ""
		};
	}
	nameSanitizer;
	language = "typescript";
	parent;
	root = false;
	scope = "value";
	structuralChildren;
	structuralParents;
	symbol;
	toAst() {}
	"~brand" = nodeBrand;
	$if(value, ifTrue, ifFalse) {
		if (value) {
			let result;
			try {
				result = ifTrue?.(this, value);
			} catch {}
			if (result === void 0) try {
				result = ifTrue?.(value);
			} catch {}
			if (result === void 0) try {
				result = ifTrue?.();
			} catch {}
			return result ?? this;
		}
		if (ifFalse) {
			let result;
			try {
				result = ifFalse?.(this, value);
			} catch {}
			if (result === void 0) try {
				result = ifFalse?.(value);
			} catch {}
			if (result === void 0) try {
				result = ifFalse?.();
			} catch {}
			return result ?? this;
		}
		return this;
	}
	$maybeId(expr) {
		return typeof expr === "string" ? ts.factory.createIdentifier(expr) : expr;
	}
	$name(name) {
		const value = fromRef(name);
		if (isSymbol(value)) try {
			return value.finalName;
		} catch {
			return value.name;
		}
		return String(value);
	}
	$node(value) {
		if (value === void 0) return;
		if (isRef(value)) value = fromRef(value);
		if (isSymbol(value)) return this.$maybeId(value.finalName);
		if (typeof value === "string") return this.$maybeId(value);
		if (value instanceof Array) return value.map((item) => {
			if (isRef(item)) item = fromRef(item);
			return this.unwrap(item);
		});
		return this.unwrap(value);
	}
	$type(value, args) {
		if (value === void 0) return;
		if (isRef(value)) value = fromRef(value);
		if (isSymbol(value)) return ts.factory.createTypeReferenceNode(value.finalName, args);
		if (typeof value === "string") return ts.factory.createTypeReferenceNode(value, args);
		if (typeof value === "boolean") {
			const literal = value ? ts.factory.createTrue() : ts.factory.createFalse();
			return ts.factory.createLiteralTypeNode(literal);
		}
		if (typeof value === "number") return ts.factory.createLiteralTypeNode(ts.factory.createNumericLiteral(value));
		if (value instanceof Array) return value.map((item) => this.$type(item, args));
		return this.unwrap(value);
	}
	_name;
	/** Unwraps nested nodes into raw TypeScript AST. */
	unwrap(value) {
		return isNode(value) ? value.toAst() : value;
	}
};
var TypeTsDsl = class extends TsDsl {};

//#endregion
//#region src/ts-dsl/expr/id.ts
const Mixed$52 = TsDsl;
var IdTsDsl = class extends Mixed$52 {
	"~dsl" = "IdTsDsl";
	constructor(name) {
		super();
		this.name.set(name);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	toAst() {
		return ts.factory.createIdentifier(this.name.toString());
	}
};

//#endregion
//#region src/ts-dsl/layout/newline.ts
var NewlineTsDsl = class extends TsDsl {
	"~dsl" = "NewlineTsDsl";
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	toAst() {
		return this.$node(new IdTsDsl("\n"));
	}
};

//#endregion
//#region src/ts-dsl/mixins/args.ts
/**
* Adds `.arg()` and `.args()` for managing expression arguments in call-like nodes.
*/
function ArgsMixin(Base) {
	class Args extends Base {
		_args = [];
		analyze(ctx$1) {
			super.analyze(ctx$1);
			for (const arg of this._args) ctx$1.analyze(arg);
		}
		arg(arg) {
			if (arg !== void 0) this._args.push(ref(arg));
			return this;
		}
		args(...args) {
			this._args.push(...args.filter((a) => a !== void 0).map((a) => ref(a)));
			return this;
		}
		$args() {
			return this.$node(this._args).map((arg) => this.$node(arg));
		}
	}
	return Args;
}

//#endregion
//#region src/ts-dsl/expr/prefix.ts
const Mixed$51 = TsDsl;
var PrefixTsDsl = class extends Mixed$51 {
	"~dsl" = "PrefixTsDsl";
	_expr;
	_op;
	constructor(expr, op) {
		super();
		this._expr = expr;
		this._op = op;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._expr);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Sets the operand (the expression being prefixed). */
	expr(expr) {
		this._expr = expr;
		return this;
	}
	/** Sets the operator to MinusToken for negation (`-`). */
	neg() {
		this._op = ts.SyntaxKind.MinusToken;
		return this;
	}
	/** Sets the operator to ExclamationToken for logical NOT (`!`). */
	not() {
		this._op = ts.SyntaxKind.ExclamationToken;
		return this;
	}
	/** Sets the operator (e.g. `ts.SyntaxKind.ExclamationToken` for `!`). */
	op(op) {
		this._op = op;
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createPrefixUnaryExpression(this._op, this.$node(this._expr));
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Prefix unary expression missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._expr) missing.push(".expr()");
		if (!this._op) missing.push("operator (e.g., .not(), .neg())");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/utils/factories.ts
function createFactory(name) {
	let impl;
	const slot = ((...args) => {
		if (!impl) throw new Error(`${name} factory not registered`);
		return impl(...args);
	});
	slot.set = (fn) => {
		impl = fn;
	};
	return slot;
}
const f = {
	as: createFactory("as"),
	attr: createFactory("attr"),
	await: createFactory("await"),
	call: createFactory("call"),
	new: createFactory("new"),
	return: createFactory("return"),
	type: {
		expr: createFactory("type.expr"),
		idx: createFactory("type.idx"),
		operator: createFactory("type.operator"),
		query: createFactory("type.query")
	},
	typeofExpr: createFactory("typeofExpr")
};

//#endregion
//#region src/ts-dsl/mixins/as.ts
function AsMixin(Base) {
	class As extends Base {
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		as(...args) {
			return f.as(this, ...args);
		}
	}
	return As;
}

//#endregion
//#region src/ts-dsl/expr/literal.ts
const Mixed$50 = AsMixin(TsDsl);
var LiteralTsDsl = class extends Mixed$50 {
	"~dsl" = "LiteralTsDsl";
	value;
	constructor(value) {
		super();
		this.value = value;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	toAst() {
		if (typeof this.value === "boolean") return this.value ? ts.factory.createTrue() : ts.factory.createFalse();
		if (typeof this.value === "number") {
			const expr = ts.factory.createNumericLiteral(Math.abs(this.value));
			return this.value < 0 ? this.$node(new PrefixTsDsl(expr).neg()) : expr;
		}
		if (typeof this.value === "string") return ts.factory.createStringLiteral(this.value, true);
		if (typeof this.value === "bigint") return ts.factory.createBigIntLiteral(this.value.toString());
		if (this.value === null) return ts.factory.createNull();
		throw new Error(`Unsupported literal: ${String(this.value)}`);
	}
};

//#endregion
//#region src/ts-dsl/utils/regexp.ts
/**
* Matches characters from the start as long as they're not allowed.
*/
const illegalStartCharactersRegExp = /^[^$_\p{ID_Start}]+/u;
/**
* Matches string if it contains only digits and optionally decimal point or
* leading minus sign.
*/
const numberRegExp = /^-?\d+(\.\d+)?$/;
/**
* Javascript identifier regexp pattern retrieved from
* {@link} https://developer.mozilla.org/docs/Web/JavaScript/Reference/Lexical_grammar#identifiers
*/
const validTypescriptIdentifierRegExp = /^[$_\p{ID_Start}][$\u200c\u200d\p{ID_Continue}]*$/u;
const regexp = {
	illegalStartCharacters: illegalStartCharactersRegExp,
	number: numberRegExp,
	typeScriptIdentifier: validTypescriptIdentifierRegExp
};

//#endregion
//#region src/ts-dsl/utils/keywords.ts
const browserGlobals = [
	"document",
	"history",
	"location",
	"navigator",
	"window"
];
const javaScriptGlobals = [
	"console",
	"Array",
	"Date",
	"Error",
	"Function",
	"JSON",
	"Map",
	"Math",
	"Object",
	"Promise",
	"RegExp",
	"Set",
	"WeakMap",
	"WeakSet"
];
const javaScriptKeywords = [
	"arguments",
	"async",
	"await",
	"break",
	"case",
	"catch",
	"class",
	"const",
	"continue",
	"debugger",
	"default",
	"delete",
	"do",
	"else",
	"enum",
	"eval",
	"export",
	"extends",
	"false",
	"finally",
	"for",
	"from",
	"function",
	"if",
	"implements",
	"import",
	"in",
	"instanceof",
	"interface",
	"let",
	"new",
	"null",
	"package",
	"private",
	"protected",
	"public",
	"return",
	"static",
	"super",
	"switch",
	"this",
	"throw",
	"true",
	"try",
	"typeof",
	"var",
	"void",
	"while",
	"with",
	"yield"
];
const nodeGlobals = [
	"global",
	"process",
	"Buffer"
];
const typeScriptKeywords = [
	"any",
	"as",
	"bigint",
	"boolean",
	"namespace",
	"never",
	"null",
	"number",
	"string",
	"symbol",
	"type",
	"undefined",
	"unknown",
	"void"
];
const keywords = {
	browserGlobals,
	javaScriptGlobals,
	javaScriptKeywords,
	nodeGlobals,
	typeScriptKeywords
};

//#endregion
//#region src/ts-dsl/utils/reserved.ts
var ReservedList = class {
	_array;
	_set;
	constructor(values) {
		this._array = values;
		this._set = new Set(values);
	}
	get "~values"() {
		return this._set;
	}
	/**
	* Updates the reserved list with new values.
	*
	* @param values New reserved values or a function that receives the previous
	* reserved values and returns the new ones.
	*/
	set(values) {
		const vals = typeof values === "function" ? values(this._array) : values;
		this._array = vals;
		this._set = new Set(vals);
	}
};
const runtimeReserved = new ReservedList([
	...keywords.browserGlobals,
	...keywords.javaScriptGlobals,
	...keywords.javaScriptKeywords,
	...keywords.nodeGlobals,
	...keywords.typeScriptKeywords
]);
const typeReserved = new ReservedList([...keywords.javaScriptKeywords, ...keywords.typeScriptKeywords]);
/**
* Reserved names for identifiers. These names will not be used
* for variables, functions, classes, or other identifiers in generated code.
*/
const reserved = {
	runtime: runtimeReserved,
	type: typeReserved
};

//#endregion
//#region src/ts-dsl/utils/name.ts
const safeAccessorName = (name) => {
	regexp.number.lastIndex = 0;
	if (regexp.number.test(name)) return name.startsWith("-") ? `'${name}'` : name;
	regexp.typeScriptIdentifier.lastIndex = 0;
	if (regexp.typeScriptIdentifier.test(name)) return name;
	return `'${name}'`;
};
const safeMemberName = (name) => {
	regexp.typeScriptIdentifier.lastIndex = 0;
	if (regexp.typeScriptIdentifier.test(name)) return new IdTsDsl(name);
	return new LiteralTsDsl(name);
};
const safePropName = (name) => {
	regexp.number.lastIndex = 0;
	if (regexp.number.test(name)) return name.startsWith("-") ? new LiteralTsDsl(name) : new LiteralTsDsl(Number(name));
	regexp.typeScriptIdentifier.lastIndex = 0;
	if (regexp.typeScriptIdentifier.test(name)) return new IdTsDsl(name);
	return new LiteralTsDsl(name);
};
const safeName = (name, reserved$1) => {
	let sanitized = "";
	let index;
	const first = name[0] ?? "";
	regexp.illegalStartCharacters.lastIndex = 0;
	if (regexp.illegalStartCharacters.test(first)) {
		sanitized += "_";
		index = 0;
	} else {
		sanitized += first;
		index = 1;
	}
	while (index < name.length) {
		const char = name[index] ?? "";
		sanitized += /^[\u200c\u200d\p{ID_Continue}]$/u.test(char) ? char : "_";
		index += 1;
	}
	if (reserved$1["~values"].has(sanitized)) sanitized = `${sanitized}_`;
	return sanitized || "_";
};
const safeRuntimeName = (name) => safeName(name, reserved.runtime);
const safeTypeName = (name) => safeName(name, reserved.type);

//#endregion
//#region src/ts-dsl/decl/decorator.ts
const Mixed$49 = ArgsMixin(TsDsl);
var DecoratorTsDsl = class extends Mixed$49 {
	"~dsl" = "DecoratorTsDsl";
	nameSanitizer = safeRuntimeName;
	constructor(name, ...args) {
		super();
		this.name.set(name);
		this.args(...args);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.name);
	}
	toAst() {
		const target = this.$node(this.name);
		const args = this.$args();
		return ts.factory.createDecorator(args.length ? ts.factory.createCallExpression(target, void 0, args) : target);
	}
};

//#endregion
//#region src/ts-dsl/mixins/decorator.ts
function DecoratorMixin(Base) {
	class Decorator extends Base {
		decorators = [];
		analyze(ctx$1) {
			super.analyze(ctx$1);
			for (const decorator of this.decorators) ctx$1.analyze(decorator);
		}
		decorator(name, ...args) {
			this.decorators.push(new DecoratorTsDsl(name, ...args));
			return this;
		}
		$decorators() {
			return this.$node(this.decorators);
		}
	}
	return Decorator;
}

//#endregion
//#region src/ts-dsl/utils/context.ts
function accessChainToNode(accessChain) {
	let result;
	accessChain.forEach((node, index) => {
		if (index === 0) result = node;
		else result = result.attr(node.name);
	});
	return result;
}
function getAccessChainForNode(node) {
	const accessChain = structuralToAccessChain([...getStructuralChainForNode(node, /* @__PURE__ */ new Set())]);
	if (accessChain.length === 0) return [node.clone()];
	return accessChain.map((node$1) => node$1.clone());
}
function getScope(node) {
	return node.scope ?? "value";
}
function getStructuralChainForNode(node, visited) {
	if (visited.has(node)) return [];
	visited.add(node);
	if (isStopNode(node)) return [];
	if (node.structuralParents) for (const [parent] of node.structuralParents) {
		if (getScope(parent) !== getScope(node)) continue;
		const chain = getStructuralChainForNode(parent, visited);
		if (chain.length > 0) return [...chain, node];
	}
	if (!node.root) return [];
	return [node];
}
function isAccessorNode(node) {
	return node["~dsl"] === "FieldTsDsl" || node["~dsl"] === "GetterTsDsl" || node["~dsl"] === "MethodTsDsl";
}
function isStopNode(node) {
	return node["~dsl"] === "FuncTsDsl" || node["~dsl"] === "TemplateTsDsl";
}
/**
* Fold a structural chain to an access chain by removing
* non-accessor nodes.
*/
function structuralToAccessChain(structuralChain) {
	const accessChain = [];
	structuralChain.forEach((node, index) => {
		if (index === 0) accessChain.push(node);
		else if (isAccessorNode(node)) accessChain.push(node);
	});
	return accessChain;
}
function transformAccessChain(accessChain, options = {}) {
	return accessChain.map((node, index) => {
		const transformedNode = options.transform?.(node, index, accessChain);
		if (transformedNode) return transformedNode;
		const accessNode = node.toAccessNode?.(node, options, {
			chain: accessChain,
			index,
			isLeaf: index === accessChain.length - 1,
			isRoot: index === 0,
			length: accessChain.length
		});
		if (accessNode) return accessNode;
		if (index === 0) {
			if (node["~dsl"] === "ClassTsDsl") {
				const nextNode = accessChain[index + 1];
				if (nextNode && isAccessorNode(nextNode)) {
					if (nextNode.hasModifier("static")) return $(node.name);
				}
				return $.new(node.name).args();
			}
			return $(node.name);
		}
		return node;
	});
}
var TsDslContext = class {
	/**
	* Build an expression for accessing the node.
	*
	* @param node - The node or symbol to build access for
	* @param options - Access options
	* @returns Expression for accessing the node
	*
	* @example
	* ```ts
	* ctx.access(node); // → Expression for accessing the node
	* ```
	*/
	access(node, options) {
		const n = isSymbol(node) ? node.node : node;
		if (!n) throw new Error(`Symbol ${node.name} is not resolved to a node.`);
		return accessChainToNode(transformAccessChain(getAccessChainForNode(n), options));
	}
	/**
	* Build an example.
	*
	* @param node - The node to generate an example for
	* @param options - Example options
	* @returns Full example string
	*
	* @example
	* ```ts
	* ctx.example(node, { moduleName: 'my-sdk' }); // → Full example string
	* ```
	*/
	example(node, options, astOptions) {
		if (astOptions) return TypeScriptRenderer.astToString(astOptions);
		options ||= {};
		const accessChain = getAccessChainForNode(node);
		if (options.importName) accessChain[0].name.set(options.importName);
		const importNode = $(accessChain[0].name.toString());
		const finalChain = transformAccessChain(accessChain, { context: "example" });
		const setupNode = options.importSetup ? typeof options.importSetup === "function" ? options.importSetup({
			$,
			node: importNode
		}) : options.importSetup : finalChain[0];
		const setupName = options.setupName;
		let payload = typeof options.payload === "function" ? options.payload({ $ }) : options.payload;
		payload = payload instanceof Array ? payload : payload ? [payload] : [];
		let nodes = [];
		if (setupName) nodes = [$.const(setupName).assign(setupNode), $.await(accessChainToNode([$(setupName), ...finalChain.slice(1)]).call(...payload))];
		else nodes = [$.await(accessChainToNode([setupNode, ...finalChain.slice(1)]).call(...payload))];
		const localName = importNode.name.toString();
		return TypeScriptRenderer.astToString({
			imports: [[{
				imports: !options.importKind || options.importKind === "named" ? [{
					isTypeOnly: false,
					localName,
					sourceName: localName
				}] : [],
				isTypeOnly: false,
				kind: options.importKind ?? "named",
				localName: options.importKind !== "named" ? localName : void 0,
				modulePath: options.moduleName ?? "your-package"
			}]],
			nodes,
			trailingNewline: false
		});
	}
};
const ctx = new TsDslContext();

//#endregion
//#region src/ts-dsl/layout/doc.ts
var DocTsDsl = class extends TsDsl {
	"~dsl" = "DocTsDsl";
	_lines = [];
	constructor(lines, fn) {
		super();
		if (lines) this.add(lines);
		fn?.(this);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	add(lines) {
		this._lines.push(lines);
		return this;
	}
	apply(node) {
		const lines = this._lines.reduce((lines$1, line) => {
			if (typeof line === "function") line = line(ctx);
			for (const l of typeof line === "string" ? [line] : line) if (l || l === "") lines$1.push(l);
			return lines$1;
		}, []);
		if (!lines.length) return node;
		const jsdocTexts = lines.map((line) => ts.factory.createJSDocText(`${line}\n`));
		const jsdoc = ts.factory.createJSDocComment(ts.factory.createNodeArray(jsdocTexts), void 0);
		const cleanedJsdoc = ts.createPrinter().printNode(ts.EmitHint.Unspecified, jsdoc, node.getSourceFile?.() ?? ts.createSourceFile("", "", ts.ScriptTarget.Latest)).replace("/*", "").replace("*  */", "");
		ts.addSyntheticLeadingComment(node, ts.SyntaxKind.MultiLineCommentTrivia, cleanedJsdoc, true);
		return node;
	}
	toAst() {
		return this.$node(new IdTsDsl(""));
	}
};

//#endregion
//#region src/ts-dsl/mixins/doc.ts
function DocMixin(Base) {
	class Doc extends Base {
		_doc;
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		doc(lines, fn) {
			this._doc = new DocTsDsl(lines, fn);
			return this;
		}
		$docs(node) {
			return this._doc ? this._doc.apply(node) : node;
		}
	}
	return Doc;
}

//#endregion
//#region src/ts-dsl/mixins/modifiers.ts
function modifierToKind(modifier) {
	switch (modifier) {
		case "abstract": return ts.SyntaxKind.AbstractKeyword;
		case "async": return ts.SyntaxKind.AsyncKeyword;
		case "const": return ts.SyntaxKind.ConstKeyword;
		case "declare": return ts.SyntaxKind.DeclareKeyword;
		case "default": return ts.SyntaxKind.DefaultKeyword;
		case "export": return ts.SyntaxKind.ExportKeyword;
		case "override": return ts.SyntaxKind.OverrideKeyword;
		case "private": return ts.SyntaxKind.PrivateKeyword;
		case "protected": return ts.SyntaxKind.ProtectedKeyword;
		case "public": return ts.SyntaxKind.PublicKeyword;
		case "readonly": return ts.SyntaxKind.ReadonlyKeyword;
		case "static": return ts.SyntaxKind.StaticKeyword;
	}
}
function ModifiersMixin(Base) {
	class Modifiers extends Base {
		modifiers = [];
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		hasModifier(modifier) {
			const kind = modifierToKind(modifier);
			return Boolean(this.modifiers.find((mod) => mod.kind === kind));
		}
		_m(modifier, condition) {
			if (condition) {
				const kind = modifierToKind(modifier);
				this.modifiers.push(ts.factory.createModifier(kind));
			}
			return this;
		}
	}
	return Modifiers;
}
/**
* Mixin that adds an `abstract` modifier to a node.
*/
function AbstractMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Abstract extends Mixed$53 {
		abstract(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("abstract", cond);
		}
	}
	return Abstract;
}
/**
* Mixin that adds an `async` modifier to a node.
*/
function AsyncMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Async extends Mixed$53 {
		async(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("async", cond);
		}
	}
	return Async;
}
/**
* Mixin that adds a `const` modifier to a node.
*/
function ConstMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Const extends Mixed$53 {
		const(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("const", cond);
		}
	}
	return Const;
}
/**
* Mixin that adds a `default` modifier to a node.
*/
function DefaultMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Default extends Mixed$53 {
		/**
		* Adds the `default` keyword modifier if the condition is true.
		*
		* @param condition - Whether to add the modifier (default: true).
		* @returns The target object for chaining.
		*/
		default(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("default", cond);
		}
	}
	return Default;
}
/**
* Mixin that adds an `export` modifier to a node.
*/
function ExportMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Export extends Mixed$53 {
		/**
		* Adds the `export` keyword modifier if the condition is true.
		*
		* @param condition - Whether to add the modifier (default: true).
		* @returns The target object for chaining.
		*/
		export(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			this.exported = cond;
			if (this.symbol) this.symbol.setExported(cond);
			return this._m("export", cond);
		}
	}
	return Export;
}
/**
* Mixin that adds a `private` modifier to a node.
*/
function PrivateMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Private extends Mixed$53 {
		private(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("private", cond);
		}
	}
	return Private;
}
/**
* Mixin that adds a `protected` modifier to a node.
*/
function ProtectedMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Protected extends Mixed$53 {
		protected(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("protected", cond);
		}
	}
	return Protected;
}
/**
* Mixin that adds a `public` modifier to a node.
*/
function PublicMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Public extends Mixed$53 {
		public(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("public", cond);
		}
	}
	return Public;
}
/**
* Mixin that adds a `readonly` modifier to a node.
*/
function ReadonlyMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Readonly extends Mixed$53 {
		readonly(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("readonly", cond);
		}
	}
	return Readonly;
}
/**
* Mixin that adds a `static` modifier to a node.
*/
function StaticMixin(Base) {
	const Mixed$53 = ModifiersMixin(Base);
	class Static extends Mixed$53 {
		static(condition) {
			const cond = arguments.length === 0 ? true : Boolean(condition);
			return this._m("static", cond);
		}
	}
	return Static;
}

//#endregion
//#region src/ts-dsl/type/param.ts
const Mixed$48 = TsDsl;
var TypeParamTsDsl = class extends Mixed$48 {
	"~dsl" = "TypeParamTsDsl";
	scope = "type";
	constraint;
	defaultValue;
	constructor(name, fn) {
		super();
		if (name) this.name.set(name);
		fn?.(this);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.name);
		ctx$1.analyze(this.constraint);
		ctx$1.analyze(this.defaultValue);
	}
	default(value) {
		this.defaultValue = ref(value);
		return this;
	}
	extends(constraint) {
		this.constraint = ref(constraint);
		return this;
	}
	toAst() {
		if (!this.name.toString()) throw new Error("Missing type name");
		return ts.factory.createTypeParameterDeclaration(void 0, this.$node(this.name), this.$type(this.constraint), this.$type(this.defaultValue));
	}
};

//#endregion
//#region src/ts-dsl/mixins/type-params.ts
function TypeParamsMixin(Base) {
	class TypeParams extends Base {
		_generics = [];
		analyze(ctx$1) {
			super.analyze(ctx$1);
			for (const g of this._generics) ctx$1.analyze(g);
		}
		generic(...args) {
			const g = new TypeParamTsDsl(...args);
			this._generics.push(g);
			return this;
		}
		generics(...args) {
			for (let arg of args) {
				if (typeof arg === "string" || typeof arg === "number" || isSymbol(arg) || isRef(arg)) arg = new TypeParamTsDsl(arg);
				this._generics.push(arg);
			}
			return this;
		}
		$generics() {
			return this.$node(this._generics);
		}
	}
	return TypeParams;
}

//#endregion
//#region src/ts-dsl/mixins/optional.ts
function OptionalMixin(Base) {
	class Optional extends Base {
		_optional;
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		optional(condition) {
			this._optional = arguments.length === 0 ? true : Boolean(condition);
			return this;
		}
		required(condition) {
			this._optional = arguments.length === 0 ? false : !condition;
			return this;
		}
	}
	return Optional;
}

//#endregion
//#region src/ts-dsl/mixins/value.ts
function ValueMixin(Base) {
	class Value extends Base {
		value;
		analyze(ctx$1) {
			super.analyze(ctx$1);
			ctx$1.analyze(this.value);
		}
		assign(expr) {
			this.value = expr;
			return this;
		}
		$value() {
			return this.$node(this.value);
		}
	}
	return Value;
}

//#endregion
//#region src/ts-dsl/token.ts
var TokenTsDsl = class extends TsDsl {
	"~dsl" = "TokenTsDsl";
	_kind;
	/** Sets the token kind */
	kind(kind) {
		this._kind = kind;
		return this;
	}
	/** Creates `-` */
	minus() {
		return this.kind(ts.SyntaxKind.MinusToken);
	}
	/** Creates `?` (optional) */
	optional() {
		return this.kind(ts.SyntaxKind.QuestionToken);
	}
	/** Creates `+` */
	plus() {
		return this.kind(ts.SyntaxKind.PlusToken);
	}
	/** Creates `?.` (optional chaining token) */
	questionDot() {
		return this.kind(ts.SyntaxKind.QuestionDotToken);
	}
	/** Creates `readonly` */
	readonly() {
		return this.kind(ts.SyntaxKind.ReadonlyKeyword);
	}
	/** Creates `...` (spread / rest) */
	spread() {
		return this.kind(ts.SyntaxKind.DotDotDotToken);
	}
	toAst() {
		this.$validate();
		return ts.factory.createToken(this._kind);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Token missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._kind) missing.push(".kind()");
		return missing;
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
};

//#endregion
//#region src/ts-dsl/mixins/type-args.ts
function TypeArgsMixin(Base) {
	class TypeArgs extends Base {
		_generics = [];
		analyze(ctx$1) {
			super.analyze(ctx$1);
			for (const g of this._generics) ctx$1.analyze(g);
		}
		generic(arg) {
			this._generics.push(ref(arg));
			return this;
		}
		generics(...args) {
			this._generics.push(...args.map((a) => ref(a)));
			return this;
		}
		$generics() {
			return this.$type(this._generics);
		}
	}
	return TypeArgs;
}

//#endregion
//#region src/ts-dsl/mixins/type-expr.ts
function TypeExprMixin(Base) {
	class TypeExpr extends Base {
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		idx(...args) {
			return f.type.idx(this, ...args);
		}
		keyof() {
			return f.type.operator().keyof(this);
		}
		readonly() {
			return f.type.operator().readonly(this);
		}
		returnType(...args) {
			return f.type.expr("ReturnType").generic(f.type.query(this, ...args));
		}
		typeofExpr(...args) {
			return f.typeofExpr(this, ...args);
		}
		typeofType(...args) {
			return f.type.query(this, ...args);
		}
		unique() {
			return f.type.operator().unique(this);
		}
	}
	return TypeExpr;
}

//#endregion
//#region src/ts-dsl/type/attr.ts
const Mixed$47 = TypeExprMixin(TsDsl);
var TypeAttrTsDsl = class extends Mixed$47 {
	"~dsl" = "TypeAttrTsDsl";
	scope = "type";
	_base;
	_right;
	constructor(base, right) {
		super();
		if (right) {
			this.base(base);
			this.right(right);
		} else {
			this.base();
			this.right(base);
		}
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._base);
		ctx$1.analyze(this._right);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	base(base) {
		if (isRef(base)) this._base = base;
		else this._base = base ? ref(base) : void 0;
		return this;
	}
	right(right) {
		this._right = ref(right);
		return this;
	}
	toAst() {
		this.$validate();
		const left = this.$node(this._base);
		if (!ts.isEntityName(left)) throw new Error("TypeAttrTsDsl: base must be an EntityName");
		return ts.factory.createQualifiedName(left, this.$node(this._right));
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Type attribute missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._base) missing.push(".base()");
		if (!this._right) missing.push(".right()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/type/expr.ts
const Mixed$46 = TypeArgsMixin(TypeExprMixin(TsDsl));
var TypeExprTsDsl = class extends Mixed$46 {
	"~dsl" = "TypeExprTsDsl";
	scope = "type";
	_exprInput;
	constructor(name, fn) {
		super();
		if (typeof name === "function") name(this);
		else {
			this._exprInput = name ? ref(name) : void 0;
			fn?.(this);
		}
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._exprInput);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Accesses a nested type (e.g. `Foo.Bar`). */
	attr(right) {
		this._exprInput = isNode(right) ? ref(right.base(this._exprInput)) : ref(new TypeAttrTsDsl(this._exprInput, right));
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createTypeReferenceNode(this.$type(this._exprInput), this.$generics());
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Type expression missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._exprInput) missing.push("name or .attr()");
		return missing;
	}
};
f.type.expr.set((...args) => new TypeExprTsDsl(...args));

//#endregion
//#region src/ts-dsl/decl/field.ts
const Mixed$45 = DecoratorMixin(DocMixin(OptionalMixin(PrivateMixin(ProtectedMixin(PublicMixin(ReadonlyMixin(StaticMixin(ValueMixin(TsDsl)))))))));
var FieldTsDsl = class extends Mixed$45 {
	"~dsl" = "FieldTsDsl";
	nameSanitizer = safeAccessorName;
	_type;
	constructor(name, fn) {
		super();
		this.name.set(name);
		fn?.(this);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.name);
		ctx$1.analyze(this._type);
	}
	/** Sets the field type. */
	type(type) {
		this._type = type instanceof TypeTsDsl ? type : new TypeExprTsDsl(type);
		return this;
	}
	toAst() {
		const node = ts.factory.createPropertyDeclaration([...this.$decorators(), ...this.modifiers], this.$node(this.name), this._optional ? this.$node(new TokenTsDsl().optional()) : void 0, this.$type(this._type), this.$value());
		return this.$docs(node);
	}
};

//#endregion
//#region src/ts-dsl/stmt/stmt.ts
const Mixed$44 = TsDsl;
var StmtTsDsl = class extends Mixed$44 {
	"~dsl" = "StmtTsDsl";
	_inner;
	constructor(inner) {
		super();
		this._inner = inner;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._inner);
	}
	toAst() {
		const node = this.$node(this._inner);
		return ts.isStatement(node) ? node : ts.factory.createExpressionStatement(node);
	}
};

//#endregion
//#region src/ts-dsl/mixins/do.ts
/**
* Adds `.do()` for appending statements or expressions to a body.
*/
function DoMixin(Base) {
	class Do extends Base {
		_do = [];
		analyze(ctx$1) {
			super.analyze(ctx$1);
			ctx$1.pushScope();
			try {
				for (const item of this._do) ctx$1.analyze(item);
			} finally {
				ctx$1.popScope();
			}
		}
		do(...items) {
			this._do.push(...items);
			return this;
		}
		$do() {
			return this.$node(this._do.map((item) => new StmtTsDsl(item)));
		}
	}
	return Do;
}

//#endregion
//#region src/ts-dsl/decl/pattern.ts
const Mixed$43 = TsDsl;
/**
* Builds binding patterns (e.g. `{ foo, bar }`, `[a, b, ...rest]`).
*/
var PatternTsDsl = class extends Mixed$43 {
	"~dsl" = "PatternTsDsl";
	pattern;
	_spread;
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Defines an array pattern (e.g. `[a, b, c]`). */
	array(...props) {
		this.pattern = {
			kind: "array",
			values: props[0] instanceof Array ? [...props[0]] : props
		};
		return this;
	}
	/** Defines an object pattern (e.g. `{ a, b: alias }`). */
	object(...props) {
		const entries = {};
		for (const p of props) if (typeof p === "string") entries[p] = p;
		else if (p instanceof Array) for (const n of p) entries[n] = n;
		else Object.assign(entries, p);
		this.pattern = {
			kind: "object",
			values: entries
		};
		return this;
	}
	/** Adds a spread element (e.g. `...rest`, `...options`, `...args`). */
	spread(name) {
		this._spread = name;
		return this;
	}
	toAst() {
		this.$validate();
		if (this.pattern.kind === "object") {
			const elements = Object.entries(this.pattern.values).map(([key, alias]) => key === alias ? ts.factory.createBindingElement(void 0, void 0, key, void 0) : ts.factory.createBindingElement(void 0, key, alias, void 0));
			const spread = this.createSpread();
			if (spread) elements.push(spread);
			return ts.factory.createObjectBindingPattern(elements);
		}
		if (this.pattern.kind === "array") {
			const elements = this.pattern.values.map((p) => ts.factory.createBindingElement(void 0, void 0, p, void 0));
			const spread = this.createSpread();
			if (spread) elements.push(spread);
			return ts.factory.createArrayBindingPattern(elements);
		}
		throw new Error("PatternTsDsl requires object() or array() pattern");
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Binding pattern missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this.pattern) missing.push(".array() or .object()");
		return missing;
	}
	createSpread() {
		return this._spread ? ts.factory.createBindingElement(this.$node(new TokenTsDsl().spread()), void 0, this.$node(new IdTsDsl(this._spread))) : void 0;
	}
};

//#endregion
//#region src/ts-dsl/mixins/pattern.ts
/**
* Mixin providing `.array()`, `.object()`, and `.spread()` methods for defining destructuring patterns.
*/
function PatternMixin(Base) {
	class Pattern extends Base {
		pattern;
		analyze(ctx$1) {
			super.analyze(ctx$1);
			ctx$1.analyze(this.pattern);
		}
		array(...props) {
			(this.pattern ??= new PatternTsDsl()).array(...props);
			return this;
		}
		object(...props) {
			(this.pattern ??= new PatternTsDsl()).object(...props);
			return this;
		}
		/** Adds a spread element (e.g. `...args`, `...options`) to the pattern. */
		spread(name) {
			(this.pattern ??= new PatternTsDsl()).spread(name);
			return this;
		}
		/** Renders the pattern into a `BindingName`. */
		$pattern() {
			if (!this.pattern) return;
			return this.$node(this.pattern);
		}
	}
	return Pattern;
}

//#endregion
//#region src/ts-dsl/decl/param.ts
const Mixed$42 = DecoratorMixin(OptionalMixin(PatternMixin(ValueMixin(TsDsl))));
var ParamTsDsl = class extends Mixed$42 {
	"~dsl" = "ParamTsDsl";
	_type;
	constructor(name, fn) {
		super();
		if (typeof name === "function") name(this);
		else {
			this.name.set(name);
			fn?.(this);
		}
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.name);
		ctx$1.analyze(this._type);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Sets the parameter type. */
	type(type) {
		this._type = type instanceof TypeTsDsl ? type : new TypeExprTsDsl(type);
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createParameterDeclaration(this.$decorators(), void 0, this.$pattern() ?? this.name.toString(), this._optional ? this.$node(new TokenTsDsl().optional()) : void 0, this.$type(this._type), this.$value());
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Parameter missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this.$pattern() && !this.name.toString()) missing.push("name or pattern (.array()/.object())");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/mixins/param.ts
function ParamMixin(Base) {
	class Param extends Base {
		_params = [];
		analyze(ctx$1) {
			super.analyze(ctx$1);
			for (const param of this._params) ctx$1.analyze(param);
		}
		param(name, fn) {
			const p = new ParamTsDsl(name, fn);
			this._params.push(p);
			return this;
		}
		params(...params) {
			this._params.push(...params);
			return this;
		}
		$params() {
			return this.$node(this._params);
		}
	}
	return Param;
}

//#endregion
//#region src/ts-dsl/mixins/layout.ts
function LayoutMixin(Base) {
	class Layout extends Base {
		static DEFAULT_THRESHOLD = 3;
		layout;
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		auto(threshold = Layout.DEFAULT_THRESHOLD) {
			this.layout = threshold;
			return this;
		}
		inline() {
			this.layout = false;
			return this;
		}
		pretty() {
			this.layout = true;
			return this;
		}
		$multiline(count) {
			if (this.layout === void 0) this.layout = Layout.DEFAULT_THRESHOLD;
			if (count === 0) return false;
			return typeof this.layout === "number" ? count >= this.layout : this.layout;
		}
	}
	return Layout;
}

//#endregion
//#region src/ts-dsl/stmt/block.ts
const Mixed$41 = DoMixin(LayoutMixin(TsDsl));
var BlockTsDsl = class extends Mixed$41 {
	"~dsl" = "BlockTsDsl";
	constructor(...items) {
		super();
		this.do(...items);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	toAst() {
		const statements = this.$do();
		return ts.factory.createBlock(statements, this.$multiline(statements.length));
	}
};

//#endregion
//#region src/ts-dsl/decl/init.ts
const Mixed$40 = DecoratorMixin(DoMixin(DocMixin(ParamMixin(PrivateMixin(ProtectedMixin(PublicMixin(TsDsl)))))));
var InitTsDsl = class extends Mixed$40 {
	"~dsl" = "InitTsDsl";
	constructor(fn) {
		super();
		fn?.(this);
	}
	analyze(ctx$1) {
		ctx$1.pushScope();
		try {
			super.analyze(ctx$1);
		} finally {
			ctx$1.popScope();
		}
	}
	toAst() {
		const node = ts.factory.createConstructorDeclaration([...this.$decorators(), ...this.modifiers], this.$params(), this.$node(new BlockTsDsl(...this._do).pretty()));
		return this.$docs(node);
	}
};

//#endregion
//#region src/ts-dsl/mixins/type-returns.ts
function TypeReturnsMixin(Base) {
	class TypeReturns extends Base {
		_returns;
		analyze(ctx$1) {
			super.analyze(ctx$1);
			ctx$1.analyze(this._returns);
		}
		returns(type) {
			this._returns = type instanceof TypeTsDsl ? type : new TypeExprTsDsl(type);
			return this;
		}
		$returns() {
			return this.$type(this._returns);
		}
	}
	return TypeReturns;
}

//#endregion
//#region src/ts-dsl/decl/method.ts
const Mixed$39 = AbstractMixin(AsyncMixin(DecoratorMixin(DoMixin(DocMixin(OptionalMixin(ParamMixin(PrivateMixin(ProtectedMixin(PublicMixin(StaticMixin(TypeParamsMixin(TypeReturnsMixin(TsDsl)))))))))))));
var MethodTsDsl = class extends Mixed$39 {
	"~dsl" = "MethodTsDsl";
	nameSanitizer = safeAccessorName;
	constructor(name, fn) {
		super();
		this.name.set(name);
		fn?.(this);
	}
	analyze(ctx$1) {
		ctx$1.analyze(this.name);
		ctx$1.pushScope();
		try {
			super.analyze(ctx$1);
		} finally {
			ctx$1.popScope();
		}
	}
	toAst() {
		const node = ts.factory.createMethodDeclaration([...this.$decorators(), ...this.modifiers], void 0, this.$node(this.name), this._optional ? this.$node(new TokenTsDsl().optional()) : void 0, this.$generics(), this.$params(), this.$returns(), this.$node(new BlockTsDsl(...this._do).pretty()));
		return this.$docs(node);
	}
};

//#endregion
//#region src/ts-dsl/decl/class.ts
const Mixed$38 = AbstractMixin(DecoratorMixin(DefaultMixin(DocMixin(ExportMixin(TypeParamsMixin(TsDsl))))));
var ClassTsDsl = class extends Mixed$38 {
	"~dsl" = "ClassTsDsl";
	nameSanitizer = safeRuntimeName;
	baseClass;
	body = [];
	constructor(name) {
		super();
		this.name.set(name);
		if (isSymbol(name)) name.setKind("class");
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.baseClass);
		ctx$1.analyze(this.name);
		ctx$1.pushScope();
		try {
			for (const item of this.body) ctx$1.analyze(item);
		} finally {
			ctx$1.popScope();
		}
	}
	/** Returns true if the class has any members. */
	get hasBody() {
		return this.body.length > 0;
	}
	/** Adds one or more class members (fields, methods, etc.). */
	do(...items) {
		this.body.push(...items);
		return this;
	}
	/** Records a base class to extend from. */
	extends(base) {
		this.baseClass = base ? ref(base) : void 0;
		return this;
	}
	/** Adds a class field. */
	field(name, fn) {
		const f$1 = new FieldTsDsl(name, fn);
		this.body.push(f$1);
		return this;
	}
	/** Adds a class constructor. */
	init(fn) {
		const i = typeof fn === "function" ? new InitTsDsl(fn) : fn || new InitTsDsl();
		this.body.push(i);
		return this;
	}
	/** Adds a class method. */
	method(name, fn) {
		const m = new MethodTsDsl(name, fn);
		this.body.push(m);
		return this;
	}
	/** Inserts an empty line between members for formatting. */
	newline() {
		this.body.push(new NewlineTsDsl());
		return this;
	}
	toAst() {
		const body = this.$node(this.body);
		const node = ts.factory.createClassDeclaration([...this.$decorators(), ...this.modifiers], this.$node(this.name), this.$generics(), this._heritage(), body);
		return this.$docs(node);
	}
	/** Builds heritage clauses (extends). */
	_heritage() {
		const node = this.$node(this.baseClass);
		if (!node) return [];
		return [ts.factory.createHeritageClause(ts.SyntaxKind.ExtendsKeyword, [ts.factory.createExpressionWithTypeArguments(node, void 0)])];
	}
};

//#endregion
//#region src/ts-dsl/decl/member.ts
const Mixed$37 = DocMixin(TsDsl);
var EnumMemberTsDsl = class extends Mixed$37 {
	"~dsl" = "EnumMemberTsDsl";
	_value;
	constructor(name, value) {
		super();
		this.name.set(name);
		if (typeof value === "function") value(this);
		else this.value(value);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._value);
	}
	/** Sets the enum member value. */
	value(value) {
		this._value = value;
		return this;
	}
	toAst() {
		const node = ts.factory.createEnumMember(this.$node(safeMemberName(this.name.toString())), this.$node(this._value));
		return this.$docs(node);
	}
};

//#endregion
//#region src/ts-dsl/decl/enum.ts
const Mixed$36 = ConstMixin(DocMixin(ExportMixin(TsDsl)));
var EnumTsDsl = class extends Mixed$36 {
	"~dsl" = "EnumTsDsl";
	nameSanitizer = safeRuntimeName;
	_members = [];
	constructor(name, fn) {
		super();
		this.name.set(name);
		if (isSymbol(name)) name.setKind("enum");
		fn?.(this);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.name);
		ctx$1.pushScope();
		try {
			for (const member of this._members) ctx$1.analyze(member);
		} finally {
			ctx$1.popScope();
		}
	}
	/** Adds an enum member. */
	member(name, value) {
		const m = new EnumMemberTsDsl(name, value);
		this._members.push(m);
		return this;
	}
	/** Adds multiple enum members. */
	members(...members) {
		this._members.push(...members);
		return this;
	}
	toAst() {
		const node = ts.factory.createEnumDeclaration(this.modifiers, this.$node(this.name), this.$node(this._members));
		return this.$docs(node);
	}
};

//#endregion
//#region src/ts-dsl/decl/func.ts
const Mixed$35 = AbstractMixin(AsMixin(AsyncMixin(DecoratorMixin(DoMixin(DocMixin(ParamMixin(PrivateMixin(ProtectedMixin(PublicMixin(StaticMixin(TypeParamsMixin(TypeReturnsMixin(TsDsl)))))))))))));
var ImplFuncTsDsl = class extends Mixed$35 {
	"~dsl" = "FuncTsDsl";
	nameSanitizer = safeRuntimeName;
	mode;
	constructor(name, fn) {
		super();
		if (typeof name === "function") {
			this.mode = "arrow";
			name(this);
		} else if (name) {
			this.mode = "decl";
			this.name.set(name);
			if (isSymbol(name)) name.setKind("function");
			fn?.(this);
		}
	}
	analyze(ctx$1) {
		ctx$1.pushScope();
		try {
			super.analyze(ctx$1);
			ctx$1.analyze(this.name);
		} finally {
			ctx$1.popScope();
		}
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Switches the function to an arrow function form. */
	arrow() {
		this.mode = "arrow";
		return this;
	}
	/** Switches the function to a function declaration form. */
	decl() {
		this.mode = "decl";
		return this;
	}
	/** Switches the function to a function expression form. */
	expr() {
		this.mode = "expr";
		return this;
	}
	toAst() {
		this.$validate();
		const body = this.$node(new BlockTsDsl(...this._do).pretty());
		if (this.mode === "decl") {
			const node$1 = ts.factory.createFunctionDeclaration([...this.$decorators(), ...this.modifiers], void 0, this.$node(this.name), this.$generics(), this.$params(), this.$returns(), body);
			return this.$docs(node$1);
		}
		if (this.mode === "expr") {
			const node$1 = ts.factory.createFunctionExpression(this.modifiers, void 0, this.$node(this.name), this.$generics(), this.$params(), this.$returns(), body);
			return this.$docs(node$1);
		}
		const node = ts.factory.createArrowFunction(this.modifiers, this.$generics(), this.$params(), this.$returns(), void 0, body.statements.length === 1 && ts.isReturnStatement(body.statements[0]) && body.statements[0].expression ? body.statements[0].expression : body);
		return this.$docs(node);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Function ${this.mode} missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (this.mode === "decl" && !this.name.toString()) missing.push("name");
		return missing;
	}
};
const FuncTsDsl = ImplFuncTsDsl;

//#endregion
//#region src/ts-dsl/decl/getter.ts
const Mixed$34 = AbstractMixin(AsyncMixin(DecoratorMixin(DoMixin(DocMixin(ParamMixin(PrivateMixin(ProtectedMixin(PublicMixin(StaticMixin(TypeReturnsMixin(TsDsl)))))))))));
var GetterTsDsl = class extends Mixed$34 {
	"~dsl" = "GetterTsDsl";
	nameSanitizer = safeAccessorName;
	constructor(name, fn) {
		super();
		this.name.set(name);
		fn?.(this);
	}
	analyze(ctx$1) {
		ctx$1.analyze(this.name);
		ctx$1.pushScope();
		try {
			super.analyze(ctx$1);
		} finally {
			ctx$1.popScope();
		}
	}
	toAst() {
		const node = ts.factory.createGetAccessorDeclaration([...this.$decorators(), ...this.modifiers], this.$node(this.name), this.$params(), this.$returns(), this.$node(new BlockTsDsl(...this._do).pretty()));
		return this.$docs(node);
	}
};

//#endregion
//#region src/ts-dsl/decl/setter.ts
const Mixed$33 = AbstractMixin(AsyncMixin(DecoratorMixin(DoMixin(DocMixin(ParamMixin(PrivateMixin(ProtectedMixin(PublicMixin(StaticMixin(TsDsl))))))))));
var SetterTsDsl = class extends Mixed$33 {
	"~dsl" = "SetterTsDsl";
	nameSanitizer = safeAccessorName;
	constructor(name, fn) {
		super();
		this.name.set(name);
		fn?.(this);
	}
	analyze(ctx$1) {
		ctx$1.analyze(this.name);
		ctx$1.pushScope();
		try {
			super.analyze(ctx$1);
		} finally {
			ctx$1.popScope();
		}
	}
	toAst() {
		const node = ts.factory.createSetAccessorDeclaration([...this.$decorators(), ...this.modifiers], this.$node(this.name), this.$params(), this.$node(new BlockTsDsl(...this._do).pretty()));
		return this.$docs(node);
	}
};

//#endregion
//#region src/ts-dsl/expr/array.ts
const Mixed$32 = AsMixin(LayoutMixin(TsDsl));
var ArrayTsDsl = class extends Mixed$32 {
	"~dsl" = "ArrayTsDsl";
	_elements = [];
	constructor(...exprs) {
		super();
		this.elements(...exprs);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const item of this._elements) ctx$1.analyze(item.expr);
	}
	/** Adds a single array element. */
	element(expr) {
		const node = typeof expr === "string" || typeof expr === "number" || typeof expr === "boolean" ? new LiteralTsDsl(expr) : expr;
		this._elements.push({
			expr: node,
			kind: "element"
		});
		return this;
	}
	/** Adds multiple array elements. */
	elements(...exprs) {
		for (const expr of exprs) this.element(expr);
		return this;
	}
	/** Adds a spread element (`...expr`). */
	spread(expr) {
		this._elements.push({
			expr,
			kind: "spread"
		});
		return this;
	}
	toAst() {
		const elements = this._elements.map((item) => {
			const node = this.$node(item.expr);
			return item.kind === "spread" ? ts.factory.createSpreadElement(node) : node;
		});
		return ts.factory.createArrayLiteralExpression(elements, this.$multiline(this._elements.length));
	}
};

//#endregion
//#region src/ts-dsl/mixins/expr.ts
function ExprMixin(Base) {
	class Expr extends Base {
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		attr(...args) {
			return f.attr(this, ...args);
		}
		await() {
			return f.await(this);
		}
		call(...args) {
			return f.call(this, ...args);
		}
		return() {
			return f.return(this);
		}
	}
	return Expr;
}

//#endregion
//#region src/ts-dsl/expr/as.ts
const Mixed$31 = AsMixin(ExprMixin(TsDsl));
var AsTsDsl = class extends Mixed$31 {
	"~dsl" = "AsTsDsl";
	expr;
	type;
	constructor(expr, type) {
		super();
		this.expr = ref(expr);
		this.type = ref(type);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.expr);
		ctx$1.analyze(this.type);
	}
	toAst() {
		return ts.factory.createAsExpression(this.$node(this.expr), this.$type(this.type));
	}
};
f.as.set((...args) => new AsTsDsl(...args));

//#endregion
//#region src/ts-dsl/expr/binary.ts
const Mixed$30 = AsMixin(ExprMixin(TsDsl));
var BinaryTsDsl = class extends Mixed$30 {
	"~dsl" = "BinaryTsDsl";
	_base;
	_expr;
	_op;
	constructor(base, op, expr) {
		super();
		this._base = ref(base);
		this._op = op;
		if (expr) this._expr = ref(expr);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._base);
		ctx$1.analyze(this._expr);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Logical AND — `this && expr` */
	and(expr) {
		return this.opAndExpr("&&", expr);
	}
	/** Creates an assignment expression (e.g. `this = expr`). */
	assign(expr) {
		return this.opAndExpr("=", expr);
	}
	/** Nullish coalescing — `this ?? expr` */
	coalesce(expr) {
		return this.opAndExpr("??", expr);
	}
	/** Division — `this / expr` */
	div(expr) {
		return this.opAndExpr("/", expr);
	}
	/** Strict equality — `this === expr` */
	eq(expr) {
		return this.opAndExpr("===", expr);
	}
	/** Greater than — `this > expr` */
	gt(expr) {
		return this.opAndExpr(">", expr);
	}
	/** Greater than or equal — `this >= expr` */
	gte(expr) {
		return this.opAndExpr(">=", expr);
	}
	/** Loose equality — `this == expr` */
	looseEq(expr) {
		return this.opAndExpr("==", expr);
	}
	/** Loose inequality — `this != expr` */
	looseNeq(expr) {
		return this.opAndExpr("!=", expr);
	}
	/** Less than — `this < expr` */
	lt(expr) {
		return this.opAndExpr("<", expr);
	}
	/** Less than or equal — `this <= expr` */
	lte(expr) {
		return this.opAndExpr("<=", expr);
	}
	/** Subtraction — `this - expr` */
	minus(expr) {
		return this.opAndExpr("-", expr);
	}
	/** Strict inequality — `this !== expr` */
	neq(expr) {
		return this.opAndExpr("!==", expr);
	}
	/** Nullish assignment — `this ??= expr` */
	nullishAssign(expr) {
		return this.opAndExpr("??=", expr);
	}
	/** Logical OR — `this || expr` */
	or(expr) {
		return this.opAndExpr("||", expr);
	}
	/** Addition — `this + expr` */
	plus(expr) {
		return this.opAndExpr("+", expr);
	}
	/** Multiplication — `this * expr` */
	times(expr) {
		return this.opAndExpr("*", expr);
	}
	toAst() {
		this.$validate();
		const base = this.$node(this._base);
		const operator = typeof this._op === "string" ? this.opToToken(this._op) : this._op;
		return ts.factory.createBinaryExpression(base, operator, this.$node(this._expr));
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Binary expression missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._op) missing.push("operator (e.g., .eq(), .plus())");
		if (!this._expr) missing.push("right-hand expression");
		return missing;
	}
	/** Sets the binary operator and right-hand operand for this expression. */
	opAndExpr(op, expr) {
		this._expr = ref(expr);
		this._op = op;
		return this;
	}
	opToToken(op) {
		const token = {
			"!=": ts.SyntaxKind.ExclamationEqualsToken,
			"!==": ts.SyntaxKind.ExclamationEqualsEqualsToken,
			"&&": ts.SyntaxKind.AmpersandAmpersandToken,
			"*": ts.SyntaxKind.AsteriskToken,
			"+": ts.SyntaxKind.PlusToken,
			"-": ts.SyntaxKind.MinusToken,
			"/": ts.SyntaxKind.SlashToken,
			"<": ts.SyntaxKind.LessThanToken,
			"<=": ts.SyntaxKind.LessThanEqualsToken,
			"=": ts.SyntaxKind.EqualsToken,
			"==": ts.SyntaxKind.EqualsEqualsToken,
			"===": ts.SyntaxKind.EqualsEqualsEqualsToken,
			">": ts.SyntaxKind.GreaterThanToken,
			">=": ts.SyntaxKind.GreaterThanEqualsToken,
			"??": ts.SyntaxKind.QuestionQuestionToken,
			"??=": ts.SyntaxKind.QuestionQuestionEqualsToken,
			"||": ts.SyntaxKind.BarBarToken
		}[op];
		if (!token) throw new Error(`Unsupported operator: ${op}`);
		return token;
	}
};

//#endregion
//#region src/ts-dsl/mixins/operator.ts
function OperatorMixin(Base) {
	class Operator extends Base {
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		and(expr) {
			return new BinaryTsDsl(this).and(expr);
		}
		assign(expr) {
			return new BinaryTsDsl(this, "=", expr);
		}
		coalesce(expr) {
			return new BinaryTsDsl(this).coalesce(expr);
		}
		div(expr) {
			return new BinaryTsDsl(this).div(expr);
		}
		eq(expr) {
			return new BinaryTsDsl(this).eq(expr);
		}
		gt(expr) {
			return new BinaryTsDsl(this).gt(expr);
		}
		gte(expr) {
			return new BinaryTsDsl(this).gte(expr);
		}
		looseEq(expr) {
			return new BinaryTsDsl(this).looseEq(expr);
		}
		looseNeq(expr) {
			return new BinaryTsDsl(this).looseNeq(expr);
		}
		lt(expr) {
			return new BinaryTsDsl(this).lt(expr);
		}
		lte(expr) {
			return new BinaryTsDsl(this).lte(expr);
		}
		minus(expr) {
			return new BinaryTsDsl(this).minus(expr);
		}
		neq(expr) {
			return new BinaryTsDsl(this).neq(expr);
		}
		nullishAssign(expr) {
			return new BinaryTsDsl(this).nullishAssign(expr);
		}
		or(expr) {
			return new BinaryTsDsl(this).or(expr);
		}
		plus(expr) {
			return new BinaryTsDsl(this).plus(expr);
		}
		times(expr) {
			return new BinaryTsDsl(this).times(expr);
		}
	}
	return Operator;
}

//#endregion
//#region src/ts-dsl/expr/attr.ts
const Mixed$29 = AsMixin(ExprMixin(OperatorMixin(OptionalMixin(TsDsl))));
var AttrTsDsl = class extends Mixed$29 {
	"~dsl" = "AttrTsDsl";
	left;
	constructor(left, right) {
		super();
		this.left = ref(left);
		this.name.set(right);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.left);
		ctx$1.analyze(this.name);
	}
	toAst() {
		const leftNode = this.$node(this.left);
		regexp.typeScriptIdentifier.lastIndex = 0;
		const right = fromRef(this.name);
		if (!regexp.typeScriptIdentifier.test(this.name.toString())) {
			let value = isSymbol(right) ? right.finalName : right;
			if (typeof value === "string") {
				if (value.startsWith("'") && value.endsWith("'") || value.startsWith("\"") && value.endsWith("\"")) value = value.slice(1, -1);
			}
			if (this._optional) return ts.factory.createElementAccessChain(leftNode, this.$node(new TokenTsDsl().questionDot()), this.$node(new LiteralTsDsl(value)));
			return ts.factory.createElementAccessExpression(leftNode, this.$node(new LiteralTsDsl(value)));
		}
		if (this._optional) return ts.factory.createPropertyAccessChain(leftNode, this.$node(new TokenTsDsl().questionDot()), this.$node(this.name));
		return ts.factory.createPropertyAccessExpression(leftNode, this.$node(this.name));
	}
};
f.attr.set((...args) => new AttrTsDsl(...args));

//#endregion
//#region src/ts-dsl/expr/await.ts
const Mixed$28 = ExprMixin(TsDsl);
var AwaitTsDsl = class extends Mixed$28 {
	"~dsl" = "AwaitTsDsl";
	_awaitExpr;
	constructor(expr) {
		super();
		this._awaitExpr = ref(expr);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._awaitExpr);
	}
	toAst() {
		return ts.factory.createAwaitExpression(this.$node(this._awaitExpr));
	}
};
f.await.set((...args) => new AwaitTsDsl(...args));

//#endregion
//#region src/ts-dsl/expr/call.ts
const Mixed$27 = ArgsMixin(AsMixin(ExprMixin(TypeArgsMixin(TsDsl))));
var CallTsDsl = class extends Mixed$27 {
	"~dsl" = "CallTsDsl";
	_callExpr;
	constructor(expr, ...args) {
		super();
		this._callExpr = ref(expr);
		this.args(...args);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._callExpr);
	}
	toAst() {
		return ts.factory.createCallExpression(this.$node(this._callExpr), this.$generics(), this.$args());
	}
};
f.call.set((...args) => new CallTsDsl(...args));

//#endregion
//#region src/ts-dsl/expr/expr.ts
const Mixed$26 = AsMixin(ExprMixin(OperatorMixin(TypeExprMixin(TsDsl))));
var ExprTsDsl = class extends Mixed$26 {
	"~dsl" = "ExprTsDsl";
	_exprInput;
	constructor(id) {
		super();
		this._exprInput = ref(id);
		if (typeof id === "string" || isSymbol(id)) this.name.set(id);
		else if (isNode(id)) this.name.set(id.name);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._exprInput);
	}
	toAst() {
		return this.$node(this._exprInput);
	}
};

//#endregion
//#region src/ts-dsl/layout/hint.ts
var HintTsDsl = class extends TsDsl {
	"~dsl" = "HintTsDsl";
	_lines = [];
	constructor(lines, fn) {
		super();
		if (lines) this.add(lines);
		fn?.(this);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	add(lines) {
		this._lines.push(lines);
		return this;
	}
	apply(node) {
		const lines = this._lines.reduce((lines$1, line) => {
			if (typeof line === "function") line = line(ctx);
			for (const l of typeof line === "string" ? [line] : line) if (l || l === "") lines$1.push(l);
			return lines$1;
		}, []);
		if (!lines.length) return node;
		for (const line of lines) ts.addSyntheticLeadingComment(node, ts.SyntaxKind.SingleLineCommentTrivia, ` ${line}`, false);
		return node;
	}
	toAst() {
		return this.$node(new IdTsDsl(""));
	}
};

//#endregion
//#region src/ts-dsl/mixins/hint.ts
function HintMixin(Base) {
	class Hint extends Base {
		_hint;
		analyze(ctx$1) {
			super.analyze(ctx$1);
		}
		hint(lines, fn) {
			this._hint = new HintTsDsl(lines, fn);
			return this;
		}
		$hint(node) {
			return this._hint ? this._hint.apply(node) : node;
		}
	}
	return Hint;
}

//#endregion
//#region src/ts-dsl/expr/prop.ts
const Mixed$25 = DocMixin(TsDsl);
var ObjectPropTsDsl = class extends Mixed$25 {
	"~dsl" = "ObjectPropTsDsl";
	_value;
	_meta;
	constructor(meta) {
		super();
		this._meta = meta;
	}
	get kind() {
		return this._meta.kind;
	}
	get propName() {
		return this._meta.name;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._value);
	}
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	value(value) {
		if (typeof value === "function") value(this);
		else this._value = ref(value);
		return this;
	}
	toAst() {
		this.$validate();
		const node = this.$node(this._value);
		if (this._meta.kind === "spread") {
			if (ts.isStatement(node)) throw new Error("Invalid spread: object spread must be an expression, not a statement.");
			const result$1 = ts.factory.createSpreadAssignment(node);
			return this.$docs(result$1);
		}
		if (this._meta.kind === "getter") {
			const getter = new GetterTsDsl(this._meta.name).do(node);
			const result$1 = this.$node(getter);
			return this.$docs(result$1);
		}
		if (this._meta.kind === "setter") {
			const setter = new SetterTsDsl(this._meta.name).do(node);
			const result$1 = this.$node(setter);
			return this.$docs(result$1);
		}
		if (ts.isIdentifier(node) && node.text === this._meta.name) {
			const result$1 = ts.factory.createShorthandPropertyAssignment(this._meta.name);
			return this.$docs(result$1);
		}
		if (ts.isStatement(node)) throw new Error("Invalid property: object property value must be an expression, not a statement.");
		const result = ts.factory.createPropertyAssignment(this._meta.kind === "computed" ? ts.factory.createComputedPropertyName(this.$node(new IdTsDsl(this._meta.name))) : this.$node(safePropName(this._meta.name)), node);
		return this.$docs(result);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Object property${this._meta.name ? ` "${this._meta.name}"` : ""} missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._value) missing.push(".value()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/expr/object.ts
const Mixed$24 = AsMixin(ExprMixin(HintMixin(LayoutMixin(TsDsl))));
var ObjectTsDsl = class extends Mixed$24 {
	"~dsl" = "ObjectTsDsl";
	_props = /* @__PURE__ */ new Map();
	_spreadCounter = 0;
	constructor(...props) {
		super();
		this.props(...props);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const prop of this._props.values()) ctx$1.analyze(prop);
	}
	/** Returns composite key for the property. */
	_propKey(prop) {
		if (prop.kind === "spread") return `spread:${this._spreadCounter++}`;
		return `${prop.kind}:${prop.propName}`;
	}
	/** Adds a computed property (e.g. `{ [expr]: value }`), or removes if null. */
	computed(name, expr) {
		if (expr === null) this._props.delete(`computed:${name}`);
		else this._props.set(`computed:${name}`, new ObjectPropTsDsl({
			kind: "computed",
			name
		}).value(expr));
		return this;
	}
	/** Adds a getter property (e.g. `{ get foo() { ... } }`), or removes if null. */
	getter(name, stmt) {
		if (stmt === null) this._props.delete(`getter:${name}`);
		else this._props.set(`getter:${name}`, new ObjectPropTsDsl({
			kind: "getter",
			name
		}).value(stmt));
		return this;
	}
	/** Returns true if object has at least one property or spread. */
	hasProps() {
		return this._props.size > 0;
	}
	/** Returns true if object has no properties or spreads. */
	get isEmpty() {
		return this._props.size === 0;
	}
	/** Adds a property assignment, or removes if null. */
	prop(name, expr) {
		if (expr === null) this._props.delete(`prop:${name}`);
		else this._props.set(`prop:${name}`, new ObjectPropTsDsl({
			kind: "prop",
			name
		}).value(expr));
		return this;
	}
	/** Adds multiple properties. */
	props(...props) {
		for (const prop of props) this._props.set(this._propKey(prop), prop);
		return this;
	}
	/** Adds a setter property (e.g. `{ set foo(v) { ... } }`), or removes if null. */
	setter(name, stmt) {
		if (stmt === null) this._props.delete(`setter:${name}`);
		else this._props.set(`setter:${name}`, new ObjectPropTsDsl({
			kind: "setter",
			name
		}).value(stmt));
		return this;
	}
	/** Adds a spread property (e.g. `{ ...options }`). */
	spread(expr) {
		const key = `spread:${this._spreadCounter++}`;
		this._props.set(key, new ObjectPropTsDsl({ kind: "spread" }).value(expr));
		return this;
	}
	toAst() {
		const props = [...this._props.values()];
		const node = ts.factory.createObjectLiteralExpression(this.$node(props), this.$multiline(props.length));
		return this.$hint(node);
	}
};

//#endregion
//#region src/ts-dsl/expr/fromValue.ts
const fromValue$1 = (input, options) => {
	if (isNode(input)) return input;
	if (input === null || typeof input === "number" || typeof input === "boolean" || typeof input === "string" || typeof input === "bigint") return new LiteralTsDsl(input);
	if (input instanceof Array) {
		const arr = new ArrayTsDsl(...input.map((v) => fromValue$1(v, options)));
		if (options?.layout === "pretty") arr.pretty();
		return arr;
	}
	if (typeof input === "object") {
		const obj = new ObjectTsDsl();
		for (const [key, val] of Object.entries(input)) {
			const expr = fromValue$1(val, options);
			obj.prop(key, expr);
		}
		if (options?.layout === "pretty") obj.pretty();
		return obj;
	}
	throw new Error(`$.fromValue(): Unsupported input type ${String(input)}`);
};

//#endregion
//#region src/ts-dsl/expr/new.ts
const Mixed$23 = ArgsMixin(AsMixin(ExprMixin(TypeArgsMixin(TsDsl))));
var NewTsDsl = class extends Mixed$23 {
	"~dsl" = "NewTsDsl";
	_newExpr;
	constructor(expr, ...args) {
		super();
		this._newExpr = ref(expr);
		this.args(...args);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._newExpr);
	}
	toAst() {
		return ts.factory.createNewExpression(this.$node(this._newExpr), this.$generics(), this.$args());
	}
};
f.new.set((...args) => new NewTsDsl(...args));

//#endregion
//#region src/ts-dsl/expr/regexp.ts
const Mixed$22 = TsDsl;
var RegExpTsDsl = class extends Mixed$22 {
	"~dsl" = "RegExpTsDsl";
	pattern;
	flags;
	constructor(pattern, flags) {
		super();
		this.pattern = pattern;
		this.flags = flags;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	toAst() {
		const literal = `/${(this.pattern.startsWith("/") && this.pattern.endsWith("/") ? this.pattern.slice(1, -1) : this.pattern).replace(/(?<!\\)\//g, "\\/")}/${this.flags ?? ""}`;
		return ts.factory.createRegularExpressionLiteral(literal);
	}
};

//#endregion
//#region src/ts-dsl/expr/template.ts
const Mixed$21 = TsDsl;
var TemplateTsDsl = class extends Mixed$21 {
	"~dsl" = "TemplateTsDsl";
	parts = [];
	constructor(value) {
		super();
		if (value !== void 0) this.add(value);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const part of this.parts) ctx$1.analyze(part);
	}
	add(value) {
		this.parts.push(ref(value));
		return this;
	}
	toAst() {
		const parts = this.$node(this.parts.map((p) => {
			const part = fromRef(p);
			return isSymbol(part) ? part.finalName : part;
		}));
		const normalized = [];
		for (let index = 0; index < parts.length; index++) {
			const current = parts[index];
			if (typeof current === "string") {
				let merged = current;
				while (index + 1 < parts.length && typeof parts[index + 1] === "string") {
					merged += parts[index + 1];
					index++;
				}
				normalized.push(merged);
			} else if (typeof current === "number") normalized.push(String(current));
			else normalized.push(current);
		}
		if (normalized.length === 0 || typeof normalized[0] !== "string") normalized.unshift("");
		if (normalized.length === 1 && typeof normalized[0] === "string") return ts.factory.createNoSubstitutionTemplateLiteral(normalized[0]);
		if (normalized.length === 2 && typeof normalized[0] === "string" && typeof normalized[1] !== "string") return ts.factory.createTemplateExpression(ts.factory.createTemplateHead(normalized[0]), [ts.factory.createTemplateSpan(normalized[1], ts.factory.createTemplateTail(""))]);
		const head = ts.factory.createTemplateHead(normalized.shift());
		const spans = [];
		while (normalized.length) {
			const expr = normalized.shift();
			const next = typeof normalized[0] === "string" ? normalized.shift() : "";
			const isLast = normalized.length === 0;
			spans.push(ts.factory.createTemplateSpan(expr, isLast ? ts.factory.createTemplateTail(next) : ts.factory.createTemplateMiddle(next)));
		}
		return ts.factory.createTemplateExpression(head, spans);
	}
};

//#endregion
//#region src/ts-dsl/expr/ternary.ts
const Mixed$20 = TsDsl;
var TernaryTsDsl = class extends Mixed$20 {
	"~dsl" = "TernaryTsDsl";
	_condition;
	_then;
	_else;
	constructor(condition) {
		super();
		if (condition) this.condition(condition);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._condition);
		ctx$1.analyze(this._then);
		ctx$1.analyze(this._else);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	condition(condition) {
		this._condition = condition;
		return this;
	}
	do(expr) {
		this._then = expr;
		return this;
	}
	otherwise(expr) {
		this._else = expr;
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createConditionalExpression(this.$node(this._condition), void 0, this.$node(this._then), void 0, this.$node(this._else));
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Ternary expression missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._condition) missing.push(".condition()");
		if (!this._then) missing.push(".do()");
		if (!this._else) missing.push(".otherwise()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/expr/typeof.ts
const Mixed$19 = OperatorMixin(TsDsl);
var TypeOfExprTsDsl = class extends Mixed$19 {
	"~dsl" = "TypeOfExprTsDsl";
	_expr;
	constructor(expr) {
		super();
		this._expr = expr;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._expr);
	}
	toAst() {
		return ts.factory.createTypeOfExpression(this.$node(this._expr));
	}
};
f.typeofExpr.set((...args) => new TypeOfExprTsDsl(...args));

//#endregion
//#region src/ts-dsl/layout/note.ts
var NoteTsDsl = class extends TsDsl {
	"~dsl" = "NoteTsDsl";
	_lines = [];
	constructor(lines, fn) {
		super();
		if (lines) this.add(lines);
		fn?.(this);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	add(lines) {
		this._lines.push(lines);
		return this;
	}
	apply(node) {
		const lines = this._lines.reduce((lines$1, line) => {
			if (typeof line === "function") line = line(ctx);
			for (const l of typeof line === "string" ? [line] : line) if (l || l === "") lines$1.push(l);
			return lines$1;
		}, []);
		if (!lines.length) return node;
		ts.addSyntheticLeadingComment(node, ts.SyntaxKind.MultiLineCommentTrivia, `\n${lines.join("\n")}\n`, true);
		return node;
	}
	toAst() {
		return this.$node(new IdTsDsl(""));
	}
};

//#endregion
//#region src/ts-dsl/stmt/if.ts
const Mixed$18 = DoMixin(TsDsl);
var IfTsDsl = class extends Mixed$18 {
	"~dsl" = "IfTsDsl";
	_condition;
	_else;
	constructor(condition) {
		super();
		if (condition) this.condition(condition);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._condition);
		if (this._else) {
			ctx$1.pushScope();
			try {
				for (const stmt of this._else) ctx$1.analyze(stmt);
			} finally {
				ctx$1.popScope();
			}
		}
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	condition(condition) {
		this._condition = condition;
		return this;
	}
	otherwise(...items) {
		this._else = items;
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createIfStatement(this.$node(this._condition), this.$node(new BlockTsDsl(...this._do).pretty()), this._else ? this.$node(new BlockTsDsl(...this._else).pretty()) : void 0);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`If statement missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._condition) missing.push(".condition()");
		if (this._do.length === 0) missing.push(".do()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/stmt/return.ts
const Mixed$17 = TsDsl;
var ReturnTsDsl = class extends Mixed$17 {
	"~dsl" = "ReturnTsDsl";
	_returnExpr;
	constructor(expr) {
		super();
		if (expr) this._returnExpr = ref(expr);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._returnExpr);
	}
	toAst() {
		return ts.factory.createReturnStatement(this.$node(this._returnExpr));
	}
};
f.return.set((...args) => new ReturnTsDsl(...args));

//#endregion
//#region src/ts-dsl/stmt/throw.ts
const Mixed$16 = TsDsl;
var ThrowTsDsl = class extends Mixed$16 {
	"~dsl" = "ThrowTsDsl";
	error;
	msg;
	useNew;
	constructor(error, useNew = true) {
		super();
		this.error = error;
		this.useNew = useNew;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.error);
		ctx$1.analyze(this.msg);
	}
	message(value) {
		this.msg = value;
		return this;
	}
	toAst() {
		const errorNode = this.$node(this.error);
		const messageNode = this.$node(this.msg ? [this.msg] : []).map((expr) => typeof expr === "string" ? this.$node(new LiteralTsDsl(expr)) : expr);
		if (this.useNew) return ts.factory.createThrowStatement(ts.factory.createNewExpression(errorNode, void 0, messageNode));
		const args = messageNode.length ? [ts.factory.createCallExpression(errorNode, void 0, messageNode)] : [errorNode];
		return ts.factory.createThrowStatement(args[0]);
	}
};

//#endregion
//#region src/ts-dsl/stmt/try.ts
const Mixed$15 = TsDsl;
var TryTsDsl = class extends Mixed$15 {
	"~dsl" = "TryTsDsl";
	_catch;
	_catchArg;
	_finally;
	_try;
	constructor(...tryBlock) {
		super();
		this.try(...tryBlock);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		if (this._try) {
			ctx$1.pushScope();
			try {
				for (const stmt of this._try) ctx$1.analyze(stmt);
			} finally {
				ctx$1.popScope();
			}
		}
		if (this._catch || this._catchArg) {
			ctx$1.pushScope();
			try {
				ctx$1.analyze(this._catchArg);
				if (this._catch) for (const stmt of this._catch) ctx$1.analyze(stmt);
			} finally {
				ctx$1.popScope();
			}
		}
		if (this._finally) {
			ctx$1.pushScope();
			try {
				for (const stmt of this._finally) ctx$1.analyze(stmt);
			} finally {
				ctx$1.popScope();
			}
		}
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	catch(...items) {
		this._catch = items;
		return this;
	}
	catchArg(arg) {
		this._catchArg = arg;
		return this;
	}
	finally(...items) {
		this._finally = items;
		return this;
	}
	try(...items) {
		this._try = items;
		return this;
	}
	toAst() {
		this.$validate();
		const catchParam = this._catchArg ? this.$node(this._catchArg) : void 0;
		return ts.factory.createTryStatement(this.$node(new BlockTsDsl(...this._try).pretty()), ts.factory.createCatchClause(catchParam ? ts.factory.createVariableDeclaration(catchParam) : void 0, this.$node(new BlockTsDsl(...this._catch ?? []).pretty())), this._finally ? this.$node(new BlockTsDsl(...this._finally).pretty()) : void 0);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Try statement missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._try || this._try.length === 0) missing.push(".try()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/stmt/var.ts
const Mixed$14 = DefaultMixin(DocMixin(ExportMixin(HintMixin(PatternMixin(ValueMixin(TsDsl))))));
var VarTsDsl = class extends Mixed$14 {
	"~dsl" = "VarTsDsl";
	nameSanitizer = safeRuntimeName;
	kind = ts.NodeFlags.None;
	_type;
	constructor(name) {
		super();
		if (name) this.name.set(name);
		if (isSymbol(name)) name.setKind("var");
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.name);
		ctx$1.analyze(this._type);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	const() {
		this.kind = ts.NodeFlags.Const;
		return this;
	}
	let() {
		this.kind = ts.NodeFlags.Let;
		return this;
	}
	/** Sets the variable type. */
	type(type) {
		this._type = type instanceof TypeTsDsl ? type : new TypeExprTsDsl(type);
		return this;
	}
	var() {
		this.kind = ts.NodeFlags.None;
		return this;
	}
	toAst() {
		this.$validate();
		const node = ts.factory.createVariableStatement(this.modifiers, ts.factory.createVariableDeclarationList([ts.factory.createVariableDeclaration(this.$pattern() ?? this.$node(this.name), void 0, this.$type(this._type), this.$value())], this.kind));
		return this.$docs(this.$hint(node));
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Variable declaration missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this.$pattern() && !this.name.toString()) missing.push("name or pattern (.array()/.object())");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/type/alias.ts
const Mixed$13 = DocMixin(ExportMixin(TypeParamsMixin(TsDsl)));
var TypeAliasTsDsl = class extends Mixed$13 {
	"~dsl" = "TypeAliasTsDsl";
	nameSanitizer = safeTypeName;
	scope = "type";
	value;
	constructor(name, fn) {
		super();
		this.name.set(name);
		if (isSymbol(name)) name.setKind("type");
		fn?.(this);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.name);
		ctx$1.analyze(this.value);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Sets the type expression on the right-hand side of `= ...`. */
	type(node) {
		this.value = node;
		return this;
	}
	toAst() {
		this.$validate();
		const node = ts.factory.createTypeAliasDeclaration(this.modifiers, this.$node(this.name), this.$generics(), this.$type(this.value));
		return this.$docs(node);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		const name = this.name.toString();
		throw new Error(`Type alias${name ? ` "${name}"` : ""} missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this.value) missing.push(".type()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/type/and.ts
const Mixed$12 = TsDsl;
var TypeAndTsDsl = class extends Mixed$12 {
	"~dsl" = "TypeAndTsDsl";
	scope = "type";
	_types = [];
	constructor(...nodes) {
		super();
		this.types(...nodes);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const type of this._types) ctx$1.analyze(type);
	}
	types(...nodes) {
		this._types.push(...nodes.map((n) => ref(n)));
		return this;
	}
	toAst() {
		const flat = [];
		for (const node of this._types) {
			const type = this.$type(node);
			if (ts.isIntersectionTypeNode(type)) flat.push(...type.types);
			else flat.push(type);
		}
		return ts.factory.createIntersectionTypeNode(flat);
	}
};

//#endregion
//#region src/ts-dsl/type/literal.ts
const Mixed$11 = TsDsl;
var TypeLiteralTsDsl = class extends Mixed$11 {
	"~dsl" = "TypeLiteralTsDsl";
	scope = "type";
	value;
	constructor(value) {
		super();
		this.value = value;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	toAst() {
		return ts.factory.createLiteralTypeNode(this.$node(new LiteralTsDsl(this.value)));
	}
};

//#endregion
//#region src/ts-dsl/type/idx-sig.ts
const Mixed$10 = DocMixin(ReadonlyMixin(TsDsl));
var TypeIdxSigTsDsl = class extends Mixed$10 {
	"~dsl" = "TypeIdxSigTsDsl";
	scope = "type";
	_key;
	_type;
	constructor(name, fn) {
		super();
		this.name.set(name);
		fn?.(this);
	}
	/** Element kind. */
	get kind() {
		return "idxSig";
	}
	/** Index signature parameter name. */
	get propName() {
		return this.name.toString();
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._key);
		ctx$1.analyze(this._type);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Sets the key type: `[name: T]` */
	key(type) {
		this._key = type;
		return this;
	}
	/** Sets the property type. */
	type(type) {
		this._type = type;
		return this;
	}
	toAst() {
		this.$validate();
		const node = ts.factory.createIndexSignature(this.modifiers, [ts.factory.createParameterDeclaration(void 0, void 0, this.$node(this.name), void 0, this.$type(this._key))], this.$type(this._type));
		return this.$docs(node);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		const name = this.name.toString();
		throw new Error(`Index signature${name ? ` "${name}"` : ""} missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._key) missing.push(".key()");
		if (!this._type) missing.push(".​type()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/type/prop.ts
const Mixed$9 = DocMixin(OptionalMixin(ReadonlyMixin(TsDsl)));
var TypePropTsDsl = class extends Mixed$9 {
	"~dsl" = "TypePropTsDsl";
	scope = "type";
	_type;
	constructor(name, fn) {
		super();
		this.name.set(name);
		fn(this);
	}
	/** Element kind. */
	get kind() {
		return "prop";
	}
	/** Property name. */
	get propName() {
		return this.name.toString();
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._type);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Sets the property type. */
	type(type) {
		this._type = ref(type);
		return this;
	}
	toAst() {
		this.$validate();
		const name = this.name.toString();
		const node = ts.factory.createPropertySignature(this.modifiers, this.$node(safePropName(name)), this._optional ? this.$node(new TokenTsDsl().optional()) : void 0, this.$type(this._type));
		return this.$docs(node);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		const name = this.name.toString();
		throw new Error(`Type property${name ? ` "${name}"` : ""} missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._type) missing.push(".type()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/type/object.ts
const Mixed$8 = TsDsl;
var TypeObjectTsDsl = class extends Mixed$8 {
	"~dsl" = "TypeObjectTsDsl";
	scope = "type";
	_props = /* @__PURE__ */ new Map();
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const prop of this._props.values()) ctx$1.analyze(prop);
	}
	/** Returns true if object has at least one property or index signature. */
	hasProps() {
		return this._props.size > 0;
	}
	/** Adds an index signature to the object type, or removes if fn is null. */
	idxSig(name, fn) {
		const key = `idxSig:${name}`;
		if (fn === null) this._props.delete(key);
		else this._props.set(key, new TypeIdxSigTsDsl(name, fn));
		return this;
	}
	/** Returns true if object has no properties or index signatures. */
	get isEmpty() {
		return this._props.size === 0;
	}
	/** Adds a property signature, or removes if fn is null. */
	prop(name, fn) {
		const key = `prop:${name}`;
		if (fn === null) this._props.delete(key);
		else this._props.set(key, new TypePropTsDsl(name, fn));
		return this;
	}
	/** Adds multiple properties/index signatures. */
	props(...members) {
		for (const member of members) this._props.set(`${member.kind}:${member.propName}`, member);
		return this;
	}
	toAst() {
		return ts.factory.createTypeLiteralNode(this.$node([...this._props.values()]));
	}
};

//#endregion
//#region src/ts-dsl/type/tuple.ts
const Mixed$7 = TsDsl;
var TypeTupleTsDsl = class extends Mixed$7 {
	"~dsl" = "TypeTupleTsDsl";
	scope = "type";
	_elements = [];
	constructor(...nodes) {
		super();
		this.elements(...nodes);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const type of this._elements) ctx$1.analyze(type);
	}
	elements(...types) {
		this._elements.push(...types);
		return this;
	}
	toAst() {
		return ts.factory.createTupleTypeNode(this._elements.map((t) => this.$type(t)));
	}
};

//#endregion
//#region src/ts-dsl/type/fromValue.ts
const fromValue = (input) => {
	if (isNode(input)) return input;
	if (input === null) return new TypeLiteralTsDsl(input);
	if (typeof input === "number" || typeof input === "boolean" || typeof input === "string") return new TypeLiteralTsDsl(input);
	if (input instanceof Array) return new TypeTupleTsDsl(...input.map((v) => fromValue(v)));
	if (typeof input === "object") {
		const obj = new TypeObjectTsDsl();
		for (const [key, val] of Object.entries(input)) {
			const type = fromValue(val);
			obj.prop(key, (p) => p.type(type));
		}
		return obj;
	}
	throw new Error(`$.type.fromValue(): Unsupported input type ${String(input)}`);
};

//#endregion
//#region src/ts-dsl/type/func.ts
const Mixed$6 = DocMixin(ParamMixin(TypeParamsMixin(TypeReturnsMixin(TsDsl))));
var TypeFuncTsDsl = class extends Mixed$6 {
	"~dsl" = "TypeFuncTsDsl";
	scope = "type";
	analyze(ctx$1) {
		super.analyze(ctx$1);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	toAst() {
		this.$validate();
		const node = ts.factory.createFunctionTypeNode(this.$generics(), this.$params(), this.$returns());
		return this.$docs(node);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Function type missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (this.$returns() === void 0) missing.push(".returns()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/type/idx.ts
const Mixed$5 = TypeExprMixin(TsDsl);
var TypeIdxTsDsl = class extends Mixed$5 {
	"~dsl" = "TypeIdxTsDsl";
	scope = "type";
	_base;
	_index;
	constructor(base, index) {
		super();
		this.base(base);
		this.index(index);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._base);
		ctx$1.analyze(this._index);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	base(base) {
		this._base = base;
		return this;
	}
	index(index) {
		this._index = index;
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createIndexedAccessTypeNode(this.$type(this._base), this.$type(this._index));
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Indexed access type missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (this._base === void 0) missing.push(".base()");
		if (this._index === void 0) missing.push(".index()");
		return missing;
	}
};
f.type.idx.set((...args) => new TypeIdxTsDsl(...args));

//#endregion
//#region src/ts-dsl/type/mapped.ts
const Mixed$4 = TsDsl;
var TypeMappedTsDsl = class extends Mixed$4 {
	"~dsl" = "TypeMappedTsDsl";
	scope = "type";
	questionToken;
	readonlyToken;
	_key;
	_type;
	constructor(name) {
		super();
		if (name) this.name.set(name);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.questionToken);
		ctx$1.analyze(this.readonlyToken);
		ctx$1.analyze(this._key);
		ctx$1.analyze(this._type);
	}
	/** Returns true when all required builder calls are present. */
	get isValid() {
		return this.missingRequiredCalls().length === 0;
	}
	/** Sets the key constraint: `[K in Constraint]` */
	key(type) {
		this._key = type;
		return this;
	}
	/** Removes `readonly` from the mapped members (`[K in X]-readonly`). */
	mutable() {
		this.readonlyToken = new TokenTsDsl().minus();
		return this;
	}
	/** Makes `[K in X]?:` optional. */
	optional() {
		this.questionToken = new TokenTsDsl().optional();
		return this;
	}
	/** Makes `[K in X]` readonly */
	readonly() {
		this.readonlyToken = new TokenTsDsl().readonly();
		return this;
	}
	/** Removes `?` from the mapped members (`[K in X]-?:`). */
	required() {
		this.questionToken = new TokenTsDsl().minus();
		return this;
	}
	/** Sets the mapped value type: `[K in X]: ValueType` */
	type(type) {
		this._type = type;
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createMappedTypeNode(this.$node(this.readonlyToken), ts.factory.createTypeParameterDeclaration(void 0, this.$node(this.name), this.$type(this._key), void 0), void 0, this.$node(this.questionToken), this.$type(this._type), void 0);
	}
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		const name = this.name.toString();
		throw new Error(`Mapped type${name ? ` "${name}"` : ""} missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._key) missing.push(".key()");
		if (!this._type) missing.push(".​type()");
		return missing;
	}
};

//#endregion
//#region src/ts-dsl/type/operator.ts
const Mixed$3 = TsDsl;
/**
* Builds a TypeScript `TypeOperatorNode`, such as:
*
* - `keyof T`
* - `readonly U`
* - `unique V`
*
* This DSL provides both a generic `.operator()` API and convenient
* shorthand methods (`.keyof()`, `.readonly()`, `.unique()`).
*
* The node will throw during render if required fields are missing.
*/
var TypeOperatorTsDsl = class extends Mixed$3 {
	"~dsl" = "TypeOperatorTsDsl";
	scope = "type";
	_op;
	_type;
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._type);
	}
	/** Shorthand: builds `keyof T`. */
	keyof(type) {
		this.operator(ts.SyntaxKind.KeyOfKeyword);
		this.type(type);
		return this;
	}
	/** Sets the operator explicitly. */
	operator(op) {
		this._op = op;
		return this;
	}
	/** Shorthand: builds `readonly T`. */
	readonly(type) {
		this.operator(ts.SyntaxKind.ReadonlyKeyword);
		this.type(type);
		return this;
	}
	/** Sets the target type of the operator. */
	type(type) {
		this._type = type;
		return this;
	}
	/** Shorthand: builds `unique T`. */
	unique(type) {
		this.operator(ts.SyntaxKind.UniqueKeyword);
		this.type(type);
		return this;
	}
	toAst() {
		this.$validate();
		return ts.factory.createTypeOperatorNode(this._op, this.$type(this._type));
	}
	/** Throws if required fields are not set. */
	$validate() {
		const missing = this.missingRequiredCalls();
		if (missing.length === 0) return;
		throw new Error(`Type operator missing ${missing.join(" and ")}`);
	}
	missingRequiredCalls() {
		const missing = [];
		if (!this._op) missing.push(".operator()");
		if (!this._type) missing.push(".​type()");
		return missing;
	}
};
f.type.operator.set((...args) => new TypeOperatorTsDsl(...args));

//#endregion
//#region src/ts-dsl/type/or.ts
const Mixed$2 = TsDsl;
var TypeOrTsDsl = class extends Mixed$2 {
	"~dsl" = "TypeOrTsDsl";
	scope = "type";
	_types = [];
	constructor(...nodes) {
		super();
		this.types(...nodes);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const type of this._types) ctx$1.analyze(type);
	}
	types(...nodes) {
		this._types.push(...nodes.map((n) => ref(n)));
		return this;
	}
	toAst() {
		const flat = [];
		for (const node of this._types) {
			const type = this.$type(node);
			if (ts.isUnionTypeNode(type)) flat.push(...type.types);
			else flat.push(type);
		}
		return ts.factory.createUnionTypeNode(flat);
	}
};

//#endregion
//#region src/ts-dsl/type/query.ts
const Mixed$1 = TypeExprMixin(TsDsl);
var TypeQueryTsDsl = class extends Mixed$1 {
	"~dsl" = "TypeQueryTsDsl";
	scope = "type";
	_expr;
	constructor(expr) {
		super();
		this._expr = expr;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this._expr);
	}
	toAst() {
		const expr = this.$node(this._expr);
		return ts.factory.createTypeQueryNode(expr);
	}
};
f.type.query.set((...args) => new TypeQueryTsDsl(...args));

//#endregion
//#region src/ts-dsl/type/template.ts
const Mixed = TsDsl;
var TypeTemplateTsDsl = class extends Mixed {
	"~dsl" = "TypeTemplateTsDsl";
	scope = "type";
	parts = [];
	constructor(value) {
		super();
		if (value !== void 0) this.add(value);
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		for (const part of this.parts) ctx$1.analyze(part);
	}
	/** Adds a raw string segment or embedded type expression. */
	add(part) {
		this.parts.push(part);
		return this;
	}
	toAst() {
		const parts = this.$node(this.parts);
		const normalized = [];
		for (let index = 0; index < parts.length; index++) {
			const current = parts[index];
			if (typeof current === "string") {
				let merged = current;
				while (index + 1 < parts.length && typeof parts[index + 1] === "string") {
					merged += parts[index + 1];
					index++;
				}
				normalized.push(merged);
			} else normalized.push(current);
		}
		if (normalized.length === 0 || typeof normalized[0] !== "string") normalized.unshift("");
		if (normalized.length === 1 && typeof normalized[0] === "string") return ts.factory.createTemplateLiteralType(ts.factory.createTemplateHead(normalized[0]), []);
		if (normalized.length === 2 && typeof normalized[0] === "string" && typeof normalized[1] !== "string") return ts.factory.createTemplateLiteralType(ts.factory.createTemplateHead(normalized[0]), [ts.factory.createTemplateLiteralTypeSpan(normalized[1], ts.factory.createTemplateTail(""))]);
		const head = ts.factory.createTemplateHead(normalized.shift());
		const spans = [];
		while (normalized.length) {
			const type = normalized.shift();
			const next = typeof normalized[0] === "string" ? normalized.shift() : "";
			const isLast = normalized.length === 0;
			spans.push(ts.factory.createTemplateLiteralTypeSpan(type, isLast ? ts.factory.createTemplateTail(next) : ts.factory.createTemplateMiddle(next)));
		}
		return ts.factory.createTemplateLiteralType(head, spans);
	}
};

//#endregion
//#region src/ts-dsl/utils/lazy.ts
var LazyTsDsl = class extends TsDsl {
	"~dsl" = "LazyTsDsl";
	_thunk;
	constructor(thunk) {
		super();
		this._thunk = thunk;
	}
	analyze(ctx$1) {
		super.analyze(ctx$1);
		ctx$1.analyze(this.toResult());
	}
	toResult() {
		return this._thunk(ctx);
	}
	toAst() {
		return this.toResult().toAst();
	}
};

//#endregion
//#region src/ts-dsl/utils/render-utils.ts
const printer = ts.createPrinter({
	newLine: ts.NewLineKind.LineFeed,
	removeComments: false
});
const blankFile = ts.createSourceFile("", "", ts.ScriptTarget.ESNext, false, ts.ScriptKind.TS);
/** Print a TypeScript node to a string. */
function astToString(node) {
	const result = printer.printNode(ts.EmitHint.Unspecified, node, blankFile);
	try {
		/**
		* TypeScript Compiler API escapes unicode characters by default and there
		* is no way to disable this behavior
		* {@link https://github.com/microsoft/TypeScript/issues/36174}
		*/
		return result.replace(/\\u([0-9a-fA-F]{4})/g, (_, hex) => String.fromCharCode(Number.parseInt(hex, 16)));
	} catch {
		return result;
	}
}
function moduleSortKey({ file, fromFile, preferFileExtension, root }) {
	const filePath = file.finalPath.split(path.sep).join("/");
	let modulePath = fromFile.finalPath.split(path.sep).join("/");
	if (fromFile.external && !path.isAbsolute(modulePath)) return [
		0,
		0,
		modulePath
	];
	if (!modulePath.startsWith(root.split(path.sep).join("/"))) return [
		1,
		0,
		modulePath
	];
	const rel = path.relative(path.dirname(filePath), path.dirname(modulePath)).split(path.sep).join("/");
	let parentCount;
	if (!rel.startsWith("..")) {
		modulePath = `./${rel ? `${rel}/` : ""}${fromFile.name}${fromFile.extension ?? ""}`;
		parentCount = 0;
	} else {
		modulePath = `${rel}/${fromFile.name}${fromFile.extension ?? ""}`;
		parentCount = rel.split(path.sep).filter((segment) => segment === "..").length;
	}
	if (modulePath.endsWith(".ts")) modulePath = modulePath.slice(0, -3);
	if (preferFileExtension) modulePath += preferFileExtension;
	else if (modulePath.endsWith("/index")) modulePath = modulePath.slice(0, -6);
	return [
		2,
		parentCount,
		modulePath
	];
}

//#endregion
//#region src/ts-dsl/utils/render.ts
function headerToLines(header) {
	if (!header) return [];
	const lines = [];
	if (typeof header === "string") {
		lines.push(...header.split(/\r?\n/));
		return lines;
	}
	for (const line of header) lines.push(...line.split(/\r?\n/));
	return lines;
}
var TypeScriptRenderer = class TypeScriptRenderer {
	/**
	* Function to generate a file header.
	*
	* @private
	*/
	_header;
	/**
	* Whether `export * from 'module'` should be used when possible instead of named exports.
	*
	* @private
	*/
	_preferExportAll;
	/**
	* Controls whether imports/exports include a file extension (e.g., '.ts' or '.js').
	*
	* @private
	*/
	_preferFileExtension;
	/**
	* Optional function to transform module specifiers.
	*
	* @private
	*/
	_resolveModuleName;
	constructor(args = {}) {
		this._header = args.header;
		this._preferExportAll = args.preferExportAll ?? false;
		this._preferFileExtension = args.preferFileExtension ?? "";
		this._resolveModuleName = args.resolveModuleName;
	}
	render(ctx$1) {
		const header = typeof this._header === "function" ? this._header(ctx$1) : this._header;
		return TypeScriptRenderer.astToString({
			exports: this.getExports(ctx$1),
			exportsOptions: { preferExportAll: this._preferExportAll },
			header,
			imports: this.getImports(ctx$1),
			nodes: ctx$1.file.nodes
		});
	}
	supports(ctx$1) {
		return ctx$1.file.language === "typescript";
	}
	static astToString(args) {
		let text = "";
		for (const header of headerToLines(args.header)) text += `${header}\n`;
		let imports = "";
		for (const group of args.imports ?? []) {
			if (imports) imports += "\n";
			for (const imp of group) imports += `${astToString(TypeScriptRenderer.toImportAst(imp))}\n`;
		}
		text = `${text}${text && imports ? "\n" : ""}${imports}`;
		let nodes = "";
		for (const node of args.nodes ?? []) {
			if (nodes) nodes += "\n";
			nodes += `${astToString(node.toAst())}\n`;
		}
		text = `${text}${text && nodes ? "\n" : ""}${nodes}`;
		let exports = "";
		for (const group of args.exports ?? []) {
			if (!exports && nodes || exports) exports += "\n";
			for (const exp of group) exports += `${astToString(TypeScriptRenderer.toExportAst(exp, args.exportsOptions))}\n`;
		}
		text = `${text}${text && exports ? "\n" : ""}${exports}`;
		if (args.trailingNewline === false && text.endsWith("\n")) text = text.slice(0, -1);
		return text;
	}
	static toExportAst(group, options) {
		const specifiers = group.exports.map((exp) => {
			return ts.factory.createExportSpecifier(exp.isTypeOnly, exp.sourceName !== exp.exportedName ? $.id(exp.sourceName).toAst() : void 0, $.id(exp.exportedName).toAst());
		});
		const exportClause = group.namespaceExport ? ts.factory.createNamespaceExport($.id(group.namespaceExport).toAst()) : (!group.canExportAll || !options?.preferExportAll) && specifiers.length ? ts.factory.createNamedExports(specifiers) : void 0;
		return ts.factory.createExportDeclaration(void 0, group.isTypeOnly, exportClause, $.literal(group.modulePath).toAst());
	}
	static toImportAst(group) {
		const specifiers = group.imports.map((imp) => {
			return ts.factory.createImportSpecifier(imp.isTypeOnly, imp.sourceName !== imp.localName ? $.id(imp.sourceName).toAst() : void 0, $.id(imp.localName).toAst());
		});
		const importClause = ts.factory.createImportClause(group.isTypeOnly, group.kind === "default" ? $.id(group.localName ?? "").toAst() : void 0, group.kind === "namespace" ? ts.factory.createNamespaceImport($.id(group.localName ?? "").toAst()) : specifiers.length > 0 ? ts.factory.createNamedImports(specifiers) : void 0);
		return ts.factory.createImportDeclaration(void 0, importClause, $.literal(group.modulePath).toAst());
	}
	getExports(ctx$1) {
		const groups = /* @__PURE__ */ new Map();
		for (const exp of ctx$1.file.exports) {
			const sortKey = moduleSortKey({
				file: ctx$1.file,
				fromFile: exp.from,
				preferFileExtension: this._preferFileExtension,
				root: ctx$1.project.root
			});
			const modulePath = this._resolveModuleName?.(sortKey[2]) ?? sortKey[2];
			const [groupIndex] = sortKey;
			if (!groups.has(groupIndex)) groups.set(groupIndex, /* @__PURE__ */ new Map());
			const moduleMap = groups.get(groupIndex);
			if (!moduleMap.has(modulePath)) moduleMap.set(modulePath, {
				group: {
					canExportAll: exp.canExportAll,
					exports: exp.exports,
					isTypeOnly: exp.isTypeOnly,
					modulePath,
					namespaceExport: exp.namespaceExport
				},
				sortKey
			});
		}
		return Array.from(groups.entries()).sort((a, b) => a[0] - b[0]).map(([, moduleMap]) => {
			const entries = Array.from(moduleMap.values());
			entries.sort((a, b) => {
				const d = a.sortKey[1] - b.sortKey[1];
				return d !== 0 ? d : a.group.modulePath.localeCompare(b.group.modulePath);
			});
			return entries.map((e) => {
				const group = e.group;
				if (group.namespaceExport) group.exports = [];
				else {
					if (!group.exports.find((exp) => !exp.isTypeOnly)) {
						group.isTypeOnly = true;
						for (const exp of group.exports) exp.isTypeOnly = false;
					}
					group.exports.sort((a, b) => a.exportedName.localeCompare(b.exportedName));
				}
				return group;
			});
		});
	}
	getImports(ctx$1) {
		const groups = /* @__PURE__ */ new Map();
		for (const imp of ctx$1.file.imports) {
			const sortKey = moduleSortKey({
				file: ctx$1.file,
				fromFile: imp.from,
				preferFileExtension: this._preferFileExtension,
				root: ctx$1.project.root
			});
			const modulePath = this._resolveModuleName?.(sortKey[2]) ?? sortKey[2];
			const [groupIndex] = sortKey;
			if (!groups.has(groupIndex)) groups.set(groupIndex, /* @__PURE__ */ new Map());
			const moduleMap = groups.get(groupIndex);
			if (!moduleMap.has(modulePath)) moduleMap.set(modulePath, {
				group: {
					imports: [],
					isTypeOnly: false,
					kind: imp.kind,
					modulePath
				},
				sortKey
			});
			const group = moduleMap.get(modulePath).group;
			if (imp.kind !== "named") {
				group.isTypeOnly = imp.isTypeOnly;
				group.kind = imp.kind;
				group.localName = imp.localName;
			} else group.imports.push(...imp.imports);
		}
		return Array.from(groups.entries()).sort((a, b) => a[0] - b[0]).map(([, moduleMap]) => {
			const entries = Array.from(moduleMap.values());
			entries.sort((a, b) => {
				const d = a.sortKey[1] - b.sortKey[1];
				return d !== 0 ? d : a.group.modulePath.localeCompare(b.group.modulePath);
			});
			return entries.map((e) => {
				const group = e.group;
				if (group.kind === "namespace") group.imports = [];
				else {
					if (!group.imports.find((imp) => !imp.isTypeOnly)) {
						group.isTypeOnly = true;
						for (const imp of group.imports) imp.isTypeOnly = false;
					}
					group.imports.sort((a, b) => a.localName.localeCompare(b.localName));
				}
				return group;
			});
		});
	}
};

//#endregion
//#region src/ts-dsl/index.ts
const tsDsl = {
	array: (...args) => new ArrayTsDsl(...args),
	as: (...args) => new AsTsDsl(...args),
	attr: (...args) => new AttrTsDsl(...args),
	await: (...args) => new AwaitTsDsl(...args),
	binary: (...args) => new BinaryTsDsl(...args),
	block: (...args) => new BlockTsDsl(...args),
	call: (...args) => new CallTsDsl(...args),
	class: (...args) => new ClassTsDsl(...args),
	const: (...args) => new VarTsDsl(...args).const(),
	decorator: (...args) => new DecoratorTsDsl(...args),
	doc: (...args) => new DocTsDsl(...args),
	enum: (...args) => new EnumTsDsl(...args),
	expr: (...args) => new ExprTsDsl(...args),
	field: (...args) => new FieldTsDsl(...args),
	fromValue: (...args) => fromValue$1(...args),
	func: ((nameOrFn, fn) => {
		if (nameOrFn === void 0) return new FuncTsDsl();
		if (typeof nameOrFn !== "string") return new FuncTsDsl(nameOrFn);
		if (fn === void 0) return new FuncTsDsl(nameOrFn);
		return new FuncTsDsl(nameOrFn, fn);
	}),
	getter: (...args) => new GetterTsDsl(...args),
	hint: (...args) => new HintTsDsl(...args),
	id: (...args) => new IdTsDsl(...args),
	if: (...args) => new IfTsDsl(...args),
	init: (...args) => new InitTsDsl(...args),
	lazy: (...args) => new LazyTsDsl(...args),
	let: (...args) => new VarTsDsl(...args).let(),
	literal: (...args) => new LiteralTsDsl(...args),
	member: (...args) => new EnumMemberTsDsl(...args),
	method: (...args) => new MethodTsDsl(...args),
	neg: (...args) => new PrefixTsDsl(...args).neg(),
	new: (...args) => new NewTsDsl(...args),
	newline: (...args) => new NewlineTsDsl(...args),
	not: (...args) => new PrefixTsDsl(...args).not(),
	note: (...args) => new NoteTsDsl(...args),
	object: (...args) => new ObjectTsDsl(...args),
	param: (...args) => new ParamTsDsl(...args),
	pattern: (...args) => new PatternTsDsl(...args),
	prefix: (...args) => new PrefixTsDsl(...args),
	prop: (...args) => new ObjectPropTsDsl(...args),
	regexp: (...args) => new RegExpTsDsl(...args),
	return: (...args) => new ReturnTsDsl(...args),
	setter: (...args) => new SetterTsDsl(...args),
	stmt: (...args) => new StmtTsDsl(...args),
	template: (...args) => new TemplateTsDsl(...args),
	ternary: (...args) => new TernaryTsDsl(...args),
	throw: (...args) => new ThrowTsDsl(...args),
	token: (...args) => new TokenTsDsl(...args),
	try: (...args) => new TryTsDsl(...args),
	type: Object.assign((...args) => new TypeExprTsDsl(...args), {
		alias: (...args) => new TypeAliasTsDsl(...args),
		and: (...args) => new TypeAndTsDsl(...args),
		attr: (...args) => new TypeAttrTsDsl(...args),
		expr: (...args) => new TypeExprTsDsl(...args),
		fromValue: (...args) => fromValue(...args),
		func: (...args) => new TypeFuncTsDsl(...args),
		idx: (...args) => new TypeIdxTsDsl(...args),
		literal: (...args) => new TypeLiteralTsDsl(...args),
		mapped: (...args) => new TypeMappedTsDsl(...args),
		object: (...args) => new TypeObjectTsDsl(...args),
		operator: (...args) => new TypeOperatorTsDsl(...args),
		or: (...args) => new TypeOrTsDsl(...args),
		param: (...args) => new TypeParamTsDsl(...args),
		query: (...args) => new TypeQueryTsDsl(...args),
		template: (...args) => new TypeTemplateTsDsl(...args),
		tuple: (...args) => new TypeTupleTsDsl(...args)
	}),
	typeofExpr: (...args) => new TypeOfExprTsDsl(...args),
	var: (...args) => new VarTsDsl(...args)
};
const $ = Object.assign((...args) => new ExprTsDsl(...args), tsDsl);

//#endregion
//#region src/plugins/@hey-api/sdk/shared/class.ts
const createRegistryClass = ({ plugin, sdkSymbol, symbol }) => {
	const symbolDefaultKey = plugin.symbol("defaultKey");
	const symbolInstances = plugin.symbol("instances");
	return $.class(symbol).generic("T").field(symbolDefaultKey, (f$1) => f$1.private().readonly().assign($.literal("default"))).newline().field(symbolInstances, (f$1) => f$1.private().readonly().type($.type("Map").generics("string", "T")).assign($.new("Map"))).newline().method("get", (m) => m.returns("T").param("key", (p) => p.type("string").optional()).do($.const("instance").assign($("this").attr(symbolInstances).attr("get").call($("key").coalesce($("this").attr(symbolDefaultKey)))), $.if($.not("instance")).do($.throw("Error").message($.template("No SDK client found. Create one with \"new ").add(sdkSymbol).add("()\" to fix this error."))), $.return("instance"))).newline().method("set", (m) => m.returns("void").param("value", (p) => p.type("T")).param("key", (p) => p.type("string").optional()).do($("this").attr(symbolInstances).attr("set").call($("key").coalesce($("this").attr(symbolDefaultKey)), "value")));
};
const createClientClass = ({ plugin, symbol }) => {
	const symClient = plugin.getSymbol({ category: "client" });
	const optionalClient = Boolean(plugin.config.client && symClient);
	const symbolClient = plugin.external("client.Client");
	return $.class(symbol).field("client", (f$1) => f$1.protected().type(symbolClient)).newline().init((i) => i.param("args", (p) => p.optional(optionalClient).type($.type.object().prop("client", (p$1) => p$1.optional(optionalClient).type(symbolClient)))).do($("this").attr("client").assign($("args").attr("client").optional(optionalClient).$if(optionalClient, (a) => a.coalesce(symClient)))));
};

//#endregion
//#region src/plugins/@hey-api/sdk/shared/constants.ts
const nuxtTypeComposable = "TComposable";
const nuxtTypeDefault = "DefaultT";
const nuxtTypeResponse = "ResT";

//#endregion
//#region src/plugins/@hey-api/sdk/shared/auth.ts
const securitySchemeObjectToAuthObject = ({ securitySchemeObject }) => {
	if (securitySchemeObject.type === "openIdConnect") return {
		scheme: "bearer",
		type: "http"
	};
	if (securitySchemeObject.type === "oauth2") {
		if (securitySchemeObject.flows.password || securitySchemeObject.flows.authorizationCode || securitySchemeObject.flows.clientCredentials || securitySchemeObject.flows.implicit) return {
			scheme: "bearer",
			type: "http"
		};
		return;
	}
	if (securitySchemeObject.type === "apiKey") {
		if (securitySchemeObject.in === "header") return {
			name: securitySchemeObject.name,
			type: "apiKey"
		};
		if (securitySchemeObject.in === "query" || securitySchemeObject.in == "cookie") return {
			in: securitySchemeObject.in,
			name: securitySchemeObject.name,
			type: "apiKey"
		};
		return;
	}
	if (securitySchemeObject.type === "http") {
		const scheme = securitySchemeObject.scheme.toLowerCase();
		if (scheme === "bearer" || scheme === "basic") return {
			scheme,
			type: "http"
		};
		return;
	}
};
const operationAuth = ({ operation, plugin }) => {
	if (!operation.security || !plugin.config.auth) return [];
	const auth = [];
	for (const securitySchemeObject of operation.security) {
		const authObject = securitySchemeObjectToAuthObject({ securitySchemeObject });
		if (authObject) auth.push(authObject);
		else if (securitySchemeObject.type !== "mutualTLS") console.warn(`❗️ SDK warning: unsupported security scheme. Please open an issue if you'd like it added https://github.com/hey-api/openapi-ts/issues\n${JSON.stringify(securitySchemeObject, null, 2)}`);
	}
	return auth;
};

//#endregion
//#region src/plugins/@hey-api/sdk/shared/signature.ts
/**
* Collects and resolves all operation parameters for flattened SDK signatures.
* - Prefixes all conflicting names with their location (e.g. path_foo, query_foo)
* - Returns a flat map of resolved parameter names to their metadata
*/
function getSignatureParameters({ operation }) {
	const locations = [
		"header",
		"path",
		"query"
	];
	const nameToLocations = {};
	const addParameter = (name, location) => {
		if (!nameToLocations[name]) nameToLocations[name] = /* @__PURE__ */ new Set();
		nameToLocations[name].add(location);
	};
	for (const location of locations) {
		const parameters = operation.parameters?.[location];
		if (parameters) for (const key in parameters) {
			const parameter = parameters[key];
			addParameter(parameter.name, location);
		}
	}
	if (operation.body) if (!operation.body.schema.logicalOperator && operation.body.schema.type === "object" && operation.body.schema.properties) {
		const properties = operation.body.schema.properties;
		for (const key in properties) addParameter(key, "body");
	} else if (operation.body.schema.$ref) addParameter(toCase(refToName(operation.body.schema.$ref), "camelCase"), "body");
	else addParameter("body", "body");
	const conflicts = /* @__PURE__ */ new Set();
	for (const name in nameToLocations) if (nameToLocations[name].size > 1) conflicts.add(name);
	const signatureParameters = {};
	const fields = [];
	for (const location of locations) {
		const parameters = operation.parameters?.[location];
		if (parameters) for (const key in parameters) {
			const parameter = parameters[key];
			const originalName = parameter.name;
			const name = conflicts.has(originalName) ? `${location}_${originalName}` : originalName;
			const signatureParameter = {
				isRequired: parameter.required ?? false,
				name,
				schema: parameter.schema
			};
			if (name !== originalName) signatureParameter.originalName = originalName;
			signatureParameters[name] = signatureParameter;
			fields.push({
				in: location === "header" ? "headers" : location,
				key: name,
				...name !== originalName ? { map: originalName } : {}
			});
		}
	}
	if (operation.body) {
		const location = "body";
		if (!operation.body.schema.logicalOperator && operation.body.schema.type === "object" && operation.body.schema.properties) {
			const properties = operation.body.schema.properties;
			for (const originalName in properties) {
				const property = properties[originalName];
				const name = conflicts.has(originalName) ? `${location}_${originalName}` : originalName;
				const signatureParameter = {
					isRequired: property.required?.includes(originalName) ?? false,
					name,
					schema: property
				};
				if (name !== originalName) signatureParameter.originalName = originalName;
				signatureParameters[name] = signatureParameter;
				fields.push({
					in: location,
					key: name,
					...name !== originalName ? { map: originalName } : {}
				});
			}
		} else if (operation.body.schema.$ref) {
			const originalName = toCase(refToName(operation.body.schema.$ref), "camelCase");
			const name = conflicts.has(originalName) ? `${location}_${originalName}` : originalName;
			const signatureParameter = {
				isRequired: operation.body.required ?? false,
				name,
				schema: operation.body.schema
			};
			if (name !== originalName) signatureParameter.originalName = originalName;
			signatureParameters[name] = signatureParameter;
			fields.push({
				key: name,
				map: "body"
			});
		} else {
			signatureParameters.body = {
				isRequired: operation.body.required ?? false,
				name: "body",
				schema: operation.body.schema
			};
			fields.push({
				key: "body",
				map: "body"
			});
		}
	}
	if (!Object.keys(signatureParameters).length) return;
	return {
		fields,
		parameters: signatureParameters
	};
}

//#endregion
//#region src/plugins/@hey-api/sdk/shared/validator.ts
const createRequestValidator = ({ operation, plugin }) => {
	if (!plugin.config.validator.request) return;
	const validator = plugin.getPluginOrThrow(plugin.config.validator.request);
	if (!validator.api.createRequestValidator) return;
	return validator.api.createRequestValidator({
		operation,
		plugin: validator
	});
};
const createResponseValidator = ({ operation, plugin }) => {
	if (!plugin.config.validator.response) return;
	const validator = plugin.getPluginOrThrow(plugin.config.validator.response);
	if (!validator.api.createResponseValidator) return;
	return validator.api.createResponseValidator({
		operation,
		plugin: validator
	});
};

//#endregion
//#region src/plugins/@hey-api/sdk/shared/operation.ts
/** TODO: needs complete refactor */
const operationOptionsType = ({ isDataAllowed = true, operation, plugin, throwOnError }) => {
	const isNuxtClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-nuxt";
	const symbolDataType = isDataAllowed ? plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "typescript"
	}) : void 0;
	const symbolOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client-options",
		tool: "sdk"
	});
	if (isNuxtClient) {
		const symbolResponseType = plugin.querySymbol({
			category: "type",
			resource: "operation",
			resourceId: operation.id,
			role: "response"
		});
		return $.type(symbolOptions).generic(nuxtTypeComposable).generic(isDataAllowed ? symbolDataType ?? "unknown" : "never").generic(symbolResponseType ?? "unknown").generic(nuxtTypeDefault);
	}
	if (throwOnError) return $.type(symbolOptions).generic(isDataAllowed ? symbolDataType ?? "unknown" : "never").generic(throwOnError);
	return $.type(symbolOptions).$if(!isDataAllowed || symbolDataType, (t) => t.generic(isDataAllowed ? symbolDataType : "never"));
};
function operationParameters({ isRequiredOptions, operation, plugin }) {
	const result = {
		argNames: [],
		fields: [],
		parameters: []
	};
	const pluginTypeScript = plugin.getPluginOrThrow("@hey-api/typescript");
	const isNuxtClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-nuxt";
	if (plugin.config.paramsStructure === "flat") {
		const signature = getSignatureParameters({
			operation,
			plugin
		});
		const flatParams = $.type.object();
		if (signature) {
			let isParametersRequired = false;
			for (const key in signature.parameters) {
				const parameter = signature.parameters[key];
				if (parameter.isRequired) isParametersRequired = true;
				flatParams.prop(parameter.name, (p) => p.required(parameter.isRequired).type(pluginTypeScript.api.schemaToType({
					plugin: pluginTypeScript,
					schema: parameter.schema,
					state: refs({ path: [] })
				})));
			}
			result.argNames.push("parameters");
			for (const field of signature.fields) result.fields.push(field);
			result.parameters.push($.param("parameters", (p) => p.required(isParametersRequired).type(flatParams)));
		}
	}
	result.parameters.push($.param("options", (p) => p.required(isRequiredOptions).type(operationOptionsType({
		isDataAllowed: plugin.config.paramsStructure === "grouped",
		operation,
		plugin,
		throwOnError: isNuxtClient ? void 0 : "ThrowOnError"
	}))));
	return result;
}
/**
* Infers `responseType` value from provided response content type. This is
* an adapted version of `getParseAs()` from the Fetch API client.
*
* From Axios documentation:
* `responseType` indicates the type of data that the server will respond with
* options are: 'arraybuffer', 'document', 'json', 'text', 'stream'
* browser only: 'blob'
*/
const getResponseType = (contentType) => {
	if (!contentType) return;
	const cleanContent = contentType.split(";")[0]?.trim();
	if (!cleanContent) return;
	if (cleanContent.startsWith("application/json") || cleanContent.endsWith("+json")) return "json";
	if ([
		"application/",
		"audio/",
		"image/",
		"video/"
	].some((type) => cleanContent.startsWith(type))) return "blob";
	if (cleanContent.startsWith("text/")) return "text";
};
function operationStatements({ isRequiredOptions, opParameters, operation, plugin }) {
	const client = getClientPlugin(getTypedConfig(plugin));
	const isNuxtClient = client.name === "@hey-api/client-nuxt";
	const symbolResponseType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: isNuxtClient ? "response" : "responses"
	});
	const symbolErrorType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: isNuxtClient ? "error" : "errors"
	});
	const reqOptions = $.object();
	if (operation.body) {
		const isBinaryFormat = operation.body.schema?.format === "binary";
		switch (operation.body.type) {
			case "form-data": {
				const symbol = plugin.external("client.formDataBodySerializer");
				reqOptions.spread(symbol);
				break;
			}
			case "json":
				if (isBinaryFormat) reqOptions.prop("bodySerializer", $.literal(null));
				break;
			case "text":
			case "octet-stream":
				reqOptions.prop("bodySerializer", $.literal(null));
				break;
			case "url-search-params": {
				const symbol = plugin.external("client.urlSearchParamsBodySerializer");
				reqOptions.spread(symbol);
				break;
			}
			default:
				if (isBinaryFormat) reqOptions.prop("bodySerializer", $.literal(null));
				break;
		}
	}
	const paramSerializers = $.object();
	for (const name in operation.parameters?.query) {
		const parameter = operation.parameters.query[name];
		if (parameter.schema.type === "array" || parameter.schema.type === "tuple") {
			if (parameter.style !== "form" || !parameter.explode) paramSerializers.prop(parameter.name, $.object().prop("array", $.object().$if(parameter.explode === false, (o) => o.prop("explode", $.literal(parameter.explode))).$if(parameter.style !== "form", (o) => o.prop("style", $.literal(parameter.style)))));
		} else if (parameter.schema.type === "object") {
			if (parameter.style !== "deepObject" || !parameter.explode) paramSerializers.prop(parameter.name, $.object().prop("object", $.object().$if(parameter.explode === false, (o) => o.prop("explode", $.literal(parameter.explode))).$if(parameter.style !== "deepObject", (o) => o.prop("style", $.literal(parameter.style)))));
		}
	}
	if (paramSerializers.hasProps()) reqOptions.prop("querySerializer", $.object().prop("parameters", paramSerializers));
	const requestValidator = createRequestValidator({
		operation,
		plugin
	});
	const responseValidator = createResponseValidator({
		operation,
		plugin
	});
	if (requestValidator) reqOptions.prop("requestValidator", requestValidator.arrow());
	if (plugin.config.transformer) {
		const query = {
			category: "transform",
			resource: "operation",
			resourceId: operation.id,
			role: "response"
		};
		if (plugin.isSymbolRegistered(query)) {
			const ref$1 = plugin.referenceSymbol(query);
			reqOptions.prop("responseTransformer", $(ref$1));
		}
	}
	let hasServerSentEvents = false;
	let responseTypeValue;
	for (const statusCode in operation.responses) {
		const response = operation.responses[statusCode];
		if (!responseTypeValue && client.name === "@hey-api/client-axios") {
			if (statusCodeToGroup({ statusCode }) === "2XX") {
				responseTypeValue = getResponseType(response.mediaType);
				if (responseTypeValue) reqOptions.prop("responseType", $.literal(responseTypeValue));
			}
		}
		if (response.mediaType === "text/event-stream") hasServerSentEvents = true;
	}
	if (responseValidator) reqOptions.prop("responseValidator", responseValidator.arrow());
	if (plugin.config.responseStyle === "data") reqOptions.prop("responseStyle", $.literal(plugin.config.responseStyle));
	const auth = operationAuth({
		context: plugin.context,
		operation,
		plugin
	});
	if (auth.length) reqOptions.prop("security", $.fromValue(auth));
	reqOptions.prop("url", $.literal(operation.path));
	reqOptions.spread("options");
	const statements = [];
	const hasParams = opParameters.argNames.length;
	if (hasParams) {
		const args = [];
		const config = [];
		for (const argName of opParameters.argNames) args.push($(argName));
		for (const field of opParameters.fields) {
			const shape = $.object();
			if ("in" in field) shape.prop("in", $.literal(field.in));
			if ("key" in field) {
				if (field.key) shape.prop("key", $.literal(field.key));
				if (field.map) shape.prop("map", $.literal(field.map));
			}
			config.push(shape);
		}
		const symbol = plugin.external("client.buildClientParams");
		statements.push($.const("params").assign($(symbol).call($.array(...args), $.array($.object().prop("args", $.array(...config))))));
		reqOptions.spread("params");
	}
	if (operation.body) {
		const parameterContentType = operation.parameters?.header?.["content-type"];
		if (!Boolean(parameterContentType?.required)) {
			const headers = $.object().pretty().prop(parameterContentType?.name ?? "Content-Type", $.literal(operation.body.type === "form-data" ? null : operation.body.mediaType)).spread($("options").attr("headers").required(isRequiredOptions));
			if (hasParams) headers.spread($("params").attr("headers"));
			reqOptions.prop("headers", headers);
		}
	}
	const symbolClient = plugin.config.client ? plugin.getSymbol({ category: "client" }) : void 0;
	let clientExpression;
	const optionsClient = $("options").attr("client").required(isRequiredOptions);
	if (isInstance(plugin)) clientExpression = optionsClient.coalesce($("this").attr("client"));
	else if (symbolClient) clientExpression = optionsClient.coalesce(symbolClient);
	else clientExpression = optionsClient;
	let functionName = hasServerSentEvents ? clientExpression.attr("sse") : clientExpression;
	functionName = functionName.attr(operation.method);
	statements.push($.return(functionName.call(reqOptions).$if(isNuxtClient, (f$1) => f$1.generic(nuxtTypeComposable).generic($.type.or(symbolResponseType ?? "unknown", nuxtTypeDefault)).generic(symbolErrorType ?? "unknown").generic(nuxtTypeDefault), (f$1) => f$1.generic(symbolResponseType ?? "unknown").generic(symbolErrorType ?? "unknown").generic("ThrowOnError")).$if(plugin.config.responseStyle === "data", (f$1) => f$1.generic($.type.literal(plugin.config.responseStyle)))));
	return statements;
}

//#endregion
//#region src/plugins/@hey-api/sdk/v1/node.ts
const source$1 = globalThis.Symbol("@hey-api/sdk");
function isInstance(plugin) {
	const config = plugin.config.operations;
	return config.container === "class" && config.methods === "instance" && config.strategy !== "flat";
}
function attachComment$1(args) {
	const { node, operation, plugin } = args;
	return node.$if(plugin.config.comments && createOperationComment(operation), (n, v) => n.doc(v));
}
function createShellMeta(node) {
	return {
		category: "utility",
		resource: "class",
		resourceId: node.getPath().join("."),
		tool: "sdk"
	};
}
function createFnSymbol(plugin, item) {
	const { operation, path: path$1, tags } = item.data;
	const name = item.location[item.location.length - 1];
	return plugin.symbol(applyNaming(name, plugin.config.operations.methodName), { meta: {
		category: "sdk",
		path: path$1,
		resource: "operation",
		resourceId: operation.id,
		tags,
		tool: "sdk"
	} });
}
function childToNode(resource, plugin) {
	const refChild = plugin.referenceSymbol(createShellMeta(resource));
	const memberNameStr = toCase(refChild.name, plugin.config.operations.methodName.casing ?? "camelCase");
	const memberName = plugin.symbol(memberNameStr);
	if (isInstance(plugin)) {
		const privateName = plugin.symbol(`_${memberNameStr}`);
		return [$.field(privateName, (f$1) => f$1.private().optional().type(refChild)), $.getter(memberName, (g) => g.returns(refChild).do($("this").attr(privateName).nullishAssign($.new(refChild).args($.object().prop("client", $("this").attr("client")))).return()))];
	}
	if (plugin.isSymbolRegistered(refChild.id)) return [$.field(memberName, (f$1) => f$1.static().assign($(refChild)))];
	return [$.getter(memberName, (g) => g.public().static().do($.return(refChild)))];
}
function createShell(plugin) {
	const isAngularClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-angular";
	return { define: (node) => {
		const symbol = plugin.symbol(applyNaming(node.name, node.isRoot ? plugin.config.operations.containerName : plugin.config.operations.segmentName), { meta: createShellMeta(node) });
		const c = $.class(symbol).export().$if(isInstance(plugin), (c$1) => c$1.extends(plugin.referenceSymbol({
			category: "utility",
			resource: "class",
			resourceId: "HeyApiClient",
			tool: "sdk"
		}))).$if(isAngularClient && node.isRoot, (c$1) => c$1.decorator(plugin.external("@angular/core.Injectable"), $.object().prop("providedIn", $.literal("root"))));
		const dependencies = [];
		if (node.isRoot && isInstance(plugin)) enrichRootClass({
			dependencies,
			node: c,
			plugin,
			symbol
		});
		return {
			dependencies,
			node: c
		};
	} };
}
function enrichRootClass(args) {
	const { dependencies, node, plugin, symbol } = args;
	const symbolClient = plugin.symbol("HeyApiClient", { meta: {
		category: "utility",
		resource: "class",
		resourceId: "HeyApiClient",
		tool: "sdk"
	} });
	dependencies.push(createClientClass({
		plugin,
		symbol: symbolClient
	}));
	const symbolRegistry = plugin.symbol("HeyApiRegistry", { meta: {
		category: "utility",
		resource: "class",
		resourceId: "HeyApiRegistry",
		tool: "sdk"
	} });
	dependencies.push(createRegistryClass({
		plugin,
		sdkSymbol: symbol,
		symbol: symbolRegistry
	}));
	const isClientRequired = !plugin.config.client || !plugin.getSymbol({ category: "client" });
	const registry = plugin.symbol("__registry");
	node.toAccessNode = (node$1, options) => {
		if (options.context) return;
		return $(node$1.name).attr(registry).attr("get").call();
	};
	node.do($.field(registry, (f$1) => f$1.public().static().readonly().assign($.new(symbolRegistry).generic(symbol))), $.newline(), $.init((i) => i.param("args", (p) => p.required(isClientRequired).type($.type.object().prop("client", (p$1) => p$1.required(isClientRequired).type(plugin.external("client.Client"))).prop("key", (p$1) => p$1.optional().type("string")))).do($("super").call("args"), $(symbol).attr(registry).attr("set").call("this", $("args").attr("key").required(isClientRequired)))));
}
function exampleIntent(node, operation, plugin) {
	const config = plugin.config.examples;
	if (!config.enabled) return;
	plugin.intent({ async run(context) {
		const { payload } = config;
		let example = ctx.example(node, {
			...config,
			payload: (ctx$1) => typeof payload === "function" ? payload(operation, ctx$1) : payload
		});
		if (config.transform) example = await config.transform(example, operation);
		if (example) context.setExample(operation, {
			lang: config.language,
			source: example
		});
	} });
}
function implementFn(args) {
	const { node, operation, plugin } = args;
	const client = getClientPlugin(getTypedConfig(plugin));
	const isNuxtClient = client.name === "@hey-api/client-nuxt";
	const isRequiredOptions = isOperationOptionsRequired({
		context: plugin.context,
		operation
	});
	const opParameters = operationParameters({
		isRequiredOptions,
		operation,
		plugin
	});
	const statements = operationStatements({
		isRequiredOptions,
		opParameters,
		operation,
		plugin
	});
	return node.$if(isNuxtClient, (m) => m.generic(nuxtTypeComposable, (t) => t.extends(plugin.external("client.Composable")).default($.type.literal("$fetch"))).generic(nuxtTypeDefault, (t) => t.$if(plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "response"
	}), (t$1, s) => t$1.extends(s).default(s), (t$1) => t$1.default("undefined"))), (m) => m.generic("ThrowOnError", (t) => t.extends("boolean").default(("throwOnError" in client.config ? client.config.throwOnError : false) ?? false))).params(...opParameters.parameters).do(...statements);
}
function toNode(model, plugin) {
	if (model.virtual) {
		const nodes$1 = [];
		for (const item of model.itemsFrom(source$1)) {
			const { operation } = item.data;
			let node$1 = $.const(createFnSymbol(plugin, item)).export().assign(implementFn({
				node: $.func(),
				operation,
				plugin
			}));
			node$1 = attachComment$1({
				node: node$1,
				operation,
				plugin
			});
			nodes$1.push(node$1);
			exampleIntent(node$1, operation, plugin);
		}
		return { nodes: nodes$1 };
	}
	if (!model.shell) return { nodes: [] };
	const nodes = [];
	const isAngularClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-angular";
	const shell = model.shell.define(model);
	const node = shell.node;
	let index = 0;
	for (const item of model.itemsFrom(source$1)) {
		const { operation } = item.data;
		if (node["~dsl"] === "VarTsDsl") {} else {
			if (index > 0 || node.hasBody) node.newline();
			const method = implementFn({
				node: $.method(createFnSymbol(plugin, item), (m) => attachComment$1({
					node: m,
					operation,
					plugin
				}).public().static(!isAngularClient && !isInstance(plugin))),
				operation,
				plugin
			});
			node.do(method);
			exampleIntent(method, operation, plugin);
		}
		index += 1;
	}
	for (const child of model.children.values()) if (node["~dsl"] === "VarTsDsl") {} else {
		if (node.hasBody) node.newline();
		node.do(...childToNode(child, plugin));
	}
	nodes.push(node);
	return {
		dependencies: shell.dependencies,
		nodes
	};
}

//#endregion
//#region src/plugins/shared/utils/operation.ts
function createOperationComment(operation) {
	const comments = [];
	if (operation.summary) comments.push(escapeComment(operation.summary));
	if (operation.description) {
		if (comments.length) comments.push("");
		comments.push(escapeComment(operation.description));
	}
	if (operation.deprecated) {
		if (comments.length) comments.push("");
		comments.push("@deprecated");
	}
	return comments.length ? comments : void 0;
}
/**
* TODO: replace with plugin logic...
*
* @deprecated this needs to be refactored
*/
function isOperationOptionsRequired({ context, operation }) {
	const config = getTypedConfig(context);
	const isNuxtClient = getClientPlugin(config).name === "@hey-api/client-nuxt";
	const plugin = config.plugins["@hey-api/sdk"];
	if (plugin) {
		if (!plugin.config.client && !isInstance(plugin)) return true;
		if (plugin.config.paramsStructure === "flat") return false;
	}
	return isNuxtClient || hasOperationDataRequired(operation);
}
function hasOperationSse({ operation }) {
	for (const statusCode in operation.responses) if (operation.responses[statusCode].mediaType === "text/event-stream") return true;
	return false;
}

//#endregion
//#region src/plugins/@angular/common/shared/node.ts
const source = globalThis.Symbol("@angular/common");
function attachComment(args) {
	const { node, operation, plugin } = args;
	return node.$if(plugin.config.comments && createOperationComment(operation), (n, v) => n.doc(v));
}
function createHttpRequestFnMeta(operation) {
	return {
		category: "utility",
		resource: "operation",
		resourceId: operation.id,
		role: "request",
		tool: "angular"
	};
}
function createHttpRequestShellMeta(node) {
	return {
		category: "utility",
		resource: "shell",
		resourceId: node.getPath().join("."),
		role: "request",
		tool: "angular"
	};
}
function createHttpResourceFnMeta(operation) {
	return {
		category: "utility",
		resource: "operation",
		resourceId: operation.id,
		role: "resource",
		tool: "angular"
	};
}
function createHttpResourceShellMeta(node) {
	return {
		category: "utility",
		resource: "shell",
		resourceId: node.getPath().join("."),
		role: "resource",
		tool: "angular"
	};
}
function createHttpRequestFnSymbol(plugin, item) {
	const { operation } = item.data;
	const name = item.location[item.location.length - 1];
	return plugin.symbol(applyNaming(name, plugin.config.httpRequests.methodName), { meta: createHttpRequestFnMeta(operation) });
}
function createHttpResourceFnSymbol(plugin, item) {
	const { operation } = item.data;
	const name = item.location[item.location.length - 1];
	return plugin.symbol(applyNaming(name, plugin.config.httpResources.methodName), { meta: createHttpResourceFnMeta(operation) });
}
function childToHttpRequestNode(resource, plugin) {
	const refChild = plugin.referenceSymbol(createHttpRequestShellMeta(resource));
	const memberNameStr = toCase(refChild.name, "camelCase");
	const memberName = plugin.symbol(memberNameStr);
	const privateName = plugin.symbol(`_${memberNameStr}`);
	return [$.field(privateName, (f$1) => f$1.private().optional().type(refChild)), $.getter(memberName, (g) => g.returns(refChild).do($("this").attr(privateName).nullishAssign($.new(refChild).args()).return()))];
}
function childToHttpResourceNode(resource, plugin) {
	const refChild = plugin.referenceSymbol(createHttpResourceShellMeta(resource));
	const memberNameStr = toCase(refChild.name, "camelCase");
	const memberName = plugin.symbol(memberNameStr);
	const privateName = plugin.symbol(`_${memberNameStr}`);
	return [$.field(privateName, (f$1) => f$1.private().optional().type(refChild)), $.getter(memberName, (g) => g.returns(refChild).do($("this").attr(privateName).nullishAssign($.new(refChild).args()).return()))];
}
function createHttpRequestShell(plugin) {
	const isAngularClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-angular";
	const symbolInjectable = plugin.external("@angular/core.Injectable");
	return { define: (node) => {
		const symbol = plugin.symbol(applyNaming(node.name, node.isRoot ? plugin.config.httpRequests.containerName : plugin.config.httpRequests.segmentName), { meta: createHttpRequestShellMeta(node) });
		return {
			dependencies: [],
			node: $.class(symbol).export().$if(isAngularClient && node.isRoot, (c) => c.decorator(symbolInjectable, $.object().prop("providedIn", $.literal("root"))))
		};
	} };
}
function createHttpResourceShell(plugin) {
	const isAngularClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-angular";
	const symbolInjectable = plugin.external("@angular/core.Injectable");
	return { define: (node) => {
		const symbol = plugin.symbol(applyNaming(node.name, node.isRoot ? plugin.config.httpResources.containerName : plugin.config.httpResources.segmentName), { meta: createHttpResourceShellMeta(node) });
		return {
			dependencies: [],
			node: $.class(symbol).export().$if(isAngularClient && node.isRoot, (c) => c.decorator(symbolInjectable, $.object().prop("providedIn", $.literal("root"))))
		};
	} };
}
function implementHttpRequestFn(args) {
	const { node, operation, plugin } = args;
	const isRequiredOptions = isOperationOptionsRequired({
		context: plugin.context,
		operation
	});
	const symbolHttpRequest = plugin.external("@angular/common/http.HttpRequest");
	const symbolClient = plugin.getSymbol({ category: "client" });
	const symbolOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client-options",
		tool: "sdk"
	});
	const symbolDataType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "typescript"
	});
	const symbolResponseType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "response"
	});
	return node.param("options", (p) => p.required(isRequiredOptions).type($.type(symbolOptions).generic(symbolDataType ?? "unknown").generic("ThrowOnError"))).generic("ThrowOnError", (g) => g.extends("boolean").default(false)).returns($.type(symbolHttpRequest).generic(symbolResponseType ?? "unknown")).do($.return($("options").attr("client").optional().$if(symbolClient, (c, s) => c.coalesce(s)).attr("requestOptions").call($.object().prop("responseStyle", $.literal("data")).prop("method", $.literal(operation.method.toUpperCase())).prop("url", $.literal(operation.path)).spread("options")).generic(symbolResponseType ?? "unknown").generic("ThrowOnError")));
}
function implementHttpResourceFn(args) {
	const { node, operation, plugin } = args;
	const isRequiredOptions = isOperationOptionsRequired({
		context: plugin.context,
		operation
	});
	const symbolHttpResource = plugin.external("@angular/common/http.httpResource");
	const symbolInject = plugin.external("@angular/core.inject");
	const symbolOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client-options",
		tool: "sdk"
	});
	const symbolDataType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "typescript"
	});
	const symbolResponseType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "response"
	});
	return node.param("options", (p) => p.required(isRequiredOptions).type($.type.func().returns($.type.or($.type(symbolOptions).generic(symbolDataType ?? "unknown").generic("ThrowOnError"), $.type("undefined"))))).generic("ThrowOnError", (g) => g.extends("boolean").default(false)).do($.return($(symbolHttpResource).call($.func().do($.const("opts").assign($.ternary("options").do($("options").call()).otherwise($.id("undefined"))), $.return($.ternary("opts").do($.lazy((ctx$1) => ctx$1.access(plugin.referenceSymbol(createHttpRequestFnMeta(operation)), { transform: (node$1, index) => index === 0 ? node$1["~dsl"] === "ClassTsDsl" ? $(symbolInject).call($(node$1.name)) : $(node$1.name) : node$1 }).call("opts"))).otherwise($.id("undefined"))))).generic(symbolResponseType ?? "unknown")));
}
function toHttpRequestNode(model, plugin) {
	if (model.virtual) {
		const nodes$1 = [];
		for (const item of model.itemsFrom(source)) {
			const { operation } = item.data;
			let node$1 = $.const(createHttpRequestFnSymbol(plugin, item)).export().assign(implementHttpRequestFn({
				node: $.func(),
				operation,
				plugin
			}));
			node$1 = attachComment({
				node: node$1,
				operation,
				plugin
			});
			nodes$1.push(node$1);
		}
		return { nodes: nodes$1 };
	}
	if (!model.shell) return { nodes: [] };
	const nodes = [];
	const shell = model.shell.define(model);
	const node = shell.node;
	let index = 0;
	for (const item of model.itemsFrom(source)) {
		const { operation } = item.data;
		if (index > 0 || node.hasBody) node.newline();
		node.do(implementHttpRequestFn({
			node: $.method(createHttpRequestFnSymbol(plugin, item), (m) => attachComment({
				node: m,
				operation,
				plugin
			}).public()),
			operation,
			plugin
		}));
		index += 1;
	}
	for (const child of model.children.values()) {
		if (node.hasBody) node.newline();
		node.do(...childToHttpRequestNode(child, plugin));
	}
	nodes.push(node);
	return {
		dependencies: shell.dependencies,
		nodes
	};
}
function toHttpResourceNode(model, plugin) {
	if (model.virtual) {
		const nodes$1 = [];
		for (const item of model.itemsFrom(source)) {
			const { operation } = item.data;
			let node$1 = $.const(createHttpResourceFnSymbol(plugin, item)).export().assign(implementHttpResourceFn({
				node: $.func(),
				operation,
				plugin
			}));
			node$1 = attachComment({
				node: node$1,
				operation,
				plugin
			});
			nodes$1.push(node$1);
		}
		return { nodes: nodes$1 };
	}
	if (!model.shell) return { nodes: [] };
	const nodes = [];
	const shell = model.shell.define(model);
	const node = shell.node;
	let index = 0;
	for (const item of model.itemsFrom(source)) {
		const { operation } = item.data;
		if (index > 0 || node.hasBody) node.newline();
		node.do(implementHttpResourceFn({
			node: $.method(createHttpResourceFnSymbol(plugin, item), (m) => attachComment({
				node: m,
				operation,
				plugin
			}).public()),
			operation,
			plugin
		}));
		index += 1;
	}
	for (const child of model.children.values()) {
		if (node.hasBody) node.newline();
		node.do(...childToHttpResourceNode(child, plugin));
	}
	nodes.push(node);
	return {
		dependencies: shell.dependencies,
		nodes
	};
}

//#endregion
//#region src/plugins/@angular/common/plugin.ts
const handler$11 = ({ plugin }) => {
	plugin.symbol("HttpRequest", {
		external: "@angular/common/http",
		kind: "type",
		meta: {
			category: "external",
			resource: "@angular/common/http.HttpRequest"
		}
	});
	plugin.symbol("inject", {
		external: "@angular/core",
		meta: {
			category: "external",
			resource: "@angular/core.inject"
		}
	});
	plugin.symbol("Injectable", {
		external: "@angular/core",
		meta: {
			category: "external",
			resource: "@angular/core.Injectable"
		}
	});
	plugin.symbol("httpResource", {
		external: "@angular/common/http",
		meta: {
			category: "external",
			resource: "@angular/common/http.httpResource"
		}
	});
	const httpRequestStructure = new StructureModel();
	const httpResourceStructure = new StructureModel();
	if (plugin.config.httpRequests.enabled) {
		const shell = createHttpRequestShell(plugin);
		const strategy = resolveHttpRequestsStrategy(plugin);
		plugin.forEach("operation", ({ operation }) => {
			httpRequestStructure.insert({
				data: { operation },
				locations: strategy(operation).map((path$1) => ({
					path: path$1,
					shell
				})),
				source
			});
		}, { order: "declarations" });
	}
	if (plugin.config.httpResources.enabled) {
		const shell = createHttpResourceShell(plugin);
		const strategy = resolveHttpResourcesStrategy(plugin);
		plugin.forEach("operation", ({ operation }) => {
			httpResourceStructure.insert({
				data: { operation },
				locations: strategy(operation).map((path$1) => ({
					path: path$1,
					shell
				})),
				source
			});
		}, { order: "declarations" });
	}
	const allDependencies = [];
	const allNodes = [];
	for (const node of httpRequestStructure.walk()) {
		const { dependencies, nodes } = toHttpRequestNode(node, plugin);
		allDependencies.push(...dependencies ?? []);
		allNodes.push(...nodes);
	}
	for (const node of httpResourceStructure.walk()) {
		const { dependencies, nodes } = toHttpResourceNode(node, plugin);
		allDependencies.push(...dependencies ?? []);
		allNodes.push(...nodes);
	}
	const uniqueDependencies = /* @__PURE__ */ new Map();
	for (const dep of allDependencies) if (dep.symbol) uniqueDependencies.set(dep.symbol.id, dep);
	for (const dep of uniqueDependencies.values()) plugin.node(dep);
	for (const node of allNodes) plugin.node(node);
};

//#endregion
//#region src/plugins/@angular/common/config.ts
const defaultConfig$23 = {
	config: {
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/client-angular", "@hey-api/sdk"],
	handler: handler$11,
	name: "@angular/common",
	resolveConfig: (plugin, context) => {
		plugin.config.httpRequests = resolveHttpRequests(plugin.config, context);
		plugin.config.httpResources = resolveHttpResources(plugin.config, context);
	}
};
/**
* Type helper for `@angular/common` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$23 = definePluginConfig(defaultConfig$23);

//#endregion
//#region src/plugins/@faker-js/faker/api.ts
var Api$4 = class {
	toNode(_schema) {}
	toNodeRef(_schema) {}
};

//#endregion
//#region src/plugins/@faker-js/faker/config.ts
const defaultConfig$22 = {
	api: new Api$4(),
	config: {
		case: "camelCase",
		includeInEntry: false
	},
	handler: () => {},
	name: "@faker-js/faker",
	resolveConfig: (plugin, context) => {
		plugin.config.definitions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "v{{name}}"
			},
			mappers,
			value: plugin.config.definitions
		});
	},
	tags: ["mocker"]
};
/**
* Type helper for Faker plugin, returns {@link Plugin.Config} object
*/
const defineConfig$22 = definePluginConfig(defaultConfig$22);

//#endregion
//#region src/plugins/@hey-api/client-core/config.ts
const clientDefaultConfig = {
	baseUrl: true,
	bundle: true,
	includeInEntry: false
};
const clientDefaultMeta = {
	dependencies: ["@hey-api/typescript"],
	tags: ["client"]
};

//#endregion
//#region src/generate/client.ts
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
/**
* Dev mode: 'src' appears after 'dist' (or dist doesn't exist), and 'generate' follows 'src'
*/
function isDevMode() {
	const normalized = __dirname.split(path.sep);
	const srcIndex = normalized.lastIndexOf("src");
	const distIndex = normalized.lastIndexOf("dist");
	return srcIndex !== -1 && srcIndex > distIndex && srcIndex === normalized.length - 2 && normalized[srcIndex + 1] === "generate";
}
/**
* Returns paths to client bundle files based on execution context
*/
function getClientBundlePaths(pluginName) {
	const clientName = pluginName.slice(16);
	if (isDevMode()) {
		const pluginsDir = path.resolve(__dirname, "..", "plugins", "@hey-api");
		return {
			clientPath: path.resolve(pluginsDir, `client-${clientName}`, "bundle"),
			corePath: path.resolve(pluginsDir, "client-core", "bundle")
		};
	}
	return {
		clientPath: path.resolve(__dirname, "clients", clientName),
		corePath: path.resolve(__dirname, "clients", "core")
	};
}
/**
* Returns absolute path to the client folder. This is hard-coded for now.
*/
function clientFolderAbsolutePath(config) {
	const client = getClientPlugin(config);
	if ("bundle" in client.config && client.config.bundle) {
		const renamed = config._FRAGILE_CLIENT_BUNDLE_RENAMED;
		return path.resolve(config.output.path, "client", `${renamed?.get("index") ?? "index"}.ts`);
	}
	return client.name;
}
/**
* Recursively copies files and directories.
* This is a PnP-compatible alternative to fs.cpSync that works with Yarn PnP's
* virtualized filesystem.
*/
function copyRecursivePnP(src, dest) {
	if (fs.statSync(src).isDirectory()) {
		if (!fs.existsSync(dest)) fs.mkdirSync(dest, { recursive: true });
		const files = fs.readdirSync(src);
		for (const file of files) copyRecursivePnP(path.join(src, file), path.join(dest, file));
	} else {
		const content = fs.readFileSync(src);
		fs.writeFileSync(dest, content);
	}
}
function renameFile({ filePath, project, renamed }) {
	const extension = path.extname(filePath);
	const name = path.basename(filePath, extension);
	const renamedName = project.fileName?.(name) || name;
	if (renamedName !== name) {
		const outputPath = path.dirname(filePath);
		fs.renameSync(filePath, path.resolve(outputPath, `${renamedName}${extension}`));
		renamed.set(name, renamedName);
	}
}
function replaceImports({ filePath, isDevMode: isDevMode$1, meta, renamed }) {
	let content = fs.readFileSync(filePath, "utf8");
	if (isDevMode$1) {
		content = content.replace(/from\s+['"]\.\.\/\.\.\/client-core\/bundle\//g, "from '../core/");
		content = content.replace(/from\s+['"]\.\.\/\.\.\/client-core\/bundle['"]/g, "from '../core'");
	}
	content = content.replace(/from\s+['"](\.\.?\/[^'"]*?)['"]/g, (match, importPath) => {
		const importIndex = match.indexOf(importPath);
		const extension = path.extname(importPath);
		const fileName = path.basename(importPath, extension);
		const importDir = path.dirname(importPath);
		const replacedName = (renamed.get(fileName) ?? fileName) + (meta.importFileExtension ? meta.importFileExtension : extension);
		return match.slice(0, importIndex) + [importDir, replacedName].filter(Boolean).join("/") + match.slice(importIndex + importPath.length);
	});
	content = `// This file is auto-generated by @hey-api/openapi-ts

${content}`;
	fs.writeFileSync(filePath, content, "utf8");
}
/**
* Creates a `client` folder containing the same modules as the client package.
*/
function generateClientBundle({ meta, outputPath, plugin, project }) {
	const renamed = /* @__PURE__ */ new Map();
	const devMode = isDevMode();
	if (plugin.name.startsWith("@hey-api/client-")) {
		const { clientPath, corePath } = getClientBundlePaths(plugin.name);
		const coreOutputPath = path.resolve(outputPath, "core");
		ensureDirSync(coreOutputPath);
		copyRecursivePnP(corePath, coreOutputPath);
		const clientOutputPath = path.resolve(outputPath, "client");
		ensureDirSync(clientOutputPath);
		copyRecursivePnP(clientPath, clientOutputPath);
		if (project) {
			const copiedCoreFiles = fs.readdirSync(coreOutputPath);
			for (const file of copiedCoreFiles) renameFile({
				filePath: path.resolve(coreOutputPath, file),
				project,
				renamed
			});
			const copiedClientFiles = fs.readdirSync(clientOutputPath);
			for (const file of copiedClientFiles) renameFile({
				filePath: path.resolve(clientOutputPath, file),
				project,
				renamed
			});
		}
		const coreFiles = fs.readdirSync(coreOutputPath);
		for (const file of coreFiles) replaceImports({
			filePath: path.resolve(coreOutputPath, file),
			isDevMode: devMode,
			meta,
			renamed
		});
		const clientFiles = fs.readdirSync(clientOutputPath);
		for (const file of clientFiles) replaceImports({
			filePath: path.resolve(clientOutputPath, file),
			isDevMode: devMode,
			meta,
			renamed
		});
		return renamed;
	}
	const clientSrcPath = path.isAbsolute(plugin.name) ? path.dirname(plugin.name) : void 0;
	if (clientSrcPath) {
		const dirPath$1 = path.resolve(outputPath, "client");
		ensureDirSync(dirPath$1);
		copyRecursivePnP(clientSrcPath, dirPath$1);
		return;
	}
	const clientModulePathComponents = path.normalize(__require.resolve(plugin.name)).split(path.sep);
	const clientDistPath = clientModulePathComponents.slice(0, clientModulePathComponents.indexOf("dist") + 1).join(path.sep);
	const distFiles = [
		clientModulePathComponents[clientModulePathComponents.length - 1],
		"index.d.mts",
		"index.d.cts"
	];
	const dirPath = path.resolve(outputPath, "client");
	ensureDirSync(dirPath);
	for (const file of distFiles) fs.copyFileSync(path.resolve(clientDistPath, file), path.resolve(dirPath, file));
}

//#endregion
//#region src/plugins/@hey-api/client-core/client.ts
const resolveBaseUrlString = ({ plugin }) => {
	const { baseUrl } = plugin.config;
	if (baseUrl === false) return;
	if (typeof baseUrl === "string") return baseUrl;
	const { servers } = plugin.context.ir;
	if (!servers) return;
	return servers[typeof baseUrl === "number" ? baseUrl : 0]?.url;
};
const createClient = ({ plugin }) => {
	const clientModule = clientFolderAbsolutePath(getTypedConfig(plugin));
	const symbolCreateClient = plugin.symbol("createClient", { external: clientModule });
	const symbolCreateConfig = plugin.symbol("createConfig", { external: clientModule });
	const symbolClientOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client",
		role: "options"
	});
	const { runtimeConfigPath } = plugin.config;
	const symbolCreateClientConfig = runtimeConfigPath ? plugin.symbol("createClientConfig", { external: runtimeConfigPath }) : void 0;
	const defaultVals = $.object();
	const resolvedBaseUrl = resolveBaseUrlString({ plugin });
	if (resolvedBaseUrl) {
		const url = parseUrl(resolvedBaseUrl);
		if (url.protocol && url.host && !resolvedBaseUrl.includes("{")) defaultVals.prop(getClientBaseUrlKey(getTypedConfig(plugin)), $.literal(resolvedBaseUrl));
		else if (resolvedBaseUrl !== "/" && resolvedBaseUrl.startsWith("/")) {
			const baseUrl = resolvedBaseUrl.endsWith("/") ? resolvedBaseUrl.slice(0, -1) : resolvedBaseUrl;
			defaultVals.prop(getClientBaseUrlKey(getTypedConfig(plugin)), $.literal(baseUrl));
		}
	}
	if ("throwOnError" in plugin.config && plugin.config.throwOnError) defaultVals.prop("throwOnError", $.literal(true));
	const createConfigParameters = [$(symbolCreateConfig).call(defaultVals.hasProps() ? defaultVals : void 0).generic(symbolClientOptions)];
	const symbolClient = plugin.symbol("client", { meta: { category: "client" } });
	const statement = $.const(symbolClient).export().assign($(symbolCreateClient).$if(symbolCreateClientConfig, (c, s) => c.call($(s).call(...createConfigParameters)), (c) => c.call(...createConfigParameters)));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/@hey-api/client-core/createClientConfig.ts
const createClientConfigType = ({ plugin }) => {
	const clientModule = clientFolderAbsolutePath(getTypedConfig(plugin));
	const symbolClientOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client",
		role: "options"
	});
	const symbolConfig = plugin.symbol("Config", {
		external: clientModule,
		kind: "type"
	});
	const symbolDefaultClientOptions = plugin.symbol("ClientOptions", {
		external: clientModule,
		kind: "type"
	});
	const symbolCreateClientConfig = plugin.symbol("CreateClientConfig");
	const typeCreateClientConfig = $.type.alias(symbolCreateClientConfig).export().doc([
		"The `createClientConfig()` function will be called on client initialization",
		"and the returned object will become the client's initial configuration.",
		"",
		"You may want to initialize your client this way instead of calling",
		"`setConfig()`. This is useful for example if you're using Next.js",
		"to ensure your client always has the correct values."
	]).generic("T", (g) => g.extends(symbolDefaultClientOptions).default(symbolClientOptions)).type($.type.func().param("override", (p) => p.optional().type($.type(symbolConfig).generic($.type.and(symbolDefaultClientOptions, "T")))).returns($.type(symbolConfig).generic($.type.and($.type("Required").generic(symbolDefaultClientOptions), "T"))));
	plugin.node(typeCreateClientConfig);
};

//#endregion
//#region src/plugins/@hey-api/client-core/plugin.ts
const clientPluginHandler = ({ plugin }) => {
	createClientConfigType({ plugin });
	createClient({ plugin });
};

//#endregion
//#region src/plugins/@hey-api/client-angular/config.ts
const defaultConfig$21 = {
	...clientDefaultMeta,
	config: {
		...clientDefaultConfig,
		throwOnError: false
	},
	handler: clientPluginHandler,
	name: "@hey-api/client-angular"
};
/**
* Type helper for `@hey-api/client-angular` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$21 = definePluginConfig(defaultConfig$21);

//#endregion
//#region src/plugins/@hey-api/client-axios/config.ts
const defaultConfig$20 = {
	...clientDefaultMeta,
	config: {
		...clientDefaultConfig,
		throwOnError: false
	},
	handler: clientPluginHandler,
	name: "@hey-api/client-axios"
};
/**
* Type helper for `@hey-api/client-axios` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$20 = definePluginConfig(defaultConfig$20);

//#endregion
//#region src/plugins/@hey-api/client-fetch/config.ts
const defaultConfig$19 = {
	...clientDefaultMeta,
	config: {
		...clientDefaultConfig,
		throwOnError: false
	},
	handler: clientPluginHandler,
	name: "@hey-api/client-fetch"
};
/**
* Type helper for `@hey-api/client-fetch` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$19 = definePluginConfig(defaultConfig$19);

//#endregion
//#region src/plugins/@hey-api/client-ky/config.ts
const defaultConfig$18 = {
	...clientDefaultMeta,
	config: {
		...clientDefaultConfig,
		throwOnError: false
	},
	handler: clientPluginHandler,
	name: "@hey-api/client-ky"
};
/**
* Type helper for `@hey-api/client-ky` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$18 = definePluginConfig(defaultConfig$18);

//#endregion
//#region src/plugins/@hey-api/client-next/config.ts
const defaultConfig$17 = {
	...clientDefaultMeta,
	config: {
		...clientDefaultConfig,
		throwOnError: false
	},
	handler: clientPluginHandler,
	name: "@hey-api/client-next"
};
/**
* Type helper for `@hey-api/client-next` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$17 = definePluginConfig(defaultConfig$17);

//#endregion
//#region src/plugins/@hey-api/client-nuxt/config.ts
const defaultConfig$16 = {
	...clientDefaultMeta,
	config: clientDefaultConfig,
	handler: clientPluginHandler,
	name: "@hey-api/client-nuxt"
};
/**
* Type helper for `@hey-api/client-nuxt` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$16 = definePluginConfig(defaultConfig$16);

//#endregion
//#region src/plugins/@hey-api/client-ofetch/config.ts
const defaultConfig$15 = {
	...clientDefaultMeta,
	config: {
		...clientDefaultConfig,
		throwOnError: false
	},
	handler: clientPluginHandler,
	name: "@hey-api/client-ofetch"
};
/**
* Type helper for `@hey-api/client-ofetch` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$15 = definePluginConfig(defaultConfig$15);

//#endregion
//#region src/plugins/@hey-api/schemas/plugin.ts
const stripSchema = ({ plugin, schema }) => {
	if (plugin.config.type === "form") {
		if (schema.description) delete schema.description;
		if (schema["x-enum-descriptions"]) delete schema["x-enum-descriptions"];
		if (schema["x-enum-varnames"]) delete schema["x-enum-varnames"];
		if (schema["x-enumNames"]) delete schema["x-enumNames"];
		if (schema.title) delete schema.title;
	}
};
const schemaToJsonSchemaDraft_04 = ({ context, plugin, schema: _schema }) => {
	if (Array.isArray(_schema)) return _schema.map((item) => schemaToJsonSchemaDraft_04({
		context,
		plugin,
		schema: item
	}));
	const schema = structuredClone(_schema);
	if (schema.$ref) {
		schema.$ref = decodeURI(schema.$ref);
		return schema;
	}
	stripSchema({
		plugin,
		schema
	});
	if (schema.additionalProperties && typeof schema.additionalProperties !== "boolean") schema.additionalProperties = schemaToJsonSchemaDraft_04({
		context,
		plugin,
		schema: schema.additionalProperties
	});
	if (schema.allOf) schema.allOf = schema.allOf.map((item) => schemaToJsonSchemaDraft_04({
		context,
		plugin,
		schema: item
	}));
	if (schema.items) schema.items = schemaToJsonSchemaDraft_04({
		context,
		plugin,
		schema: schema.items
	});
	if (schema.properties) for (const name in schema.properties) {
		const property = schema.properties[name];
		if (typeof property !== "boolean") schema.properties[name] = schemaToJsonSchemaDraft_04({
			context,
			plugin,
			schema: property
		});
	}
	return schema;
};
const schemaToJsonSchemaDraft_05 = ({ context, plugin, schema: _schema }) => {
	if (Array.isArray(_schema)) return _schema.map((item) => schemaToJsonSchemaDraft_05({
		context,
		plugin,
		schema: item
	}));
	const schema = structuredClone(_schema);
	if ("$ref" in schema) {
		schema.$ref = decodeURI(schema.$ref);
		return schema;
	}
	stripSchema({
		plugin,
		schema
	});
	if (schema.additionalProperties && typeof schema.additionalProperties !== "boolean") schema.additionalProperties = schemaToJsonSchemaDraft_05({
		context,
		plugin,
		schema: schema.additionalProperties
	});
	if (schema.allOf) schema.allOf = schema.allOf.map((item) => schemaToJsonSchemaDraft_05({
		context,
		plugin,
		schema: item
	}));
	if (schema.anyOf) schema.anyOf = schema.anyOf.map((item) => schemaToJsonSchemaDraft_05({
		context,
		plugin,
		schema: item
	}));
	if (schema.items) schema.items = schemaToJsonSchemaDraft_05({
		context,
		plugin,
		schema: schema.items
	});
	if (schema.oneOf) schema.oneOf = schema.oneOf.map((item) => schemaToJsonSchemaDraft_05({
		context,
		plugin,
		schema: item
	}));
	if (schema.properties) for (const name in schema.properties) {
		const property = schema.properties[name];
		if (typeof property !== "boolean") schema.properties[name] = schemaToJsonSchemaDraft_05({
			context,
			plugin,
			schema: property
		});
	}
	return schema;
};
const schemaToJsonSchema2020_12 = ({ context, plugin, schema: _schema }) => {
	if (Array.isArray(_schema)) return _schema.map((item) => schemaToJsonSchema2020_12({
		context,
		plugin,
		schema: item
	}));
	const schema = structuredClone(_schema);
	stripSchema({
		plugin,
		schema
	});
	if (schema.$ref) schema.$ref = decodeURI(schema.$ref);
	if (schema.additionalProperties && typeof schema.additionalProperties !== "boolean") schema.additionalProperties = schemaToJsonSchema2020_12({
		context,
		plugin,
		schema: schema.additionalProperties
	});
	if (schema.allOf) schema.allOf = schema.allOf.map((item) => schemaToJsonSchema2020_12({
		context,
		plugin,
		schema: item
	}));
	if (schema.anyOf) schema.anyOf = schema.anyOf.map((item) => schemaToJsonSchema2020_12({
		context,
		plugin,
		schema: item
	}));
	if (schema.items) schema.items = schemaToJsonSchema2020_12({
		context,
		plugin,
		schema: schema.items
	});
	if (schema.oneOf) schema.oneOf = schema.oneOf.map((item) => schemaToJsonSchema2020_12({
		context,
		plugin,
		schema: item
	}));
	if (schema.prefixItems) schema.prefixItems = schema.prefixItems.map((item) => schemaToJsonSchema2020_12({
		context,
		plugin,
		schema: item
	}));
	if (schema.properties) for (const name in schema.properties) {
		const property = schema.properties[name];
		if (typeof property !== "boolean") schema.properties[name] = schemaToJsonSchema2020_12({
			context,
			plugin,
			schema: property
		});
	}
	return schema;
};
const schemaName = ({ name, plugin, schema }) => {
	let customName = "";
	if (plugin.config.nameBuilder) if (typeof plugin.config.nameBuilder === "function") customName = plugin.config.nameBuilder(name, schema);
	else customName = plugin.config.nameBuilder.replace("{{name}}", name);
	if (!customName) customName = `${name}Schema`;
	return customName;
};
const schemasV2_0_X = ({ context, plugin }) => {
	if (!context.spec.definitions) return;
	for (const name in context.spec.definitions) {
		const schema = context.spec.definitions[name];
		const symbol = plugin.symbol(schemaName({
			name,
			plugin,
			schema
		}), { meta: {
			category: "schema",
			resource: "definition",
			resourceId: name,
			tool: "json-schema"
		} });
		const obj = schemaToJsonSchemaDraft_04({
			context,
			plugin,
			schema
		});
		const statement = $.const(symbol).export().assign($($.fromValue(obj, { layout: "pretty" })).as("const"));
		plugin.node(statement);
	}
};
const schemasV3_0_X = ({ context, plugin }) => {
	if (!context.spec.components) return;
	for (const name in context.spec.components.schemas) {
		const schema = context.spec.components.schemas[name];
		const symbol = plugin.symbol(schemaName({
			name,
			plugin,
			schema
		}), { meta: {
			category: "schema",
			resource: "definition",
			resourceId: name,
			tool: "json-schema"
		} });
		const obj = schemaToJsonSchemaDraft_05({
			context,
			plugin,
			schema
		});
		const statement = $.const(symbol).export().assign($($.fromValue(obj, { layout: "pretty" })).as("const"));
		plugin.node(statement);
	}
};
const schemasV3_1_X = ({ context, plugin }) => {
	if (!context.spec.components) return;
	for (const name in context.spec.components.schemas) {
		const schema = context.spec.components.schemas[name];
		const symbol = plugin.symbol(schemaName({
			name,
			plugin,
			schema
		}), { meta: {
			category: "schema",
			resource: "definition",
			resourceId: name,
			tool: "json-schema"
		} });
		const obj = schemaToJsonSchema2020_12({
			context,
			plugin,
			schema
		});
		const statement = $.const(symbol).export().assign($($.fromValue(obj, { layout: "pretty" })).as("const"));
		plugin.node(statement);
	}
};
const handler$10 = ({ plugin }) => {
	if ("swagger" in plugin.context.spec) {
		schemasV2_0_X({
			context: plugin.context,
			plugin
		});
		return;
	}
	if (satisfies(plugin.context.spec.openapi, ">=3.0.0 <3.1.0")) {
		schemasV3_0_X({
			context: plugin.context,
			plugin
		});
		return;
	}
	if (satisfies(plugin.context.spec.openapi, ">=3.1.0")) {
		schemasV3_1_X({
			context: plugin.context,
			plugin
		});
		return;
	}
	throw new Error("Unsupported OpenAPI specification");
};

//#endregion
//#region src/plugins/@hey-api/schemas/config.ts
const defaultConfig$14 = {
	config: {
		includeInEntry: false,
		nameBuilder: (name) => `${name}Schema`,
		type: "json"
	},
	handler: handler$10,
	name: "@hey-api/schemas"
};
/**
* Type helper for `@hey-api/schemas` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$14 = definePluginConfig(defaultConfig$14);

//#endregion
//#region src/plugins/@hey-api/sdk/examples/config.ts
function resolveExamples(config, context) {
	return context.valueToObject({
		defaultValue: {
			enabled: Boolean(config.examples),
			language: "JavaScript"
		},
		mappers: { boolean: (enabled) => ({ enabled }) },
		value: config.examples
	});
}

//#endregion
//#region src/plugins/@hey-api/sdk/operations/config.ts
function resolveOperations(config, context) {
	if (config.asClass !== void 0) log.warnDeprecated({
		context: "@hey-api/sdk",
		field: "asClass",
		replacement: ["operations: { strategy: \"byTags\" }", "operations: { strategy: \"single\" }"]
	});
	if (config.classNameBuilder !== void 0) log.warnDeprecated({
		context: "@hey-api/sdk",
		field: "classNameBuilder",
		replacement: "operations: { containerName: \"...\" }"
	});
	if (config.classStructure !== void 0) log.warnDeprecated({
		context: "@hey-api/sdk",
		field: "classStructure",
		replacement: ["operations: { nesting: \"operationId\" }", "operations: { nesting: \"id\" }"]
	});
	if (config.instance !== void 0) log.warnDeprecated({
		context: "@hey-api/sdk",
		field: "instance",
		replacement: `operations: { strategy: "single", containerName: "${config.instance || "Name"}", methods: "instance" }`
	});
	if (config.methodNameBuilder !== void 0) log.warnDeprecated({
		context: "@hey-api/sdk",
		field: "methodNameBuilder",
		replacement: "operations: { methodName: \"...\" }"
	});
	if (config.operationId !== void 0) log.warnDeprecated({
		context: "@hey-api/sdk",
		field: "operationId",
		replacement: ["operations: { nesting: \"operationId\" }", "operations: { nesting: \"id\" }"]
	});
	const legacy = mapLegacyToConfig(config);
	return normalizeConfig(config.operations, legacy, context);
}
function normalizeConfig(input, legacy, context) {
	if (!input || typeof input === "string" || typeof input === "function") input = { strategy: input };
	const strategy = legacy.strategy ?? input.strategy ?? "flat";
	const methods = strategy === "single" ? "instance" : "static";
	return context.valueToObject({
		defaultValue: {
			container: "class",
			methods,
			nesting: "operationId",
			nestingDelimiters: /[./]/,
			strategy,
			strategyDefaultTag: "default"
		},
		mappers: { object(value) {
			value.containerName = context.valueToObject({
				defaultValue: strategy === "single" ? {
					casing: "PascalCase",
					name: "Sdk"
				} : { casing: "PascalCase" },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.containerName
			});
			value.methodName = context.valueToObject({
				defaultValue: { casing: "camelCase" },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.methodName
			});
			value.segmentName = context.valueToObject({
				defaultValue: { casing: "PascalCase" },
				mappers: {
					function: (name) => ({ name }),
					string: (name) => ({ name })
				},
				value: value.segmentName
			});
			return value;
		} },
		value: {
			...legacy,
			...input
		}
	});
}
function mapLegacyToConfig(config) {
	let strategy;
	if (config.instance) strategy = "single";
	else if (config.asClass) strategy = "byTags";
	else if (config.instance === false || config.asClass === false) strategy = "flat";
	let containerName;
	let segmentName;
	if (config.instance) {
		let name = typeof config.instance === "string" ? config.instance : "Sdk";
		segmentName = { casing: "PascalCase" };
		if (config.classNameBuilder) {
			segmentName.name = config.classNameBuilder;
			if (typeof config.classNameBuilder === "string") name = config.classNameBuilder.replace("{{name}}", name);
			else name = config.classNameBuilder(name);
		}
		containerName = {
			casing: "PascalCase",
			name
		};
	} else if (config.classNameBuilder) {
		containerName = {
			casing: "PascalCase",
			name: config.classNameBuilder
		};
		segmentName = { ...containerName };
	} else if (config.asClass) {
		containerName = { casing: "PascalCase" };
		segmentName = { ...containerName };
	}
	let methods;
	if (config.instance) methods = "instance";
	else if (config.asClass) methods = "static";
	let nesting;
	if (config.classStructure === "off" || config.operationId === false) nesting = "id";
	else if (config.classStructure === "auto") nesting = "operationId";
	let methodName;
	if (config.methodNameBuilder) methodName = {
		casing: "camelCase",
		name: config.methodNameBuilder
	};
	return {
		containerName,
		methodName,
		methods,
		nesting,
		segmentName,
		strategy
	};
}

//#endregion
//#region src/plugins/@hey-api/sdk/operations/resolve.ts
function resolvePath(plugin) {
	if (plugin.config.operations.nesting === "id") return OperationPath.id();
	if (plugin.config.operations.nesting === "operationId") return OperationPath.fromOperationId({
		delimiters: plugin.config.operations.nestingDelimiters,
		fallback: OperationPath.id()
	});
	return plugin.config.operations.nesting;
}
function resolveStrategy(plugin) {
	if (plugin.config.operations.strategy === "flat") return OperationStrategy.flat({ path: (operation) => [resolvePath(plugin)(operation).join(".")] });
	if (plugin.config.operations.strategy === "single") {
		const root = plugin.config.operations.containerName;
		return OperationStrategy.single({
			path: resolvePath(plugin),
			root: typeof root.name === "string" ? root.name : root.name?.("") ?? ""
		});
	}
	if (plugin.config.operations.strategy === "byTags") return OperationStrategy.byTags({
		fallback: plugin.config.operations.strategyDefaultTag,
		path: resolvePath(plugin)
	});
	return plugin.config.operations.strategy;
}

//#endregion
//#region src/plugins/@hey-api/sdk/shared/typeOptions.ts
const createTypeOptions = ({ plugin }) => {
	const clientModule = clientFolderAbsolutePath(getTypedConfig(plugin));
	const client = getClientPlugin(getTypedConfig(plugin));
	const isNuxtClient = client.name === "@hey-api/client-nuxt";
	const symbolTDataShape = plugin.symbol("TDataShape", {
		external: clientModule,
		kind: "type"
	});
	const symbolClient = plugin.symbol("Client", {
		external: clientModule,
		kind: "type",
		meta: {
			category: "external",
			resource: "client.Client",
			tool: client.name
		}
	});
	const symbolClientOptions = plugin.symbol("Options", {
		external: clientModule,
		kind: "type"
	});
	const symbolOptions = plugin.symbol("Options", { meta: {
		category: "type",
		resource: "client-options",
		tool: "sdk"
	} });
	const typeOptions = $.type.alias(symbolOptions).export().$if(isNuxtClient, (t) => t.generic("TComposable", (g) => g.extends(plugin.external("client.Composable")).default($.type.literal("$fetch"))).generic("TData", (g) => g.extends(symbolTDataShape).default(symbolTDataShape)).generic(nuxtTypeResponse, (g) => g.default("unknown")).generic(nuxtTypeDefault, (g) => g.default("undefined")), (t) => t.generic("TData", (g) => g.extends(symbolTDataShape).default(symbolTDataShape)).generic("ThrowOnError", (g) => g.extends("boolean").default("boolean"))).type($.type.and($.type(symbolClientOptions).$if(isNuxtClient, (t) => t.generic("TComposable").generic("TData").generic(nuxtTypeResponse).generic(nuxtTypeDefault), (t) => t.generic("TData").generic("ThrowOnError")), $.type.object().prop("client", (p) => p.doc([
		"You can provide a client instance returned by `createClient()` instead of",
		"individual options. This might be also useful if you want to implement a",
		"custom client."
	]).required(!plugin.config.client && !isInstance(plugin)).type(symbolClient)).prop("meta", (p) => p.doc(["You can pass arbitrary values through the `meta` object. This can be", "used to access values that aren't defined as part of the SDK function."]).optional().type($.type("Record").generics("string", "unknown")))));
	plugin.node(typeOptions);
};

//#endregion
//#region src/plugins/@hey-api/sdk/v1/plugin.ts
const handlerV1$2 = ({ plugin }) => {
	const clientModule = clientFolderAbsolutePath(getTypedConfig(plugin));
	const client = getClientPlugin(getTypedConfig(plugin));
	const isAngularClient = client.name === "@hey-api/client-angular";
	const isNuxtClient = client.name === "@hey-api/client-nuxt";
	plugin.symbol("formDataBodySerializer", {
		external: clientModule,
		meta: {
			category: "external",
			resource: "client.formDataBodySerializer",
			tool: client.name
		}
	});
	plugin.symbol("urlSearchParamsBodySerializer", {
		external: clientModule,
		meta: {
			category: "external",
			resource: "client.urlSearchParamsBodySerializer",
			tool: client.name
		}
	});
	plugin.symbol("buildClientParams", {
		external: clientModule,
		meta: {
			category: "external",
			resource: "client.buildClientParams",
			tool: client.name
		}
	});
	if (isNuxtClient) plugin.symbol("Composable", {
		external: clientModule,
		kind: "type",
		meta: {
			category: "external",
			resource: "client.Composable",
			tool: client.name
		}
	});
	if (isAngularClient) plugin.symbol("Injectable", {
		external: "@angular/core",
		meta: {
			category: "external",
			resource: "@angular/core.Injectable"
		}
	});
	createTypeOptions({ plugin });
	const structure = new StructureModel();
	const shell = createShell(plugin);
	const strategy = resolveStrategy(plugin);
	plugin.forEach("operation", (event) => {
		structure.insert({
			data: {
				operation: event.operation,
				path: event._path,
				tags: event.tags
			},
			locations: strategy(event.operation).map((path$1) => ({
				path: path$1,
				shell
			})),
			source: source$1
		});
	}, { order: "declarations" });
	const allDependencies = [];
	const allNodes = [];
	for (const node of structure.walk()) {
		const { dependencies, nodes } = toNode(node, plugin);
		allDependencies.push(...dependencies ?? []);
		allNodes.push(...nodes);
	}
	const uniqueDependencies = /* @__PURE__ */ new Map();
	for (const dep of allDependencies) if (dep.symbol) uniqueDependencies.set(dep.symbol.id, dep);
	for (const dep of uniqueDependencies.values()) plugin.node(dep);
	for (const node of allNodes) plugin.node(node);
};

//#endregion
//#region src/plugins/@hey-api/sdk/plugin.ts
const handler$9 = (args) => handlerV1$2(args);

//#endregion
//#region src/plugins/@hey-api/sdk/config.ts
const defaultConfig$13 = {
	config: {
		auth: true,
		client: true,
		comments: true,
		includeInEntry: true,
		paramsStructure: "grouped",
		responseStyle: "fields",
		transformer: false,
		validator: false,
		response: "body"
	},
	dependencies: ["@hey-api/typescript"],
	handler: handler$9,
	name: "@hey-api/sdk",
	resolveConfig: (plugin, context) => {
		if (plugin.config.client) {
			if (typeof plugin.config.client === "boolean") plugin.config.client = context.pluginByTag("client", { defaultPlugin: "@hey-api/client-fetch" });
			plugin.dependencies.add(plugin.config.client);
		} else plugin.config.client = false;
		if (plugin.config.transformer) {
			if (typeof plugin.config.transformer === "boolean") plugin.config.transformer = context.pluginByTag("transformer");
			plugin.dependencies.add(plugin.config.transformer);
		} else plugin.config.transformer = false;
		if (typeof plugin.config.validator !== "object") plugin.config.validator = {
			request: plugin.config.validator,
			response: plugin.config.validator
		};
		if (plugin.config.validator.request) {
			if (typeof plugin.config.validator.request === "boolean") plugin.config.validator.request = context.pluginByTag("validator");
			plugin.dependencies.add(plugin.config.validator.request);
		} else plugin.config.validator.request = false;
		if (plugin.config.validator.response) {
			if (typeof plugin.config.validator.response === "boolean") plugin.config.validator.response = context.pluginByTag("validator");
			plugin.dependencies.add(plugin.config.validator.response);
		} else plugin.config.validator.response = false;
		plugin.config.examples = resolveExamples(plugin.config, context);
		plugin.config.operations = resolveOperations(plugin.config, context);
	}
};
/**
* Type helper for `@hey-api/sdk` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$13 = definePluginConfig(defaultConfig$13);

//#endregion
//#region src/plugins/@hey-api/transformers/expressions.ts
const bigIntExpressions = ({ dataExpression, schema }) => {
	if (schema.type !== "integer" || schema.format !== "int64") return;
	const bigIntCallExpression = dataExpression !== void 0 ? $("BigInt").call($.expr(dataExpression).attr("toString").call()) : void 0;
	if (bigIntCallExpression) {
		if (typeof dataExpression === "string") return [bigIntCallExpression];
		if (dataExpression) return [$.expr(dataExpression).assign(bigIntCallExpression)];
	}
};
const dateExpressions = ({ dataExpression, schema }) => {
	if (schema.type !== "string" || !(schema.format === "date" || schema.format === "date-time")) return;
	if (typeof dataExpression === "string") return [$.new("Date").arg(dataExpression)];
	if (dataExpression) return [$.expr(dataExpression).assign($.new("Date").arg(dataExpression))];
};

//#endregion
//#region src/plugins/@hey-api/transformers/plugin.ts
const dataVariableName = "data";
const buildingSymbols = /* @__PURE__ */ new Set();
const isNodeReturnStatement = (node) => node["~dsl"] === "ReturnTsDsl";
const schemaResponseTransformerNodes = ({ plugin, schema }) => {
	const nodes = processSchemaType({
		dataExpression: $(dataVariableName),
		plugin,
		schema
	});
	if (nodes.length) {
		const last = nodes[nodes.length - 1];
		if (!isNodeReturnStatement(last)) nodes.push($.return(dataVariableName));
	}
	return nodes;
};
const processSchemaType = ({ dataExpression, plugin, schema }) => {
	if (schema.$ref) {
		const query = {
			category: "transform",
			resource: "definition",
			resourceId: schema.$ref
		};
		const symbol = plugin.getSymbol(query) ?? plugin.symbol(applyNaming(refToName(schema.$ref), {
			case: "camelCase",
			name: "{{name}}SchemaResponseTransformer"
		}), { meta: query });
		if (!symbol.node && !buildingSymbols.has(symbol.id)) {
			buildingSymbols.add(symbol.id);
			try {
				const nodes = schemaResponseTransformerNodes({
					plugin,
					schema: plugin.context.resolveIrRef(schema.$ref)
				});
				if (nodes.length) {
					const node = $.const(symbol).assign($.func().param(dataVariableName, (p) => p.type("any")).do(...nodes));
					plugin.node(node);
				}
			} finally {
				buildingSymbols.delete(symbol.id);
			}
		}
		if (symbol.node || buildingSymbols.has(symbol.id)) {
			const callExpression = $(plugin.referenceSymbol(query)).call(dataExpression);
			if (dataExpression) {
				if (typeof dataExpression === "string" && dataExpression === "item") return [$.return(callExpression)];
				return [typeof dataExpression === "string" ? callExpression : $(dataExpression).assign(callExpression)];
			}
		}
		return [];
	}
	if (schema.type === "array") {
		if (!dataExpression || typeof dataExpression === "string") return [];
		const nodes = !schema.items ? [] : processSchemaType({
			dataExpression: "item",
			plugin,
			schema: schema.items?.[0] ? schema.items[0] : {
				...schema,
				type: void 0
			}
		});
		if (!nodes.length) return [];
		const mapCallbackStatements = nodes;
		if (!mapCallbackStatements.some((stmt) => isNodeReturnStatement(stmt))) mapCallbackStatements.push($.return("item"));
		return [$(dataExpression).assign($(dataExpression).attr("map").call($.func().param("item", (p) => p.type("any")).do(...mapCallbackStatements)))];
	}
	if (schema.type === "object") {
		let nodes = [];
		const required = schema.required ?? [];
		for (const name in schema.properties) {
			const property = schema.properties[name];
			const propertyAccessExpression = $(dataExpression || dataVariableName).attr(name);
			const propertyNodes = processSchemaType({
				dataExpression: propertyAccessExpression,
				plugin,
				schema: property
			});
			if (!propertyNodes.length) continue;
			const noNullableTypesInSchema = !property.items?.find((x) => x.type === "null");
			if (required.includes(name) && noNullableTypesInSchema) nodes = nodes.concat(propertyNodes);
			else nodes.push($.if(propertyAccessExpression).do(...propertyNodes));
		}
		return nodes;
	}
	if (schema.items) {
		if (schema.items.length === 1) return processSchemaType({
			dataExpression: "item",
			plugin,
			schema: schema.items[0]
		});
		let arrayNodes = [];
		if (schema.logicalOperator === "and" || schema.items.length === 2 && schema.items.find((item) => item.type === "null" || item.type === "void")) {
			for (const item of schema.items) {
				const nodes = processSchemaType({
					dataExpression: dataExpression || "item",
					plugin,
					schema: item
				});
				if (nodes.length) if (dataExpression) arrayNodes = arrayNodes.concat(nodes);
				else arrayNodes.push($.if("item").do(...nodes), $.return("item"));
			}
			return arrayNodes;
		}
		if (schema.type !== "enum") {
			const hasSimpleTypes = (schema.items ?? []).every((item) => [
				"boolean",
				"integer",
				"null",
				"number",
				"string"
			].includes(item.type));
			const isProcessable = (schema.items ?? []).every((item) => {
				if (item.$ref) return true;
				if (item.logicalOperator === "and" && item.items) return true;
				return false;
			});
			if (!hasSimpleTypes && !isProcessable) console.warn(`❗️ Transformers warning: schema ${JSON.stringify(schema)} is too complex and won't be currently processed. This will likely produce an incomplete transformer which is not what you want. Please open an issue if you'd like this improved https://github.com/hey-api/openapi-ts/issues`);
		}
	}
	for (const transformer of plugin.config.transformers) {
		const t = transformer({
			config: plugin.config,
			dataExpression,
			schema
		});
		if (t) return t;
	}
	return [];
};
const handler$8 = ({ plugin }) => {
	plugin.forEach("operation", ({ operation }) => {
		const { response } = operationResponsesMap(operation);
		if (!response) return;
		if (response.items && response.items.length > 1 && response.logicalOperator !== "and") {
			if (plugin.context.config.logs.level === "debug") console.warn(`❗️ Transformers warning: route ${createOperationKey(operation)} has ${response.items.length} non-void success responses. This is currently not handled and we will not generate a response transformer. Please open an issue if you'd like this feature https://github.com/hey-api/openapi-ts/issues`);
			return;
		}
		const symbolResponse = plugin.querySymbol({
			category: "type",
			resource: "operation",
			resourceId: operation.id,
			role: "response"
		});
		if (!symbolResponse) return;
		const nodes = schemaResponseTransformerNodes({
			plugin,
			schema: response
		});
		if (!nodes.length) return;
		const symbol = plugin.symbol(applyNaming(operation.id, {
			case: "camelCase",
			name: "{{name}}ResponseTransformer"
		}), { meta: {
			category: "transform",
			resource: "operation",
			resourceId: operation.id,
			role: "response"
		} });
		const value = $.const(symbol).export().assign($.func().async().param(dataVariableName, (p) => p.type("any")).returns($.type("Promise").generic(symbolResponse)).do(...nodes));
		plugin.node(value);
	}, { order: "declarations" });
};

//#endregion
//#region src/plugins/@hey-api/transformers/config.ts
const defaultConfig$12 = {
	config: {
		bigInt: true,
		dates: true,
		includeInEntry: false,
		transformers: [],
		typeTransformers: []
	},
	dependencies: ["@hey-api/typescript"],
	handler: handler$8,
	name: "@hey-api/transformers",
	resolveConfig: (plugin) => {
		if (!plugin.config.transformers) plugin.config.transformers = [];
		if (plugin.config.dates) plugin.config.transformers = [...plugin.config.transformers, dateExpressions];
		if (plugin.config.bigInt) plugin.config.transformers = [...plugin.config.transformers, bigIntExpressions];
	},
	tags: ["transformer"]
};
/**
* Type helper for `@hey-api/transformers`, returns {@link Plugin.Config} object
*/
const defineConfig$12 = definePluginConfig(defaultConfig$12);

//#endregion
//#region src/plugins/@hey-api/typescript/shared/clientOptions.ts
const serverToBaseUrlType = ({ server }) => {
	const url = parseUrl(server.url);
	if (url.protocol && url.host) return $.type.literal(server.url);
	return $.type.template().add(url.protocol || $.type("string")).add("://").add(url.host || $.type("string")).add(url.port ? `:${url.port}` : "").add(url.path || "");
};
const createClientOptions = ({ nodeIndex, plugin, servers }) => {
	const client = getClientPlugin(getTypedConfig(plugin));
	const types = servers.map((server) => serverToBaseUrlType({ server }));
	if (!servers.length) types.push($.type("string"));
	else if (!("strictBaseUrl" in client.config && client.config.strictBaseUrl)) types.push($.type.and($.type("string"), $.type.object()));
	const symbol = plugin.symbol(applyNaming("ClientOptions", { case: plugin.config.case }), { meta: {
		category: "type",
		resource: "client",
		role: "options",
		tool: "typescript"
	} });
	const node = $.type.alias(symbol).export().type($.type.object().prop(getClientBaseUrlKey(getTypedConfig(plugin)), (p) => p.type($.type.or(...types))));
	plugin.node(node, nodeIndex);
};

//#endregion
//#region src/plugins/shared/utils/schema.ts
const createSchemaComment = (schema) => {
	const comments = [];
	if (schema.title) comments.push(escapeComment(schema.title));
	if (schema.description) {
		if (comments.length) comments.push("");
		comments.push(escapeComment(schema.description));
	}
	if (schema.deprecated) {
		if (comments.length) comments.push("");
		comments.push("@deprecated");
	}
	return comments.length ? comments : void 0;
};

//#endregion
//#region src/plugins/@hey-api/typescript/shared/export.ts
const schemaToEnumObject = ({ plugin, schema }) => {
	const keyCounts = {};
	const typeofItems = [];
	return {
		obj: (schema.items ?? []).map((item, index) => {
			const typeOfItemConst = typeof item.const;
			if (!typeofItems.includes(typeOfItemConst)) typeofItems.push(typeOfItemConst);
			let key;
			if (item.title) key = item.title;
			else if (typeOfItemConst === "number" || typeOfItemConst === "string") key = `${item.const}`;
			else if (typeOfItemConst === "boolean") key = item.const ? "true" : "false";
			else if (item.const === null) key = "null";
			else key = `${index}`;
			if (key) {
				key = toCase(key, plugin.config.enums.case, { stripLeadingSeparators: false });
				regexp.number.lastIndex = 0;
				if (regexp.number.test(key) && plugin.config.enums.enabled && (plugin.config.enums.mode === "typescript" || plugin.config.enums.mode === "typescript-const")) key = `_${key}`;
				const keyCount = (keyCounts[key] ?? 0) + 1;
				keyCounts[key] = keyCount;
				if (keyCount > 1) {
					const nameConflictResolver = plugin.context.config.output?.nameConflictResolver;
					if (nameConflictResolver) {
						const resolvedName = nameConflictResolver({
							attempt: keyCount - 1,
							baseName: key
						});
						if (resolvedName !== null) key = resolvedName;
						else key = `${key}${keyCount}`;
					} else key = `${key}${keyCount}`;
				}
			}
			return {
				key,
				schema: item
			};
		}),
		typeofItems
	};
};
const exportType = ({ plugin, schema, state, type }) => {
	const $ref = pathToJsonPointer(fromRef(state.path));
	if (schema.type === "enum" && plugin.config.enums.enabled) {
		const enumObject = schemaToEnumObject({
			plugin,
			schema
		});
		if (plugin.config.enums.mode === "javascript") {
			if (plugin.config.enums.constantsIgnoreNull && enumObject.typeofItems.includes("object")) enumObject.obj = enumObject.obj.filter((item) => item.schema.const !== null);
			const symbolObject = plugin.symbol(applyNaming(refToName($ref), plugin.config.definitions), { meta: {
				category: "utility",
				path: fromRef(state.path),
				resource: "definition",
				resourceId: $ref,
				tags: fromRef(state.tags),
				tool: "typescript"
			} });
			const objectNode = $.const(symbolObject).export().$if(plugin.config.comments && createSchemaComment(schema), (c, v) => c.doc(v)).assign($.object(...enumObject.obj.map((item) => $.prop({
				kind: "prop",
				name: item.key
			}).$if(plugin.config.comments && createSchemaComment(item.schema), (p, v) => p.doc(v)).value($.fromValue(item.schema.const)))).as("const"));
			plugin.node(objectNode);
			const symbol$1 = plugin.symbol(applyNaming(refToName($ref), plugin.config.definitions), { meta: {
				category: "type",
				path: fromRef(state.path),
				resource: "definition",
				resourceId: $ref,
				tags: fromRef(state.tags),
				tool: "typescript"
			} });
			const node$1 = $.type.alias(symbol$1).export().$if(plugin.config.comments && createSchemaComment(schema), (t, v) => t.doc(v)).type($.type(symbol$1).idx($.type(symbol$1).typeofType().keyof()).typeofType());
			plugin.node(node$1);
			return;
		} else if (plugin.config.enums.mode === "typescript" || plugin.config.enums.mode === "typescript-const") {
			if (!enumObject.typeofItems.some((type$1) => type$1 !== "number" && type$1 !== "string")) {
				const symbol$1 = plugin.symbol(applyNaming(refToName($ref), plugin.config.definitions), { meta: {
					category: "type",
					path: fromRef(state.path),
					resource: "definition",
					resourceId: $ref,
					tags: fromRef(state.tags),
					tool: "typescript"
				} });
				const enumNode = $.enum(symbol$1).export().$if(plugin.config.comments && createSchemaComment(schema), (e, v) => e.doc(v)).const(plugin.config.enums.mode === "typescript-const").members(...enumObject.obj.map((item) => $.member(item.key).$if(plugin.config.comments && createSchemaComment(item.schema), (m, v) => m.doc(v)).value($.fromValue(item.schema.const))));
				plugin.node(enumNode);
				return;
			}
		}
	}
	const symbol = plugin.symbol(applyNaming(refToName($ref), plugin.config.definitions), { meta: {
		category: "type",
		path: fromRef(state.path),
		resource: "definition",
		resourceId: $ref,
		tags: fromRef(state.tags),
		tool: "typescript"
	} });
	const node = $.type.alias(symbol).export().$if(plugin.config.comments && createSchemaComment(schema), (t, v) => t.doc(v)).type(type);
	plugin.node(node);
};

//#endregion
//#region src/plugins/@hey-api/typescript/shared/operation.ts
const irParametersToIrSchema = ({ parameters }) => {
	const irSchema = { type: "object" };
	if (parameters) {
		const properties = {};
		const required = [];
		for (const key in parameters) {
			const parameter = parameters[key];
			properties[parameter.name] = deduplicateSchema({
				detectFormat: false,
				schema: parameter.schema
			});
			if (parameter.required) required.push(parameter.name);
		}
		irSchema.properties = properties;
		if (required.length) irSchema.required = required;
	}
	return irSchema;
};
const operationToDataType$1 = ({ operation, plugin, state }) => {
	const data = { type: "object" };
	const dataRequired = [];
	if (!data.properties) data.properties = {};
	if (operation.body) {
		data.properties.body = operation.body.schema;
		if (operation.body.required) dataRequired.push("body");
	} else data.properties.body = { type: "never" };
	if (operation.parameters?.header) {
		data.properties.headers = irParametersToIrSchema({ parameters: operation.parameters.header });
		if (data.properties.headers.required) dataRequired.push("headers");
	}
	if (operation.parameters?.path) {
		data.properties.path = irParametersToIrSchema({ parameters: operation.parameters.path });
		if (data.properties.path.required) dataRequired.push("path");
	} else data.properties.path = { type: "never" };
	if (operation.parameters?.query) {
		data.properties.query = irParametersToIrSchema({ parameters: operation.parameters.query });
		if (data.properties.query.required) dataRequired.push("query");
	} else data.properties.query = { type: "never" };
	data.properties.url = {
		const: operation.path,
		type: "string"
	};
	dataRequired.push("url");
	data.required = dataRequired;
	const symbol = plugin.symbol(applyNaming(operation.id, plugin.config.requests), { meta: {
		category: "type",
		path: fromRef(state.path),
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tags: fromRef(state.tags),
		tool: "typescript"
	} });
	const node = $.type.alias(symbol).export().type(irSchemaToAst$5({
		plugin,
		schema: data,
		state
	}));
	plugin.node(node);
};
const operationToType = ({ operation, plugin, state }) => {
	operationToDataType$1({
		operation,
		plugin,
		state
	});
	const { error, errors, response, responses } = operationResponsesMap(operation);
	if (errors) {
		const symbolErrors = plugin.symbol(applyNaming(operation.id, plugin.config.errors), { meta: {
			category: "type",
			path: fromRef(state.path),
			resource: "operation",
			resourceId: operation.id,
			role: "errors",
			tags: fromRef(state.tags),
			tool: "typescript"
		} });
		const node = $.type.alias(symbolErrors).export().type(irSchemaToAst$5({
			plugin,
			schema: errors,
			state
		}));
		plugin.node(node);
		if (error) {
			const symbol = plugin.symbol(applyNaming(operation.id, {
				case: plugin.config.errors.case,
				name: plugin.config.errors.error
			}), { meta: {
				category: "type",
				path: fromRef(state.path),
				resource: "operation",
				resourceId: operation.id,
				role: "error",
				tags: fromRef(state.tags),
				tool: "typescript"
			} });
			const node$1 = $.type.alias(symbol).export().type($.type(symbolErrors).idx($.type(symbolErrors).keyof()));
			plugin.node(node$1);
		}
	}
	if (responses) {
		const symbolResponses = plugin.symbol(applyNaming(operation.id, plugin.config.responses), { meta: {
			category: "type",
			path: fromRef(state.path),
			resource: "operation",
			resourceId: operation.id,
			role: "responses",
			tags: fromRef(state.tags),
			tool: "typescript"
		} });
		const node = $.type.alias(symbolResponses).export().type(irSchemaToAst$5({
			plugin,
			schema: responses,
			state
		}));
		plugin.node(node);
		if (response) {
			const symbol = plugin.symbol(applyNaming(operation.id, {
				case: plugin.config.responses.case,
				name: plugin.config.responses.response
			}), { meta: {
				category: "type",
				path: fromRef(state.path),
				resource: "operation",
				resourceId: operation.id,
				role: "response",
				tags: fromRef(state.tags),
				tool: "typescript"
			} });
			const node$1 = $.type.alias(symbol).export().type($.type(symbolResponses).idx($.type(symbolResponses).keyof()));
			plugin.node(node$1);
		}
	}
};

//#endregion
//#region src/plugins/@hey-api/typescript/shared/webhook.ts
const operationToDataType = ({ operation, plugin, state }) => {
	const data = { type: "object" };
	const dataRequired = [];
	if (!data.properties) data.properties = {};
	if (operation.body) {
		const symbolWebhookPayload = plugin.symbol(applyNaming(operation.id, {
			case: plugin.config.webhooks.case,
			name: plugin.config.webhooks.payload
		}), { meta: {
			category: "type",
			path: fromRef(state.path),
			resource: "webhook",
			resourceId: operation.id,
			role: "data",
			tags: fromRef(state.tags),
			tool: "typescript"
		} });
		const node$1 = $.type.alias(symbolWebhookPayload).export().$if(plugin.config.comments && createSchemaComment(operation.body.schema), (t, v) => t.doc(v)).type(irSchemaToAst$5({
			plugin,
			schema: operation.body.schema,
			state
		}));
		plugin.node(node$1);
		data.properties.body = { symbolRef: symbolWebhookPayload };
		dataRequired.push("body");
	} else data.properties.body = { type: "never" };
	data.properties.key = {
		const: operation.path,
		type: "string"
	};
	dataRequired.push("key");
	data.properties.path = { type: "never" };
	data.properties.query = { type: "never" };
	data.required = dataRequired;
	const symbolWebhookRequest = plugin.symbol(applyNaming(operation.id, plugin.config.webhooks), { meta: {
		category: "type",
		path: fromRef(state.path),
		resource: "webhook",
		resourceId: operation.id,
		role: "data",
		tags: fromRef(state.tags),
		tool: "typescript"
	} });
	const node = $.type.alias(symbolWebhookRequest).export().type(irSchemaToAst$5({
		plugin,
		schema: data,
		state
	}));
	plugin.node(node);
	return symbolWebhookRequest;
};
const webhookToType = ({ operation, plugin, state }) => {
	return operationToDataType({
		operation,
		plugin,
		state
	});
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/array.ts
const arrayToAst$4 = ({ plugin, schema, state }) => {
	if (!schema.items) return $.type("Array").generic($.type(plugin.config.topType));
	schema = deduplicateSchema({
		detectFormat: true,
		schema
	});
	const itemTypes = [];
	if (schema.items) schema.items.forEach((item, index) => {
		const type = irSchemaToAst$5({
			plugin,
			schema: item,
			state: {
				...state,
				path: ref([
					...fromRef(state.path),
					"items",
					index
				])
			}
		});
		itemTypes.push(type);
	});
	if (itemTypes.length === 1) return $.type("Array").generic(itemTypes[0]);
	return schema.logicalOperator === "and" ? $.type("Array").generic($.type.and(...itemTypes)) : $.type("Array").generic($.type.or(...itemTypes));
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/boolean.ts
function booleanToAst$4({ schema }) {
	if (schema.const !== void 0) return $.type.literal(schema.const);
	return $.type("boolean");
}

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/enum.ts
const enumToAst$4 = ({ plugin, schema, state }) => {
	return irSchemaToAst$5({
		plugin,
		schema: {
			...schema,
			type: void 0
		},
		state
	});
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/never.ts
const neverToAst$4 = (_args) => {
	return $.type("never");
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/null.ts
const nullToAst$5 = (_args) => {
	return $.type.literal(null);
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/number.ts
const numberToAst = ({ plugin, schema }) => {
	if (schema.const !== void 0) return $.type.literal(schema.const);
	if (schema.type === "integer" && schema.format === "int64") {
		if (plugin.getPlugin("@hey-api/transformers")?.config.bigInt) return $.type("bigint");
	}
	return $.type("number");
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/object.ts
const objectToAst$5 = ({ plugin, schema, state }) => {
	const shape = $.type.object();
	const required = schema.required ?? [];
	let indexSchemas = [];
	let hasOptionalProperties = false;
	for (const name in schema.properties) {
		const property = schema.properties[name];
		const propertyType = irSchemaToAst$5({
			plugin,
			schema: property,
			state: {
				...state,
				path: ref([
					...fromRef(state.path),
					"properties",
					name
				])
			}
		});
		const isRequired = required.includes(name);
		shape.prop(name, (p) => p.$if(plugin.config.comments && createSchemaComment(property), (p$1, v) => p$1.doc(v)).readonly(property.accessScope === "read").required(isRequired).type(propertyType));
		indexSchemas.push(property);
		if (!isRequired) hasOptionalProperties = true;
	}
	if (schema.patternProperties) for (const pattern in schema.patternProperties) {
		const ir = schema.patternProperties[pattern];
		indexSchemas.unshift(ir);
	}
	const hasPatterns = !!schema.patternProperties && Object.keys(schema.patternProperties).length > 0;
	const addPropsRaw = schema.additionalProperties;
	const addPropsObj = addPropsRaw !== false && addPropsRaw ? addPropsRaw : void 0;
	if (hasPatterns || !!addPropsObj && (addPropsObj.type !== "never" || !indexSchemas.length)) {
		const addProps = addPropsObj;
		if (addProps && addProps.type !== "never") indexSchemas.unshift(addProps);
		else if (!hasPatterns && !indexSchemas.length && addProps && addProps.type === "never") indexSchemas = [addProps];
		if (hasOptionalProperties) indexSchemas.push({ type: "undefined" });
		const type = indexSchemas.length === 1 ? irSchemaToAst$5({
			plugin,
			schema: indexSchemas[0],
			state
		}) : irSchemaToAst$5({
			plugin,
			schema: {
				items: indexSchemas,
				logicalOperator: "or"
			},
			state
		});
		if (schema.propertyNames?.$ref) return $.type.mapped("key").key(irSchemaToAst$5({
			plugin,
			schema: { $ref: schema.propertyNames.$ref },
			state
		})).optional().type(type);
		shape.idxSig("key", (i) => i.key("string").type(type));
	}
	return shape;
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/string.ts
const stringToAst$1 = ({ plugin, schema }) => {
	if (schema.const !== void 0) return $.type.literal(schema.const);
	if (schema.format) {
		if (schema.format === "binary") return $.type.or($.type("Blob"), $.type("File"));
		if (schema.format === "date-time" || schema.format === "date") {
			if (plugin.getPlugin("@hey-api/transformers")?.config.dates) return $.type("Date");
		}
		if (schema.format === "typeid" && typeof schema.example === "string") {
			const parts = String(schema.example).split("_");
			parts.pop();
			const type = parts.join("_");
			const query = {
				category: "type",
				resource: "type-id",
				resourceId: type,
				tool: "typescript"
			};
			if (!plugin.getSymbol(query)) {
				const queryTypeId = {
					category: "type",
					resource: "type-id",
					tool: "typescript",
					variant: "container"
				};
				if (!plugin.getSymbol(queryTypeId)) {
					const symbolTypeId$1 = plugin.symbol("TypeID", { meta: queryTypeId });
					const nodeTypeId = $.type.alias(symbolTypeId$1).export().generic("T", (g) => g.extends("string")).type($.type.template().add($.type("T")).add("_").add($.type("string")));
					plugin.node(nodeTypeId);
				}
				const symbolTypeId = plugin.referenceSymbol(queryTypeId);
				const symbolTypeName = plugin.symbol(toCase(`${type}_id`, plugin.config.case), { meta: query });
				const node = $.type.alias(symbolTypeName).export().type($.type(symbolTypeId).generic($.type.literal(type)));
				plugin.node(node);
			}
			const symbol = plugin.referenceSymbol(query);
			return $.type(symbol);
		}
	}
	return $.type("string");
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/tuple.ts
const tupleToAst$4 = ({ plugin, schema, state }) => {
	let itemTypes = [];
	if (schema.const && Array.isArray(schema.const)) itemTypes = schema.const.map((value) => $.type.fromValue(value));
	else if (schema.items) schema.items.forEach((item, index) => {
		const type = irSchemaToAst$5({
			plugin,
			schema: item,
			state: {
				...state,
				path: ref([
					...fromRef(state.path),
					"items",
					index
				])
			}
		});
		itemTypes.push(type);
	});
	return $.type.tuple(...itemTypes);
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/undefined.ts
const undefinedToAst$4 = (_args) => {
	return $.type("undefined");
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/unknown.ts
const unknownToAst$4 = ({ plugin }) => {
	return $.type(plugin.config.topType);
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/void.ts
const voidToAst$4 = (_args) => {
	return $.type("void");
};

//#endregion
//#region src/plugins/@hey-api/typescript/v1/toAst/index.ts
function irSchemaWithTypeToAst$5({ schema, ...args }) {
	const transformersPlugin = args.plugin.getPlugin("@hey-api/transformers");
	if (transformersPlugin?.config.typeTransformers) for (const typeTransformer of transformersPlugin.config.typeTransformers) {
		const typeNode = typeTransformer({ schema });
		if (typeNode) return typeNode;
	}
	switch (schema.type) {
		case "array": return arrayToAst$4({
			...args,
			schema
		});
		case "boolean": return booleanToAst$4({
			...args,
			schema
		});
		case "enum": return enumToAst$4({
			...args,
			schema
		});
		case "integer":
		case "number": return numberToAst({
			...args,
			schema
		});
		case "never": return neverToAst$4({
			...args,
			schema
		});
		case "null": return nullToAst$5({
			...args,
			schema
		});
		case "object": return objectToAst$5({
			...args,
			schema
		});
		case "string": return stringToAst$1({
			...args,
			schema
		});
		case "tuple": return tupleToAst$4({
			...args,
			schema
		});
		case "undefined": return undefinedToAst$4({
			...args,
			schema
		});
		case "unknown": return unknownToAst$4({
			...args,
			schema
		});
		case "void": return voidToAst$4({
			...args,
			schema
		});
	}
}

//#endregion
//#region src/plugins/@hey-api/typescript/v1/plugin.ts
function irSchemaToAst$5({ plugin, schema, schemaExtractor, state }) {
	if (schemaExtractor && !schema.$ref) {
		const extracted = schemaExtractor({
			meta: {
				resource: "definition",
				resourceId: pathToJsonPointer(fromRef(state.path))
			},
			path: fromRef(state.path),
			schema
		});
		if (extracted !== schema) schema = extracted;
	}
	if (schema.symbolRef) {
		const baseType = $.type(schema.symbolRef);
		if (schema.omit && schema.omit.length > 0) {
			const omittedKeys = schema.omit.length === 1 ? $.type.literal(schema.omit[0]) : $.type.or(...schema.omit.map((key) => $.type.literal(key)));
			return $.type("Omit").generics(baseType, omittedKeys);
		}
		return baseType;
	}
	if (schema.$ref) {
		const symbol = plugin.referenceSymbol({
			category: "type",
			resource: "definition",
			resourceId: schema.$ref
		});
		const baseType = $.type(symbol);
		if (schema.omit && schema.omit.length > 0) {
			const omittedKeys = schema.omit.length === 1 ? $.type.literal(schema.omit[0]) : $.type.or(...schema.omit.map((key) => $.type.literal(key)));
			return $.type("Omit").generics(baseType, omittedKeys);
		}
		return baseType;
	}
	if (schema.type) return irSchemaWithTypeToAst$5({
		plugin,
		schema,
		state
	});
	if (schema.items) {
		schema = deduplicateSchema({
			detectFormat: false,
			schema
		});
		if (schema.items) {
			const itemTypes = schema.items.map((item) => irSchemaToAst$5({
				plugin,
				schema: item,
				state
			}));
			return schema.logicalOperator === "and" ? $.type.and(...itemTypes) : $.type.or(...itemTypes);
		}
		return irSchemaToAst$5({
			plugin,
			schema,
			state
		});
	}
	return irSchemaWithTypeToAst$5({
		plugin,
		schema: { type: "unknown" },
		state
	});
}
function handleComponent$1({ plugin, schema, state }) {
	exportType({
		plugin,
		schema,
		state,
		type: irSchemaToAst$5({
			plugin,
			schema,
			state
		})
	});
}
const handlerV1$1 = ({ plugin }) => {
	const nodeClientIndex = plugin.node(null);
	const nodeWebhooksIndex = plugin.node(null);
	const servers = [];
	const webhooks = [];
	plugin.forEach("operation", "parameter", "requestBody", "schema", "server", "webhook", (event) => {
		const state = refs({
			path: event._path,
			tags: event.tags
		});
		switch (event.type) {
			case "operation":
				operationToType({
					operation: event.operation,
					plugin,
					state
				});
				break;
			case "parameter":
				handleComponent$1({
					plugin,
					schema: event.parameter.schema,
					state
				});
				break;
			case "requestBody":
				handleComponent$1({
					plugin,
					schema: event.requestBody.schema,
					state
				});
				break;
			case "schema":
				handleComponent$1({
					plugin,
					schema: event.schema,
					state
				});
				break;
			case "server":
				servers.push(event.server);
				break;
			case "webhook":
				webhooks.push(webhookToType({
					operation: event.operation,
					plugin,
					state
				}));
				break;
		}
	}, { order: "declarations" });
	createClientOptions({
		nodeIndex: nodeClientIndex,
		plugin,
		servers
	});
	if (webhooks.length > 0) {
		const symbol = plugin.symbol(applyNaming("Webhooks", { case: plugin.config.case }), { meta: {
			category: "type",
			resource: "webhook",
			tool: "typescript",
			variant: "container"
		} });
		const node = $.type.alias(symbol).export().type($.type.or(...webhooks));
		plugin.node(node, nodeWebhooksIndex);
	}
};

//#endregion
//#region src/plugins/@hey-api/typescript/api.ts
var Api$3 = class {
	schemaToType(args) {
		return irSchemaToAst$5(args);
	}
};

//#endregion
//#region src/plugins/@hey-api/typescript/plugin.ts
const handler$7 = (args) => handlerV1$1(args);

//#endregion
//#region src/plugins/@hey-api/typescript/config.ts
const defaultConfig$11 = {
	api: new Api$3(),
	config: {
		case: "PascalCase",
		comments: true,
		includeInEntry: true,
		topType: "unknown"
	},
	handler: handler$7,
	name: "@hey-api/typescript",
	resolveConfig: (plugin, context) => {
		plugin.config.definitions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				name: "{{name}}"
			},
			mappers: {
				function: (name) => ({ name }),
				string: (name) => ({ name })
			},
			value: plugin.config.definitions
		});
		plugin.config.enums = context.valueToObject({
			defaultValue: {
				case: "SCREAMING_SNAKE_CASE",
				constantsIgnoreNull: false,
				enabled: Boolean(plugin.config.enums),
				mode: "javascript"
			},
			mappers: {
				boolean: (enabled) => ({ enabled }),
				string: (mode) => ({ mode })
			},
			value: plugin.config.enums
		});
		plugin.config.errors = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				error: "{{name}}Error",
				name: "{{name}}Errors"
			},
			mappers: {
				function: (name) => ({ name }),
				string: (name) => ({ name })
			},
			value: plugin.config.errors
		});
		plugin.config.requests = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				name: "{{name}}Data"
			},
			mappers: {
				function: (name) => ({ name }),
				string: (name) => ({ name })
			},
			value: plugin.config.requests
		});
		plugin.config.responses = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				name: "{{name}}Responses",
				response: "{{name}}Response"
			},
			mappers: {
				function: (name) => ({ name }),
				string: (name) => ({ name })
			},
			value: plugin.config.responses
		});
		plugin.config.webhooks = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				name: "{{name}}WebhookRequest",
				payload: "{{name}}WebhookPayload"
			},
			mappers: {
				function: (name) => ({ name }),
				string: (name) => ({ name })
			},
			value: plugin.config.webhooks
		});
	}
};
/**
* Type helper for `@hey-api/typescript` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$11 = definePluginConfig(defaultConfig$11);

//#endregion
//#region src/plugins/@pinia/colada/meta.ts
const handleMeta$1 = (plugin, operation, configPath) => {
	const metaFn = plugin.config[configPath].meta;
	if (!metaFn) return;
	const metaObject = metaFn(operation);
	if (!Object.keys(metaObject).length) return;
	return $.fromValue(metaObject);
};

//#endregion
//#region src/plugins/@pinia/colada/useType.ts
const useTypeData$1 = ({ operation, plugin }) => {
	return operationOptionsType({
		operation,
		plugin: plugin.getPluginOrThrow("@hey-api/sdk")
	});
};
const useTypeError$1 = ({ operation, plugin }) => {
	const client = getClientPlugin(getTypedConfig(plugin));
	const symbolError = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "error"
	}) || "Error";
	if (client.name === "@hey-api/client-axios") {
		const symbol = plugin.external("axios.AxiosError");
		return $.type(symbol).generic(symbolError);
	}
	return $.type(symbolError);
};
const useTypeResponse$1 = ({ operation, plugin }) => {
	const symbolResponseType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "response"
	});
	return $.type(symbolResponseType ?? "unknown");
};

//#endregion
//#region src/plugins/@pinia/colada/utils.ts
const getPublicTypeData = ({ isNuxtClient, operation, plugin }) => {
	const typeData = useTypeData$1({
		operation,
		plugin
	});
	return isNuxtClient ? $.type("Omit").generic(typeData).generic("composable") : typeData;
};

//#endregion
//#region src/plugins/@pinia/colada/mutationOptions.ts
const createMutationOptions$1 = ({ operation, plugin }) => {
	const symbolMutationOptionsType = plugin.external(`${plugin.name}.UseMutationOptions`);
	const isNuxtClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-nuxt";
	const typeData = getPublicTypeData({
		isNuxtClient,
		operation,
		plugin
	});
	const options = plugin.symbol("options");
	const fnOptions$1 = plugin.symbol("vars");
	const awaitSdkFn = $.lazy((ctx$1) => ctx$1.access(plugin.referenceSymbol({
		category: "sdk",
		resource: "operation",
		resourceId: operation.id
	})).call($.object().pretty().spread(options).spread(fnOptions$1).prop("throwOnError", $.literal(true))).await());
	const statements = [];
	if (plugin.getPluginOrThrow("@hey-api/sdk").config.responseStyle === "data") statements.push($.return(awaitSdkFn));
	else statements.push($.const().object("data").assign(awaitSdkFn), $.return("data"));
	const mutationOpts = $.object().pretty().prop("mutation", $.func().async().param(fnOptions$1, (p) => p.$if(isNuxtClient, (f$1) => f$1.type($.type("Partial").generic(typeData)))).do(...statements)).$if(handleMeta$1(plugin, operation, "mutationOptions"), (o, v) => o.prop("meta", v));
	const symbolMutationOptions = plugin.symbol(applyNaming(operation.id, plugin.config.mutationOptions));
	const statement = $.const(symbolMutationOptions).export().$if(plugin.config.comments && createOperationComment(operation), (c, v) => c.doc(v)).assign($.func().param(options, (p) => p.optional().type($.type("Partial").generic(typeData))).returns($.type(symbolMutationOptionsType).generic(useTypeResponse$1({
		operation,
		plugin
	})).generic(typeData).generic(useTypeError$1({
		operation,
		plugin
	}))).do($.return(mutationOpts)));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/@pinia/colada/queryKey.ts
const TOptionsType$1 = "TOptions";
const createQueryKeyFunction$1 = ({ plugin }) => {
	const symbolCreateQueryKey = plugin.symbol(applyNaming("createQueryKey", { case: plugin.config.case }), { meta: {
		category: "utility",
		resource: "createQueryKey",
		tool: plugin.name
	} });
	const symbolQueryKeyType = plugin.referenceSymbol({
		category: "type",
		resource: "QueryKey",
		tool: plugin.name
	});
	const symbolJsonValue = plugin.external(`${plugin.name}._JSONValue`);
	const returnType = $.type(symbolQueryKeyType).generic(TOptionsType$1).idx(0);
	const baseUrlKey = getClientBaseUrlKey(getTypedConfig(plugin));
	const symbolOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client-options",
		tool: "sdk"
	});
	const symbolClient = plugin.getSymbol({ category: "client" });
	const clientModule = clientFolderAbsolutePath(getTypedConfig(plugin));
	const symbolSerializeQueryValue = plugin.symbol("serializeQueryKeyValue", {
		external: clientModule,
		meta: {
			category: "external",
			resource: `${clientModule}.serializeQueryKeyValue`
		}
	});
	const fn = $.const(symbolCreateQueryKey).assign($.func().param("id", (p) => p.type("string")).param("options", (p) => p.optional().type(TOptionsType$1)).param("tags", (p) => p.optional().type("ReadonlyArray<string>")).returns($.type.tuple(returnType)).generic(TOptionsType$1, (g) => g.extends(symbolOptions)).do($.const("params").type(returnType).assign($.object().prop("_id", "id").prop(baseUrlKey, $("options").attr(baseUrlKey).optional().or($("options").attr("client").optional().$if(symbolClient, (a, v) => a.coalesce(v)).attr("getConfig").call().attr(baseUrlKey))).as(returnType)), $.if("tags").do($("params").attr("tags").assign($("tags").as("unknown").as(symbolJsonValue))), $.if($("options").attr("body").optional().neq($.id("undefined"))).do($.const("normalizedBody").assign($(symbolSerializeQueryValue).call($("options").attr("body"))), $.if($("normalizedBody").neq($.id("undefined"))).do($("params").attr("body").assign("normalizedBody"))), $.if($("options").attr("path").optional()).do($("params").attr("path").assign($("options").attr("path"))), $.if($("options").attr("query").optional().neq($.id("undefined"))).do($.const("normalizedQuery").assign($(symbolSerializeQueryValue).call($("options").attr("query"))), $.if($("normalizedQuery").neq($.id("undefined"))).do($("params").attr("query").assign("normalizedQuery"))), $.return($.array($("params")))));
	plugin.node(fn);
};
const createQueryKeyLiteral$1 = ({ id, operation, plugin }) => {
	const config = plugin.config.queryKeys;
	let tagsExpression;
	if (config.tags && operation.tags && operation.tags.length > 0) tagsExpression = $.array(...operation.tags.map((tag) => $.literal(tag)));
	return $(plugin.referenceSymbol({
		category: "utility",
		resource: "createQueryKey",
		tool: plugin.name
	})).call($.literal(id), "options", tagsExpression);
};
const createQueryKeyType$1 = ({ plugin }) => {
	const symbolJsonValue = plugin.external(`${plugin.name}._JSONValue`);
	const symbolOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client-options",
		tool: "sdk"
	});
	const symbolQueryKeyType = plugin.symbol("QueryKey", { meta: {
		category: "type",
		resource: "QueryKey",
		tool: plugin.name
	} });
	const queryKeyType = $.type.alias(symbolQueryKeyType).export().generic(TOptionsType$1, (g) => g.extends($.type(symbolOptions))).type($.type.tuple($.type.and($.type(`Pick<${TOptionsType$1}, 'path'>`), $.type.object().prop("_id", (p) => p.type("string")).prop(getClientBaseUrlKey(getTypedConfig(plugin)), (p) => p.optional().type(symbolJsonValue)).prop("body", (p) => p.optional().type(symbolJsonValue)).prop("query", (p) => p.optional().type(symbolJsonValue)).prop("tags", (p) => p.optional().type(symbolJsonValue)))));
	plugin.node(queryKeyType);
};
const queryKeyStatement$1 = ({ operation, plugin, symbol }) => {
	const isNuxtClient = getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-nuxt";
	return $.const(symbol).export().assign($.func().param("options", (p) => p.required(hasOperationDataRequired(operation)).type(getPublicTypeData({
		isNuxtClient,
		operation,
		plugin
	}))).do(createQueryKeyLiteral$1({
		id: operation.id,
		operation,
		plugin
	}).return()));
};

//#endregion
//#region src/plugins/@pinia/colada/queryOptions.ts
const optionsParamName$2 = "options";
const fnOptions = "context";
const createQueryOptions$1 = ({ operation, plugin }) => {
	if (hasOperationSse({ operation })) return;
	const isRequiredOptions = isOperationOptionsRequired({
		context: plugin.context,
		operation
	});
	if (!plugin.getSymbol({
		category: "utility",
		resource: "createQueryKey",
		tool: plugin.name
	})) {
		createQueryKeyType$1({ plugin });
		createQueryKeyFunction$1({ plugin });
	}
	let keyExpression;
	if (plugin.config.queryKeys.enabled) {
		const symbolQueryKey = plugin.symbol(applyNaming(operation.id, plugin.config.queryKeys));
		const node = queryKeyStatement$1({
			operation,
			plugin,
			symbol: symbolQueryKey
		});
		plugin.node(node);
		keyExpression = $(symbolQueryKey).call(optionsParamName$2);
	} else {
		const symbolCreateQueryKey = plugin.referenceSymbol({
			category: "utility",
			resource: "createQueryKey",
			tool: plugin.name
		});
		let tagsExpr;
		if (plugin.config.queryKeys.tags && operation.tags && operation.tags.length > 0) tagsExpr = $.array(...operation.tags.map((t) => $.literal(t)));
		keyExpression = $(symbolCreateQueryKey).call($.literal(operation.id), optionsParamName$2, tagsExpr);
	}
	const typeData = getPublicTypeData({
		isNuxtClient: getClientPlugin(getTypedConfig(plugin)).name === "@hey-api/client-nuxt",
		operation,
		plugin
	});
	const awaitSdkFn = $.lazy((ctx$1) => ctx$1.access(plugin.referenceSymbol({
		category: "sdk",
		resource: "operation",
		resourceId: operation.id
	})).call($.object().spread(optionsParamName$2).spread(fnOptions).prop("throwOnError", $.literal(true))).await());
	const statements = [];
	if (plugin.getPluginOrThrow("@hey-api/sdk").config.responseStyle === "data") statements.push($.return(awaitSdkFn));
	else statements.push($.const().object("data").assign(awaitSdkFn), $.return("data"));
	const queryOpts = $.object().pretty().prop("key", keyExpression).prop("query", $.func().async().param(fnOptions).do(...statements)).$if(handleMeta$1(plugin, operation, "queryOptions"), (o, v) => o.prop("meta", v));
	const symbolQueryOptionsFn = plugin.symbol(applyNaming(operation.id, plugin.config.queryOptions), { meta: {
		category: "hook",
		resource: "operation",
		resourceId: operation.id,
		role: "queryOptions",
		tool: plugin.name
	} });
	const symbolDefineQueryOptions = plugin.external(`${plugin.name}.defineQueryOptions`);
	const statement = $.const(symbolQueryOptionsFn).export().$if(plugin.config.comments && createOperationComment(operation), (c, v) => c.doc(v)).assign($(symbolDefineQueryOptions).call($.func().param(optionsParamName$2, (p) => p.required(isRequiredOptions).type(typeData)).do($.return(queryOpts))));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/@pinia/colada/v0/plugin.ts
const handlerV0 = ({ plugin }) => {
	plugin.symbol("defineQueryOptions", {
		external: plugin.name,
		meta: {
			category: "external",
			resource: `${plugin.name}.defineQueryOptions`
		}
	});
	plugin.symbol("UseMutationOptions", {
		external: plugin.name,
		kind: "type",
		meta: {
			category: "external",
			resource: `${plugin.name}.UseMutationOptions`
		}
	});
	plugin.symbol("UseQueryOptions", {
		external: plugin.name,
		kind: "type",
		meta: {
			category: "external",
			resource: `${plugin.name}.UseQueryOptions`
		}
	});
	plugin.symbol("_JSONValue", {
		external: plugin.name,
		kind: "type",
		meta: {
			category: "external",
			resource: `${plugin.name}._JSONValue`
		}
	});
	plugin.symbol("AxiosError", {
		external: "axios",
		kind: "type",
		meta: {
			category: "external",
			resource: "axios.AxiosError"
		}
	});
	plugin.forEach("operation", ({ operation }) => {
		if (plugin.hooks.operation.isQuery(operation)) {
			if (plugin.config.queryOptions.enabled) createQueryOptions$1({
				operation,
				plugin
			});
		}
		if (plugin.hooks.operation.isMutation(operation)) {
			if (plugin.config.mutationOptions.enabled) createMutationOptions$1({
				operation,
				plugin
			});
		}
	}, { order: "declarations" });
};

//#endregion
//#region src/plugins/@pinia/colada/plugin.ts
const handler$6 = (args) => handlerV0(args);

//#endregion
//#region src/plugins/@pinia/colada/config.ts
const defaultConfig$10 = {
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/typescript", "@hey-api/sdk"],
	handler: handler$6,
	name: "@pinia/colada",
	resolveConfig: (plugin, context) => {
		plugin.config.mutationOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Mutation"
			},
			mappers,
			value: plugin.config.mutationOptions
		});
		plugin.config.queryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}QueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.queryKeys
		});
		plugin.config.queryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Query"
			},
			mappers,
			value: plugin.config.queryOptions
		});
	}
};
/**
* Type helper for `@pinia/colada` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$10 = definePluginConfig(defaultConfig$10);

//#endregion
//#region src/plugins/@tanstack/query-core/shared/useType.ts
const useTypeData = ({ operation, plugin }) => {
	return operationOptionsType({
		operation,
		plugin: plugin.getPluginOrThrow("@hey-api/sdk")
	});
};
const useTypeError = ({ operation, plugin }) => {
	const client = getClientPlugin(getTypedConfig(plugin));
	const symbolError = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "error"
	}) || plugin.external(`${plugin.name}.DefaultError`);
	if (client.name === "@hey-api/client-axios") {
		const symbol = plugin.external("axios.AxiosError");
		return $.type(symbol).generic(symbolError);
	}
	return $.type(symbolError);
};
const useTypeResponse = ({ operation, plugin }) => {
	const symbolResponseType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "response"
	});
	return $.type(symbolResponseType ?? "unknown");
};

//#endregion
//#region src/plugins/@tanstack/query-core/queryKey.ts
const TOptionsType = "TOptions";
const createQueryKeyFunction = ({ plugin }) => {
	const symbolCreateQueryKey = plugin.symbol(applyNaming("createQueryKey", { case: plugin.config.case }), { meta: {
		category: "utility",
		resource: "createQueryKey",
		tool: plugin.name
	} });
	const symbolQueryKeyType = plugin.referenceSymbol({
		category: "type",
		resource: "QueryKey",
		tool: plugin.name
	});
	const baseUrlKey = getClientBaseUrlKey(getTypedConfig(plugin));
	const symbolClient = plugin.getSymbol({ category: "client" });
	const symbolOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client-options",
		tool: "sdk"
	});
	const returnType = $.type(symbolQueryKeyType).generic(TOptionsType).idx(0);
	const fn = $.const(symbolCreateQueryKey).assign($.func().param("id", (p) => p.type("string")).param("options", (p) => p.optional().type(TOptionsType)).param("infinite", (p) => p.optional().type("boolean")).param("tags", (p) => p.optional().type("ReadonlyArray<string>")).generic(TOptionsType, (g) => g.extends(symbolOptions)).returns($.type.tuple(returnType)).do($.const("params").type(returnType).assign($.object().prop("_id", "id").prop(baseUrlKey, $("options").attr(baseUrlKey).optional().or($("options").attr("client").optional().$if(symbolClient, (a, v) => a.coalesce(v)).attr("getConfig").call().attr(baseUrlKey))).as(returnType)), $.if("infinite").do($("params").attr("_infinite").assign("infinite")), $.if("tags").do($("params").attr("tags").assign("tags")), $.if($("options").attr("body").optional()).do($("params").attr("body").assign($("options").attr("body"))), $.if($("options").attr("headers").optional()).do($("params").attr("headers").assign($("options").attr("headers"))), $.if($("options").attr("path").optional()).do($("params").attr("path").assign($("options").attr("path"))), $.if($("options").attr("query").optional()).do($("params").attr("query").assign($("options").attr("query"))), $.return($.array().element($("params")))));
	plugin.node(fn);
};
const createQueryKeyLiteral = ({ id, isInfinite, operation, plugin }) => {
	const config = isInfinite ? plugin.config.infiniteQueryKeys : plugin.config.queryKeys;
	let tagsArray;
	if (config.tags && operation.tags && operation.tags.length > 0) tagsArray = $.array().elements(...operation.tags);
	return $(plugin.referenceSymbol({
		category: "utility",
		resource: "createQueryKey",
		tool: plugin.name
	})).call($.literal(id), "options", isInfinite || tagsArray ? $.literal(Boolean(isInfinite)) : void 0, tagsArray);
};
const createQueryKeyType = ({ plugin }) => {
	const symbolOptions = plugin.referenceSymbol({
		category: "type",
		resource: "client-options",
		tool: "sdk"
	});
	const symbolQueryKeyType = plugin.symbol("QueryKey", { meta: {
		category: "type",
		resource: "QueryKey",
		tool: plugin.name
	} });
	const queryKeyType = $.type.alias(symbolQueryKeyType).export().generic(TOptionsType, (g) => g.extends(symbolOptions)).type($.type.tuple($.type.and($.type(`Pick<${TOptionsType}, '${getClientBaseUrlKey(getTypedConfig(plugin))}' | 'body' | 'headers' | 'path' | 'query'>`), $.type.object().prop("_id", (p) => p.type("string")).prop("_infinite", (p) => p.optional().type("boolean")).prop("tags", (p) => p.optional().type("ReadonlyArray<string>")))));
	plugin.node(queryKeyType);
};
const queryKeyStatement = ({ isInfinite, operation, plugin, symbol, typeQueryKey }) => {
	const typeData = useTypeData({
		operation,
		plugin
	});
	return $.const(symbol).export().assign($.func().param("options", (p) => p.required(hasOperationDataRequired(operation)).type(typeData)).$if(isInfinite && typeQueryKey, (f$1, v) => f$1.returns(v)).do(createQueryKeyLiteral({
		id: operation.id,
		isInfinite,
		operation,
		plugin
	}).return()));
};

//#endregion
//#region src/plugins/@tanstack/query-core/shared/meta.ts
const handleMeta = (plugin, operation, configPath) => {
	const metaFn = plugin.config[configPath].meta;
	if (!metaFn) return;
	const metaObject = metaFn(operation);
	if (!Object.keys(metaObject).length) return;
	return $.fromValue(metaObject);
};

//#endregion
//#region src/plugins/@tanstack/query-core/v5/infiniteQueryOptions.ts
const createInfiniteParamsFunction = ({ plugin }) => {
	const symbolCreateInfiniteParams = plugin.symbol(applyNaming("createInfiniteParams", { case: plugin.config.case }), { meta: {
		category: "utility",
		resource: "createInfiniteParams",
		tool: plugin.name
	} });
	const fn = $.const(symbolCreateInfiniteParams).assign($.func().generic("K", (g) => g.extends($.type("Pick").generics($.type("QueryKey").generic("Options").idx(0), $.type.or($.type.literal("body"), $.type.literal("headers"), $.type.literal("path"), $.type.literal("query"))))).param("queryKey", (p) => p.type("QueryKey<Options>")).param("page", (p) => p.type("K")).do($.const("params").assign($.object().spread($("queryKey").attr(0))), $.if($("page").attr("body")).do($("params").attr("body").assign($.object().pretty().spread($("queryKey").attr(0).attr("body").as("any")).spread($("page").attr("body").as("any")))), $.if($("page").attr("headers")).do($("params").attr("headers").assign($.object().pretty().spread($("queryKey").attr(0).attr("headers")).spread($("page").attr("headers")))), $.if($("page").attr("path")).do($("params").attr("path").assign($.object().pretty().spread($("queryKey").attr(0).attr("path").as("any")).spread($("page").attr("path").as("any")))), $.if($("page").attr("query")).do($("params").attr("query").assign($.object().pretty().spread($("queryKey").attr(0).attr("query").as("any")).spread($("page").attr("query").as("any")))), $.return($("params").as("unknown").as($("page").typeofType()))));
	plugin.node(fn);
};
const createInfiniteQueryOptions = ({ operation, plugin }) => {
	const pagination = operationPagination({
		context: plugin.context,
		operation
	});
	if (!pagination) return;
	const isRequiredOptions = isOperationOptionsRequired({
		context: plugin.context,
		operation
	});
	if (!plugin.getSymbol({
		category: "utility",
		resource: "createQueryKey",
		tool: plugin.name
	})) {
		createQueryKeyType({ plugin });
		createQueryKeyFunction({ plugin });
	}
	if (!plugin.getSymbol({
		category: "utility",
		resource: "createInfiniteParams",
		tool: plugin.name
	})) createInfiniteParamsFunction({ plugin });
	const symbolInfiniteQueryOptions = plugin.external(`${plugin.name}.infiniteQueryOptions`);
	const symbolInfiniteDataType = plugin.external(`${plugin.name}.InfiniteData`);
	const typeData = useTypeData({
		operation,
		plugin
	});
	const typeResponse = useTypeResponse({
		operation,
		plugin
	});
	const symbolQueryKeyType = plugin.referenceSymbol({
		category: "type",
		resource: "QueryKey",
		tool: plugin.name
	});
	const typeQueryKey = $.type(symbolQueryKeyType).generic(typeData);
	const typePageObjectParam = $.type("Pick").generics(typeQueryKey.idx(0), $.type.or($.type.literal("body"), $.type.literal("headers"), $.type.literal("path"), $.type.literal("query")));
	const pluginTypeScript = plugin.getPluginOrThrow("@hey-api/typescript");
	const type = pluginTypeScript.api.schemaToType({
		plugin: pluginTypeScript,
		schema: pagination.schema,
		state: { path: ref([]) }
	});
	const symbolInfiniteQueryKey = plugin.symbol(applyNaming(operation.id, plugin.config.infiniteQueryKeys));
	const node = queryKeyStatement({
		isInfinite: true,
		operation,
		plugin,
		symbol: symbolInfiniteQueryKey,
		typeQueryKey
	});
	plugin.node(node);
	const awaitSdkFn = $.lazy((ctx$1) => ctx$1.access(plugin.referenceSymbol({
		category: "sdk",
		resource: "operation",
		resourceId: operation.id
	})).call($.object().spread("options").spread("params").prop("signal", $("signal")).prop("throwOnError", $.literal(true))).await());
	const symbolCreateInfiniteParams = plugin.referenceSymbol({
		category: "utility",
		resource: "createInfiniteParams",
		tool: plugin.name
	});
	const statements = [$.const("page").type(typePageObjectParam).hint("@ts-ignore").assign($.ternary($("pageParam").typeofExpr().eq($.literal("object"))).do("pageParam").otherwise($.object().pretty().prop(pagination.in, $.object().pretty().prop(pagination.name, $("pageParam"))))), $.const("params").assign($(symbolCreateInfiniteParams).call("queryKey", "page"))];
	if (plugin.getPluginOrThrow("@hey-api/sdk").config.responseStyle === "data") statements.push($.return(awaitSdkFn));
	else statements.push($.const().object("data").assign(awaitSdkFn), $.return("data"));
	const symbolInfiniteQueryOptionsFn = plugin.symbol(applyNaming(operation.id, plugin.config.infiniteQueryOptions));
	const statement = $.const(symbolInfiniteQueryOptionsFn).export().$if(plugin.config.comments && createOperationComment(operation), (c, v) => c.doc(v)).assign($.func().param("options", (p) => p.required(isRequiredOptions).type(typeData)).do($.return($(symbolInfiniteQueryOptions).call($.object().pretty().hint("@ts-ignore").prop("queryFn", $.func().async().param((p) => p.object("pageParam", "queryKey", "signal")).do(...statements)).prop("queryKey", $(symbolInfiniteQueryKey).call("options")).$if(handleMeta(plugin, operation, "infiniteQueryOptions"), (o, v) => o.prop("meta", v))).generics(typeResponse, useTypeError({
		operation,
		plugin
	}), $.type(symbolInfiniteDataType).generic(typeResponse), typeQueryKey, $.type.or(type, typePageObjectParam)))));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/@tanstack/query-core/v5/mutationOptions.ts
const createMutationOptions = ({ operation, plugin }) => {
	const symbolMutationOptionsType = plugin.external(`${plugin.name}.MutationOptions`);
	const typeData = useTypeData({
		operation,
		plugin
	});
	const mutationType = $.type(symbolMutationOptionsType).generic(useTypeResponse({
		operation,
		plugin
	})).generic(useTypeError({
		operation,
		plugin
	})).generic(typeData);
	const fnOptions$1 = "fnOptions";
	const awaitSdkFn = $.lazy((ctx$1) => ctx$1.access(plugin.referenceSymbol({
		category: "sdk",
		resource: "operation",
		resourceId: operation.id
	})).call($.object().spread("options").spread(fnOptions$1).prop("throwOnError", $.literal(true))).await());
	const statements = [];
	if (plugin.getPluginOrThrow("@hey-api/sdk").config.responseStyle === "data") statements.push($.return(awaitSdkFn));
	else statements.push($.const().object("data").assign(awaitSdkFn), $.return("data"));
	const mutationOptionsFn = "mutationOptions";
	const symbolMutationOptions = plugin.symbol(applyNaming(operation.id, plugin.config.mutationOptions), { meta: {
		category: "hook",
		resource: "operation",
		resourceId: operation.id,
		role: "mutationOptions",
		tool: plugin.name
	} });
	const statement = $.const(symbolMutationOptions).export().$if(plugin.config.comments && createOperationComment(operation), (c, v) => c.doc(v)).assign($.func().param("options", (p) => p.optional().type($.type("Partial").generic(typeData))).returns(mutationType).do($.const(mutationOptionsFn).type(mutationType).assign($.object().pretty().prop("mutationFn", $.func().async().param(fnOptions$1).do(...statements)).$if(handleMeta(plugin, operation, "mutationOptions"), (c, v) => c.prop("meta", v))), $(mutationOptionsFn).return()));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/@tanstack/query-core/v5/queryOptions.ts
const optionsParamName$1 = "options";
const createQueryOptions = ({ operation, plugin }) => {
	if (hasOperationSse({ operation })) return;
	const isRequiredOptions = isOperationOptionsRequired({
		context: plugin.context,
		operation
	});
	if (!plugin.getSymbol({
		category: "utility",
		resource: "createQueryKey",
		tool: plugin.name
	})) {
		createQueryKeyType({ plugin });
		createQueryKeyFunction({ plugin });
	}
	const symbolQueryOptions = plugin.external(`${plugin.name}.queryOptions`);
	const symbolQueryKey = plugin.symbol(applyNaming(operation.id, plugin.config.queryKeys));
	const node = queryKeyStatement({
		isInfinite: false,
		operation,
		plugin,
		symbol: symbolQueryKey
	});
	plugin.node(node);
	const typeResponse = useTypeResponse({
		operation,
		plugin
	});
	const awaitSdkFn = $.lazy((ctx$1) => ctx$1.access(plugin.referenceSymbol({
		category: "sdk",
		resource: "operation",
		resourceId: operation.id
	})).call($.object().spread(optionsParamName$1).spread($("queryKey").attr(0)).prop("signal", $("signal")).prop("throwOnError", $.literal(true))).await());
	const statements = [];
	if (plugin.getPluginOrThrow("@hey-api/sdk").config.responseStyle === "data") statements.push($.return(awaitSdkFn));
	else statements.push($.const().object("data").assign(awaitSdkFn), $.return("data"));
	const queryOptionsObj = $.object().pretty().prop("queryFn", $.func().async().param((p) => p.object("queryKey", "signal")).do(...statements)).prop("queryKey", $(symbolQueryKey).call(optionsParamName$1)).$if(handleMeta(plugin, operation, "queryOptions"), (o, v) => o.prop("meta", v));
	const symbolQueryOptionsFn = plugin.symbol(applyNaming(operation.id, plugin.config.queryOptions), { meta: {
		category: "hook",
		resource: "operation",
		resourceId: operation.id,
		role: "queryOptions",
		tool: plugin.name
	} });
	const statement = $.const(symbolQueryOptionsFn).export(plugin.config.queryOptions.exported).$if(plugin.config.comments && createOperationComment(operation), (c, v) => c.doc(v)).assign($.func().param(optionsParamName$1, (p) => p.required(isRequiredOptions).type(useTypeData({
		operation,
		plugin
	}))).do($(symbolQueryOptions).call(queryOptionsObj).generics(typeResponse, useTypeError({
		operation,
		plugin
	}), typeResponse, $(symbolQueryKey).returnType()).return()));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/@tanstack/query-core/v5/useQuery.ts
const optionsParamName = "options";
const createUseQuery = ({ operation, plugin }) => {
	if (hasOperationSse({ operation })) return;
	if (!("useQuery" in plugin.config)) return;
	const symbolUseQueryFn = plugin.symbol(applyNaming(operation.id, plugin.config.useQuery));
	const symbolUseQuery = plugin.external(`${plugin.name}.useQuery`);
	const isRequiredOptions = isOperationOptionsRequired({
		context: plugin.context,
		operation
	});
	const typeData = useTypeData({
		operation,
		plugin
	});
	const symbolQueryOptionsFn = plugin.referenceSymbol({
		category: "hook",
		resource: "operation",
		resourceId: operation.id,
		role: "queryOptions",
		tool: plugin.name
	});
	const statement = $.const(symbolUseQueryFn).export().$if(plugin.config.comments && createOperationComment(operation), (c, v) => c.doc(v)).assign($.func().param(optionsParamName, (p) => p.required(isRequiredOptions).type(typeData)).do($(symbolUseQuery).call($(symbolQueryOptionsFn).call(optionsParamName)).return()));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/@tanstack/query-core/v5/plugin.ts
const handlerV5 = ({ plugin }) => {
	plugin.symbol("DefaultError", {
		external: plugin.name,
		kind: "type",
		meta: {
			category: "external",
			resource: `${plugin.name}.DefaultError`
		}
	});
	plugin.symbol("InfiniteData", {
		external: plugin.name,
		kind: "type",
		meta: {
			category: "external",
			resource: `${plugin.name}.InfiniteData`
		}
	});
	const mutationsType = plugin.name === "@tanstack/angular-query-experimental" || plugin.name === "@tanstack/svelte-query" || plugin.name === "@tanstack/solid-query" ? "MutationOptions" : "UseMutationOptions";
	plugin.symbol(mutationsType, {
		external: plugin.name,
		kind: "type",
		meta: {
			category: "external",
			resource: `${plugin.name}.MutationOptions`
		}
	});
	plugin.symbol("infiniteQueryOptions", {
		external: plugin.name,
		meta: {
			category: "external",
			resource: `${plugin.name}.infiniteQueryOptions`
		}
	});
	plugin.symbol("queryOptions", {
		external: plugin.name,
		meta: {
			category: "external",
			resource: `${plugin.name}.queryOptions`
		}
	});
	plugin.symbol("useQuery", {
		external: plugin.name,
		meta: {
			category: "external",
			resource: `${plugin.name}.useQuery`
		}
	});
	plugin.symbol("AxiosError", {
		external: "axios",
		kind: "type",
		meta: {
			category: "external",
			resource: "axios.AxiosError"
		}
	});
	plugin.forEach("operation", ({ operation }) => {
		if (plugin.hooks.operation.isQuery(operation)) {
			if (plugin.config.queryOptions.enabled) createQueryOptions({
				operation,
				plugin
			});
			if (plugin.config.infiniteQueryOptions.enabled) createInfiniteQueryOptions({
				operation,
				plugin
			});
			if ("useQuery" in plugin.config && plugin.config.useQuery.enabled) createUseQuery({
				operation,
				plugin
			});
		}
		if (plugin.hooks.operation.isMutation(operation)) {
			if (plugin.config.mutationOptions.enabled) createMutationOptions({
				operation,
				plugin
			});
		}
	}, { order: "declarations" });
};

//#endregion
//#region src/plugins/@tanstack/query-core/plugin.ts
const handler$5 = (args) => handlerV5(args);

//#endregion
//#region src/plugins/@tanstack/angular-query-experimental/config.ts
const defaultConfig$9 = {
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/sdk", "@hey-api/typescript"],
	handler: handler$5,
	name: "@tanstack/angular-query-experimental",
	resolveConfig: (plugin, context) => {
		plugin.config.infiniteQueryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteQueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.infiniteQueryKeys
		});
		plugin.config.infiniteQueryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteOptions"
			},
			mappers,
			value: plugin.config.infiniteQueryOptions
		});
		plugin.config.mutationOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Mutation"
			},
			mappers,
			value: plugin.config.mutationOptions
		});
		plugin.config.queryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}QueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.queryKeys
		});
		plugin.config.queryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				exported: true,
				name: "{{name}}Options"
			},
			mappers,
			value: plugin.config.queryOptions
		});
	}
};
/**
* Type helper for `@tanstack/angular-query-experimental` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$9 = definePluginConfig(defaultConfig$9);

//#endregion
//#region src/plugins/@tanstack/react-query/config.ts
const defaultConfig$8 = {
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/sdk", "@hey-api/typescript"],
	handler: handler$5,
	name: "@tanstack/react-query",
	resolveConfig: (plugin, context) => {
		plugin.config.infiniteQueryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteQueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.infiniteQueryKeys
		});
		plugin.config.infiniteQueryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteOptions"
			},
			mappers,
			value: plugin.config.infiniteQueryOptions
		});
		plugin.config.mutationOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Mutation"
			},
			mappers,
			value: plugin.config.mutationOptions
		});
		plugin.config.queryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}QueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.queryKeys
		});
		plugin.config.queryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				exported: true,
				name: "{{name}}Options"
			},
			mappers,
			value: plugin.config.queryOptions
		});
		plugin.config.useQuery = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: false,
				name: "use{{name}}Query"
			},
			mappers: {
				boolean: (enabled) => ({ enabled }),
				function: (name) => ({
					enabled: true,
					name
				}),
				object: (fields) => ({
					enabled: true,
					...fields
				}),
				string: (name) => ({
					enabled: true,
					name
				})
			},
			value: plugin.config.useQuery
		});
		if (plugin.config.useQuery.enabled) {
			if (!plugin.config.queryOptions.enabled) {
				plugin.config.queryOptions.enabled = true;
				plugin.config.queryOptions.exported = false;
			}
		}
	}
};
/**
* Type helper for `@tanstack/react-query` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$8 = definePluginConfig(defaultConfig$8);

//#endregion
//#region src/plugins/@tanstack/solid-query/config.ts
const defaultConfig$7 = {
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/sdk", "@hey-api/typescript"],
	handler: handler$5,
	name: "@tanstack/solid-query",
	resolveConfig: (plugin, context) => {
		plugin.config.infiniteQueryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteQueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.infiniteQueryKeys
		});
		plugin.config.infiniteQueryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteOptions"
			},
			mappers,
			value: plugin.config.infiniteQueryOptions
		});
		plugin.config.mutationOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Mutation"
			},
			mappers,
			value: plugin.config.mutationOptions
		});
		plugin.config.queryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}QueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.queryKeys
		});
		plugin.config.queryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				exported: true,
				name: "{{name}}Options"
			},
			mappers,
			value: plugin.config.queryOptions
		});
	}
};
/**
* Type helper for `@tanstack/solid-query` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$7 = definePluginConfig(defaultConfig$7);

//#endregion
//#region src/plugins/@tanstack/svelte-query/config.ts
const defaultConfig$6 = {
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/sdk", "@hey-api/typescript"],
	handler: handler$5,
	name: "@tanstack/svelte-query",
	resolveConfig: (plugin, context) => {
		plugin.config.infiniteQueryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteQueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.infiniteQueryKeys
		});
		plugin.config.infiniteQueryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteOptions"
			},
			mappers,
			value: plugin.config.infiniteQueryOptions
		});
		plugin.config.mutationOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Mutation"
			},
			mappers,
			value: plugin.config.mutationOptions
		});
		plugin.config.queryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}QueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.queryKeys
		});
		plugin.config.queryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				exported: true,
				name: "{{name}}Options"
			},
			mappers,
			value: plugin.config.queryOptions
		});
	}
};
/**
* Type helper for `@tanstack/svelte-query` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$6 = definePluginConfig(defaultConfig$6);

//#endregion
//#region src/plugins/@tanstack/vue-query/config.ts
const defaultConfig$5 = {
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/sdk", "@hey-api/typescript"],
	handler: handler$5,
	name: "@tanstack/vue-query",
	resolveConfig: (plugin, context) => {
		plugin.config.infiniteQueryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteQueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.infiniteQueryKeys
		});
		plugin.config.infiniteQueryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteOptions"
			},
			mappers,
			value: plugin.config.infiniteQueryOptions
		});
		plugin.config.mutationOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Mutation"
			},
			mappers,
			value: plugin.config.mutationOptions
		});
		plugin.config.queryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}QueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.queryKeys
		});
		plugin.config.queryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				exported: true,
				name: "{{name}}Options"
			},
			mappers,
			value: plugin.config.queryOptions
		});
	}
};
/**
* Type helper for `@tanstack/vue-query` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$5 = definePluginConfig(defaultConfig$5);

//#endregion
//#region src/plugins/arktype/v2/api.ts
const createRequestValidatorV2 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "arktype"
	});
	if (!symbol) return;
	const dataParameterName = "data";
	return $.func().async().param(dataParameterName).do($(symbol).attr("parseAsync").call(dataParameterName).await().return());
};
const createResponseValidatorV2 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "responses",
		tool: "arktype"
	});
	if (!symbol) return;
	const dataParameterName = "data";
	return $.func().async().param(dataParameterName).do($(symbol).attr("parseAsync").call(dataParameterName).await().return());
};

//#endregion
//#region src/plugins/arktype/api.ts
var Api$2 = class {
	createRequestValidator(args) {
		return createRequestValidatorV2(args);
	}
	createResponseValidator(args) {
		return createResponseValidatorV2(args);
	}
};

//#endregion
//#region src/plugins/arktype/constants.ts
const identifiers$2 = {
	keywords: {
		false: "false",
		true: "true"
	},
	number: {
		Infinity: "Infinity",
		NaN: "NaN",
		NegativeInfinity: "NegativeInfinity",
		epoch: "epoch",
		integer: "integer",
		safe: "safe"
	},
	primitives: {
		bigint: "bigint",
		boolean: "boolean",
		keywords: "keywords",
		null: "null",
		number: "number",
		string: "string",
		symbol: "symbol",
		undefined: "undefined",
		unit: "unit"
	},
	string: {
		NFC: "NFC",
		NFD: "NFD",
		NFKC: "NFKC",
		NFKD: "NFKD",
		alpha: "alpha",
		alphanumeric: "alphanumeric",
		base64: "base64",
		capitalize: "capitalize",
		creditCard: "creditCard",
		date: "date",
		digits: "digits",
		email: "email",
		epoch: "epoch",
		hex: "hex",
		integer: "integer",
		ip: "ip",
		iso: "iso",
		json: "json",
		lower: "lower",
		normalize: "normalize",
		numeric: "numeric",
		parse: "parse",
		preformatted: "preformatted",
		regex: "regex",
		semver: "semver",
		trim: "trim",
		upper: "upper",
		url: "url",
		uuid: "uuid",
		v1: "v1",
		v2: "v2",
		v3: "v3",
		v4: "v4",
		v5: "v5",
		v6: "v6",
		v7: "v7",
		v8: "v8"
	},
	type: {
		$: "$",
		allows: "allows",
		and: "and",
		array: "array",
		as: "as",
		assert: "assert",
		brand: "brand",
		configure: "configure",
		default: "default",
		describe: "describe",
		description: "description",
		equals: "equals",
		exclude: "exclude",
		expression: "expression",
		extends: "extends",
		extract: "extract",
		filter: "filter",
		from: "from",
		ifEquals: "ifEquals",
		ifExtends: "ifExtends",
		infer: "infer",
		inferIn: "inferIn",
		intersect: "intersect",
		json: "json",
		meta: "meta",
		narrow: "narrow",
		onDeepUndeclaredKey: "onDeepUndeclaredKey",
		onUndeclaredKey: "onUndeclaredKey",
		optional: "optional",
		or: "or",
		overlaps: "overlaps",
		pipe: "pipe",
		select: "select",
		to: "to",
		toJsonSchema: "toJsonSchema"
	}
};

//#endregion
//#region src/plugins/arktype/shared/export.ts
function exportAst$2({ ast, plugin, schema, symbol, typeInferSymbol }) {
	const type = plugin.external("arktype.type");
	const statement = $.const(symbol).export().$if(plugin.config.comments && createSchemaComment(schema), (c, v) => c.doc(v)).assign($(type).call(ast.def ? $.literal(ast.def) : ast.expression));
	plugin.node(statement);
	if (typeInferSymbol) {
		const inferType = $.type.alias(typeInferSymbol).export().type($.type(symbol).attr(identifiers$2.type.infer).typeofType());
		plugin.node(inferType);
	}
}

//#endregion
//#region src/plugins/arktype/v2/toAst/null.ts
const nullToAst$4 = (_args) => {
	const result = {};
	result.def = identifiers$2.primitives.null;
	return result;
};

//#endregion
//#region src/plugins/arktype/v2/toAst/object.ts
const objectToAst$4 = ({ plugin, schema, state }) => {
	const result = {};
	const shape = $.object().pretty();
	const required = schema.required ?? [];
	for (const name in schema.properties) {
		const property = schema.properties[name];
		const isRequired = required.includes(name);
		const propertyAst = irSchemaToAst$4({
			optional: !isRequired,
			plugin,
			schema: property,
			state: {
				...state,
				path: ref([
					...fromRef(state.path),
					"properties",
					name
				])
			}
		});
		if (propertyAst.hasLazyExpression) result.hasLazyExpression = true;
		shape.prop(isRequired ? name : `${name}?`, propertyAst.expression);
	}
	if (schema.additionalProperties && (!schema.properties || !Object.keys(schema.properties).length)) {
		const additionalAst = irSchemaToAst$4({
			plugin,
			schema: schema.additionalProperties,
			state: {
				...state,
				path: ref([...fromRef(state.path), "additionalProperties"])
			}
		});
		result.expression = $("TODO").attr("record").call($("TODO").attr("string").call(), additionalAst.expression);
		if (additionalAst.hasLazyExpression) result.hasLazyExpression = true;
		return result;
	}
	result.expression = shape;
	if (result.hasLazyExpression) return {
		...result,
		typeName: "TODO"
	};
	return result;
};

//#endregion
//#region src/plugins/arktype/v2/toAst/string.ts
const stringToAst = ({ schema }) => {
	const result = {};
	if (typeof schema.const === "string") {
		result.def = schema.const;
		return result;
	}
	let def = identifiers$2.primitives.string;
	if (schema.format) switch (schema.format) {
		case "date":
		case "date-time":
		case "time":
			def = `${def}.${identifiers$2.string.date}.${identifiers$2.string.iso}`;
			break;
		case "email":
			def = `${def}.${identifiers$2.string.email}`;
			break;
		case "ipv4":
			def = `${def}.${identifiers$2.string.ip}.${identifiers$2.string.v4}`;
			break;
		case "ipv6":
			def = `${def}.${identifiers$2.string.ip}.${identifiers$2.string.v6}`;
			break;
		case "uri":
			def = `${def}.${identifiers$2.string.url}`;
			break;
		case "uuid":
			def = `${def}.${identifiers$2.string.uuid}`;
			break;
	}
	if (schema.minLength === schema.maxLength && schema.minLength !== void 0) def = `${schema.minLength} <= ${def} <= ${schema.maxLength}`;
	else if (schema.maxLength !== void 0) {
		def = `${def} <= ${schema.maxLength}`;
		if (schema.minLength !== void 0) def = `${schema.minLength} <= ${def}`;
	} else if (schema.minLength !== void 0) def = `${def} >= ${schema.minLength}`;
	if (schema.pattern) def = `/${schema.pattern}/`;
	result.def = def;
	return result;
};

//#endregion
//#region src/plugins/arktype/v2/toAst/index.ts
function irSchemaWithTypeToAst$4({ schema, ...args }) {
	switch (schema.type) {
		case "null": return nullToAst$4({
			...args,
			schema
		});
		case "object": return objectToAst$4({
			...args,
			schema
		});
		case "string": return stringToAst({
			...args,
			schema
		});
	}
	return {
		def: "",
		expression: $(args.plugin.external("arktype.type")).call($.object().prop("name", $.literal("string")).prop("platform", $.literal("'android' | 'ios'")).prop("versions?", $.literal("(number | string)[]"))),
		hasLazyExpression: false
	};
}

//#endregion
//#region src/plugins/arktype/v2/plugin.ts
function irSchemaToAst$4({ plugin, schema, schemaExtractor, state }) {
	if (schemaExtractor && !schema.$ref) {
		const extracted = schemaExtractor({
			meta: {
				resource: "definition",
				resourceId: pathToJsonPointer(fromRef(state.path))
			},
			path: fromRef(state.path),
			schema
		});
		if (extracted !== schema) schema = extracted;
	}
	let ast = {};
	if (schema.$ref) {
		const query = {
			category: "schema",
			resource: "definition",
			resourceId: schema.$ref,
			tool: "arktype"
		};
		const refSymbol = plugin.referenceSymbol(query);
		if (plugin.isSymbolRegistered(query)) {
			const ref$1 = $(refSymbol);
			ast.expression = ref$1;
		} else {
			const lazyExpression = $("TODO").attr("TODO").call($.func().returns("any").do($.return(refSymbol)));
			ast.expression = lazyExpression;
			ast.hasLazyExpression = true;
			state.hasLazyExpression["~ref"] = true;
		}
	} else if (schema.type) {
		const typeAst = irSchemaWithTypeToAst$4({
			plugin,
			schema,
			state
		});
		ast.def = typeAst.def;
		ast.expression = typeAst.expression;
		ast.hasLazyExpression = typeAst.hasLazyExpression;
		if (plugin.config.metadata && schema.description) {}
	} else if (schema.items) {
		schema = deduplicateSchema({ schema });
		if (schema.items) {} else ast = irSchemaToAst$4({
			plugin,
			schema,
			state
		});
	} else {
		const typeAst = irSchemaWithTypeToAst$4({
			plugin,
			schema: { type: "unknown" },
			state
		});
		ast.def = typeAst.def;
		ast.expression = typeAst.expression;
	}
	if (!ast.expression) {
		const typeAst = irSchemaWithTypeToAst$4({
			plugin,
			schema: { type: "unknown" },
			state
		});
		ast.expression = typeAst.expression;
	}
	return ast;
}
function handleComponent({ plugin, schema, state }) {
	const $ref = pathToJsonPointer(fromRef(state.path));
	const ast = irSchemaToAst$4({
		plugin,
		schema,
		state
	});
	const baseName = refToName($ref);
	exportAst$2({
		ast,
		plugin,
		schema,
		symbol: plugin.symbol(applyNaming(baseName, plugin.config.definitions), { meta: {
			category: "schema",
			path: fromRef(state.path),
			resource: "definition",
			resourceId: $ref,
			tags: fromRef(state.tags),
			tool: "arktype"
		} }),
		typeInferSymbol: plugin.config.definitions.types.infer.enabled ? plugin.symbol(applyNaming(baseName, plugin.config.definitions.types.infer), { meta: {
			category: "type",
			path: fromRef(state.path),
			resource: "definition",
			resourceId: $ref,
			tool: "arktype",
			variant: "infer"
		} }) : void 0
	});
}
const handlerV2$1 = ({ plugin }) => {
	plugin.symbol("type", {
		external: "arktype",
		meta: {
			category: "external",
			resource: "arktype.type"
		}
	});
	plugin.forEach("operation", "parameter", "requestBody", "schema", "webhook", (event) => {
		const state = refs({
			hasLazyExpression: false,
			path: event._path,
			tags: event.tags
		});
		switch (event.type) {
			case "parameter":
				handleComponent({
					plugin,
					schema: event.parameter.schema,
					state
				});
				break;
			case "requestBody":
				handleComponent({
					plugin,
					schema: event.requestBody.schema,
					state
				});
				break;
			case "schema":
				handleComponent({
					plugin,
					schema: event.schema,
					state
				});
				break;
		}
	});
};

//#endregion
//#region src/plugins/arktype/plugin.ts
const handler$4 = (args) => handlerV2$1(args);

//#endregion
//#region src/plugins/arktype/config.ts
const defaultConfig$4 = {
	api: new Api$2(),
	config: {
		case: "PascalCase",
		comments: true,
		includeInEntry: false,
		metadata: false
	},
	handler: handler$4,
	name: "arktype",
	resolveConfig: (plugin, context) => {
		plugin.config.types = context.valueToObject({
			defaultValue: { infer: {
				case: "PascalCase",
				enabled: false
			} },
			mappers: { object: (fields, defaultValue) => ({
				...fields,
				infer: context.valueToObject({
					defaultValue: {
						...defaultValue.infer,
						enabled: fields.infer !== void 0 ? Boolean(fields.infer) : defaultValue.infer.enabled
					},
					mappers,
					value: fields.infer
				})
			}) },
			value: plugin.config.types
		});
		plugin.config.definitions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				enabled: true,
				name: "{{name}}",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.definitions
		});
		plugin.config.requests = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				enabled: true,
				name: "{{name}}Data",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}Data"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.requests
		});
		plugin.config.responses = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				enabled: true,
				name: "{{name}}Response",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}Response"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.responses
		});
		plugin.config.webhooks = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "PascalCase",
				enabled: true,
				name: "{{name}}WebhookRequest",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}WebhookRequest"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.webhooks
		});
	},
	tags: ["validator"]
};
/**
* Type helper for Arktype plugin, returns {@link Plugin.Config} object
*/
const defineConfig$4 = definePluginConfig(defaultConfig$4);

//#endregion
//#region src/plugins/fastify/plugin.ts
const operationToRouteHandler = ({ operation, plugin }) => {
	const type = $.type.object();
	const symbolDataType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "typescript"
	});
	if (symbolDataType) {
		if (operation.body) type.prop("Body", (p) => p.required(operation.body.required).type($.type(symbolDataType).idx($.type.literal("body"))));
		if (operation.parameters) {
			if (operation.parameters.header) type.prop("Headers", (p) => p.required(hasParameterGroupObjectRequired(operation.parameters.header)).type($.type(symbolDataType).idx($.type.literal("headers"))));
			if (operation.parameters.path) type.prop("Params", (p) => p.required(hasParameterGroupObjectRequired(operation.parameters.path)).type($.type(symbolDataType).idx($.type.literal("path"))));
			if (operation.parameters.query) type.prop("Querystring", (p) => p.required(hasParameterGroupObjectRequired(operation.parameters.query)).type($.type(symbolDataType).idx($.type.literal("query"))));
		}
	}
	const { errors, responses } = operationResponsesMap(operation);
	let errorsTypeReference;
	const symbolErrorType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "errors"
	});
	if (symbolErrorType && errors && errors.properties) {
		const keys = Object.keys(errors.properties);
		if (keys.length) {
			if (!keys.includes("default")) errorsTypeReference = $.type(symbolErrorType);
			else if (keys.length > 1) errorsTypeReference = $.type("Omit", (t) => t.generics($.type(symbolErrorType), $.type.literal("default")));
		}
	}
	let responsesTypeReference = void 0;
	const symbolResponseType = plugin.querySymbol({
		category: "type",
		resource: "operation",
		resourceId: operation.id,
		role: "responses"
	});
	if (symbolResponseType && responses && responses.properties) {
		const keys = Object.keys(responses.properties);
		if (keys.length) {
			if (!keys.includes("default")) responsesTypeReference = $.type(symbolResponseType);
			else if (keys.length > 1) responsesTypeReference = $.type("Omit", (t) => t.generics($.type(symbolResponseType), $.type.literal("default")));
		}
	}
	const replyTypes = [errorsTypeReference, responsesTypeReference].filter((t) => t !== void 0);
	if (replyTypes.length) type.prop("Reply", (p) => p.type($.type.and(...replyTypes)));
	if (type.isEmpty) return;
	const symbolRouteHandler = plugin.referenceSymbol({
		category: "type",
		resource: "route-handler",
		tool: "fastify"
	});
	return {
		name: operation.id,
		type: $.type(symbolRouteHandler, (t) => t.generic(type))
	};
};
const handler$3 = ({ plugin }) => {
	plugin.symbol("RouteHandler", {
		external: "fastify",
		kind: "type",
		meta: {
			category: "type",
			resource: "route-handler",
			tool: "fastify"
		}
	});
	const symbolRouteHandlers = plugin.symbol("RouteHandlers");
	const type = $.type.object();
	plugin.forEach("operation", ({ operation }) => {
		const routeHandler = operationToRouteHandler({
			operation,
			plugin
		});
		if (routeHandler) type.prop(routeHandler.name, (p) => p.type(routeHandler.type));
	}, { order: "declarations" });
	const node = $.type.alias(symbolRouteHandlers).export().type(type);
	plugin.node(node);
};

//#endregion
//#region src/plugins/fastify/config.ts
const defaultConfig$3 = {
	config: { includeInEntry: false },
	dependencies: ["@hey-api/typescript"],
	handler: handler$3,
	name: "fastify"
};
/**
* Type helper for `fastify` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$3 = definePluginConfig(defaultConfig$3);

//#endregion
//#region src/plugins/swr/v2/useSwr.ts
const createUseSwr = ({ operation, plugin }) => {
	if (hasOperationSse({ operation })) return;
	const symbolUseSwr = plugin.external("swr");
	const symbolUseQueryFn = plugin.symbol(applyNaming(operation.id, plugin.config.useSwr));
	const awaitSdkFn = $.lazy((ctx$1) => ctx$1.access(plugin.referenceSymbol({
		category: "sdk",
		resource: "operation",
		resourceId: operation.id
	})).call($.object().prop("throwOnError", $.literal(true))).await());
	const statements = [];
	if (plugin.getPluginOrThrow("@hey-api/sdk").config.responseStyle === "data") statements.push($.return(awaitSdkFn));
	else statements.push($.const().object("data").assign(awaitSdkFn), $.return("data"));
	const statement = $.const(symbolUseQueryFn).export().$if(plugin.config.comments && createOperationComment(operation), (c, v) => c.doc(v)).assign($.func().do($(symbolUseSwr).call($.literal(operation.path), $.func().async().do(...statements)).return()));
	plugin.node(statement);
};

//#endregion
//#region src/plugins/swr/v2/plugin.ts
const handlerV2 = ({ plugin }) => {
	plugin.symbol("useSWR", {
		external: "swr",
		importKind: "default",
		kind: "function",
		meta: {
			category: "external",
			resource: "swr"
		}
	});
	plugin.forEach("operation", ({ operation }) => {
		if (plugin.hooks.operation.isQuery(operation)) {
			if (plugin.config.useSwr.enabled) createUseSwr({
				operation,
				plugin
			});
		}
	}, { order: "declarations" });
};

//#endregion
//#region src/plugins/swr/plugin.ts
const handler$2 = (args) => handlerV2(args);

//#endregion
//#region src/plugins/swr/config.ts
const defaultConfig$2 = {
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false
	},
	dependencies: ["@hey-api/sdk", "@hey-api/typescript"],
	handler: handler$2,
	name: "swr",
	resolveConfig: (plugin, context) => {
		plugin.config.infiniteQueryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteQueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.infiniteQueryKeys
		});
		plugin.config.infiniteQueryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}InfiniteOptions"
			},
			mappers,
			value: plugin.config.infiniteQueryOptions
		});
		plugin.config.mutationOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}Mutation"
			},
			mappers,
			value: plugin.config.mutationOptions
		});
		plugin.config.queryKeys = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "{{name}}QueryKey",
				tags: false
			},
			mappers,
			value: plugin.config.queryKeys
		});
		plugin.config.queryOptions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				exported: true,
				name: "{{name}}Options"
			},
			mappers,
			value: plugin.config.queryOptions
		});
		plugin.config.useSwr = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "use{{name}}"
			},
			mappers: {
				boolean: (enabled) => ({ enabled }),
				function: (name) => ({
					enabled: true,
					name
				}),
				object: (fields) => ({
					enabled: true,
					...fields
				}),
				string: (name) => ({
					enabled: true,
					name
				})
			},
			value: plugin.config.useSwr
		});
		if (plugin.config.useSwr.enabled) {
			if (!plugin.config.queryOptions.enabled) {
				plugin.config.queryOptions.enabled = true;
				plugin.config.queryOptions.exported = false;
			}
		}
	}
};
/**
* Type helper for `swr` plugin, returns {@link Plugin.Config} object
*/
const defineConfig$2 = definePluginConfig(defaultConfig$2);

//#endregion
//#region src/plugins/valibot/v1/constants.ts
const identifiers$1 = {
	actions: {
		args: "args",
		base64: "base64",
		bic: "bic",
		brand: "brand",
		bytes: "bytes",
		check: "check",
		checkItems: "checkItems",
		creditCard: "creditCard",
		cuid2: "cuid2",
		decimal: "decimal",
		description: "description",
		digits: "digits",
		email: "email",
		emoji: "emoji",
		empty: "empty",
		endsWith: "endsWith",
		entries: "entries",
		everyItem: "everyItem",
		excludes: "excludes",
		filterItems: "filterItems",
		findItem: "findItem",
		finite: "finite",
		flavor: "flavor",
		graphemes: "graphemes",
		gtValue: "gtValue",
		hash: "hash",
		hexColor: "hexColor",
		hexadecimal: "hexadecimal",
		imei: "imei",
		includes: "includes",
		integer: "integer",
		ip: "ip",
		ipv4: "ipv4",
		ipv6: "ipv6",
		isoDate: "isoDate",
		isoDateTime: "isoDateTime",
		isoTime: "isoTime",
		isoTimeSecond: "isoTimeSecond",
		isoTimestamp: "isoTimestamp",
		isoWeek: "isoWeek",
		length: "length",
		ltValue: "ltValue",
		mac: "mac",
		mac48: "mac48",
		mac64: "mac64",
		mapItems: "mapItems",
		maxBytes: "maxBytes",
		maxEntries: "maxEntries",
		maxGraphemes: "maxGraphemes",
		maxLength: "maxLength",
		maxSize: "maxSize",
		maxValue: "maxValue",
		maxWords: "maxWords",
		metadata: "metadata",
		mimeType: "mimeType",
		minBytes: "minBytes",
		minEntries: "minEntries",
		minGraphemes: "minGraphemes",
		minLength: "minLength",
		minSize: "minSize",
		minValue: "minValue",
		minWords: "minWords",
		multipleOf: "multipleOf",
		nanoid: "nanoid",
		nonEmpty: "nonEmpty",
		normalize: "normalize",
		notBytes: "notBytes",
		notEntries: "notEntries",
		notGraphemes: "notGraphemes",
		notLength: "notLength",
		notSize: "notSize",
		notValue: "notValue",
		notValues: "notValues",
		notWords: "notWords",
		octal: "octal",
		parseJson: "parseJson",
		partialCheck: "partialCheck",
		rawCheck: "rawCheck",
		rawTransform: "rawTransform",
		readonly: "readonly",
		reduceItems: "reduceItems",
		regex: "regex",
		returns: "returns",
		rfcEmail: "rfcEmail",
		safeInteger: "safeInteger",
		size: "size",
		slug: "slug",
		someItem: "someItem",
		sortItems: "sortItems",
		startsWith: "startsWith",
		stringifyJson: "stringifyJson",
		title: "title",
		toLowerCase: "toLowerCase",
		toMaxValue: "toMaxValue",
		toMinValue: "toMinValue",
		toUpperCase: "toUpperCase",
		transform: "transform",
		trim: "trim",
		trimEnd: "trimEnd",
		trimStart: "trimStart",
		ulid: "ulid",
		url: "url",
		uuid: "uuid",
		value: "value",
		values: "values",
		words: "words"
	},
	async: {
		argsAsync: "argsAsync",
		arrayAsync: "arrayAsync",
		awaitAsync: "awaitAsync",
		checkAsync: "checkAsync",
		checkItemsAsync: "checkItemsAsync",
		customAsync: "customAsync",
		exactOptionalAsync: "exactOptionalAsync",
		fallbackAsync: "fallbackAsync",
		forwardAsync: "forwardAsync",
		getDefaultsAsync: "getDefaultsAsync",
		getFallbacksAsync: "getFallbacksAsync",
		intersectAsync: "intersectAsync",
		lazyAsync: "lazyAsync",
		looseObjectAsync: "looseObjectAsync",
		looseTupleAsync: "looseTupleAsync",
		mapAsync: "mapAsync",
		nonNullableAsync: "nonNullableAsync",
		nonNullishAsync: "nonNullishAsync",
		nonOptionalAsync: "nonOptionalAsync",
		nullableAsync: "nullableAsync",
		nullishAsync: "nullishAsync",
		objectAsync: "objectAsync",
		objectWithRestAsync: "objectWithRestAsync",
		optionalAsync: "optionalAsync",
		parseAsync: "parseAsync",
		parserAsync: "parserAsync",
		partialAsync: "partialAsync",
		partialCheckAsync: "partialCheckAsync",
		pipeAsync: "pipeAsync",
		rawCheckAsync: "rawCheckAsync",
		rawTransformAsync: "rawTransformAsync",
		recordAsync: "recordAsync",
		requiredAsync: "requiredAsync",
		returnsAsync: "returnsAsync",
		safeParseAsync: "safeParseAsync",
		safeParserAsync: "safeParserAsync",
		setAsync: "setAsync",
		strictObjectAsync: "strictObjectAsync",
		strictTupleAsync: "strictTupleAsync",
		transformAsync: "transformAsync",
		tupleAsync: "tupleAsync",
		tupleWithRestAsync: "tupleWithRestAsync",
		undefinedableAsync: "undefinedableAsync",
		unionAsync: "unionAsync",
		variantAsync: "variantAsync"
	},
	methods: {
		assert: "assert",
		config: "config",
		fallback: "fallback",
		flatten: "flatten",
		forward: "forward",
		getDefault: "getDefault",
		getDefaults: "getDefaults",
		getDescription: "getDescription",
		getFallback: "getFallback",
		getFallbacks: "getFallbacks",
		getMetadata: "getMetadata",
		getTitle: "getTitle",
		is: "is",
		keyof: "keyof",
		message: "message",
		omit: "omit",
		parse: "parse",
		parser: "parser",
		partial: "partial",
		pick: "pick",
		pipe: "pipe",
		required: "required",
		safeParse: "safeParse",
		safeParser: "safeParser",
		summarize: "summarize",
		unwrap: "unwrap"
	},
	schemas: {
		any: "any",
		array: "array",
		bigInt: "bigint",
		blob: "blob",
		boolean: "boolean",
		custom: "custom",
		date: "date",
		enum: "enum",
		exactOptional: "exactOptional",
		file: "file",
		function: "function",
		instance: "instance",
		intersect: "intersect",
		lazy: "lazy",
		literal: "literal",
		looseObject: "looseObject",
		looseTuple: "looseTuple",
		map: "map",
		nan: "nan",
		never: "never",
		nonNullable: "nonNullable",
		nonNullish: "nonNullish",
		nonOptional: "nonOptional",
		null: "null",
		nullable: "nullable",
		nullish: "nullish",
		number: "number",
		object: "object",
		objectWithRest: "objectWithRest",
		optional: "optional",
		picklist: "picklist",
		promise: "promise",
		record: "record",
		set: "set",
		strictObject: "strictObject",
		strictTuple: "strictTuple",
		string: "string",
		symbol: "symbol",
		tuple: "tuple",
		tupleWithRest: "tupleWithRest",
		undefined: "undefined",
		undefinedable: "undefinedable",
		union: "union",
		unknown: "unknown",
		variant: "variant",
		void: "void"
	},
	storages: {},
	types: { GenericSchema: "GenericSchema" },
	utils: {}
};

//#endregion
//#region src/plugins/valibot/shared/pipes.ts
const pipesToNode = (pipes$1, plugin) => {
	if (!(pipes$1 instanceof Array)) return pipes$1;
	if (pipes$1.length === 1) return pipes$1[0];
	return $(plugin.external("valibot.v")).attr(identifiers$1.methods.pipe).call(...pipes$1);
};
const pushPipes = (target, pipes$1) => {
	if (pipes$1 instanceof Array) target.push(...pipes$1);
	else target.push(pipes$1);
	return target;
};
/**
* Functions for working with pipes.
*/
const pipes = {
	push: pushPipes,
	toNode: pipesToNode
};

//#endregion
//#region src/plugins/valibot/v1/api.ts
const validatorResolver$3 = (ctx$1) => {
	const { schema, v } = ctx$1.symbols;
	return $(v).attr(identifiers$1.async.parseAsync).call(schema, "data").await().return();
};
const createRequestValidatorV1 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "valibot"
	});
	if (!symbol) return;
	const ctx$1 = {
		$,
		operation,
		pipes: {
			...pipes,
			current: []
		},
		plugin,
		symbols: {
			schema: symbol,
			v: plugin.external("valibot.v")
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.request, validatorResolver$3];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};
const createResponseValidatorV1 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "responses",
		tool: "valibot"
	});
	if (!symbol) return;
	const ctx$1 = {
		$,
		operation,
		pipes: {
			...pipes,
			current: []
		},
		plugin,
		symbols: {
			schema: symbol,
			v: plugin.external("valibot.v")
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.response, validatorResolver$3];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};

//#endregion
//#region src/plugins/valibot/api.ts
var Api$1 = class {
	createRequestValidator(args) {
		return createRequestValidatorV1(args);
	}
	createResponseValidator(args) {
		return createResponseValidatorV1(args);
	}
};

//#endregion
//#region src/plugins/shared/utils/coerce.ts
const shouldCoerceToBigInt = (format) => format === "int64" || format === "uint64";
const maybeBigInt = (value, format) => {
	if (!shouldCoerceToBigInt(format)) return $.fromValue(value);
	if (typeof value === "string") {
		if (value.endsWith("n")) value = value.slice(0, -1);
		return $("BigInt").call($.fromValue(value));
	}
	if (typeof value === "number") return $("BigInt").call($.fromValue(value));
	return $.fromValue(value);
};

//#endregion
//#region src/plugins/valibot/shared/operation-schema.ts
function buildOperationSchema$1(operation) {
	const requiredProperties = /* @__PURE__ */ new Set();
	const schema = {
		properties: {
			body: { type: "never" },
			path: { type: "never" },
			query: { type: "never" }
		},
		type: "object"
	};
	if (operation.parameters) for (const location of [
		"header",
		"path",
		"query"
	]) {
		const params = operation.parameters[location];
		if (!params) continue;
		const properties = {};
		const required = [];
		const propKey = location === "header" ? "headers" : location;
		for (const key in params) {
			const parameter = params[key];
			properties[parameter.name] = parameter.schema;
			if (parameter.required) {
				required.push(parameter.name);
				requiredProperties.add(propKey);
			}
		}
		if (Object.keys(properties).length) schema.properties[propKey] = {
			properties,
			required,
			type: "object"
		};
	}
	if (operation.body) {
		schema.properties.body = operation.body.schema;
		if (operation.body.required) requiredProperties.add("body");
	}
	schema.required = [...requiredProperties];
	return {
		required: schema.required,
		schema
	};
}

//#endregion
//#region src/plugins/valibot/shared/operation.ts
function irOperationToAst$1({ operation, path: path$1, plugin, processor, tags }) {
	if (plugin.config.requests.enabled) {
		const { schema } = buildOperationSchema$1(operation);
		processor.process({
			meta: {
				resource: "operation",
				resourceId: operation.id,
				role: "data"
			},
			naming: plugin.config.requests,
			namingAnchor: operation.id,
			path: path$1,
			plugin,
			schema,
			tags
		});
	}
	if (plugin.config.responses.enabled) {
		if (operation.responses) {
			const { response } = operationResponsesMap(operation);
			if (response) processor.process({
				meta: {
					resource: "operation",
					resourceId: operation.id,
					role: "responses"
				},
				naming: plugin.config.responses,
				namingAnchor: operation.id,
				path: [...path$1, "responses"],
				plugin,
				schema: response,
				tags
			});
		}
	}
}

//#endregion
//#region src/plugins/valibot/shared/webhook.ts
function irWebhookToAst$1({ operation, path: path$1, plugin, processor, tags }) {
	if (plugin.config.webhooks.enabled) {
		const { schema } = buildOperationSchema$1(operation);
		processor.process({
			meta: {
				resource: "webhook",
				resourceId: operation.id,
				role: "data"
			},
			naming: plugin.config.webhooks,
			namingAnchor: operation.id,
			path: path$1,
			plugin,
			schema,
			tags
		});
	}
}

//#endregion
//#region src/plugins/valibot/shared/export.ts
function exportAst$1({ ast, meta, naming, namingAnchor, path: path$1, plugin, schema, state, tags }) {
	const v = plugin.external("valibot.v");
	const name = pathToName(path$1, { anchor: namingAnchor });
	const symbol = plugin.symbol(applyNaming(name, naming), { meta: {
		category: "schema",
		path: path$1,
		tags,
		tool: "valibot",
		...meta
	} });
	const statement = $.const(symbol).export().$if(plugin.config.comments && createSchemaComment(schema), (c, v$1) => c.doc(v$1)).$if(state.hasLazyExpression["~ref"], (c) => c.type($.type(v).attr(ast.typeName || identifiers$1.types.GenericSchema))).assign(pipesToNode(ast.pipes, plugin));
	plugin.node(statement);
}

//#endregion
//#region src/plugins/valibot/v1/processor.ts
function createProcessor$3(plugin) {
	const processor = createSchemaProcessor();
	const hooks = [plugin.config["~hooks"]?.schemas, plugin.context.config.parser.hooks.schemas];
	function extractor(ctx$1) {
		if (processor.hasEmitted(ctx$1.path)) return ctx$1.schema;
		for (const hook of hooks) if (hook?.shouldExtract?.(ctx$1)) {
			process$1({
				namingAnchor: processor.context.anchor,
				tags: processor.context.tags,
				...ctx$1
			});
			return { $ref: pathToJsonPointer(ctx$1.path) };
		}
		return ctx$1.schema;
	}
	function process$1(ctx$1) {
		if (!processor.markEmitted(ctx$1.path)) return;
		processor.withContext({
			anchor: ctx$1.namingAnchor,
			tags: ctx$1.tags
		}, () => {
			const state = refs({
				hasLazyExpression: false,
				path: ctx$1.path,
				tags: ctx$1.tags
			});
			const ast = irSchemaToAst$3({
				plugin,
				schema: ctx$1.schema,
				schemaExtractor: extractor,
				state
			});
			exportAst$1({
				...ctx$1,
				ast,
				plugin,
				state
			});
		});
	}
	return { process: process$1 };
}

//#endregion
//#region src/plugins/valibot/v1/toAst/unknown.ts
function unknownToAst$3({ plugin }) {
	return $(plugin.external("valibot.v")).attr(identifiers$1.schemas.unknown).call();
}

//#endregion
//#region src/plugins/valibot/v1/toAst/array.ts
function arrayToAst$3(options) {
	const { plugin } = options;
	let { schema } = options;
	const result = { pipes: [] };
	const v = plugin.external("valibot.v");
	const functionName = $(v).attr(identifiers$1.schemas.array);
	if (!schema.items) {
		const expression = functionName.call(unknownToAst$3({
			...options,
			schema: { type: "unknown" }
		}));
		result.pipes.push(expression);
	} else {
		schema = deduplicateSchema({ schema });
		const itemExpressions = schema.items.map((item, index) => {
			const itemAst = irSchemaToAst$3({
				...options,
				schema: item,
				state: {
					...options.state,
					path: ref([
						...fromRef(options.state.path),
						"items",
						index
					])
				}
			});
			if (itemAst.hasLazyExpression) result.hasLazyExpression = true;
			return pipesToNode(itemAst.pipes, plugin);
		});
		if (itemExpressions.length === 1) {
			const expression = functionName.call(...itemExpressions);
			result.pipes.push(expression);
		} else {
			if (schema.logicalOperator === "and") {}
			const expression = functionName.call(unknownToAst$3({
				...options,
				schema: { type: "unknown" }
			}));
			result.pipes.push(expression);
		}
	}
	if (schema.minItems === schema.maxItems && schema.minItems !== void 0) {
		const expression = $(v).attr(identifiers$1.actions.length).call($.fromValue(schema.minItems));
		result.pipes.push(expression);
	} else {
		if (schema.minItems !== void 0) {
			const expression = $(v).attr(identifiers$1.actions.minLength).call($.fromValue(schema.minItems));
			result.pipes.push(expression);
		}
		if (schema.maxItems !== void 0) {
			const expression = $(v).attr(identifiers$1.actions.maxLength).call($.fromValue(schema.maxItems));
			result.pipes.push(expression);
		}
	}
	return result;
}

//#endregion
//#region src/plugins/valibot/v1/toAst/boolean.ts
function booleanToAst$3({ plugin, schema }) {
	const pipes$1 = [];
	const v = plugin.external("valibot.v");
	if (typeof schema.const === "boolean") {
		pipes$1.push($(v).attr(identifiers$1.schemas.literal).call($.literal(schema.const)));
		return pipesToNode(pipes$1, plugin);
	}
	pipes$1.push($(v).attr(identifiers$1.schemas.boolean).call());
	return pipesToNode(pipes$1, plugin);
}

//#endregion
//#region src/plugins/valibot/v1/toAst/enum.ts
function itemsNode$3(ctx$1) {
	const { schema } = ctx$1;
	const enumMembers = [];
	let isNullable = false;
	for (const item of schema.items ?? []) if (item.type === "string" && typeof item.const === "string") enumMembers.push($.literal(item.const));
	else if (item.type === "null" || item.const === null) isNullable = true;
	return {
		enumMembers,
		isNullable
	};
}
function baseNode$15(ctx$1) {
	const { symbols } = ctx$1;
	const { v } = symbols;
	const { enumMembers } = ctx$1.nodes.items(ctx$1);
	return $(v).attr(identifiers$1.schemas.picklist).call($.array(...enumMembers));
}
function nullableNode$3(ctx$1) {
	const { symbols } = ctx$1;
	const { v } = symbols;
	const { isNullable } = ctx$1.nodes.items(ctx$1);
	if (!isNullable) return;
	const currentNode = ctx$1.pipes.toNode(ctx$1.pipes.current, ctx$1.plugin);
	return $(v).attr(identifiers$1.schemas.nullable).call(currentNode);
}
function enumResolver$3(ctx$1) {
	const { enumMembers } = ctx$1.nodes.items(ctx$1);
	if (!enumMembers.length) return ctx$1.pipes.current;
	const baseExpression = ctx$1.nodes.base(ctx$1);
	ctx$1.pipes.push(ctx$1.pipes.current, baseExpression);
	const nullableExpression = ctx$1.nodes.nullable(ctx$1);
	if (nullableExpression) return nullableExpression;
	return ctx$1.pipes.current;
}
function enumToAst$3({ plugin, schema, state }) {
	const v = plugin.external("valibot.v");
	const { enumMembers } = itemsNode$3({
		$,
		nodes: {
			base: baseNode$15,
			items: itemsNode$3,
			nullable: nullableNode$3
		},
		pipes: {
			...pipes,
			current: []
		},
		plugin,
		schema,
		symbols: { v },
		utils: { state }
	});
	if (!enumMembers.length) return unknownToAst$3({
		plugin,
		schema: { type: "unknown" },
		state
	});
	const ctx$1 = {
		$,
		nodes: {
			base: baseNode$15,
			items: itemsNode$3,
			nullable: nullableNode$3
		},
		pipes: {
			...pipes,
			current: []
		},
		plugin,
		schema,
		symbols: { v },
		utils: { state }
	};
	const resolver = plugin.config["~resolvers"]?.enum;
	const node = resolver?.(ctx$1) ?? enumResolver$3(ctx$1);
	return ctx$1.pipes.toNode(node, plugin);
}

//#endregion
//#region src/plugins/valibot/v1/toAst/never.ts
function neverToAst$3({ plugin }) {
	return $(plugin.external("valibot.v")).attr(identifiers$1.schemas.never).call();
}

//#endregion
//#region src/plugins/valibot/v1/toAst/null.ts
function nullToAst$3({ plugin }) {
	return $(plugin.external("valibot.v")).attr(identifiers$1.schemas.null).call();
}

//#endregion
//#region src/plugins/shared/utils/formats.ts
const rangeErrors = (format, range) => ({
	maxError: `Invalid value: Expected ${format} to be <= ${range[1]}`,
	minError: `Invalid value: Expected ${format} to be >= ${range[0]}`
});
const integerRange = {
	int16: [-32768, 32767],
	int32: [-2147483648, 2147483647],
	int64: ["-9223372036854775808", "9223372036854775807"],
	int8: [-128, 127],
	uint16: [0, 65535],
	uint32: [0, 4294967295],
	uint64: ["0", "18446744073709551615"],
	uint8: [0, 255]
};
const getIntegerLimit = (format) => {
	if (!format) return;
	const range = integerRange[format];
	if (!range) return;
	const errors = rangeErrors(format, range);
	return {
		maxValue: range[1],
		minValue: range[0],
		...errors
	};
};

//#endregion
//#region src/plugins/valibot/v1/toAst/number.ts
function baseNode$14(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (ctx$1.utils.shouldCoerceToBigInt(schema.format)) return [$(v).attr(identifiers$1.schemas.union).call($.array($(v).attr(identifiers$1.schemas.number).call(), $(v).attr(identifiers$1.schemas.string).call(), $(v).attr(identifiers$1.schemas.bigInt).call())), $(v).attr(identifiers$1.actions.transform).call($.func().param("x").do($("BigInt").call("x").return()))];
	const pipes$1 = [];
	pipes$1.push($(v).attr(identifiers$1.schemas.number).call());
	if (schema.type === "integer") pipes$1.push($(v).attr(identifiers$1.actions.integer).call());
	return pipes$1;
}
function constNode$7(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (schema.const === void 0) return;
	return $(v).attr(identifiers$1.schemas.literal).call(ctx$1.utils.maybeBigInt(schema.const, schema.format));
}
function maxNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (schema.exclusiveMaximum !== void 0) return $(v).attr(identifiers$1.actions.ltValue).call(ctx$1.utils.maybeBigInt(schema.exclusiveMaximum, schema.format));
	if (schema.maximum !== void 0) return $(v).attr(identifiers$1.actions.maxValue).call(ctx$1.utils.maybeBigInt(schema.maximum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return $(v).attr(identifiers$1.actions.maxValue).call(ctx$1.utils.maybeBigInt(limit.maxValue, schema.format), $.literal(limit.maxError));
}
function minNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (schema.exclusiveMinimum !== void 0) return $(v).attr(identifiers$1.actions.gtValue).call(ctx$1.utils.maybeBigInt(schema.exclusiveMinimum, schema.format));
	if (schema.minimum !== void 0) return $(v).attr(identifiers$1.actions.minValue).call(ctx$1.utils.maybeBigInt(schema.minimum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return $(v).attr(identifiers$1.actions.minValue).call(ctx$1.utils.maybeBigInt(limit.minValue, schema.format), $.literal(limit.minError));
}
function numberResolver$3(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) return ctx$1.pipes.push(ctx$1.pipes.current, constNode$8);
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.pipes.push(ctx$1.pipes.current, baseNode$16);
	const minNode$4 = ctx$1.nodes.min(ctx$1);
	if (minNode$4) ctx$1.pipes.push(ctx$1.pipes.current, minNode$4);
	const maxNode$4 = ctx$1.nodes.max(ctx$1);
	if (maxNode$4) ctx$1.pipes.push(ctx$1.pipes.current, maxNode$4);
	return ctx$1.pipes.current;
}
function numberToNode$3({ plugin, schema }) {
	const ctx$1 = {
		$,
		nodes: {
			base: baseNode$14,
			const: constNode$7,
			max: maxNode$3,
			min: minNode$3
		},
		pipes: {
			...pipes,
			current: []
		},
		plugin,
		schema,
		symbols: { v: plugin.external("valibot.v") },
		utils: {
			getIntegerLimit,
			maybeBigInt,
			shouldCoerceToBigInt
		}
	};
	const resolver = plugin.config["~resolvers"]?.number;
	const node = resolver?.(ctx$1) ?? numberResolver$3(ctx$1);
	return ctx$1.pipes.toNode(node, plugin);
}

//#endregion
//#region src/plugins/valibot/v1/toAst/object.ts
function additionalPropertiesNode$3(ctx$1) {
	const { plugin, schema } = ctx$1;
	if (!schema.additionalProperties || !schema.additionalProperties.type) return;
	if (schema.additionalProperties.type === "never") return null;
	const additionalAst = irSchemaToAst$3({
		...ctx$1,
		schema: schema.additionalProperties,
		state: {
			...ctx$1.utils.state,
			path: ref([...fromRef(ctx$1.utils.state.path), "additionalProperties"])
		}
	});
	if (additionalAst.hasLazyExpression) ctx$1.utils.ast.hasLazyExpression = true;
	return pipes.toNode(additionalAst.pipes, plugin);
}
function baseNode$13(ctx$1) {
	const { nodes, symbols } = ctx$1;
	const { v } = symbols;
	const additional = nodes.additionalProperties(ctx$1);
	const shape = nodes.shape(ctx$1);
	if (additional === null) return $(v).attr(identifiers$1.schemas.strictObject).call(shape);
	if (additional) {
		if (shape.isEmpty) return $(v).attr(identifiers$1.schemas.record).call($(v).attr(identifiers$1.schemas.string).call(), additional);
		return $(v).attr(identifiers$1.schemas.objectWithRest).call(shape, additional);
	}
	return $(v).attr(identifiers$1.schemas.object).call(shape);
}
function objectResolver$3(ctx$1) {
	return ctx$1.nodes.base(ctx$1);
}
function shapeNode$3(ctx$1) {
	const { plugin, schema } = ctx$1;
	const shape = $.object().pretty();
	for (const name in schema.properties) {
		const property = schema.properties[name];
		const propertyAst = irSchemaToAst$3({
			...ctx$1,
			optional: !schema.required?.includes(name),
			schema: property,
			state: {
				...ctx$1.utils.state,
				path: ref([
					...fromRef(ctx$1.utils.state.path),
					"properties",
					name
				])
			}
		});
		if (propertyAst.hasLazyExpression) ctx$1.utils.ast.hasLazyExpression = true;
		shape.prop(name, pipes.toNode(propertyAst.pipes, plugin));
	}
	return shape;
}
function objectToAst$3(options) {
	const { plugin } = options;
	const ctx$1 = {
		...options,
		$,
		nodes: {
			additionalProperties: additionalPropertiesNode$3,
			base: baseNode$13,
			shape: shapeNode$3
		},
		pipes: {
			...pipes,
			current: []
		},
		symbols: { v: plugin.external("valibot.v") },
		utils: {
			ast: {},
			state: options.state
		}
	};
	const resolver = plugin.config["~resolvers"]?.object;
	const node = resolver?.(ctx$1) ?? objectResolver$3(ctx$1);
	ctx$1.utils.ast.pipes = [ctx$1.pipes.toNode(node, plugin)];
	return ctx$1.utils.ast;
}

//#endregion
//#region src/plugins/valibot/v1/toAst/string.ts
function baseNode$12(ctx$1) {
	const { v } = ctx$1.symbols;
	return $(v).attr(identifiers$1.schemas.string).call();
}
function constNode$6(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (typeof schema.const !== "string") return;
	return $(v).attr(identifiers$1.schemas.literal).call($.literal(schema.const));
}
function formatNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	switch (schema.format) {
		case "date": return $(v).attr(identifiers$1.actions.isoDate).call();
		case "date-time": return $(v).attr(identifiers$1.actions.isoTimestamp).call();
		case "email": return $(v).attr(identifiers$1.actions.email).call();
		case "ipv4":
		case "ipv6": return $(v).attr(identifiers$1.actions.ip).call();
		case "time": return $(v).attr(identifiers$1.actions.isoTimeSecond).call();
		case "uri": return $(v).attr(identifiers$1.actions.url).call();
		case "uuid": return $(v).attr(identifiers$1.actions.uuid).call();
	}
}
function lengthNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (schema.minLength === void 0 || schema.minLength !== schema.maxLength) return;
	return $(v).attr(identifiers$1.actions.length).call($.literal(schema.minLength));
}
function maxLengthNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (schema.maxLength === void 0) return;
	return $(v).attr(identifiers$1.actions.maxLength).call($.literal(schema.maxLength));
}
function minLengthNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (schema.minLength === void 0) return;
	return $(v).attr(identifiers$1.actions.minLength).call($.literal(schema.minLength));
}
function patternNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { v } = symbols;
	if (!schema.pattern) return;
	return $(v).attr(identifiers$1.actions.regex).call($.regexp(schema.pattern));
}
function stringResolver$3(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) return ctx$1.pipes.push(ctx$1.pipes.current, constNode$8);
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.pipes.push(ctx$1.pipes.current, baseNode$16);
	const formatNode$4 = ctx$1.nodes.format(ctx$1);
	if (formatNode$4) ctx$1.pipes.push(ctx$1.pipes.current, formatNode$4);
	const lengthNode$4 = ctx$1.nodes.length(ctx$1);
	if (lengthNode$4) ctx$1.pipes.push(ctx$1.pipes.current, lengthNode$4);
	else {
		const minLengthNode$4 = ctx$1.nodes.minLength(ctx$1);
		if (minLengthNode$4) ctx$1.pipes.push(ctx$1.pipes.current, minLengthNode$4);
		const maxLengthNode$4 = ctx$1.nodes.maxLength(ctx$1);
		if (maxLengthNode$4) ctx$1.pipes.push(ctx$1.pipes.current, maxLengthNode$4);
	}
	const patternNode$4 = ctx$1.nodes.pattern(ctx$1);
	if (patternNode$4) ctx$1.pipes.push(ctx$1.pipes.current, patternNode$4);
	return ctx$1.pipes.current;
}
function stringToNode$3({ plugin, schema }) {
	const ctx$1 = {
		$,
		nodes: {
			base: baseNode$12,
			const: constNode$6,
			format: formatNode$3,
			length: lengthNode$3,
			maxLength: maxLengthNode$3,
			minLength: minLengthNode$3,
			pattern: patternNode$3
		},
		pipes: {
			...pipes,
			current: []
		},
		plugin,
		schema,
		symbols: { v: plugin.external("valibot.v") }
	};
	const resolver = plugin.config["~resolvers"]?.string;
	const node = resolver?.(ctx$1) ?? stringResolver$3(ctx$1);
	return ctx$1.pipes.toNode(node, plugin);
}

//#endregion
//#region src/plugins/valibot/v1/toAst/tuple.ts
function tupleToAst$3(options) {
	const { plugin, schema } = options;
	const result = {};
	const v = plugin.external("valibot.v");
	if (schema.const && Array.isArray(schema.const)) {
		const tupleElements = schema.const.map((value) => $(v).attr(identifiers$1.schemas.literal).call($.fromValue(value)));
		result.pipes = [$(v).attr(identifiers$1.schemas.tuple).call($.array(...tupleElements))];
		return result;
	}
	if (schema.items) {
		const tupleElements = schema.items.map((item, index) => {
			const schemaPipes = irSchemaToAst$3({
				...options,
				schema: item,
				state: {
					...options.state,
					path: ref([
						...fromRef(options.state.path),
						"items",
						index
					])
				}
			});
			if (schemaPipes.hasLazyExpression) result.hasLazyExpression = true;
			return pipesToNode(schemaPipes.pipes, plugin);
		});
		result.pipes = [$(v).attr(identifiers$1.schemas.tuple).call($.array(...tupleElements))];
		return result;
	}
	return { pipes: [unknownToAst$3({
		...options,
		schema: { type: "unknown" }
	})] };
}

//#endregion
//#region src/plugins/valibot/v1/toAst/undefined.ts
function undefinedToAst$3({ plugin }) {
	return $(plugin.external("valibot.v")).attr(identifiers$1.schemas.undefined).call();
}

//#endregion
//#region src/plugins/valibot/v1/toAst/void.ts
function voidToAst$3({ plugin }) {
	return $(plugin.external("valibot.v")).attr(identifiers$1.schemas.void).call();
}

//#endregion
//#region src/plugins/valibot/v1/toAst/index.ts
function irSchemaWithTypeToAst$3({ schema, ...args }) {
	switch (schema.type) {
		case "array": return { expression: pipesToNode(arrayToAst$3({
			...args,
			schema
		}).pipes, args.plugin) };
		case "boolean": return { expression: booleanToAst$3({
			...args,
			schema
		}) };
		case "enum": return { expression: enumToAst$3({
			...args,
			schema
		}) };
		case "integer":
		case "number": return { expression: numberToNode$3({
			...args,
			schema
		}) };
		case "never": return { expression: neverToAst$3({
			...args,
			schema
		}) };
		case "null": return { expression: nullToAst$3({
			...args,
			schema
		}) };
		case "object": return { expression: pipesToNode(objectToAst$3({
			...args,
			schema
		}).pipes, args.plugin) };
		case "string": return { expression: shouldCoerceToBigInt(schema.format) ? numberToNode$3({
			...args,
			schema: {
				...schema,
				type: "number"
			}
		}) : stringToNode$3({
			...args,
			schema
		}) };
		case "tuple": return { expression: pipesToNode(tupleToAst$3({
			...args,
			schema
		}).pipes, args.plugin) };
		case "undefined": return { expression: undefinedToAst$3({
			...args,
			schema
		}) };
		case "unknown": return { expression: unknownToAst$3({
			...args,
			schema
		}) };
		case "void": return { expression: voidToAst$3({
			...args,
			schema
		}) };
	}
}

//#endregion
//#region src/plugins/valibot/v1/plugin.ts
function irSchemaToAst$3({ optional, plugin, schema, schemaExtractor, state }) {
	if (schemaExtractor && !schema.$ref) {
		const extracted = schemaExtractor({
			meta: {
				resource: "definition",
				resourceId: pathToJsonPointer(fromRef(state.path))
			},
			naming: plugin.config.definitions,
			path: fromRef(state.path),
			plugin,
			schema
		});
		if (extracted !== schema) schema = extracted;
	}
	const ast = { pipes: [] };
	const v = plugin.external("valibot.v");
	if (schema.$ref) {
		const query = {
			category: "schema",
			resource: "definition",
			resourceId: schema.$ref,
			tool: "valibot"
		};
		const refSymbol = plugin.referenceSymbol(query);
		if (plugin.isSymbolRegistered(query)) {
			const ref$1 = $(refSymbol);
			ast.pipes.push(ref$1);
		} else {
			const lazyExpression = $(v).attr(identifiers$1.schemas.lazy).call($.func().do($(refSymbol).return()));
			ast.pipes.push(lazyExpression);
			state.hasLazyExpression["~ref"] = true;
		}
	} else if (schema.type) {
		const typeAst = irSchemaWithTypeToAst$3({
			plugin,
			schema,
			schemaExtractor,
			state
		});
		ast.typeName = typeAst.anyType;
		ast.pipes.push(typeAst.expression);
		if (plugin.config.metadata && schema.description) {
			const expression = $(v).attr(identifiers$1.actions.metadata).call($.object().prop("description", $.literal(schema.description)));
			ast.pipes.push(expression);
		}
	} else if (schema.items) {
		schema = deduplicateSchema({ schema });
		if (schema.items) {
			const itemsAst = schema.items.map((item, index) => {
				return pipesToNode(irSchemaToAst$3({
					plugin,
					schema: item,
					schemaExtractor,
					state: {
						...state,
						path: ref([
							...fromRef(state.path),
							"items",
							index
						])
					}
				}).pipes, plugin);
			});
			if (schema.logicalOperator === "and") {
				const intersectExpression = $(v).attr(identifiers$1.schemas.intersect).call($.array(...itemsAst));
				ast.pipes.push(intersectExpression);
			} else {
				const unionExpression = $(v).attr(identifiers$1.schemas.union).call($.array(...itemsAst));
				ast.pipes.push(unionExpression);
			}
		} else {
			const schemaPipes = irSchemaToAst$3({
				plugin,
				schema,
				state
			});
			ast.pipes.push(...schemaPipes.pipes);
		}
	} else {
		const typeAst = irSchemaWithTypeToAst$3({
			plugin,
			schema: { type: "unknown" },
			schemaExtractor,
			state
		});
		ast.typeName = typeAst.anyType;
		ast.pipes.push(typeAst.expression);
	}
	if (ast.pipes.length) {
		if (schema.accessScope === "read") {
			const readonlyExpression = $(v).attr(identifiers$1.actions.readonly).call();
			ast.pipes.push(readonlyExpression);
		}
		if (schema.default !== void 0) ast.pipes = [$(v).attr(identifiers$1.schemas.optional).call(pipesToNode(ast.pipes, plugin), schema.type === "integer" || schema.type === "number" ? maybeBigInt(schema.default, schema.format) : $.fromValue(schema.default))];
		else if (optional) ast.pipes = [$(v).attr(identifiers$1.schemas.optional).call(pipesToNode(ast.pipes, plugin))];
	}
	return ast;
}
const handlerV1 = ({ plugin }) => {
	plugin.symbol("v", {
		external: "valibot",
		importKind: "namespace",
		meta: {
			category: "external",
			resource: "valibot.v"
		}
	});
	const processor = createProcessor$3(plugin);
	plugin.forEach("operation", "parameter", "requestBody", "schema", "webhook", (event) => {
		switch (event.type) {
			case "operation":
				irOperationToAst$1({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
			case "parameter":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.parameter.schema,
					tags: event.tags
				});
				break;
			case "requestBody":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.requestBody.schema,
					tags: event.tags
				});
				break;
			case "schema":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.schema,
					tags: event.tags
				});
				break;
			case "webhook":
				irWebhookToAst$1({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
		}
	});
};

//#endregion
//#region src/plugins/valibot/plugin.ts
const handler$1 = (args) => handlerV1(args);

//#endregion
//#region src/plugins/valibot/config.ts
const defaultConfig$1 = {
	api: new Api$1(),
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false,
		metadata: false
	},
	handler: handler$1,
	name: "valibot",
	resolveConfig: (plugin, context) => {
		plugin.config.definitions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "v{{name}}"
			},
			mappers,
			value: plugin.config.definitions
		});
		plugin.config.requests = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "v{{name}}Data"
			},
			mappers,
			value: plugin.config.requests
		});
		plugin.config.responses = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "v{{name}}Response"
			},
			mappers,
			value: plugin.config.responses
		});
		plugin.config.webhooks = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "v{{name}}WebhookRequest"
			},
			mappers,
			value: plugin.config.webhooks
		});
	},
	tags: ["validator"]
};
/**
* Type helper for Valibot plugin, returns {@link Plugin.Config} object
*/
const defineConfig$1 = definePluginConfig(defaultConfig$1);

//#endregion
//#region src/plugins/zod/constants.ts
const identifiers = {
	ZodMiniOptional: "ZodMiniOptional",
	ZodOptional: "ZodOptional",
	_default: "_default",
	and: "and",
	array: "array",
	bigint: "bigint",
	boolean: "boolean",
	check: "check",
	coerce: "coerce",
	date: "date",
	datetime: "datetime",
	default: "default",
	describe: "describe",
	email: "email",
	enum: "enum",
	globalRegistry: "globalRegistry",
	gt: "gt",
	gte: "gte",
	infer: "infer",
	int: "int",
	intersection: "intersection",
	ip: "ip",
	ipv4: "ipv4",
	ipv6: "ipv6",
	iso: "iso",
	lazy: "lazy",
	length: "length",
	literal: "literal",
	lt: "lt",
	lte: "lte",
	max: "max",
	maxLength: "maxLength",
	maximum: "maximum",
	min: "min",
	minLength: "minLength",
	minimum: "minimum",
	never: "never",
	null: "null",
	nullable: "nullable",
	number: "number",
	object: "object",
	optional: "optional",
	parseAsync: "parseAsync",
	readonly: "readonly",
	record: "record",
	regex: "regex",
	register: "register",
	string: "string",
	time: "time",
	tuple: "tuple",
	undefined: "undefined",
	union: "union",
	unknown: "unknown",
	url: "url",
	uuid: "uuid",
	void: "void"
};

//#endregion
//#region src/plugins/zod/mini/api.ts
const validatorResolver$2 = (ctx$1) => {
	const { schema } = ctx$1.symbols;
	return $(schema).attr(identifiers.parseAsync).call("data").await().return();
};
const createRequestValidatorMini = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "zod"
	});
	if (!symbol) return;
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		operation,
		plugin,
		symbols: {
			schema: symbol,
			z
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.request, validatorResolver$2];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};
const createResponseValidatorMini = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "responses",
		tool: "zod"
	});
	if (!symbol) return;
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		operation,
		plugin,
		symbols: {
			schema: symbol,
			z
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.response, validatorResolver$2];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};

//#endregion
//#region src/plugins/zod/v3/api.ts
const validatorResolver$1 = (ctx$1) => {
	const { schema } = ctx$1.symbols;
	return $(schema).attr(identifiers.parseAsync).call("data").await().return();
};
const createRequestValidatorV3 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "zod"
	});
	if (!symbol) return;
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		operation,
		plugin,
		symbols: {
			schema: symbol,
			z
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.request, validatorResolver$1];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};
const createResponseValidatorV3 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "responses",
		tool: "zod"
	});
	if (!symbol) return;
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		operation,
		plugin,
		symbols: {
			schema: symbol,
			z
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.response, validatorResolver$1];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};

//#endregion
//#region src/plugins/zod/v4/api.ts
const validatorResolver = (ctx$1) => {
	const { schema } = ctx$1.symbols;
	return $(schema).attr(identifiers.parseAsync).call("data").await().return();
};
const createRequestValidatorV4 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "data",
		tool: "zod"
	});
	if (!symbol) return;
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		operation,
		plugin,
		symbols: {
			schema: symbol,
			z
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.request, validatorResolver];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};
const createResponseValidatorV4 = ({ operation, plugin }) => {
	const symbol = plugin.getSymbol({
		category: "schema",
		resource: "operation",
		resourceId: operation.id,
		role: "responses",
		tool: "zod"
	});
	if (!symbol) return;
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		operation,
		plugin,
		symbols: {
			schema: symbol,
			z
		}
	};
	const validator = plugin.config["~resolvers"]?.validator;
	const candidates = [typeof validator === "function" ? validator : validator?.response, validatorResolver];
	for (const candidate of candidates) {
		const statements = candidate?.(ctx$1);
		if (statements === null) return;
		if (statements !== void 0) return $.func().async().param("data").do(...statements instanceof Array ? statements : [statements]);
	}
};

//#endregion
//#region src/plugins/zod/api.ts
var Api = class {
	createRequestValidator(args) {
		const { plugin } = args;
		switch (plugin.config.compatibilityVersion) {
			case 3: return createRequestValidatorV3(args);
			case "mini": return createRequestValidatorMini(args);
			case 4:
			default: return createRequestValidatorV4(args);
		}
	}
	createResponseValidator(args) {
		const { plugin } = args;
		switch (plugin.config.compatibilityVersion) {
			case 3: return createResponseValidatorV3(args);
			case "mini": return createResponseValidatorMini(args);
			case 4:
			default: return createResponseValidatorV4(args);
		}
	}
};

//#endregion
//#region src/plugins/zod/shared/module.ts
function getZodModule({ plugin }) {
	const version = plugin.package.getVersion("zod");
	if (version) {
		if (plugin.package.satisfies(version, "<4.0.0")) switch (plugin.config.compatibilityVersion) {
			case 3:
			default: return "zod";
			case 4: return "zod/v4";
			case "mini": return "zod/v4-mini";
		}
	}
	switch (plugin.config.compatibilityVersion) {
		case 3: return "zod/v3";
		case 4:
		default: return "zod";
		case "mini": return "zod/mini";
	}
}

//#endregion
//#region src/plugins/zod/shared/operation-schema.ts
function buildOperationSchema(operation) {
	const requiredProperties = /* @__PURE__ */ new Set();
	const schema = {
		properties: {
			body: { type: "never" },
			path: { type: "never" },
			query: { type: "never" }
		},
		type: "object"
	};
	if (operation.parameters) for (const location of [
		"header",
		"path",
		"query"
	]) {
		const params = operation.parameters[location];
		if (!params) continue;
		const properties = {};
		const required = [];
		const propKey = location === "header" ? "headers" : location;
		for (const key in params) {
			const parameter = params[key];
			properties[parameter.name] = parameter.schema;
			if (parameter.required) {
				required.push(parameter.name);
				requiredProperties.add(propKey);
			}
		}
		if (Object.keys(properties).length) schema.properties[propKey] = {
			properties,
			required,
			type: "object"
		};
	}
	if (operation.body) {
		schema.properties.body = operation.body.schema;
		if (operation.body.required) requiredProperties.add("body");
	}
	schema.required = [...requiredProperties];
	return {
		required: schema.required,
		schema
	};
}

//#endregion
//#region src/plugins/zod/shared/operation.ts
function irOperationToAst({ operation, path: path$1, plugin, processor, tags }) {
	if (plugin.config.requests.enabled) {
		const { schema } = buildOperationSchema(operation);
		processor.process({
			meta: {
				resource: "operation",
				resourceId: operation.id,
				role: "data"
			},
			naming: plugin.config.requests,
			namingAnchor: operation.id,
			path: path$1,
			plugin,
			schema,
			tags
		});
	}
	if (plugin.config.responses.enabled) {
		if (operation.responses) {
			const { response } = operationResponsesMap(operation);
			if (response) processor.process({
				meta: {
					resource: "operation",
					resourceId: operation.id,
					role: "responses"
				},
				naming: plugin.config.responses,
				namingAnchor: operation.id,
				path: [...path$1, "responses"],
				plugin,
				schema: response,
				tags
			});
		}
	}
}

//#endregion
//#region src/plugins/zod/shared/webhook.ts
function irWebhookToAst({ operation, path: path$1, plugin, processor, tags }) {
	if (plugin.config.webhooks.enabled) {
		const { schema } = buildOperationSchema(operation);
		processor.process({
			meta: {
				resource: "webhook",
				resourceId: operation.id,
				role: "data"
			},
			naming: plugin.config.webhooks,
			namingAnchor: operation.id,
			path: path$1,
			plugin,
			schema,
			tags
		});
	}
}

//#endregion
//#region src/plugins/zod/shared/export.ts
function exportAst({ ast, meta, naming, namingAnchor, path: path$1, plugin, schema, tags }) {
	const z = plugin.external("zod.z");
	const name = pathToName(path$1, { anchor: namingAnchor });
	const symbol = plugin.symbol(applyNaming(name, naming), { meta: {
		category: "schema",
		path: path$1,
		tags,
		tool: "zod",
		...meta
	} });
	const typeInferSymbol = naming.types.infer.enabled ? plugin.symbol(applyNaming(name, naming.types.infer), { meta: {
		category: "type",
		path: path$1,
		tags,
		tool: "zod",
		variant: "infer",
		...meta
	} }) : void 0;
	const statement = $.const(symbol).export().$if(plugin.config.comments && createSchemaComment(schema), (c, v) => c.doc(v)).$if(ast.typeName, (c, v) => c.type($.type(z).attr(v))).assign(ast.expression);
	plugin.node(statement);
	if (typeInferSymbol) {
		const inferType = $.type.alias(typeInferSymbol).export().type($.type(z).attr(identifiers.infer).generic($(symbol).typeofType()));
		plugin.node(inferType);
	}
}

//#endregion
//#region src/plugins/zod/mini/processor.ts
function createProcessor$2(plugin) {
	const processor = createSchemaProcessor();
	const hooks = [plugin.config["~hooks"]?.schemas, plugin.context.config.parser.hooks.schemas];
	function extractor(ctx$1) {
		if (processor.hasEmitted(ctx$1.path)) return ctx$1.schema;
		for (const hook of hooks) if (hook?.shouldExtract?.(ctx$1)) {
			process$1({
				namingAnchor: processor.context.anchor,
				tags: processor.context.tags,
				...ctx$1
			});
			return { $ref: pathToJsonPointer(ctx$1.path) };
		}
		return ctx$1.schema;
	}
	function process$1(ctx$1) {
		if (!processor.markEmitted(ctx$1.path)) return;
		processor.withContext({
			anchor: ctx$1.namingAnchor,
			tags: ctx$1.tags
		}, () => {
			const state = refs({
				hasLazyExpression: false,
				path: ctx$1.path,
				tags: ctx$1.tags
			});
			const ast = irSchemaToAst$2({
				plugin,
				schema: ctx$1.schema,
				schemaExtractor: extractor,
				state
			});
			exportAst({
				...ctx$1,
				ast,
				plugin,
				state
			});
		});
	}
	return { process: process$1 };
}

//#endregion
//#region src/plugins/zod/mini/toAst/unknown.ts
const unknownToAst$2 = ({ plugin }) => {
	const z = plugin.external("zod.z");
	const result = {};
	result.expression = $(z).attr(identifiers.unknown).call();
	return result;
};

//#endregion
//#region src/plugins/zod/mini/toAst/array.ts
const arrayToAst$2 = ({ plugin, schema, state }) => {
	const z = plugin.external("zod.z");
	const result = {};
	const functionName = $(z).attr(identifiers.array);
	if (!schema.items) result.expression = functionName.call(unknownToAst$2({
		plugin,
		schema: { type: "unknown" },
		state
	}).expression);
	else {
		schema = deduplicateSchema({ schema });
		const itemExpressions = schema.items.map((item, index) => {
			const itemAst = irSchemaToAst$2({
				plugin,
				schema: item,
				state: {
					...state,
					path: ref([
						...fromRef(state.path),
						"items",
						index
					])
				}
			});
			if (itemAst.hasLazyExpression) result.hasLazyExpression = true;
			return itemAst.expression;
		});
		if (itemExpressions.length === 1) result.expression = functionName.call(...itemExpressions);
		else if (schema.logicalOperator === "and") {
			const firstSchema = schema.items[0];
			let intersectionExpression;
			if (firstSchema.logicalOperator === "or" || firstSchema.type && firstSchema.type !== "object") intersectionExpression = $(z).attr(identifiers.intersection).call(...itemExpressions);
			else {
				intersectionExpression = itemExpressions[0];
				for (let i = 1; i < itemExpressions.length; i++) intersectionExpression = $(z).attr(identifiers.intersection).call(intersectionExpression, itemExpressions[i]);
			}
			result.expression = functionName.call(intersectionExpression);
		} else result.expression = $(z).attr(identifiers.array).call($(z).attr(identifiers.union).call($.array(...itemExpressions)));
	}
	const checks = [];
	if (schema.minItems === schema.maxItems && schema.minItems !== void 0) checks.push($(z).attr(identifiers.length).call($.fromValue(schema.minItems)));
	else {
		if (schema.minItems !== void 0) checks.push($(z).attr(identifiers.minLength).call($.fromValue(schema.minItems)));
		if (schema.maxItems !== void 0) checks.push($(z).attr(identifiers.maxLength).call($.fromValue(schema.maxItems)));
	}
	if (checks.length > 0) result.expression = result.expression.attr(identifiers.check).call(...checks);
	return result;
};

//#endregion
//#region src/plugins/zod/mini/toAst/boolean.ts
function booleanToAst$2({ plugin, schema }) {
	const result = {};
	let chain;
	const z = plugin.external("zod.z");
	if (typeof schema.const === "boolean") {
		chain = $(z).attr(identifiers.literal).call($.literal(schema.const));
		result.expression = chain;
		return result;
	}
	chain = $(z).attr(identifiers.boolean).call();
	result.expression = chain;
	return result;
}

//#endregion
//#region src/plugins/zod/mini/toAst/enum.ts
function itemsNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	const enumMembers = [];
	const literalMembers = [];
	let isNullable = false;
	let allStrings = true;
	for (const item of schema.items ?? []) if (item.type === "string" && typeof item.const === "string") {
		const literal = $.literal(item.const);
		enumMembers.push(literal);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if ((item.type === "number" || item.type === "integer") && typeof item.const === "number") {
		allStrings = false;
		const literal = $.literal(item.const);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if (item.type === "boolean" && typeof item.const === "boolean") {
		allStrings = false;
		const literal = $.literal(item.const);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if (item.type === "null" || item.const === null) isNullable = true;
	return {
		allStrings,
		enumMembers,
		isNullable,
		literalMembers
	};
}
function baseNode$11(ctx$1) {
	const { symbols } = ctx$1;
	const { z } = symbols;
	const { allStrings, enumMembers, literalMembers } = ctx$1.nodes.items(ctx$1);
	if (allStrings && enumMembers.length > 0) return $(z).attr(identifiers.enum).call($.array(...enumMembers));
	else if (literalMembers.length === 1) return literalMembers[0];
	else return $(z).attr(identifiers.union).call($.array(...literalMembers));
}
function nullableNode$2(ctx$1) {
	const { chain, symbols } = ctx$1;
	const { z } = symbols;
	const { isNullable } = ctx$1.nodes.items(ctx$1);
	if (!isNullable) return;
	return $(z).attr(identifiers.nullable).call(chain.current);
}
function enumResolver$2(ctx$1) {
	const { literalMembers } = ctx$1.nodes.items(ctx$1);
	if (!literalMembers.length) return ctx$1.chain.current;
	const baseExpression = ctx$1.nodes.base(ctx$1);
	ctx$1.chain.current = baseExpression;
	const nullableExpression = ctx$1.nodes.nullable(ctx$1);
	if (nullableExpression) ctx$1.chain.current = nullableExpression;
	return ctx$1.chain.current;
}
const enumToAst$2 = ({ plugin, schema, state }) => {
	const z = plugin.external("zod.z");
	const { literalMembers } = itemsNode$2({
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$11,
			items: itemsNode$2,
			nullable: nullableNode$2
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast: {},
			state
		}
	});
	if (!literalMembers.length) return unknownToAst$2({
		plugin,
		schema: { type: "unknown" },
		state
	});
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$11,
			items: itemsNode$2,
			nullable: nullableNode$2
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast: {},
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.enum;
	return { expression: resolver?.(ctx$1) ?? enumResolver$2(ctx$1) };
};

//#endregion
//#region src/plugins/zod/mini/toAst/never.ts
const neverToAst$2 = ({ plugin }) => {
	const z = plugin.external("zod.z");
	const result = {};
	result.expression = $(z).attr(identifiers.never).call();
	return result;
};

//#endregion
//#region src/plugins/zod/mini/toAst/null.ts
const nullToAst$2 = ({ plugin }) => {
	const z = plugin.external("zod.z");
	const result = {};
	result.expression = $(z).attr(identifiers.null).call();
	return result;
};

//#endregion
//#region src/plugins/zod/mini/toAst/number.ts
function baseNode$10(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (ctx$1.utils.shouldCoerceToBigInt(schema.format)) return $(z).attr(identifiers.coerce).attr(identifiers.bigint).call();
	let chain = $(z).attr(identifiers.number).call();
	if (schema.type === "integer") chain = $(z).attr(identifiers.int).call();
	return chain;
}
function constNode$5(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.const === void 0) return;
	return $(z).attr(identifiers.literal).call(ctx$1.utils.maybeBigInt(schema.const, schema.format));
}
function maxNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.exclusiveMaximum !== void 0) return $(z).attr(identifiers.lt).call(ctx$1.utils.maybeBigInt(schema.exclusiveMaximum, schema.format));
	if (schema.maximum !== void 0) return $(z).attr(identifiers.lte).call(ctx$1.utils.maybeBigInt(schema.maximum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return $(z).attr(identifiers.maximum).call(ctx$1.utils.maybeBigInt(limit.maxValue, schema.format), $.object().prop("error", $.literal(limit.maxError)));
}
function minNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.exclusiveMinimum !== void 0) return $(z).attr(identifiers.gt).call(ctx$1.utils.maybeBigInt(schema.exclusiveMinimum, schema.format));
	if (schema.minimum !== void 0) return $(z).attr(identifiers.gte).call(ctx$1.utils.maybeBigInt(schema.minimum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return $(z).attr(identifiers.minimum).call(ctx$1.utils.maybeBigInt(limit.minValue, schema.format), $.object().prop("error", $.literal(limit.minError)));
}
function numberResolver$2(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) {
		ctx$1.chain.current = constNode$8;
		return ctx$1.chain.current;
	}
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.chain.current = baseNode$16;
	const checks = [];
	const minNode$4 = ctx$1.nodes.min(ctx$1);
	if (minNode$4) checks.push(minNode$4);
	const maxNode$4 = ctx$1.nodes.max(ctx$1);
	if (maxNode$4) checks.push(maxNode$4);
	if (checks.length > 0) ctx$1.chain.current = ctx$1.chain.current.attr(identifiers.check).call(...checks);
	return ctx$1.chain.current;
}
const numberToNode$2 = ({ plugin, schema, state }) => {
	const ast = {};
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$10,
			const: constNode$5,
			max: maxNode$2,
			min: minNode$2
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast,
			getIntegerLimit,
			maybeBigInt,
			shouldCoerceToBigInt,
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.number;
	ast.expression = resolver?.(ctx$1) ?? numberResolver$2(ctx$1);
	return ast;
};

//#endregion
//#region src/plugins/zod/mini/toAst/object.ts
function additionalPropertiesNode$2(ctx$1) {
	const { plugin, schema } = ctx$1;
	if (!schema.additionalProperties || schema.properties && Object.keys(schema.properties).length > 0) return;
	const additionalAst = irSchemaToAst$2({
		plugin,
		schema: schema.additionalProperties,
		state: {
			...ctx$1.utils.state,
			path: ref([...fromRef(ctx$1.utils.state.path), "additionalProperties"])
		}
	});
	if (additionalAst.hasLazyExpression) ctx$1.utils.ast.hasLazyExpression = true;
	return additionalAst.expression;
}
function baseNode$9(ctx$1) {
	const { nodes, symbols } = ctx$1;
	const { z } = symbols;
	const additional = nodes.additionalProperties(ctx$1);
	const shape = nodes.shape(ctx$1);
	if (additional) return $(z).attr(identifiers.record).call($(z).attr(identifiers.string).call(), additional);
	return $(z).attr(identifiers.object).call(shape);
}
function objectResolver$2(ctx$1) {
	return ctx$1.nodes.base(ctx$1);
}
function shapeNode$2(ctx$1) {
	const { plugin, schema } = ctx$1;
	const shape = $.object().pretty();
	for (const name in schema.properties) {
		const property = schema.properties[name];
		const propertyAst = irSchemaToAst$2({
			optional: !schema.required?.includes(name),
			plugin,
			schema: property,
			state: {
				...ctx$1.utils.state,
				path: ref([
					...fromRef(ctx$1.utils.state.path),
					"properties",
					name
				])
			}
		});
		if (propertyAst.hasLazyExpression) {
			ctx$1.utils.ast.hasLazyExpression = true;
			shape.getter(name, propertyAst.expression.return());
		} else shape.prop(name, propertyAst.expression);
	}
	return shape;
}
const objectToAst$2 = ({ plugin, schema, state }) => {
	const ast = {};
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			additionalProperties: additionalPropertiesNode$2,
			base: baseNode$9,
			shape: shapeNode$2
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast,
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.object;
	ast.expression = resolver?.(ctx$1) ?? objectResolver$2(ctx$1);
	return ast;
};

//#endregion
//#region src/plugins/zod/mini/toAst/string.ts
function baseNode$8(ctx$1) {
	const { z } = ctx$1.symbols;
	return $(z).attr(identifiers.string).call();
}
function constNode$4(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (typeof schema.const !== "string") return;
	return $(z).attr(identifiers.literal).call($.literal(schema.const));
}
function formatNode$2(ctx$1) {
	const { plugin, schema, symbols } = ctx$1;
	const { z } = symbols;
	switch (schema.format) {
		case "date": return $(z).attr(identifiers.iso).attr(identifiers.date).call();
		case "date-time": {
			const obj = $.object().$if(plugin.config.dates.offset, (o) => o.prop("offset", $.literal(true))).$if(plugin.config.dates.local, (o) => o.prop("local", $.literal(true)));
			return $(z).attr(identifiers.iso).attr(identifiers.datetime).call(obj.hasProps() ? obj : void 0);
		}
		case "email": return $(z).attr(identifiers.email).call();
		case "ipv4": return $(z).attr(identifiers.ipv4).call();
		case "ipv6": return $(z).attr(identifiers.ipv6).call();
		case "time": return $(z).attr(identifiers.iso).attr(identifiers.time).call();
		case "uri": return $(z).attr(identifiers.url).call();
		case "uuid": return $(z).attr(identifiers.uuid).call();
	}
}
function lengthNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.minLength === void 0 || schema.minLength !== schema.maxLength) return;
	return $(z).attr(identifiers.length).call($.literal(schema.minLength));
}
function maxLengthNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.maxLength === void 0) return;
	return $(z).attr(identifiers.maxLength).call($.literal(schema.maxLength));
}
function minLengthNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.minLength === void 0) return;
	return $(z).attr(identifiers.minLength).call($.literal(schema.minLength));
}
function patternNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (!schema.pattern) return;
	return $(z).attr(identifiers.regex).call($.regexp(schema.pattern));
}
function stringResolver$2(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) {
		ctx$1.chain.current = constNode$8;
		return ctx$1.chain.current;
	}
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.chain.current = baseNode$16;
	const formatNode$4 = ctx$1.nodes.format(ctx$1);
	if (formatNode$4) ctx$1.chain.current = formatNode$4;
	const checks = [];
	const lengthNode$4 = ctx$1.nodes.length(ctx$1);
	if (lengthNode$4) checks.push(lengthNode$4);
	else {
		const minLengthNode$4 = ctx$1.nodes.minLength(ctx$1);
		if (minLengthNode$4) checks.push(minLengthNode$4);
		const maxLengthNode$4 = ctx$1.nodes.maxLength(ctx$1);
		if (maxLengthNode$4) checks.push(maxLengthNode$4);
	}
	const patternNode$4 = ctx$1.nodes.pattern(ctx$1);
	if (patternNode$4) checks.push(patternNode$4);
	if (checks.length > 0) ctx$1.chain.current = ctx$1.chain.current.attr(identifiers.check).call(...checks);
	return ctx$1.chain.current;
}
const stringToNode$2 = ({ plugin, schema }) => {
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$8,
			const: constNode$4,
			format: formatNode$2,
			length: lengthNode$2,
			maxLength: maxLengthNode$2,
			minLength: minLengthNode$2,
			pattern: patternNode$2
		},
		plugin,
		schema,
		symbols: { z }
	};
	const resolver = plugin.config["~resolvers"]?.string;
	return { expression: resolver?.(ctx$1) ?? stringResolver$2(ctx$1) };
};

//#endregion
//#region src/plugins/zod/mini/toAst/tuple.ts
const tupleToAst$2 = ({ plugin, schema, state }) => {
	const z = plugin.external("zod.z");
	const result = {};
	if (schema.const && Array.isArray(schema.const)) {
		const tupleElements$1 = schema.const.map((value) => $(z).attr(identifiers.literal).call($.fromValue(value)));
		result.expression = $(z).attr(identifiers.tuple).call($.array(...tupleElements$1));
		return result;
	}
	const tupleElements = [];
	if (schema.items) schema.items.forEach((item, index) => {
		const itemSchema = irSchemaToAst$2({
			plugin,
			schema: item,
			state: {
				...state,
				path: ref([
					...fromRef(state.path),
					"items",
					index
				])
			}
		});
		tupleElements.push(itemSchema.expression);
		if (itemSchema.hasLazyExpression) result.hasLazyExpression = true;
	});
	result.expression = $(z).attr(identifiers.tuple).call($.array(...tupleElements));
	return result;
};

//#endregion
//#region src/plugins/zod/mini/toAst/undefined.ts
const undefinedToAst$2 = ({ plugin }) => {
	const z = plugin.external("zod.z");
	const result = {};
	result.expression = $(z).attr(identifiers.undefined).call();
	return result;
};

//#endregion
//#region src/plugins/zod/mini/toAst/void.ts
const voidToAst$2 = ({ plugin }) => {
	const z = plugin.external("zod.z");
	const result = {};
	result.expression = $(z).attr(identifiers.void).call();
	return result;
};

//#endregion
//#region src/plugins/zod/mini/toAst/index.ts
function irSchemaWithTypeToAst$2({ schema, ...args }) {
	switch (schema.type) {
		case "array": return arrayToAst$2({
			...args,
			schema
		});
		case "boolean": return booleanToAst$2({
			...args,
			schema
		});
		case "enum": return enumToAst$2({
			...args,
			schema
		});
		case "integer":
		case "number": return numberToNode$2({
			...args,
			schema
		});
		case "never": return neverToAst$2({
			...args,
			schema
		});
		case "null": return nullToAst$2({
			...args,
			schema
		});
		case "object": return objectToAst$2({
			...args,
			schema
		});
		case "string": return shouldCoerceToBigInt(schema.format) ? numberToNode$2({
			...args,
			schema: {
				...schema,
				type: "number"
			}
		}) : stringToNode$2({
			...args,
			schema
		});
		case "tuple": return tupleToAst$2({
			...args,
			schema
		});
		case "undefined": return undefinedToAst$2({
			...args,
			schema
		});
		case "unknown": return unknownToAst$2({
			...args,
			schema
		});
		case "void": return voidToAst$2({
			...args,
			schema
		});
	}
}

//#endregion
//#region src/plugins/zod/mini/plugin.ts
function irSchemaToAst$2({ optional, plugin, schema, schemaExtractor, state }) {
	if (schemaExtractor && !schema.$ref) {
		const extracted = schemaExtractor({
			meta: {
				resource: "definition",
				resourceId: pathToJsonPointer(fromRef(state.path))
			},
			naming: plugin.config.definitions,
			path: fromRef(state.path),
			plugin,
			schema
		});
		if (extracted !== schema) schema = extracted;
	}
	let ast = {};
	const z = plugin.external("zod.z");
	if (schema.$ref) {
		const query = {
			category: "schema",
			resource: "definition",
			resourceId: schema.$ref,
			tool: "zod"
		};
		const refSymbol = plugin.referenceSymbol(query);
		if (plugin.isSymbolRegistered(query)) ast.expression = $(refSymbol);
		else {
			ast.expression = $(z).attr(identifiers.lazy).call($.func().returns("any").do($(refSymbol).return()));
			ast.hasLazyExpression = true;
			state.hasLazyExpression["~ref"] = true;
		}
	} else if (schema.type) {
		const typeAst = irSchemaWithTypeToAst$2({
			plugin,
			schema,
			state
		});
		ast.expression = typeAst.expression;
		ast.hasLazyExpression = typeAst.hasLazyExpression;
		if (plugin.config.metadata && schema.description) ast.expression = ast.expression.attr(identifiers.register).call($(z).attr(identifiers.globalRegistry), $.object().pretty().prop("description", $.literal(schema.description)));
	} else if (schema.items) {
		schema = deduplicateSchema({ schema });
		if (schema.items) {
			const itemSchemas = schema.items.map((item, index) => irSchemaToAst$2({
				plugin,
				schema: item,
				state: {
					...state,
					path: ref([
						...fromRef(state.path),
						"items",
						index
					])
				}
			}));
			if (schema.logicalOperator === "and") {
				const firstSchema = schema.items[0];
				if (firstSchema.logicalOperator === "or" || firstSchema.type && firstSchema.type !== "object") ast.expression = $(z).attr(identifiers.intersection).call(...itemSchemas.map((schema$1) => schema$1.expression));
				else {
					ast.expression = itemSchemas[0].expression;
					itemSchemas.slice(1).forEach((schema$1) => {
						ast.expression = $(z).attr(identifiers.intersection).call(ast.expression, schema$1.hasLazyExpression ? $(z).attr(identifiers.lazy).call($.func().do(schema$1.expression.return())) : schema$1.expression);
					});
				}
			} else ast.expression = $(z).attr(identifiers.union).call($.array().pretty().elements(...itemSchemas.map((schema$1) => schema$1.expression)));
		} else ast = irSchemaToAst$2({
			plugin,
			schema,
			state
		});
	} else {
		const typeAst = irSchemaWithTypeToAst$2({
			plugin,
			schema: { type: "unknown" },
			state
		});
		ast.expression = typeAst.expression;
	}
	if (ast.expression) {
		if (schema.accessScope === "read") ast.expression = $(z).attr(identifiers.readonly).call(ast.expression);
		if (optional) {
			ast.expression = $(z).attr(identifiers.optional).call(ast.expression);
			ast.typeName = identifiers.ZodMiniOptional;
		}
		if (schema.default !== void 0) ast.expression = $(z).attr(identifiers._default).call(ast.expression, schema.type === "integer" || schema.type === "number" ? maybeBigInt(schema.default, schema.format) : $.fromValue(schema.default));
	}
	return ast;
}
const handlerMini = ({ plugin }) => {
	plugin.symbol("z", {
		external: getZodModule({ plugin }),
		importKind: "namespace",
		meta: {
			category: "external",
			resource: "zod.z"
		}
	});
	const processor = createProcessor$2(plugin);
	plugin.forEach("operation", "parameter", "requestBody", "schema", "webhook", (event) => {
		switch (event.type) {
			case "operation":
				irOperationToAst({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
			case "parameter":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.parameter.schema,
					tags: event.tags
				});
				break;
			case "requestBody":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.requestBody.schema,
					tags: event.tags
				});
				break;
			case "schema":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.schema,
					tags: event.tags
				});
				break;
			case "webhook":
				irWebhookToAst({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
		}
	});
};

//#endregion
//#region src/plugins/zod/v3/processor.ts
function createProcessor$1(plugin) {
	const processor = createSchemaProcessor();
	const hooks = [plugin.config["~hooks"]?.schemas, plugin.context.config.parser.hooks.schemas];
	function extractor(ctx$1) {
		if (processor.hasEmitted(ctx$1.path)) return ctx$1.schema;
		for (const hook of hooks) if (hook?.shouldExtract?.(ctx$1)) {
			process$1({
				namingAnchor: processor.context.anchor,
				tags: processor.context.tags,
				...ctx$1
			});
			return { $ref: pathToJsonPointer(ctx$1.path) };
		}
		return ctx$1.schema;
	}
	function process$1(ctx$1) {
		if (!processor.markEmitted(ctx$1.path)) return;
		processor.withContext({
			anchor: ctx$1.namingAnchor,
			tags: ctx$1.tags
		}, () => {
			const state = refs({
				hasLazyExpression: false,
				path: ctx$1.path,
				tags: ctx$1.tags
			});
			const ast = irSchemaToAst$1({
				plugin,
				schema: ctx$1.schema,
				schemaExtractor: extractor,
				state
			});
			exportAst({
				...ctx$1,
				ast,
				plugin,
				state
			});
		});
	}
	return { process: process$1 };
}

//#endregion
//#region src/plugins/zod/v3/toAst/unknown.ts
const unknownToAst$1 = ({ plugin }) => {
	return $(plugin.external("zod.z")).attr(identifiers.unknown).call();
};

//#endregion
//#region src/plugins/zod/v3/toAst/array.ts
const arrayToAst$1 = ({ plugin, schema, state }) => {
	const z = plugin.external("zod.z");
	const functionName = $(z).attr(identifiers.array);
	let arrayExpression;
	let hasLazyExpression = false;
	if (!schema.items) arrayExpression = functionName.call(unknownToAst$1({
		plugin,
		schema: { type: "unknown" },
		state
	}));
	else {
		schema = deduplicateSchema({ schema });
		const itemExpressions = schema.items.map((item, index) => {
			const itemAst = irSchemaToAst$1({
				plugin,
				schema: item,
				state: {
					...state,
					path: ref([
						...fromRef(state.path),
						"items",
						index
					])
				}
			});
			if (itemAst.hasLazyExpression) hasLazyExpression = true;
			return itemAst.expression;
		});
		if (itemExpressions.length === 1) arrayExpression = functionName.call(...itemExpressions);
		else if (schema.logicalOperator === "and") {
			const firstSchema = schema.items[0];
			let intersectionExpression;
			if (firstSchema.logicalOperator === "or" || firstSchema.type && firstSchema.type !== "object") intersectionExpression = $(z).attr(identifiers.intersection).call(...itemExpressions);
			else {
				intersectionExpression = itemExpressions[0];
				for (let i = 1; i < itemExpressions.length; i++) intersectionExpression = intersectionExpression.attr(identifiers.and).call(itemExpressions[i]);
			}
			arrayExpression = functionName.call(intersectionExpression);
		} else arrayExpression = $(z).attr(identifiers.array).call($(z).attr(identifiers.union).call($.array(...itemExpressions)));
	}
	if (schema.minItems === schema.maxItems && schema.minItems !== void 0) arrayExpression = arrayExpression.attr(identifiers.length).call($.fromValue(schema.minItems));
	else {
		if (schema.minItems !== void 0) arrayExpression = arrayExpression.attr(identifiers.min).call($.fromValue(schema.minItems));
		if (schema.maxItems !== void 0) arrayExpression = arrayExpression.attr(identifiers.max).call($.fromValue(schema.maxItems));
	}
	return {
		expression: arrayExpression,
		hasLazyExpression
	};
};

//#endregion
//#region src/plugins/zod/v3/toAst/boolean.ts
function booleanToAst$1({ plugin, schema }) {
	let chain;
	const z = plugin.external("zod.z");
	if (typeof schema.const === "boolean") {
		chain = $(z).attr(identifiers.literal).call($.literal(schema.const));
		return chain;
	}
	chain = $(z).attr(identifiers.boolean).call();
	return chain;
}

//#endregion
//#region src/plugins/zod/v3/toAst/enum.ts
function itemsNode$1(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	const enumMembers = [];
	const literalMembers = [];
	let isNullable = false;
	let allStrings = true;
	for (const item of schema.items ?? []) if (item.type === "string" && typeof item.const === "string") {
		const literal = $.literal(item.const);
		enumMembers.push(literal);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if ((item.type === "number" || item.type === "integer") && typeof item.const === "number") {
		allStrings = false;
		const literal = $.literal(item.const);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if (item.type === "boolean" && typeof item.const === "boolean") {
		allStrings = false;
		const literal = $.literal(item.const);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if (item.type === "null" || item.const === null) isNullable = true;
	return {
		allStrings,
		enumMembers,
		isNullable,
		literalMembers
	};
}
function baseNode$7(ctx$1) {
	const { symbols } = ctx$1;
	const { z } = symbols;
	const { allStrings, enumMembers, literalMembers } = ctx$1.nodes.items(ctx$1);
	if (allStrings && enumMembers.length > 0) return $(z).attr(identifiers.enum).call($.array(...enumMembers));
	else if (literalMembers.length === 1) return literalMembers[0];
	else return $(z).attr(identifiers.union).call($.array(...literalMembers));
}
function nullableNode$1(ctx$1) {
	const { chain } = ctx$1;
	const { isNullable } = ctx$1.nodes.items(ctx$1);
	if (!isNullable) return;
	return chain.current.attr(identifiers.nullable).call();
}
function enumResolver$1(ctx$1) {
	const { literalMembers } = ctx$1.nodes.items(ctx$1);
	if (!literalMembers.length) return ctx$1.chain.current;
	const baseExpression = ctx$1.nodes.base(ctx$1);
	ctx$1.chain.current = baseExpression;
	const nullableExpression = ctx$1.nodes.nullable(ctx$1);
	if (nullableExpression) ctx$1.chain.current = nullableExpression;
	return ctx$1.chain.current;
}
const enumToAst$1 = ({ plugin, schema, state }) => {
	const z = plugin.external("zod.z");
	const { literalMembers } = itemsNode$1({
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$7,
			items: itemsNode$1,
			nullable: nullableNode$1
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast: {},
			state
		}
	});
	if (!literalMembers.length) return unknownToAst$1({
		plugin,
		schema: { type: "unknown" },
		state
	});
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$7,
			items: itemsNode$1,
			nullable: nullableNode$1
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast: {},
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.enum;
	return resolver?.(ctx$1) ?? enumResolver$1(ctx$1);
};

//#endregion
//#region src/plugins/zod/v3/toAst/never.ts
const neverToAst$1 = ({ plugin }) => {
	return $(plugin.external("zod.z")).attr(identifiers.never).call();
};

//#endregion
//#region src/plugins/zod/v3/toAst/null.ts
const nullToAst$1 = ({ plugin }) => {
	return $(plugin.external("zod.z")).attr(identifiers.null).call();
};

//#endregion
//#region src/plugins/zod/v3/toAst/number.ts
function baseNode$6(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (ctx$1.utils.shouldCoerceToBigInt(schema.format)) return $(z).attr(identifiers.coerce).attr(identifiers.bigint).call();
	let chain = $(z).attr(identifiers.number).call();
	if (schema.type === "integer") chain = chain.attr(identifiers.int).call();
	return chain;
}
function constNode$3(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.const === void 0) return;
	return $(z).attr(identifiers.literal).call(ctx$1.utils.maybeBigInt(schema.const, schema.format));
}
function maxNode$1(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.exclusiveMaximum !== void 0) return chain.current.attr(identifiers.lt).call(ctx$1.utils.maybeBigInt(schema.exclusiveMaximum, schema.format));
	if (schema.maximum !== void 0) return chain.current.attr(identifiers.lte).call(ctx$1.utils.maybeBigInt(schema.maximum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return chain.current.attr(identifiers.max).call(ctx$1.utils.maybeBigInt(limit.maxValue, schema.format), $.object().prop("message", $.literal(limit.maxError)));
}
function minNode$1(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.exclusiveMinimum !== void 0) return chain.current.attr(identifiers.gt).call(ctx$1.utils.maybeBigInt(schema.exclusiveMinimum, schema.format));
	if (schema.minimum !== void 0) return chain.current.attr(identifiers.gte).call(ctx$1.utils.maybeBigInt(schema.minimum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return chain.current.attr(identifiers.min).call(ctx$1.utils.maybeBigInt(limit.minValue, schema.format), $.object().prop("message", $.literal(limit.minError)));
}
function numberResolver$1(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) {
		ctx$1.chain.current = constNode$8;
		return ctx$1.chain.current;
	}
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.chain.current = baseNode$16;
	const minNode$4 = ctx$1.nodes.min(ctx$1);
	if (minNode$4) ctx$1.chain.current = minNode$4;
	const maxNode$4 = ctx$1.nodes.max(ctx$1);
	if (maxNode$4) ctx$1.chain.current = maxNode$4;
	return ctx$1.chain.current;
}
const numberToNode$1 = ({ plugin, schema, state }) => {
	const ast = {};
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$6,
			const: constNode$3,
			max: maxNode$1,
			min: minNode$1
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast,
			getIntegerLimit,
			maybeBigInt,
			shouldCoerceToBigInt,
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.number;
	return resolver?.(ctx$1) ?? numberResolver$1(ctx$1);
};

//#endregion
//#region src/plugins/zod/v3/toAst/object.ts
function additionalPropertiesNode$1(ctx$1) {
	const { plugin, schema } = ctx$1;
	if (!schema.additionalProperties || schema.properties && Object.keys(schema.properties).length > 0) return;
	const additionalAst = irSchemaToAst$1({
		plugin,
		schema: schema.additionalProperties,
		state: {
			...ctx$1.utils.state,
			path: ref([...fromRef(ctx$1.utils.state.path), "additionalProperties"])
		}
	});
	if (additionalAst.hasLazyExpression) ctx$1.utils.ast.hasLazyExpression = true;
	return additionalAst.expression;
}
function baseNode$5(ctx$1) {
	const { nodes, symbols } = ctx$1;
	const { z } = symbols;
	const additional = nodes.additionalProperties(ctx$1);
	const shape = nodes.shape(ctx$1);
	if (additional) return $(z).attr(identifiers.record).call(additional);
	return $(z).attr(identifiers.object).call(shape);
}
function objectResolver$1(ctx$1) {
	return ctx$1.nodes.base(ctx$1);
}
function shapeNode$1(ctx$1) {
	const { plugin, schema } = ctx$1;
	const shape = $.object().pretty();
	for (const name in schema.properties) {
		const property = schema.properties[name];
		const propertyAst = irSchemaToAst$1({
			optional: !schema.required?.includes(name),
			plugin,
			schema: property,
			state: {
				...ctx$1.utils.state,
				path: ref([
					...fromRef(ctx$1.utils.state.path),
					"properties",
					name
				])
			}
		});
		if (propertyAst.hasLazyExpression) ctx$1.utils.ast.hasLazyExpression = true;
		shape.prop(name, propertyAst.expression);
	}
	return shape;
}
const objectToAst$1 = ({ plugin, schema, state }) => {
	const ast = {};
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			additionalProperties: additionalPropertiesNode$1,
			base: baseNode$5,
			shape: shapeNode$1
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast,
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.object;
	ast.expression = resolver?.(ctx$1) ?? objectResolver$1(ctx$1);
	return {
		...ast,
		anyType: "AnyZodObject"
	};
};

//#endregion
//#region src/plugins/zod/v3/toAst/string.ts
function baseNode$4(ctx$1) {
	const { z } = ctx$1.symbols;
	return $(z).attr(identifiers.string).call();
}
function constNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (typeof schema.const !== "string") return;
	return $(z).attr(identifiers.literal).call($.literal(schema.const));
}
function formatNode$1(ctx$1) {
	const { chain, plugin, schema } = ctx$1;
	switch (schema.format) {
		case "date": return chain.current.attr(identifiers.date).call();
		case "date-time": {
			const obj = $.object().$if(plugin.config.dates.offset, (o) => o.prop("offset", $.literal(true))).$if(plugin.config.dates.local, (o) => o.prop("local", $.literal(true)));
			return chain.current.attr(identifiers.datetime).call(obj.hasProps() ? obj : void 0);
		}
		case "email": return chain.current.attr(identifiers.email).call();
		case "ipv4":
		case "ipv6": return chain.current.attr(identifiers.ip).call();
		case "time": return chain.current.attr(identifiers.time).call();
		case "uri": return chain.current.attr(identifiers.url).call();
		case "uuid": return chain.current.attr(identifiers.uuid).call();
	}
}
function lengthNode$1(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.minLength === void 0 || schema.minLength !== schema.maxLength) return;
	return chain.current.attr(identifiers.length).call($.literal(schema.minLength));
}
function maxLengthNode$1(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.maxLength === void 0) return;
	return chain.current.attr(identifiers.max).call($.literal(schema.maxLength));
}
function minLengthNode$1(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.minLength === void 0) return;
	return chain.current.attr(identifiers.min).call($.literal(schema.minLength));
}
function patternNode$1(ctx$1) {
	const { chain, schema } = ctx$1;
	if (!schema.pattern) return;
	return chain.current.attr(identifiers.regex).call($.regexp(schema.pattern));
}
function stringResolver$1(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) {
		ctx$1.chain.current = constNode$8;
		return ctx$1.chain.current;
	}
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.chain.current = baseNode$16;
	const formatNode$4 = ctx$1.nodes.format(ctx$1);
	if (formatNode$4) ctx$1.chain.current = formatNode$4;
	const lengthNode$4 = ctx$1.nodes.length(ctx$1);
	if (lengthNode$4) ctx$1.chain.current = lengthNode$4;
	else {
		const minLengthNode$4 = ctx$1.nodes.minLength(ctx$1);
		if (minLengthNode$4) ctx$1.chain.current = minLengthNode$4;
		const maxLengthNode$4 = ctx$1.nodes.maxLength(ctx$1);
		if (maxLengthNode$4) ctx$1.chain.current = maxLengthNode$4;
	}
	const patternNode$4 = ctx$1.nodes.pattern(ctx$1);
	if (patternNode$4) ctx$1.chain.current = patternNode$4;
	return ctx$1.chain.current;
}
const stringToNode$1 = ({ plugin, schema }) => {
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$4,
			const: constNode$2,
			format: formatNode$1,
			length: lengthNode$1,
			maxLength: maxLengthNode$1,
			minLength: minLengthNode$1,
			pattern: patternNode$1
		},
		plugin,
		schema,
		symbols: { z }
	};
	const resolver = plugin.config["~resolvers"]?.string;
	return resolver?.(ctx$1) ?? stringResolver$1(ctx$1);
};

//#endregion
//#region src/plugins/zod/v3/toAst/tuple.ts
const tupleToAst$1 = ({ plugin, schema, state }) => {
	const z = plugin.external("zod.z");
	let hasLazyExpression = false;
	if (schema.const && Array.isArray(schema.const)) {
		const tupleElements$1 = schema.const.map((value) => $(z).attr(identifiers.literal).call($.fromValue(value)));
		return {
			expression: $(z).attr(identifiers.tuple).call($.array(...tupleElements$1)),
			hasLazyExpression
		};
	}
	const tupleElements = [];
	if (schema.items) schema.items.forEach((item, index) => {
		const itemSchema = irSchemaToAst$1({
			plugin,
			schema: item,
			state: {
				...state,
				path: ref([
					...fromRef(state.path),
					"items",
					index
				])
			}
		});
		tupleElements.push(itemSchema.expression);
		if (itemSchema.hasLazyExpression) hasLazyExpression = true;
	});
	return {
		expression: $(z).attr(identifiers.tuple).call($.array(...tupleElements)),
		hasLazyExpression
	};
};

//#endregion
//#region src/plugins/zod/v3/toAst/undefined.ts
const undefinedToAst$1 = ({ plugin }) => {
	return $(plugin.external("zod.z")).attr(identifiers.undefined).call();
};

//#endregion
//#region src/plugins/zod/v3/toAst/void.ts
const voidToAst$1 = ({ plugin }) => {
	return $(plugin.external("zod.z")).attr(identifiers.void).call();
};

//#endregion
//#region src/plugins/zod/v3/toAst/index.ts
function irSchemaWithTypeToAst$1({ schema, ...args }) {
	switch (schema.type) {
		case "array": return arrayToAst$1({
			...args,
			schema
		});
		case "boolean": return { expression: booleanToAst$1({
			...args,
			schema
		}) };
		case "enum": return { expression: enumToAst$1({
			...args,
			schema
		}) };
		case "integer":
		case "number": return { expression: numberToNode$1({
			...args,
			schema
		}) };
		case "never": return { expression: neverToAst$1({
			...args,
			schema
		}) };
		case "null": return { expression: nullToAst$1({
			...args,
			schema
		}) };
		case "object": return objectToAst$1({
			...args,
			schema
		});
		case "string": return { expression: shouldCoerceToBigInt(schema.format) ? numberToNode$1({
			...args,
			schema: {
				...schema,
				type: "number"
			}
		}) : stringToNode$1({
			...args,
			schema
		}) };
		case "tuple": return tupleToAst$1({
			...args,
			schema
		});
		case "undefined": return { expression: undefinedToAst$1({
			...args,
			schema
		}) };
		case "unknown": return { expression: unknownToAst$1({
			...args,
			schema
		}) };
		case "void": return { expression: voidToAst$1({
			...args,
			schema
		}) };
	}
}

//#endregion
//#region src/plugins/zod/v3/plugin.ts
function irSchemaToAst$1({ optional, plugin, schema, schemaExtractor, state }) {
	if (schemaExtractor && !schema.$ref) {
		const extracted = schemaExtractor({
			meta: {
				resource: "definition",
				resourceId: pathToJsonPointer(fromRef(state.path))
			},
			naming: plugin.config.definitions,
			path: fromRef(state.path),
			plugin,
			schema
		});
		if (extracted !== schema) schema = extracted;
	}
	let ast = {};
	const z = plugin.external("zod.z");
	if (schema.$ref) {
		const query = {
			category: "schema",
			resource: "definition",
			resourceId: schema.$ref,
			tool: "zod"
		};
		const refSymbol = plugin.referenceSymbol(query);
		if (plugin.isSymbolRegistered(query)) ast.expression = $(refSymbol);
		else {
			ast.expression = $(z).attr(identifiers.lazy).call($.func().do($(refSymbol).return()));
			ast.hasLazyExpression = true;
			state.hasLazyExpression["~ref"] = true;
		}
	} else if (schema.type) {
		const typeAst = irSchemaWithTypeToAst$1({
			plugin,
			schema,
			state
		});
		ast.expression = typeAst.expression;
		ast.typeName = typeAst.anyType;
		if (plugin.config.metadata && schema.description) ast.expression = ast.expression.attr(identifiers.describe).call($.literal(schema.description));
	} else if (schema.items) {
		schema = deduplicateSchema({ schema });
		if (schema.items) {
			const itemTypes = schema.items.map((item, index) => {
				return irSchemaToAst$1({
					plugin,
					schema: item,
					state: {
						...state,
						path: ref([
							...fromRef(state.path),
							"items",
							index
						])
					}
				}).expression;
			});
			if (schema.logicalOperator === "and") {
				const firstSchema = schema.items[0];
				if (firstSchema.logicalOperator === "or" || firstSchema.type && firstSchema.type !== "object") ast.expression = $(z).attr(identifiers.intersection).call(...itemTypes);
				else {
					ast.expression = itemTypes[0];
					itemTypes.slice(1).forEach((item) => {
						ast.expression = ast.expression.attr(identifiers.and).call(item);
					});
				}
			} else ast.expression = $(z).attr(identifiers.union).call($.array().pretty().elements(...itemTypes));
		} else ast = irSchemaToAst$1({
			plugin,
			schema,
			state
		});
	} else {
		const typeAst = irSchemaWithTypeToAst$1({
			plugin,
			schema: { type: "unknown" },
			state
		});
		ast.expression = typeAst.expression;
		ast.typeName = typeAst.anyType;
	}
	if (ast.expression) {
		if (schema.accessScope === "read") ast.expression = ast.expression.attr(identifiers.readonly).call();
		if (optional) ast.expression = ast.expression.attr(identifiers.optional).call();
		if (schema.default !== void 0) ast.expression = ast.expression.attr(identifiers.default).call(schema.type === "integer" || schema.type === "number" ? maybeBigInt(schema.default, schema.format) : $.fromValue(schema.default));
	}
	if (state.hasLazyExpression["~ref"]) {
		if (!ast.typeName) ast.typeName = "ZodTypeAny";
	} else if (ast.typeName) ast.typeName = void 0;
	return ast;
}
const handlerV3 = ({ plugin }) => {
	plugin.symbol("z", {
		external: getZodModule({ plugin }),
		meta: {
			category: "external",
			resource: "zod.z"
		}
	});
	const processor = createProcessor$1(plugin);
	plugin.forEach("operation", "parameter", "requestBody", "schema", "webhook", (event) => {
		switch (event.type) {
			case "operation":
				irOperationToAst({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
			case "parameter":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.parameter.schema,
					tags: event.tags
				});
				break;
			case "requestBody":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.requestBody.schema,
					tags: event.tags
				});
				break;
			case "schema":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.schema,
					tags: event.tags
				});
				break;
			case "webhook":
				irWebhookToAst({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
		}
	});
};

//#endregion
//#region src/plugins/zod/v4/processor.ts
function createProcessor(plugin) {
	const processor = createSchemaProcessor();
	const hooks = [plugin.config["~hooks"]?.schemas, plugin.context.config.parser.hooks.schemas];
	function extractor(ctx$1) {
		if (processor.hasEmitted(ctx$1.path)) return ctx$1.schema;
		for (const hook of hooks) if (hook?.shouldExtract?.(ctx$1)) {
			process$1({
				namingAnchor: processor.context.anchor,
				tags: processor.context.tags,
				...ctx$1
			});
			return { $ref: pathToJsonPointer(ctx$1.path) };
		}
		return ctx$1.schema;
	}
	function process$1(ctx$1) {
		if (!processor.markEmitted(ctx$1.path)) return;
		processor.withContext({
			anchor: ctx$1.namingAnchor,
			tags: ctx$1.tags
		}, () => {
			const state = refs({
				hasLazyExpression: false,
				path: ctx$1.path,
				tags: ctx$1.tags
			});
			const ast = irSchemaToAst({
				plugin,
				schema: ctx$1.schema,
				schemaExtractor: extractor,
				state
			});
			exportAst({
				...ctx$1,
				ast,
				plugin,
				state
			});
		});
	}
	return { process: process$1 };
}

//#endregion
//#region src/plugins/zod/v4/toAst/unknown.ts
const unknownToAst = ({ plugin }) => {
	const result = {};
	result.expression = $(plugin.external("zod.z")).attr(identifiers.unknown).call();
	return result;
};

//#endregion
//#region src/plugins/zod/v4/toAst/array.ts
const arrayToAst = ({ plugin, schema, state }) => {
	const result = {};
	const z = plugin.external("zod.z");
	const functionName = $(z).attr(identifiers.array);
	if (!schema.items) result.expression = functionName.call(unknownToAst({
		plugin,
		schema: { type: "unknown" },
		state
	}).expression);
	else {
		schema = deduplicateSchema({ schema });
		const itemExpressions = schema.items.map((item, index) => {
			const itemAst = irSchemaToAst({
				plugin,
				schema: item,
				state: {
					...state,
					path: ref([
						...fromRef(state.path),
						"items",
						index
					])
				}
			});
			if (itemAst.hasLazyExpression) result.hasLazyExpression = true;
			return itemAst.expression;
		});
		if (itemExpressions.length === 1) result.expression = functionName.call(...itemExpressions);
		else if (schema.logicalOperator === "and") {
			const firstSchema = schema.items[0];
			let intersectionExpression;
			if (firstSchema.logicalOperator === "or" || firstSchema.type && firstSchema.type !== "object") intersectionExpression = $(z).attr(identifiers.intersection).call(...itemExpressions);
			else {
				intersectionExpression = itemExpressions[0];
				for (let i = 1; i < itemExpressions.length; i++) intersectionExpression = intersectionExpression.attr(identifiers.and).call(itemExpressions[i]);
			}
			result.expression = functionName.call(intersectionExpression);
		} else result.expression = $(z).attr(identifiers.array).call($(z).attr(identifiers.union).call($.array(...itemExpressions)));
	}
	if (schema.minItems === schema.maxItems && schema.minItems !== void 0) result.expression = result.expression.attr(identifiers.length).call($.fromValue(schema.minItems));
	else {
		if (schema.minItems !== void 0) result.expression = result.expression.attr(identifiers.min).call($.fromValue(schema.minItems));
		if (schema.maxItems !== void 0) result.expression = result.expression.attr(identifiers.max).call($.fromValue(schema.maxItems));
	}
	return result;
};

//#endregion
//#region src/plugins/zod/v4/toAst/boolean.ts
function booleanToAst({ plugin, schema }) {
	const result = {};
	let chain;
	const z = plugin.external("zod.z");
	if (typeof schema.const === "boolean") {
		chain = $(z).attr(identifiers.literal).call($.literal(schema.const));
		result.expression = chain;
		return result;
	}
	chain = $(z).attr(identifiers.boolean).call();
	result.expression = chain;
	return result;
}

//#endregion
//#region src/plugins/zod/v4/toAst/enum.ts
function itemsNode(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	const enumMembers = [];
	const literalMembers = [];
	let isNullable = false;
	let allStrings = true;
	for (const item of schema.items ?? []) if (item.type === "string" && typeof item.const === "string") {
		const literal = $.literal(item.const);
		enumMembers.push(literal);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if ((item.type === "number" || item.type === "integer") && typeof item.const === "number") {
		allStrings = false;
		const literal = $.literal(item.const);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if (item.type === "boolean" && typeof item.const === "boolean") {
		allStrings = false;
		const literal = $.literal(item.const);
		literalMembers.push($(z).attr(identifiers.literal).call(literal));
	} else if (item.type === "null" || item.const === null) isNullable = true;
	return {
		allStrings,
		enumMembers,
		isNullable,
		literalMembers
	};
}
function baseNode$3(ctx$1) {
	const { symbols } = ctx$1;
	const { z } = symbols;
	const { allStrings, enumMembers, literalMembers } = ctx$1.nodes.items(ctx$1);
	if (allStrings && enumMembers.length > 0) return $(z).attr(identifiers.enum).call($.array(...enumMembers));
	else if (literalMembers.length === 1) return literalMembers[0];
	else return $(z).attr(identifiers.union).call($.array(...literalMembers));
}
function nullableNode(ctx$1) {
	const { chain, symbols } = ctx$1;
	const { z } = symbols;
	const { isNullable } = ctx$1.nodes.items(ctx$1);
	if (!isNullable) return;
	return $(z).attr(identifiers.nullable).call(chain.current);
}
function enumResolver(ctx$1) {
	const { literalMembers } = ctx$1.nodes.items(ctx$1);
	if (!literalMembers.length) return ctx$1.chain.current;
	const baseExpression = ctx$1.nodes.base(ctx$1);
	ctx$1.chain.current = baseExpression;
	const nullableExpression = ctx$1.nodes.nullable(ctx$1);
	if (nullableExpression) ctx$1.chain.current = nullableExpression;
	return ctx$1.chain.current;
}
const enumToAst = ({ plugin, schema, state }) => {
	const z = plugin.external("zod.z");
	const { literalMembers } = itemsNode({
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$3,
			items: itemsNode,
			nullable: nullableNode
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast: {},
			state
		}
	});
	if (!literalMembers.length) return unknownToAst({
		plugin,
		schema: { type: "unknown" },
		state
	});
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$3,
			items: itemsNode,
			nullable: nullableNode
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast: {},
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.enum;
	return { expression: resolver?.(ctx$1) ?? enumResolver(ctx$1) };
};

//#endregion
//#region src/plugins/zod/v4/toAst/never.ts
const neverToAst = ({ plugin }) => {
	const result = {};
	result.expression = $(plugin.external("zod.z")).attr(identifiers.never).call();
	return result;
};

//#endregion
//#region src/plugins/zod/v4/toAst/null.ts
const nullToAst = ({ plugin }) => {
	const result = {};
	result.expression = $(plugin.external("zod.z")).attr(identifiers.null).call();
	return result;
};

//#endregion
//#region src/plugins/zod/v4/toAst/number.ts
function baseNode$2(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (ctx$1.utils.shouldCoerceToBigInt(schema.format)) return $(z).attr(identifiers.coerce).attr(identifiers.bigint).call();
	let chain = $(z).attr(identifiers.number).call();
	if (schema.type === "integer") chain = $(z).attr(identifiers.int).call();
	return chain;
}
function constNode$1(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (schema.const === void 0) return;
	return $(z).attr(identifiers.literal).call(ctx$1.utils.maybeBigInt(schema.const, schema.format));
}
function maxNode(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.exclusiveMaximum !== void 0) return chain.current.attr(identifiers.lt).call(ctx$1.utils.maybeBigInt(schema.exclusiveMaximum, schema.format));
	if (schema.maximum !== void 0) return chain.current.attr(identifiers.lte).call(ctx$1.utils.maybeBigInt(schema.maximum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return chain.current.attr(identifiers.max).call(ctx$1.utils.maybeBigInt(limit.maxValue, schema.format), $.object().prop("error", $.literal(limit.maxError)));
}
function minNode(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.exclusiveMinimum !== void 0) return chain.current.attr(identifiers.gt).call(ctx$1.utils.maybeBigInt(schema.exclusiveMinimum, schema.format));
	if (schema.minimum !== void 0) return chain.current.attr(identifiers.gte).call(ctx$1.utils.maybeBigInt(schema.minimum, schema.format));
	const limit = ctx$1.utils.getIntegerLimit(schema.format);
	if (limit) return chain.current.attr(identifiers.min).call(ctx$1.utils.maybeBigInt(limit.minValue, schema.format), $.object().prop("error", $.literal(limit.minError)));
}
function numberResolver(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) {
		ctx$1.chain.current = constNode$8;
		return ctx$1.chain.current;
	}
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.chain.current = baseNode$16;
	const minNode$4 = ctx$1.nodes.min(ctx$1);
	if (minNode$4) ctx$1.chain.current = minNode$4;
	const maxNode$4 = ctx$1.nodes.max(ctx$1);
	if (maxNode$4) ctx$1.chain.current = maxNode$4;
	return ctx$1.chain.current;
}
const numberToNode = ({ plugin, schema, state }) => {
	const ast = {};
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode$2,
			const: constNode$1,
			max: maxNode,
			min: minNode
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast,
			getIntegerLimit,
			maybeBigInt,
			shouldCoerceToBigInt,
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.number;
	ast.expression = resolver?.(ctx$1) ?? numberResolver(ctx$1);
	return ast;
};

//#endregion
//#region src/plugins/zod/v4/toAst/object.ts
function additionalPropertiesNode(ctx$1) {
	const { plugin, schema } = ctx$1;
	if (!schema.additionalProperties || schema.properties && Object.keys(schema.properties).length > 0) return;
	const additionalAst = irSchemaToAst({
		plugin,
		schema: schema.additionalProperties,
		state: {
			...ctx$1.utils.state,
			path: ref([...fromRef(ctx$1.utils.state.path), "additionalProperties"])
		}
	});
	if (additionalAst.hasLazyExpression) ctx$1.utils.ast.hasLazyExpression = true;
	return additionalAst.expression;
}
function baseNode$1(ctx$1) {
	const { nodes, symbols } = ctx$1;
	const { z } = symbols;
	const additional = nodes.additionalProperties(ctx$1);
	const shape = nodes.shape(ctx$1);
	if (additional) return $(z).attr(identifiers.record).call($(z).attr(identifiers.string).call(), additional);
	return $(z).attr(identifiers.object).call(shape);
}
function objectResolver(ctx$1) {
	return ctx$1.nodes.base(ctx$1);
}
function shapeNode(ctx$1) {
	const { plugin, schema } = ctx$1;
	const shape = $.object().pretty();
	for (const name in schema.properties) {
		const property = schema.properties[name];
		const propertyAst = irSchemaToAst({
			optional: !schema.required?.includes(name),
			plugin,
			schema: property,
			state: {
				...ctx$1.utils.state,
				path: ref([
					...fromRef(ctx$1.utils.state.path),
					"properties",
					name
				])
			}
		});
		if (propertyAst.hasLazyExpression) {
			ctx$1.utils.ast.hasLazyExpression = true;
			shape.getter(name, propertyAst.expression.return());
		} else shape.prop(name, propertyAst.expression);
	}
	return shape;
}
const objectToAst = ({ plugin, schema, state }) => {
	const ast = {};
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			additionalProperties: additionalPropertiesNode,
			base: baseNode$1,
			shape: shapeNode
		},
		plugin,
		schema,
		symbols: { z },
		utils: {
			ast,
			state
		}
	};
	const resolver = plugin.config["~resolvers"]?.object;
	ast.expression = resolver?.(ctx$1) ?? objectResolver(ctx$1);
	return ast;
};

//#endregion
//#region src/plugins/zod/v4/toAst/string.ts
function baseNode(ctx$1) {
	const { z } = ctx$1.symbols;
	return $(z).attr(identifiers.string).call();
}
function constNode(ctx$1) {
	const { schema, symbols } = ctx$1;
	const { z } = symbols;
	if (typeof schema.const !== "string") return;
	return $(z).attr(identifiers.literal).call($.literal(schema.const));
}
function formatNode(ctx$1) {
	const { plugin, schema, symbols } = ctx$1;
	const { z } = symbols;
	switch (schema.format) {
		case "date": return $(z).attr(identifiers.iso).attr(identifiers.date).call();
		case "date-time": {
			const obj = $.object().$if(plugin.config.dates.offset, (o) => o.prop("offset", $.literal(true))).$if(plugin.config.dates.local, (o) => o.prop("local", $.literal(true)));
			return $(z).attr(identifiers.iso).attr(identifiers.datetime).call(obj.hasProps() ? obj : void 0);
		}
		case "email": return $(z).attr(identifiers.email).call();
		case "ipv4": return $(z).attr(identifiers.ipv4).call();
		case "ipv6": return $(z).attr(identifiers.ipv6).call();
		case "time": return $(z).attr(identifiers.iso).attr(identifiers.time).call();
		case "uri": return $(z).attr(identifiers.url).call();
		case "uuid": return $(z).attr(identifiers.uuid).call();
	}
}
function lengthNode(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.minLength === void 0 || schema.minLength !== schema.maxLength) return;
	return chain.current.attr(identifiers.length).call($.literal(schema.minLength));
}
function maxLengthNode(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.maxLength === void 0) return;
	return chain.current.attr(identifiers.max).call($.literal(schema.maxLength));
}
function minLengthNode(ctx$1) {
	const { chain, schema } = ctx$1;
	if (schema.minLength === void 0) return;
	return chain.current.attr(identifiers.min).call($.literal(schema.minLength));
}
function patternNode(ctx$1) {
	const { chain, schema } = ctx$1;
	if (!schema.pattern) return;
	return chain.current.attr(identifiers.regex).call($.regexp(schema.pattern));
}
function stringResolver(ctx$1) {
	const constNode$8 = ctx$1.nodes.const(ctx$1);
	if (constNode$8) {
		ctx$1.chain.current = constNode$8;
		return ctx$1.chain.current;
	}
	const baseNode$16 = ctx$1.nodes.base(ctx$1);
	if (baseNode$16) ctx$1.chain.current = baseNode$16;
	const formatNode$4 = ctx$1.nodes.format(ctx$1);
	if (formatNode$4) ctx$1.chain.current = formatNode$4;
	const lengthNode$4 = ctx$1.nodes.length(ctx$1);
	if (lengthNode$4) ctx$1.chain.current = lengthNode$4;
	else {
		const minLengthNode$4 = ctx$1.nodes.minLength(ctx$1);
		if (minLengthNode$4) ctx$1.chain.current = minLengthNode$4;
		const maxLengthNode$4 = ctx$1.nodes.maxLength(ctx$1);
		if (maxLengthNode$4) ctx$1.chain.current = maxLengthNode$4;
	}
	const patternNode$4 = ctx$1.nodes.pattern(ctx$1);
	if (patternNode$4) ctx$1.chain.current = patternNode$4;
	return ctx$1.chain.current;
}
const stringToNode = ({ plugin, schema }) => {
	const z = plugin.external("zod.z");
	const ctx$1 = {
		$,
		chain: { current: $(z) },
		nodes: {
			base: baseNode,
			const: constNode,
			format: formatNode,
			length: lengthNode,
			maxLength: maxLengthNode,
			minLength: minLengthNode,
			pattern: patternNode
		},
		plugin,
		schema,
		symbols: { z }
	};
	const resolver = plugin.config["~resolvers"]?.string;
	return { expression: resolver?.(ctx$1) ?? stringResolver(ctx$1) };
};

//#endregion
//#region src/plugins/zod/v4/toAst/tuple.ts
const tupleToAst = ({ plugin, schema, state }) => {
	const result = {};
	const z = plugin.external("zod.z");
	if (schema.const && Array.isArray(schema.const)) {
		const tupleElements$1 = schema.const.map((value) => $(z).attr(identifiers.literal).call($.fromValue(value)));
		result.expression = $(z).attr(identifiers.tuple).call($.array(...tupleElements$1));
		return result;
	}
	const tupleElements = [];
	if (schema.items) schema.items.forEach((item, index) => {
		const itemSchema = irSchemaToAst({
			plugin,
			schema: item,
			state: {
				...state,
				path: ref([
					...fromRef(state.path),
					"items",
					index
				])
			}
		});
		tupleElements.push(itemSchema.expression);
		if (itemSchema.hasLazyExpression) result.hasLazyExpression = true;
	});
	result.expression = $(z).attr(identifiers.tuple).call($.array(...tupleElements));
	return result;
};

//#endregion
//#region src/plugins/zod/v4/toAst/undefined.ts
const undefinedToAst = ({ plugin }) => {
	const result = {};
	result.expression = $(plugin.external("zod.z")).attr(identifiers.undefined).call();
	return result;
};

//#endregion
//#region src/plugins/zod/v4/toAst/void.ts
const voidToAst = ({ plugin }) => {
	const result = {};
	result.expression = $(plugin.external("zod.z")).attr(identifiers.void).call();
	return result;
};

//#endregion
//#region src/plugins/zod/v4/toAst/index.ts
function irSchemaWithTypeToAst({ schema, ...args }) {
	switch (schema.type) {
		case "array": return arrayToAst({
			...args,
			schema
		});
		case "boolean": return booleanToAst({
			...args,
			schema
		});
		case "enum": return enumToAst({
			...args,
			schema
		});
		case "integer":
		case "number": return numberToNode({
			...args,
			schema
		});
		case "never": return neverToAst({
			...args,
			schema
		});
		case "null": return nullToAst({
			...args,
			schema
		});
		case "object": return objectToAst({
			...args,
			schema
		});
		case "string": return shouldCoerceToBigInt(schema.format) ? numberToNode({
			...args,
			schema: {
				...schema,
				type: "number"
			}
		}) : stringToNode({
			...args,
			schema
		});
		case "tuple": return tupleToAst({
			...args,
			schema
		});
		case "undefined": return undefinedToAst({
			...args,
			schema
		});
		case "unknown": return unknownToAst({
			...args,
			schema
		});
		case "void": return voidToAst({
			...args,
			schema
		});
	}
}

//#endregion
//#region src/plugins/zod/v4/plugin.ts
function irSchemaToAst({ optional, plugin, schema, schemaExtractor, state }) {
	if (schemaExtractor && !schema.$ref) {
		const extracted = schemaExtractor({
			meta: {
				resource: "definition",
				resourceId: pathToJsonPointer(fromRef(state.path))
			},
			naming: plugin.config.definitions,
			path: fromRef(state.path),
			plugin,
			schema
		});
		if (extracted !== schema) schema = extracted;
	}
	let ast = {};
	const z = plugin.external("zod.z");
	if (schema.$ref) {
		const query = {
			category: "schema",
			resource: "definition",
			resourceId: schema.$ref,
			tool: "zod"
		};
		const refSymbol = plugin.referenceSymbol(query);
		if (plugin.isSymbolRegistered(query)) ast.expression = $(refSymbol);
		else {
			ast.expression = $(z).attr(identifiers.lazy).call($.func().returns("any").do($(refSymbol).return()));
			ast.hasLazyExpression = true;
			state.hasLazyExpression["~ref"] = true;
		}
	} else if (schema.type) {
		const typeAst = irSchemaWithTypeToAst({
			plugin,
			schema,
			state
		});
		ast.expression = typeAst.expression;
		ast.hasLazyExpression = typeAst.hasLazyExpression;
		if (plugin.config.metadata && schema.description) ast.expression = ast.expression.attr(identifiers.register).call($(z).attr(identifiers.globalRegistry), $.object().pretty().prop("description", $.literal(schema.description)));
	} else if (schema.items) {
		schema = deduplicateSchema({ schema });
		if (schema.items) {
			const itemSchemas = schema.items.map((item, index) => irSchemaToAst({
				plugin,
				schema: item,
				state: {
					...state,
					path: ref([
						...fromRef(state.path),
						"items",
						index
					])
				}
			}));
			if (schema.logicalOperator === "and") {
				const firstSchema = schema.items[0];
				if (firstSchema.logicalOperator === "or" || firstSchema.type && firstSchema.type !== "object") ast.expression = $(z).attr(identifiers.intersection).call(...itemSchemas.map((schema$1) => schema$1.expression));
				else {
					ast.expression = itemSchemas[0].expression;
					itemSchemas.slice(1).forEach((schema$1) => {
						ast.expression = ast.expression.attr(identifiers.and).call(schema$1.hasLazyExpression ? $(z).attr(identifiers.lazy).call($.func().do(schema$1.expression.return())) : schema$1.expression);
					});
				}
			} else ast.expression = $(z).attr(identifiers.union).call($.array().pretty().elements(...itemSchemas.map((schema$1) => schema$1.expression)));
		} else ast = irSchemaToAst({
			plugin,
			schema,
			state
		});
	} else {
		const typeAst = irSchemaWithTypeToAst({
			plugin,
			schema: { type: "unknown" },
			state
		});
		ast.expression = typeAst.expression;
	}
	if (ast.expression) {
		if (schema.accessScope === "read") ast.expression = ast.expression.attr(identifiers.readonly).call();
		if (optional) {
			ast.expression = $(z).attr(identifiers.optional).call(ast.expression);
			ast.typeName = identifiers.ZodOptional;
		}
		if (schema.default !== void 0) ast.expression = ast.expression.attr(identifiers.default).call(schema.type === "integer" || schema.type === "number" ? maybeBigInt(schema.default, schema.format) : $.fromValue(schema.default));
	}
	return ast;
}
const handlerV4 = ({ plugin }) => {
	plugin.symbol("z", {
		external: getZodModule({ plugin }),
		importKind: "namespace",
		meta: {
			category: "external",
			resource: "zod.z"
		}
	});
	const processor = createProcessor(plugin);
	plugin.forEach("operation", "parameter", "requestBody", "schema", "webhook", (event) => {
		switch (event.type) {
			case "operation":
				irOperationToAst({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
			case "parameter":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.parameter.schema,
					tags: event.tags
				});
				break;
			case "requestBody":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.requestBody.schema,
					tags: event.tags
				});
				break;
			case "schema":
				processor.process({
					meta: {
						resource: "definition",
						resourceId: pathToJsonPointer(event._path)
					},
					naming: plugin.config.definitions,
					path: event._path,
					plugin,
					schema: event.schema,
					tags: event.tags
				});
				break;
			case "webhook":
				irWebhookToAst({
					operation: event.operation,
					path: event._path,
					plugin,
					processor,
					tags: event.tags
				});
				break;
		}
	});
};

//#endregion
//#region src/plugins/zod/plugin.ts
const handler = (args) => {
	const { plugin } = args;
	switch (plugin.config.compatibilityVersion) {
		case 3: return handlerV3(args);
		case "mini": return handlerMini(args);
		case 4:
		default: return handlerV4(args);
	}
};

//#endregion
//#region src/plugins/zod/config.ts
const defaultConfig = {
	api: new Api(),
	config: {
		case: "camelCase",
		comments: true,
		includeInEntry: false,
		metadata: false
	},
	handler,
	name: "zod",
	resolveConfig: (plugin, context) => {
		const packageName = "zod";
		const version = context.package.getVersion(packageName);
		const inferCompatibleVersion = () => {
			if (version && (version.major === 4 || version.major === 3)) return version.major;
			return 4;
		};
		const ensureCompatibleVersion = (compatibilityVersion) => {
			if (!compatibilityVersion) return inferCompatibleVersion();
			if (!version) return compatibilityVersion;
			if (compatibilityVersion === 4 || compatibilityVersion === 3 || compatibilityVersion === "mini") {
				if (!context.package.satisfies(version, ">=3.25.0 <5.0.0")) {
					const compatibleVersion = inferCompatibleVersion();
					console.warn(`🔌 ${colors.yellow("Warning:")} Installed ${colors.cyan(packageName)} ${colors.cyan(`v${version.version}`)} does not support compatibility version ${colors.yellow(String(compatibilityVersion))}, using ${colors.yellow(String(compatibleVersion))}.`);
					return compatibleVersion;
				}
			}
			return compatibilityVersion;
		};
		plugin.config.compatibilityVersion = ensureCompatibleVersion(plugin.config.compatibilityVersion);
		plugin.config.dates = context.valueToObject({
			defaultValue: {
				local: false,
				offset: false
			},
			value: plugin.config.dates
		});
		plugin.config.types = context.valueToObject({
			defaultValue: { infer: {
				case: "PascalCase",
				enabled: false
			} },
			mappers: { object: (fields, defaultValue) => ({
				...fields,
				infer: context.valueToObject({
					defaultValue: {
						...defaultValue.infer,
						enabled: fields.infer !== void 0 ? Boolean(fields.infer) : defaultValue.infer.enabled
					},
					mappers,
					value: fields.infer
				})
			}) },
			value: plugin.config.types
		});
		plugin.config.definitions = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "z{{name}}",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}ZodType"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.definitions
		});
		plugin.config.requests = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "z{{name}}Data",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}DataZodType"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.requests
		});
		plugin.config.responses = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "z{{name}}Response",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}ResponseZodType"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.responses
		});
		plugin.config.webhooks = context.valueToObject({
			defaultValue: {
				case: plugin.config.case ?? "camelCase",
				enabled: true,
				name: "z{{name}}WebhookRequest",
				types: {
					...plugin.config.types,
					infer: {
						...plugin.config.types.infer,
						name: "{{name}}WebhookRequestZodType"
					}
				}
			},
			mappers: {
				...mappers,
				object: (fields, defaultValue) => ({
					...fields,
					types: context.valueToObject({
						defaultValue: defaultValue.types,
						mappers: { object: (fields$1, defaultValue$1) => ({
							...fields$1,
							infer: context.valueToObject({
								defaultValue: {
									...defaultValue$1.infer,
									enabled: fields$1.infer !== void 0 ? Boolean(fields$1.infer) : defaultValue$1.infer.enabled
								},
								mappers,
								value: fields$1.infer
							})
						}) },
						value: fields.types
					})
				})
			},
			value: plugin.config.webhooks
		});
	},
	tags: ["validator"]
};
/**
* Type helper for Zod plugin, returns {@link Plugin.Config} object
*/
const defineConfig = definePluginConfig(defaultConfig);

//#endregion
//#region src/plugins/config.ts
const defaultPluginConfigs = {
	"@angular/common": defaultConfig$23,
	"@faker-js/faker": defaultConfig$22,
	"@hey-api/client-angular": defaultConfig$21,
	"@hey-api/client-axios": defaultConfig$20,
	"@hey-api/client-fetch": defaultConfig$19,
	"@hey-api/client-ky": defaultConfig$18,
	"@hey-api/client-next": defaultConfig$17,
	"@hey-api/client-nuxt": defaultConfig$16,
	"@hey-api/client-ofetch": defaultConfig$15,
	"@hey-api/schemas": defaultConfig$14,
	"@hey-api/sdk": defaultConfig$13,
	"@hey-api/transformers": defaultConfig$12,
	"@hey-api/typescript": defaultConfig$11,
	"@pinia/colada": defaultConfig$10,
	"@tanstack/angular-query-experimental": defaultConfig$9,
	"@tanstack/react-query": defaultConfig$8,
	"@tanstack/solid-query": defaultConfig$7,
	"@tanstack/svelte-query": defaultConfig$6,
	"@tanstack/vue-query": defaultConfig$5,
	arktype: defaultConfig$4,
	fastify: defaultConfig$3,
	swr: defaultConfig$2,
	valibot: defaultConfig$1,
	zod: defaultConfig
};

//#endregion
//#region src/config/plugins.ts
/**
* Default plugins used to generate artifacts if plugins aren't specified.
*/
const defaultPlugins = ["@hey-api/typescript", "@hey-api/sdk"];
function getPluginsConfig({ dependencies, userPlugins, userPluginsConfig }) {
	const circularReferenceTracker = /* @__PURE__ */ new Set();
	const pluginOrder = /* @__PURE__ */ new Set();
	const plugins = {};
	const dfs = (name) => {
		if (circularReferenceTracker.has(name)) throw new Error(`Circular reference detected at '${name}'`);
		if (pluginOrder.has(name)) return;
		circularReferenceTracker.add(name);
		const defaultPlugin = defaultPluginConfigs[name];
		const userPlugin = userPluginsConfig[name];
		if (!defaultPlugin && !userPlugin) throw new Error(`unknown plugin dependency "${name}" - do you need to register a custom plugin with this name?`);
		const plugin = {
			...defaultPlugin,
			...userPlugin,
			config: {
				...defaultPlugin?.config,
				...userPlugin?.config
			},
			dependencies: new Set([...defaultPlugin?.dependencies || [], ...userPlugin?.dependencies || []])
		};
		if (plugin.resolveConfig) {
			const context = {
				package: dependencyFactory(dependencies),
				pluginByTag: (tag, props = {}) => {
					const { defaultPlugin: defaultPlugin$1, errorMessage } = props;
					for (const userPlugin$1 of userPlugins) {
						const defaultConfig$24 = defaultPluginConfigs[userPlugin$1] || userPluginsConfig[userPlugin$1];
						if (defaultConfig$24 && defaultConfig$24.tags?.includes(tag) && userPlugin$1 !== name) return userPlugin$1;
					}
					if (defaultPlugin$1) {
						const defaultConfig$24 = defaultPluginConfigs[defaultPlugin$1] || userPluginsConfig[defaultPlugin$1];
						if (defaultConfig$24 && defaultConfig$24.tags?.includes(tag) && defaultPlugin$1 !== name) return defaultPlugin$1;
					}
					throw new Error(errorMessage || `missing plugin - no plugin with tag "${tag}" found`);
				},
				valueToObject
			};
			plugin.resolveConfig(plugin, context);
		}
		for (const dependency of plugin.dependencies) dfs(dependency);
		circularReferenceTracker.delete(name);
		pluginOrder.add(name);
		plugins[name] = plugin;
	};
	for (const name of userPlugins) dfs(name);
	return {
		pluginOrder: Array.from(pluginOrder),
		plugins
	};
}
function isPluginClient(plugin) {
	if (typeof plugin === "string") return plugin.startsWith("@hey-api/client");
	return plugin.name.startsWith("@hey-api/client") || plugin.tags && plugin.tags.includes("client");
}
function getPlugins({ dependencies, userConfig }) {
	const userPluginsConfig = {};
	let definedPlugins = defaultPlugins;
	if (userConfig.plugins) {
		userConfig.plugins = userConfig.plugins.filter((plugin) => typeof plugin === "string" && plugin || typeof plugin !== "string" && plugin.name);
		if (userConfig.plugins.length === 1 && isPluginClient(userConfig.plugins[0])) definedPlugins = [...defaultPlugins, ...userConfig.plugins];
		else definedPlugins = userConfig.plugins;
	}
	return getPluginsConfig({
		dependencies,
		userPlugins: definedPlugins.map((plugin) => {
			if (typeof plugin === "string") return plugin;
			const pluginName = plugin.name;
			if (pluginName) if (plugin.handler) userPluginsConfig[pluginName] = plugin;
			else {
				userPluginsConfig[pluginName] = { config: { ...plugin } };
				delete userPluginsConfig[pluginName].config.name;
			}
			return pluginName;
		}).filter(Boolean),
		userPluginsConfig
	});
}

//#endregion
//#region src/config/resolve.ts
function resolveConfig(validated, dependencies) {
	const logs = getLogs(validated.job.config.logs);
	const input = getInput(validated.job.config);
	const output = getOutput(validated.job.config);
	const parser = getParser(validated.job.config);
	output.path = path.resolve(process.cwd(), output.path);
	let plugins;
	try {
		plugins = getPlugins({
			dependencies,
			userConfig: validated.job.config
		});
	} catch (error) {
		if (error instanceof ConfigError) validated.errors.push(error);
		plugins = {
			pluginOrder: [],
			plugins: {}
		};
	}
	const config = {
		configFile: validated.job.config.configFile ?? "",
		dryRun: validated.job.config.dryRun ?? false,
		input,
		interactive: validated.job.config.interactive ?? detectInteractiveSession(),
		logs,
		output,
		parser,
		pluginOrder: plugins.pluginOrder,
		plugins: plugins.plugins
	};
	if (logs.level === "debug") {
		const jobPrefix = colors.gray(`[Job ${validated.job.index}] `);
		console.warn(`${jobPrefix}${colors.cyan("config:")}`, config);
	}
	return {
		config,
		errors: validated.errors,
		index: validated.job.index
	};
}

//#endregion
//#region src/config/validate.ts
function validateJobs(jobs) {
	return jobs.map((job) => {
		const errors = [];
		const { config } = job;
		if (!getInput(config).length) errors.push(new ConfigError("missing input - which OpenAPI specification should we use to generate your output?"));
		if (!getOutput(config).path) errors.push(new ConfigError("missing output - where should we generate your output?"));
		return {
			errors,
			job
		};
	});
}

//#endregion
//#region src/config/init.ts
/**
* @internal
*/
async function resolveJobs({ logger, userConfigs }) {
	const configs = [];
	let dependencies = {};
	const eventLoad = logger.timeEvent("load");
	for (const userConfig of userConfigs) {
		let configFile;
		if (userConfig.configFile) {
			const parts = userConfig.configFile.split(".");
			configFile = parts.slice(0, parts.length - 1).join(".");
		}
		const loaded = await loadConfigFile({
			configFile,
			logger,
			name: "openapi-ts",
			userConfig
		});
		if (!Object.keys(dependencies).length) dependencies = getProjectDependencies(loaded.foundConfig ? loaded.configFile : void 0);
		configs.push(...loaded.configs);
	}
	eventLoad.timeEnd();
	const eventBuild = logger.timeEvent("build");
	const resolvedJobs = validateJobs(expandToJobs(configs)).map((validated) => resolveConfig(validated, dependencies));
	eventBuild.timeEnd();
	return {
		dependencies,
		jobs: resolvedJobs,
		results: resolvedJobs
	};
}

//#endregion
export { postProcessors as _, clientDefaultConfig as a, TypeScriptRenderer as c, reserved as d, keywords as f, getTypedConfig as g, getClientPlugin as h, generateClientBundle as i, TsDslContext as l, TsDsl as m, defaultPlugins as n, clientDefaultMeta as o, regexp as p, clientPluginHandler as r, $ as s, resolveJobs as t, ctx as u };
//# sourceMappingURL=init-BxMu9GWq.mjs.map