import { createClient, createConfig } from '@hey-api/client-fetch';
// @transform-path ../../../package.json
import pkg from "../../../package.json" with { type: 'json' };
import { HttpError } from '../errors/HttpError.js';
import { ApiError } from '../errors/ApiError.js';
import * as sdk from '../openapi-ts/sdk.gen.js';
import { MeasurementStatus, } from '../openapi-ts/index.js';
import { wait } from '../utils.js';
export class Globalping {
    constructor({ auth, userAgent, throwApiErrors, timeout } = {}) {
        this.userAgent = userAgent !== null && userAgent !== void 0 ? userAgent : `globalping-typescript/${pkg.version} (https://github.com/jsdelivr/globalping-typescript)`;
        this.throwApiErrors = throwApiErrors !== null && throwApiErrors !== void 0 ? throwApiErrors : false;
        this.timeout = timeout !== null && timeout !== void 0 ? timeout : 30000;
        this.auth = auth;
        this.client = createClient(createConfig({
            baseUrl: 'https://api.globalping.io',
        }));
    }
    /**
     * @see {@link sdk.createMeasurement} for the API docs
     */
    createMeasurement(measurement) {
        return this.transformResult(this.request(sdk.createMeasurement, { body: measurement }));
    }
    /**
     * @see {@link sdk.getMeasurement} for the API docs
     */
    async getMeasurement(id) {
        const result = await this.transformResult(this.request(sdk.getMeasurement, { path: { id } }));
        if (!result.ok) {
            return result;
        }
        return result;
    }
    /**
     * @see {@link sdk.getMeasurement} for the API docs
     */
    async awaitMeasurement(id) {
        const getMeasurement = (eTag) => {
            return this.request(sdk.getMeasurement, {
                path: { id },
                headers: eTag ? {
                    'If-None-Match': eTag,
                } : {},
            });
        };
        const start = Date.now();
        let internalResult = await getMeasurement();
        while (internalResult.data && internalResult.data.status === MeasurementStatus.IN_PROGRESS) {
            if (Date.now() - start > 60000) {
                throw new Error(`Timed out waiting for measurement ${id} to finish.`);
            }
            await wait(500);
            const newInternalResult = await getMeasurement(internalResult.response.headers.get('ETag'));
            if (newInternalResult.response.status !== 304) {
                internalResult = newInternalResult;
            }
        }
        const result = await this.transformResult(internalResult);
        if (!result.ok) {
            return result;
        }
        const data = result.data;
        Globalping.assertMeasurementFinished(data);
        return { ...result, data };
    }
    /**
     * @see {@link sdk.listProbes} for the API docs
     */
    async listProbes() {
        return this.transformResult(this.request(sdk.listProbes));
    }
    /**
     * @see {@link sdk.getLimits} for the API docs
     */
    async getLimits() {
        return this.transformResult(this.request(sdk.getLimits));
    }
    request(fn, options) {
        const optionsWithDefaults = {
            auth: this.auth,
            timeout: this.timeout,
            userAgent: this.userAgent,
            ...(options || {}),
        };
        return fn({
            client: this.client,
            fetch: (request) => {
                var _a;
                const headers = new Headers(request.headers);
                headers.set('User-Agent', optionsWithDefaults.userAgent);
                if (optionsWithDefaults.auth) {
                    headers.set('Authorization', `Bearer ${optionsWithDefaults.auth}`);
                }
                return fetch(new Request(request, {
                    headers,
                    signal: (_a = optionsWithDefaults.signal) !== null && _a !== void 0 ? _a : AbortSignal.timeout(optionsWithDefaults.timeout),
                }));
            },
            ...optionsWithDefaults,
        });
    }
    async transformResult(requestResult) {
        const { data, error, request, response } = await requestResult;
        if (error != null) {
            if (typeof error !== 'object' || !('error' in error)) {
                throw new HttpError(request, response);
            }
            if (this.throwApiErrors) {
                throw new ApiError(request, response, error);
            }
            return { ok: false, data: error, request, response };
        }
        if (data == null) {
            throw new Error('Unexpected undefined data');
        }
        return { ok: true, data, request, response };
    }
    static assertHttpStatus(status, result) {
        if (!Globalping.isHttpStatus(status, result)) {
            throw new Error(`Expected HTTP status ${status}, got ${result.response.status}`);
        }
    }
    static assertMeasurementFinished(measurement) {
        if (!Globalping.isMeasurementFinished(measurement)) {
            throw new Error(`Expected measurement status ${MeasurementStatus.FINISHED}, got ${measurement.status}`);
        }
    }
    static assertMeasurementType(type, measurement) {
        if (!Globalping.isMeasurementType(type, measurement)) {
            throw new Error(`Expected measurement type ${type}, got ${measurement.type}`);
        }
    }
    static isHttpStatus(status, result) {
        return result.response.status === status;
    }
    static isMeasurementFinished(measurement) {
        return measurement.status === MeasurementStatus.FINISHED;
    }
    static isMeasurementType(type, measurement) {
        return measurement.type === type;
    }
}
export default Globalping;
//# sourceMappingURL=Globalping.js.map